/**************************************************************************
 * PAM module to interoperate with the SNUI authentication subsystem.
 *
 * Copyright (C) 2002, Matthew Palmer.  Released under the GPL version 2.
 * You should have received a copy of this licence with this software, if not
 * see http://www.fsf.org/copyleft/gpl.html for a full copy of the licence.
 */

static char rcsid[] = "$Id$";

#define PAM_SM_AUTH
#include <security/pam_modules.h>
#include <security/pam_misc.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>

#include "../lib/lib.h"

static int debugging_on;
#define DEBUG(x, args...){ \
				if (debugging_on) { \
					syslog(LOG_DEBUG, x , ## args); \
				}\
			}

#define PLEASE_ENTER_PASSWORD "Password: "

char hex_digit(char);
char *hash_password(char *);
int ask_for_password(pam_handle_t *);
int converse (pam_handle_t * pamh,
	      int nargs,
	      struct pam_message **message,
	      struct pam_response **response);

PAM_EXTERN int pam_sm_authenticate(pam_handle_t *pamh, int flags,
				int argc, const char **argv)
{
	char resp[1024];
	char *uid, *pass = NULL;
	int i;
	FILE *sock;
		
	for (i = 0; i < argc; i++) {
		if (!strcasecmp(argv[i], "debug")) {
			debugging_on = 1;
		}
	}

	DEBUG("Starting pam_snui::pam_sm_acct_mgmt...");

	i = pam_get_user(pamh, (const char **) &uid, "login: ");
	if (i != PAM_SUCCESS) {
		syslog(LOG_CRIT, "Failed to obtain UID");
		if (i == PAM_CONV_AGAIN) {
			DEBUG("returning PAM_INCOMPLETE");
			return PAM_INCOMPLETE;
		} else {
			DEBUG("returning PAM_AUTHINFO_UNAVAIL");
			return PAM_AUTHINFO_UNAVAIL;
		}
	}
	DEBUG("pam_snui: Username is [%s]", uid);
	
	i = pam_get_item(pamh, PAM_AUTHTOK, (const void **) &pass);
	if (i != PAM_SUCCESS) {
		syslog(LOG_ERR, "pam_get_item returned error to read password - aborting");
		DEBUG("Returning %i from pam_get_item", i);
		return i;
	}

	if (pass == NULL) {
		i = ask_for_password(pamh);
		if (i != PAM_SUCCESS) {
			DEBUG("Returning %i from ask_for_password", i);
			return i;
		}
	}
	i = pam_get_item(pamh, PAM_AUTHTOK, (const void **) &pass);
	if (i != PAM_SUCCESS) {
		syslog(LOG_ERR, "pam_get_item failed to get a password after asking for it - aborting");
		DEBUG("returning %i from pam_get_item", i);
		return i;
	}
	if (!pass) {
		syslog(LOG_ERR, "Failed to obtain a password - aborting");
		DEBUG("Returning PAM_AUTHINFO_UNAVAIL (no password)");
		return PAM_AUTHINFO_UNAVAIL;
	}
	
	pass = hash_password(pass);
	
	i = net_client_init("pam", &sock);
	if (i) {
		syslog(LOG_CRIT, "Network connection failed: %s - aborting", strerror(i));
		DEBUG("Returning PAM_AUTHINFO_UNAVAIL (network cactus)");
		return PAM_AUTHINFO_UNAVAIL;
	}

	fprintf(sock, "auth %s %s\n", uid, pass);
	net_read_response(sock, &i, resp, 1024);
	if (i == 250) {
		DEBUG("auth OK [%s]", resp);
		fclose(sock);
		DEBUG("Returning PAM_SUCCESS (yay!)");
		return PAM_SUCCESS;
	} else if (i == 450) {
		DEBUG("auth failed [%s]", resp);
	} else {
		DEBUG("unknown response %i %s", i, resp);
	}

	fclose(sock);	
	return PAM_AUTH_ERR;
}

PAM_EXTERN int pam_sm_setcred(pam_handle_t *pamh, int flags,
				int argc, const char **argv)
{
	return PAM_SUCCESS;
}

char *hash_password(char *pass)
{
	struct MD5Context ct;
	unsigned char hash[16];
	char *real_hash;
	int i;
		
	MD5Init(&ct);
	
	MD5Update(&ct, pass, strlen(pass));
	MD5Final(hash, &ct);
	
	real_hash = malloc(sizeof(char) * 33);
	if (!real_hash) {
		return NULL;
	}
	/* Now we convert the 16 octet hash to a 32 byte hex string */
	for (i = 0; i < 16; i++) {
		real_hash[2*i+1] = hash[i] & 0xF;
		real_hash[2*i] = (hash[i] & 0xF0) >> 4;
	}
	for (i = 0; i < 32; i++) {
		real_hash[i] = hex_digit(real_hash[i]);
	}
	real_hash[32] = '\0';

	return real_hash;
}

char hex_digit(char c)
{
	if (c < 10) {
		return c+'0';
	} else {
		return c-10+'a';
	}
}

/* Gratuitously nicked from libpam-mysql.  Under the GPL according to the
 * debian package docs.
 */
int ask_for_password(pam_handle_t *pamh)
{
	struct pam_message msg[1], *mesg[1];
	struct pam_response *resp=NULL;
	char *prompt=NULL;
	int i=0;
	int retval;

	prompt = malloc(strlen(PLEASE_ENTER_PASSWORD));
	if (prompt == NULL) 
	{
		syslog(LOG_ERR,"pam_snui: askForPassword(), out of memory!?");
		return PAM_BUF_ERR;
	} 
	else 
	{
		sprintf(prompt, PLEASE_ENTER_PASSWORD);
		msg[i].msg = prompt;
	}
	msg[i].msg_style = PAM_PROMPT_ECHO_OFF;
	mesg[i] = &msg[i];

	retval = converse(pamh, ++i, mesg, &resp);
	if (prompt) 
	{
	    _pam_overwrite(prompt);
	    _pam_drop(prompt);
	}
	if (retval != PAM_SUCCESS) 
	{
	    if (resp != NULL)
		_pam_drop_reply(resp,i);
	    return ((retval == PAM_CONV_AGAIN)
		    ? PAM_INCOMPLETE:PAM_AUTHINFO_UNAVAIL);
	}

	/* we have a password so set AUTHTOK
	 */
	return pam_set_item(pamh, PAM_AUTHTOK, resp->resp);
}

/* Global PAM functions stolen from other modules (libpam_mysql in this case) */


int converse(pam_handle_t *pamh, int nargs
		    , struct pam_message **message
		    , struct pam_response **response)
{
    int retval;
    struct pam_conv *conv;

    retval = pam_get_item( pamh, PAM_CONV, (const void **) &conv ) ; 
    if ( retval == PAM_SUCCESS ) 
    {
	retval = conv->conv(nargs, ( const struct pam_message ** ) message
			    , response, conv->appdata_ptr);
	if ((retval != PAM_SUCCESS) && (retval != PAM_CONV_AGAIN)) 
	{
	DEBUG("pam_snui: conversation failure [%s]"
	     , pam_strerror(pamh, retval));
	}
    } 
    else 
    {
	syslog(LOG_ERR, "pam_snui: couldn't obtain coversation function [%s]"
		 , pam_strerror(pamh, retval));
    }
    return retval;                  /* propagate error status */
}
