#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <ctype.h>
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h>

#include "../lib/lib.h"

struct thread_data {
	conf_file_t *cfile;
	FILE *conn_fd;
};

void do_command(struct thread_data *, char *searchtype, char *type, char *key);
int auth(char *, char *username, char *password);
int parse_request(char *, char **, char **, char **);
int compare(char *, int, char *);
void *do_server_thing(void *);
void strip_newline(char *);
void send_list(struct thread_data *, FILE *);
void search(struct thread_data *, FILE *, int, char *);

int main()
{
	char *port;
	conf_file_t *cfile;
	unsigned short bind_port;
	int sock, rv;
	pthread_t thread_id;

	openlog("snui_server", LOG_PID, LOG_DAEMON);

	cfile = conf_read_file(CONFIG_FILE);
	if (!cfile) {
		syslog(LOG_CRIT, "Unable to open config file %s: %m", CONFIG_FILE);
		return 0;
	}
	
	if (!conf_find_section(cfile, "server")) {
		syslog(LOG_ERR, "No server config section found");
		return 0;
	}
	
	port = conf_find_item(cfile, "bind_port");
	if (!port) {
		syslog(LOG_ERR, "No port to bind to given - aborting");
		return 0;
	}
	
	bind_port = atoi(port);

	rv = net_server_init(bind_port, &sock);
	if (rv) {
		syslog(LOG_ERR, "Error while setting up network connection: %s", strerror(rv));
		return 0;
	}

	while (1) {
		int s;
		struct thread_data *td;

		rv = net_get_connection(sock, bind_port, &s);
		if (rv) {
			syslog(LOG_ERR, "Error while waiting for connect()");
			continue;
		}
		
		td = malloc(sizeof(struct thread_data));
		if (!td) {
			syslog(LOG_ERR, "Out of memory error");
			close(s);
			continue;
		}
		td->cfile = cfile;
		td->conn_fd = fdopen(s, "a+");
		
		if (pthread_create(&thread_id, NULL, do_server_thing, (void *)td) != 0) {
			syslog(LOG_CRIT, "Failed to start connection thread");
		}
		/* I don't care what it does now */
		pthread_detach(thread_id);
	}

	return 0;
}

void *do_server_thing(void *d)
{
	struct thread_data *td;
	char buf[1024];
	char *db, *type, *key;
		
	td = (struct thread_data *) d;

	while (1) {
		fgets(buf, 1024, td->conn_fd);
		strip_newline(buf);
		if (feof(td->conn_fd)) {
			break;
		}
		if (!parse_request(buf, &db, &type, &key)) {
			syslog(LOG_ERR, "Command parse error");
			syslog(LOG_DEBUG, "Command line was [%s]", buf);
		}
	
		if (!strcasecmp(db, "quit")) {
			break;
		}
		do_command(td, db, type, key);
	}
	
	fclose(td->conn_fd);
	free(td);
	return NULL;
}

void do_command(struct thread_data *td, char *dbtype, char *type, char *key)
{
	char *dbfile;
	FILE *dbfd;
	int rv;

	if (!strcasecmp(dbtype, "passwd")) {
		dbfile = conf_find_item(td->cfile, "user_db");
	} else if (!strcasecmp(dbtype, "group")) {
		dbfile = conf_find_item(td->cfile, "group_db");
	} else if (!strcasecmp(dbtype, "auth")) {
		dbfile = conf_find_item(td->cfile, "auth_db");
		if (dbfile) {
			rv = auth(dbfile, type, key);
			if (rv < 0) {
				fprintf(td->conn_fd, "500 Error\n");
			} else if (rv > 0) {
				fprintf(td->conn_fd, "250 Auth OK\n");
			} else {
				fprintf(td->conn_fd, "450 Auth failed\n");
			}
		} else {
			syslog(LOG_ERR, "NULL auth database filename");
		}
		return;
	} else {
		fprintf(td->conn_fd, "501 Unknown command\n");
		syslog(LOG_DEBUG, "Unknown command sent");
		return;
	}

	if (!dbfile) {
		fprintf(td->conn_fd, "500 Server error\n");
		syslog(LOG_ERR, "NULL database filename");
		return;
	}

	dbfd = fopen(dbfile, "r");
	if (!dbfd) {
		syslog(LOG_ERR, "Failed to open database %s: %m", dbfile);
		fprintf(td->conn_fd, "500 Server error\n");
		return;
	}

	if (!strcmp(type, "list")) {
		send_list(td, dbfd);
	} else if (!strcmp(type, "id")) {
		syslog(LOG_DEBUG, "Looking by ID for [%s]...", key);
		search(td, dbfd, 2, key);
	} else if (!strcmp(type, "name")) {
		syslog(LOG_DEBUG, "Looking by name for [%s]...", key);
		search(td, dbfd, 0, key);
	} else {
		syslog(LOG_INFO, "Unknown search type [%s]", type);
		fprintf(td->conn_fd, "501 Unknown search type\n");
	}

	fclose(dbfd);
}

void send_list(struct thread_data *td, FILE *dbfd)
{
	char buf[1024];
	
	/* Well, at least *that* one's easy */
	while (1) {
		fgets(buf, 1024, dbfd);
		strip_newline(buf);
		if (feof(dbfd)) {
			fprintf(td->conn_fd, "\n");
			return;
		}
		/* Skip blank lines */
		if (strlen(buf)) {
			fprintf(td->conn_fd, "%s\n", buf);
		}
	}
}

/* Searches the given database FD for the given key in the nth field (numbered from zero, of course).  Prints 400 not found if we don't find it, or 200 <data line> if found.
 */
void search(struct thread_data *td, FILE *dbfd, int n, char *key)
{
	char buf[1024];
	
	while (1) {
		fgets(buf, 1024, dbfd);
		strip_newline(buf);
		if (feof(dbfd)) {
			fprintf(td->conn_fd, "400 Not found\n");
			return;
		}
		if (compare(buf, n, key)) {
			fprintf(td->conn_fd, "200 %s\n", buf);
			return;
		}
	}
}

/* Return 1 if the value in the fieldno'th field (counting the first as 0)
 * is the same as the one given, assuming that data is a colon-separated
 * line of fields.  0 if they're different.
 */
int compare(char *data, int fieldno, char *value)
{
	char *p, *q;
	int i;
	
	p = data;
	
	for (i = 0; i < fieldno; i++) {
		p = index(p, ':');
		if (!*p) {
			/* Well, it can't be nothing... */
			return 0;
		}
		p++;
	}
	q = index(p, ':');
	if (!q) {
		/* Reached end of line */
		q = p+strlen(p);
	}
	/* Different length strings can't be the same */
	if (strlen(value) != q-p) {
		return 0;
	}
	return !strncmp(p, value, q-p);
}
	
/* Check authentication information.  Return 1 if OK, 0 if failed, or -1 on
 * error.
 */
int auth(char *dbfile, char *username, char *password)
{
	FILE *dbfd;
	char buf[1024];
	
	if((username == NULL) || (password == NULL))
	{
		syslog(LOG_ERR, "username and/or password passed as NULL");
		return -1;
	}
	
	dbfd = fopen(dbfile, "r");
	if (!dbfd) {
		syslog(LOG_CRIT, "Unable to open auth DB %s: %m", dbfile);
		return -1;
	}
	
	while(1) {
		fgets(buf, 1024, dbfd);
		strip_newline(buf);
		if (feof(dbfd)) {
			fclose(dbfd);
			return 0;
		}
		if (compare(buf, 0, username) && compare(buf,1,password)) {
			fclose(dbfd);
			return 1;
		}
	}

	syslog(LOG_ERR, "can't happen - dropped out of the auth read loop");
	return -1;
}

/* Chop a command string into bits.  Command comes in of the form
 * <db> <type> [arg]
 * and we put it into the bits below.  Whitespace separated.
 * If arg is not present, arg == NULL.
 * Returns 1 if the parse went off OK, or 0 otherwise.
 */
int parse_request(char *buf, char **db, char **type, char **arg)
{
	char *p;
	
	p = buf;
	
	/* Strip any leading whitespace */
	while (isspace(*p) && *p) {
		p++;
	}
	if (!(*p)) {
		return 0;
	}
	
	*db = p;
	
	/* Skip db */
	while (!isspace(*p) && *p) {
		p++;
	}
	
	if (!(*p)) {
		return 0;
	}
	/* Make whitespace string terminator */
	while (isspace(*p) && *p) {
		*p = '\0';
		p++;
	}
	if (!(*p)) {
		return 0;
	}

	/* Type is after whitespace */
	*type = p;
	*arg = NULL;
	
	while (!isspace(*p) && *p) {
		p++;
	}
	/* It's OK if the parser stops here */
	if (!(*p)) {
		return 1;
	}

	while (isspace(*p) && *p) {
		*p = '\0';
		p++;
	}
	if (!(*p)) {
		return 1;
	}
	*arg = p;
	return 1;
}

void strip_newline(char *buf)
{
	while (buf[strlen(buf)-1] == '\n' || buf[strlen(buf)-1] == '\r') {
		buf[strlen(buf)-1] = '\0';
	}
}
