/*
 * --------------------------------------------------------------------------
 * Interface between Serial chip emulator and real terminal
 * 
 * (C) 2006 Jochen Karrer 
 *
 * State: not implemented
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * --------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include <serial.h>
#include <fio.h>
#include <configfile.h>

typedef struct FileUart {
	int fd;
        FIO_FileHandler input_fh;
        FIO_FileHandler output_fh;
        int ifh_is_active;
        int ofh_is_active;
} FileUart;

typedef struct NullUart {
	int tx_enabled;
} NullUart;

/*
 * ------------------------------------------------------
 * File descriptor based serial emulator backend 
 * ------------------------------------------------------
 */

static int
file_input(void *cd,int mask) {
	UartPort *port = (UartPort*) cd;
	if(port->sinkProc) {
		port->sinkProc(port->owner);
	}
	return 0;
}

static int
file_output(void *cd,int mask) {
	UartPort *port = (UartPort*) cd;
	if(port->srcProc) {
		port->srcProc(port->owner);
	}
	return 0;
}

static void
file_enable_rx(UartPort *port) {
	FileUart *fuart = port->backend;
        if((fuart->fd >= 0) && !(fuart->ifh_is_active)) {
                FIO_AddFileHandler(&fuart->input_fh,fuart->fd,FIO_READABLE,file_input,port);
        }
        fuart->ifh_is_active=1;
}

static void
file_disable_rx(UartPort *port) {
	FileUart *fuart = port->backend;
        if(fuart->ifh_is_active) {
                FIO_RemoveFileHandler(&fuart->input_fh);
        }
        fuart->ifh_is_active=0;
}
static void
file_enable_tx(UartPort *port) {
	FileUart *fuart = port->backend;
        if((fuart->fd>=0) && !(fuart->ofh_is_active)) {
                FIO_AddFileHandler(&fuart->output_fh,fuart->fd,FIO_WRITABLE,file_output,port);
        }
        fuart->ofh_is_active=1;
}
static void
file_disable_tx(UartPort *port) {
	FileUart *fuart = port->backend;
        if(fuart->ofh_is_active) {
                FIO_RemoveFileHandler(&fuart->output_fh);
        }
        fuart->ofh_is_active=0;
}

static void
file_close(UartPort *port) {
	FileUart *fuart = port->backend;	
	file_disable_tx(port);
	file_disable_rx(port);
	close(fuart->fd);
	fuart->fd=-1;
}

static int 
file_read(UartPort *port,uint8_t *buf,int maxlen) 
{
	FileUart *fuart = port->backend;	
	int count;
	if(fuart->fd<0) {
		return 0;
	}
	count = read(fuart->fd,buf,maxlen);
	if(count>0)
		return count;
	if((count<=0) && (errno != EAGAIN)) {
		fprintf(stderr,"Read error\n");
		file_close(port);
		return -1;
	}
	return 0;
}

static int 
file_write(UartPort *port,const uint8_t *buf,int len) 
{
	FileUart *fuart = port->backend;	
	int count;
	if(fuart->fd<0) {
		return 0;
	}
	count = write(fuart->fd,buf,len);
	if(count>0)
		return count;
	if((count<=0) && (errno != EAGAIN)) {
		fprintf(stderr,"Write error\n");
		file_close(port);
		return -1;
	}
	return 0;
}


static UartOps file_uart = {
//	void            (*set_mctrl)(struct UartPort *, uint32_t mctrl);
//      uint32_t        (*get_mctrl)(struct UartPort *);
	.stop_tx = file_disable_tx,
	.start_tx = file_enable_tx,
	.stop_rx = file_disable_rx,
	.start_rx = file_enable_rx,
 //       void            (*break_ctl)(UartPort *, int ctl);
 //       void            (*set_termios)(UartPort *, const struct termios *new);
 //       void            (*get_termios)(UartPort *, struct termios *new);
	.write = file_write,
	.read = file_read,
};


static void
TerminalConfigure(int fd) {
        struct termios term;
        if(tcgetattr(fd,&term)<0) {
                perror("can't get terminal settings\n");
                return;
        }
	/* should be raw for non stdio devices */
//	cfmakeraw(&term);
        term.c_lflag  &= ~(ECHO|ECHONL|ICANON|IEXTEN|ISIG);
        if(tcsetattr(fd,TCSAFLUSH,&term)<0) {
                perror("can't set terminal settings");
                return;
        }
}

static void
TerminalRestore(int fd) {
        struct termios term;
        if(tcgetattr(fd,&term)<0) {
                perror("can't restore terminal settings\n");
                return;
        }
        term.c_lflag |= (ECHO|ECHONL|ICANON|ISIG|IEXTEN);
        if(tcsetattr(fd,TCSAFLUSH,&term)<0) {
                perror("can't restore terminal settings");
                return;
        }
}

static void
TerminalExit(void) {
        TerminalRestore(0);
}

FileUart *
FileUart_New(const char *uart_name,const char *filename)  {
	FileUart *fiua = malloc(sizeof(FileUart));
	if(!fiua) {
		fprintf(stderr,"Out of memory\n");
	}
	memset(fiua,0,sizeof(FileUart));
	if(!strcmp(filename,"stdin")) {
		fiua->fd = 0;
	} else {
		fiua->fd = open(filename,O_RDWR);
	}
	if(fiua->fd<0) {
		fprintf(stderr,"%s: Cannot open %s\n",uart_name,filename);
		sleep(1);
		return fiua;
	} else {
		fcntl(fiua->fd,F_SETFL,O_NONBLOCK);
		fprintf(stderr,"Uart \"%s\" Connected to %s\n",uart_name,filename);
	}
	TerminalConfigure(fiua->fd);	
	atexit(TerminalExit);
	return fiua;
}

static void
null_enable_tx(UartPort *port) {
	NullUart *nua = port->backend;
	nua->tx_enabled = 1;
	while(nua->tx_enabled && port->srcProc) {
		port->srcProc(port->owner);
	}
}

static void
null_disable_tx(UartPort *port) {
	NullUart *nua = port->backend;
	nua->tx_enabled = 0;
}

static NullUart * 
NullUart_New(const char *uart_name,const char *filename)  {
	NullUart *nua = malloc(sizeof(NullUart));
	if(!nua) {
		fprintf(stderr,"Out of memory\n");
		exit(1);
	}
	nua->tx_enabled = 0;
	return nua;
}

static UartOps null_uart =  {
	.stop_tx = null_disable_tx,
	.start_tx = null_enable_tx,
	.stop_rx = NULL,
	.start_rx = NULL,
};

UartPort *
Uart_New(const char *uart_name,UartSinkProc *sinkproc,UartSrcProc *srcproc,UartStatChgProc *statproc,void *owner) 
{
	const char *filename=Config_ReadVar(uart_name,"file");
	UartPort *port = (UartPort*) malloc(sizeof(UartPort));
	port->owner = owner;
	port->sinkProc = sinkproc;
        port->srcProc = srcproc;
        port->statProc = statproc;

        if(filename) {
		port->ops = &file_uart;
		port->backend = FileUart_New(uart_name,filename);
        } else {
		port->ops = &null_uart;
		port->backend = NullUart_New(uart_name,filename);
                fprintf(stderr,"%s connected to nowhere\n",uart_name);
        }
	return port;		
}
