/*
 * -----------------------------------------------------------------------
 * Emulation of Freescale Multi layer AHB Crossbar Switch (MAX) 
 *
 * (C) 2006 Jochen Karrer
 *   Author: Jochen Karrer
 *
 * state: simply stores the register values. Has no effect 
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * -----------------------------------------------------------------------
 */

#include <errno.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include "bus.h"
#include "imx21_max.h"
#include "sgstring.h"

#define MAX_MPR(base,n)		((base)+(n)*0x100+0x000)
#define MAX_AMPR(base,n)	((base)+(n)*0x100+0x004)
#define MAX_SGPCR(base,n)	((base)+(n)*0x100+0x010)
#define		SGPCR_RO	(1<<31)
#define		SGPCR_HLP	(1<<30)
#define MAX_ASGPCR(base,n)	((base)+(n)*0x100+0x014)
#define MAX_MGPCR(base,n)	((base)+(n)*0x100+0x800)


typedef struct IMX21Max {
	uint32_t mpr[4];
	uint32_t ampr[4];
	uint32_t sgpcr[4];
	uint32_t asgpcr[4];
	uint32_t mgpcr[6];
	BusDevice bdev;
} IMX21Max;

static uint32_t
mpr_read(void *clientData,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 3;
	return max->mpr[index];
}

static void
mpr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 3;
	/* Check if it is in readonly mode first */
	if(max->sgpcr[index] & SGPCR_RO) {
		return;
	}
	max->mpr[index] = value & 0x00777777;
        return;
}
static uint32_t
ampr_read(void *clientData,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 3;
	return max->ampr[index];
}

static void
ampr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 3;
	/* Check if it is in readonly mode first */
	if(max->sgpcr[index] & SGPCR_RO) {
		return;
	}
	max->ampr[index] = value & 0x00777777;
        return;
}

static uint32_t
sgpcr_read(void *clientData,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 3;
	return max->sgpcr[index];
}

static void
sgpcr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 3;
	/* Check if it is in readonly mode first */
	if(max->sgpcr[index] & SGPCR_RO) {
		return;
	}
	max->sgpcr[index] = value & 0xc0000337;
        return;
}

static uint32_t
asgpcr_read(void *clientData,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 3;
	return max->asgpcr[index];
}

static void
asgpcr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 3;
	/* Check if it is in readonly mode first */
	if(max->sgpcr[index] & SGPCR_RO) {
		return;
	}
	max->asgpcr[index] = value & 0x40000337;
        return;
}

static uint32_t
mgpcr_read(void *clientData,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 7;
	return max->mgpcr[index];
}

static void
mgpcr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMX21Max *max = (IMX21Max*) clientData;
	int index = ((address) >> 8) & 7;
        max->mgpcr[index] = value;
	return;
}


static void
IMXMax_Unmap(void *owner,uint32_t base,uint32_t mask)
{
	int i;
	for(i=0;i<4;i++) {
		IOH_Delete32(MAX_MPR(base,i));
		IOH_Delete32(MAX_AMPR(base,i));
		IOH_Delete32(MAX_SGPCR(base,i));
		IOH_Delete32(MAX_ASGPCR(base,i));
	}
	for(i=0;i<6;i++) {
		IOH_Delete32(MAX_MGPCR(base,i));
	}
}

static void
IMXMax_Map(void *owner,uint32_t base,uint32_t mask,uint32_t mapflags)
{
	int i;
        IMX21Max *max = (IMX21Max*) owner;
	for(i=0;i<4;i++) {
		IOH_New32(MAX_MPR(base,i),mpr_read,mpr_write,max);
		IOH_New32(MAX_AMPR(base,i),ampr_read,ampr_write,max);
		IOH_New32(MAX_SGPCR(base,i),sgpcr_read,sgpcr_write,max);
		IOH_New32(MAX_ASGPCR(base,i),asgpcr_read,asgpcr_write,max);
	}
	for(i=0;i<6;i++) {
		IOH_New32(MAX_MGPCR(base,i),mgpcr_read,mgpcr_write,max);
	}
}

BusDevice *
IMX21_MaxNew(const char *name)
{
	IMX21Max *max = sg_new(IMX21Max);
	int i;
	max->bdev.first_mapping=NULL;
        max->bdev.Map=IMXMax_Map;
        max->bdev.UnMap=IMXMax_Unmap;
        max->bdev.owner=max;
        max->bdev.hw_flags=MEM_FLAG_WRITABLE|MEM_FLAG_READABLE;
	for(i=0;i<4;i++) {
		max->mpr[i] = 0x76543210;
		max->ampr[i] = 0x76543210;
		max->sgpcr[i] = 0;
		max->asgpcr[i] = 0;
	}
	for(i=0;i<6;i++) {
		max->mgpcr[i] = 0;
	}
	return &max->bdev;
}
