/* Somaplayer - Copyright (C) 2003-4 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "buffer.h"
#include "other.h"
#include "file.h"
#include "ftime.h"

#ifdef ENABLE_GTK
#include "gtk/statusbar.h"
#endif

#define PUSH	0
#define POP	1

/* Buffer initialize */
void
buffer_init (void)
{
  int len;
  char *rptr;

  if (!play->realbuf)
    if (!(play->realbuf = malloc (play->realsize)))
      fatal (_("Error: memory."));

  play->start = -1;
  play->stop = 0;

  fprintf (stderr, "\n");

  rptr=(char *)play->realbuf;

  while (play->stop < play->realsize)
    {
      len = my_read (rptr + play->stop, play->realsize - play->stop);

      fprintf (stderr, _("* Buffering: %d kbyte %3d%%\r"),
	       play->realsize / 1024, buffer_perc_size ());

#ifdef ENABLE_GTK
      if (play->graphic)
	msg_statusbar (_("Buffering: %d kbyte %3d%%"),
		       play->realsize / 1024, buffer_perc_size ());
#endif

      if (len < 1)
	{
	  play->done = ENDOFFILE;
	  break;
	}

      play->stop += len;
      play->start = 0;
    }

  if (play->stop == play->realsize)
    play->stop = 0;

  fprintf (stderr, _("* Buffering: %d kbyte %3d%%\n"), play->realsize / 1024,
	   buffer_perc_size ());

#ifdef ENABLE_GTK
  if (play->graphic)
    msg_statusbar (_("Buffering: %d kbyte %3d%%"), play->realsize / 1024,
		   buffer_perc_size ());
#endif

}

/* This is a thread that get data from the input */
void *
buffer_push (void *data)
{
  int len, ret;
  unsigned char *mem;
  char *rptr=(char *)play->realbuf;

  while (!events.quit && !events.skip)
    {

      /* Buffer_error does stat about empty of buffer. 
       * buffer_error(0) -> it asks if it realy import get all data
       *                    that the buffer needs.
       * buffer_error(1) -> (see buffer_pop) send an error to buffer_error.
       *
       * In a normal situation buffer_push inserts data inside of buffer if
       * it is empty for 20%.
       * If buffer_error recives an error, it tells to buffer push:
       * GET EVERY THING ALWAYS!
       * After X second, return to a normal situation.
       *
       * After 3 error, buffer_push gets alway all data.
       */
      if (buffer_error (0) && playing ())
	{
	  len = play->realsize - REALSIZE;

	  if (!len)
	    {
	      pthread_mutex_lock (&play->mutex);
	      pthread_cond_wait (&play->p_push, &play->mutex);
	      pthread_mutex_unlock (&play->mutex);

	      continue;
	    }
	}
      else
	{
	  if (!(len = buffer_check (PUSH)))
	    {
	      pthread_mutex_lock (&play->mutex);
	      pthread_cond_wait (&play->p_push, &play->mutex);
	      pthread_mutex_unlock (&play->mutex);

	      continue;
	    }
	}

      if (!(mem = (unsigned char *) malloc (sizeof (unsigned char) * len)))
	fatal (_("Error: memory."));

      if ((ret = my_read (mem, len)) < 1)
	{
	  play->done = ENDOFFILE;
	  break;
	}
      len = ret;

      if (play->done == ENDOFFILE)
	break;

      if (events.quit || events.skip)
	break;

      pthread_mutex_lock (&play->mutex);

      if (play->start > play->stop || (play->start < 0 && !play->stop))
	{

	  memcpy (rptr + play->stop, mem, len);
	  play->stop += len;

	  if (play->stop == play->realsize)
	    play->stop = 0;

	}
      else
	{

	  if (play->stop < play->realsize)
	    {
	      ret = play->realsize - play->stop;

	      if (ret < len)
		{
		  memcpy (rptr + play->stop, mem, ret);
		  len -= ret;
		  play->stop += ret;
		}
	      else
		{
		  memcpy (rptr + play->stop, mem, len);
		  play->stop += len;
		  len = 0;
		}

	      if (play->stop == play->realsize)
		play->stop = 0;
	    }

	  if (len && play->stop == 0 && play->start > 0)
	    {
	      memcpy (play->realbuf, mem + ret, len);
	      play->stop = len;
	      if (play->stop == play->realsize)
		play->stop = 0;
	    }
	}

      if (play->start == -1)
	play->start = 0;

      pthread_mutex_unlock (&play->mutex);

      free (mem);
    }

  pthread_exit (NULL);
}

/* This function pops data from the buffer. 
 * It is like the read(void *, size_t) with no
 * file descriptor. */
int
buffer_pop (void *a, size_t size)
{
  int len = 0;
  char *rptr=(char *)play->realbuf;
  char *dest=(char *)a;

  while (events.pause)
    {
      pthread_mutex_lock (&play->mutex);
      pthread_cond_wait (&play->p_pop, &play->mutex);
      pthread_mutex_unlock (&play->mutex);

      if (events.quit || events.skip)
	return 0;
    }

  if (events.skip || events.quit)
    return 0;

  if (play->nobuffer)
    {
      len = my_read (a, size);

      if (len < 0)
	return 0;

      return len;
    }

  if (size > play->realsize)
    size = play->realsize;

  if (!REALSIZE && play->done == ENDOFFILE)
    return 0;

  pthread_mutex_lock (&play->mutex);

  while (REALSIZE < size && play->done != ENDOFFILE)
    {
      pthread_mutex_unlock (&play->mutex);
      pthread_cond_signal (&play->p_push);

      if (events.quit || events.skip)
	return 0;

      /* ERROR: the buffer is empty!!! */
      buffer_error (1);

      pthread_mutex_lock (&play->mutex);

      sched_yield ();
    }

  /* If the input data is EOF */
  if (play->done == ENDOFFILE && size > REALSIZE)
    {
      if (events.quit || events.skip)
	{
	  pthread_mutex_unlock (&play->mutex);

	  return 0;
	}

      /* Copy al data */

      /* if a == NULL -> only move in the buffer */
      if (dest && play->start < play->stop)
	memcpy (dest, rptr + play->start, play->stop - play->start);

      else
	{
	  if (dest)
	    {
	      if (play->start < play->realsize)
		memcpy (dest, rptr + play->start, play->realsize - play->start);

	      if (play->stop != 0)
		memcpy (dest + (play->realsize - play->start), rptr, play->stop);
	    }
	}

      len = REALSIZE;

      play->start = -1;
      play->stop = 0;

      pthread_mutex_unlock (&play->mutex);
    }

  /* If there is buffer... */
  else
    {

      if (events.quit || events.skip)
	{
	  pthread_mutex_unlock (&play->mutex);

	  return 0;
	}

      len = size;

      if (play->start < play->stop || (play->realsize - play->start) >= size)
	{
		
	  if (dest)
	    memcpy (dest, rptr + play->start, size);

	  play->start += size;

	  if (play->start == play->realsize)
	    {
	      if (!play->stop)
		play->start = -1;
	      else
		play->start = 0;
	    }

	  if (play->start == play->stop)
	    {
	      play->start = -1;
	      play->stop = 0;
	    }
	}
      else
	{

	  if (play->start < play->realsize)
	    {
	      if (dest)
		memcpy (dest, rptr + play->start, play->realsize - play->start);

	      size -= play->realsize - play->start;
	    }

	  if (size > 0)
	    {
	      if (dest)
		memcpy (dest + (play->realsize - play->start), rptr, size);
	      play->start = size;
	    }

	  if (play->start == play->stop)
	    {
	      play->start = -1;
	      play->stop = 0;
	    }
	}

      pthread_mutex_unlock (&play->mutex);
    }

  if (!play->remote)
    buffer_check (POP);

  else
    pthread_cond_signal (&play->p_push);

  return len;
}

/*  Return the % of data */
int
buffer_perc_size (void)
{
  if (play->nobuffer)
    return 100;

  return REALSIZE * 100 / play->realsize;
}

/* Parse the flag of cli and return the realy dimension in bytes */
int
buffer_size (char *str)
{
  register int i;
  int part = 0;
  int k = 0;
  int m = 0;
  int b = 0;

  for (i = 0; i < strlen (str); i++)
    {
      if (str[i] >= '0' && str[i] <= '9')
	part = part * 10 + str[i] - '0';

      else if (str[i] == 'k' || str[i] == 'K')
	{
	  k += part;
	  part = 0;
	}

      else if (str[i] == 'm' || str[i] == 'M')
	{
	  m += part;
	  part = 0;
	}

      else
	fatal (_("Error in byte string %s"), str);

    }

  b += part;
  b += k * 1024;
  b += m * 1024 * 1024;

  return b;
}

/* Check the buffer. If it good idea push or pop data: */
int
buffer_check (int who)
{
  int len;

  len = play->realsize - REALSIZE;

  if (who == POP)
    {
      if (len > SIZE_BUFFER)
	pthread_cond_signal (&play->p_push);
    }

  else if (len < SIZE_BUFFER)
    return 0;

  return len;
}

/* Check the error message */
int
buffer_error (int error)
{
  static int err = 0;
  static struct timeval *t = NULL;
  static int k = 0;

  if (error)
    {

      if (err)
	k++;
      else
	err = 1;

      if (t)
	timer_clear (t);

      t = timer_start ();

    }
  else if (t && k < BUFFER_MAX_ERROR)
    {

      if (timer_time (t) > BUFFER_MAX_TIMER)
	{
	  err = 0;

	  timer_clear (t);
	  t = NULL;
	}
    }

  return err;
}

/* EOF */
