/*
  Copyright (C) 2000-2005 SKYRIX Software AG

  This file is part of SOPE.

  SOPE is free software; you can redistribute it and/or modify it under
  the terms of the GNU Lesser General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  SOPE is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
  License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with SOPE; see the file COPYING.  If not, write to the
  Free Software Foundation, 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.
*/

#import <NGObjWeb/WOxElemBuilder.h>

/*
  This builder builds all standard elements which are defined in the XHTML
  or HTML 4 namespace.

  Supported tags:
    - all other tags are represented using either WOGenericElement or
      WOGenericContainer, so this builder is "final destination" for
      all HTML related tags.
  
    <input> - the "type" attribute of the input must be static and further
              specifies the generated element
      type="submit"   maps to WOSubmitButton
      type="reset"    maps to WOResetButton
      type="image"    maps to WOImageButton
      type="radio"    maps to WORadioButton
      type="checkbox" maps to WOCheckBox
      type="file"     maps to WOFileUpload
      type="hidden"   maps to WOHiddenField
      type="password" maps to WOPasswordField
      TODO: button!
      all other       map  to WOTextField
    <a>..</a>         maps to WOHyperlink
    <img .../>        maps to WOImage
    <form .../>	      maps to WOForm
    <textarea .../>   maps to WOText
    <embed .../>      maps to WOEmbeddedObject
    <frame .../>      maps to WOFrame
    <iframe .../>     maps to WOIFrame
    <body .../>       maps to WOBody
    <entity .../>     maps to WOEntity
*/

@interface WOxHTMLElemBuilder : WOxTagClassElemBuilder
{
}

@end

#include "common.h"
#include <SaxObjC/XMLNamespaces.h>

@implementation WOxHTMLElemBuilder

- (Class)classForInputElement:(id<DOMElement>)_element {
  NSString *type;
  unsigned tl;
  unichar c1;
  
  type = [_element attribute:@"type" namespaceURI:XMLNS_XHTML];
  tl = [type length];

  if (tl == 0)
    return NSClassFromString(@"WOTextField");
  else if (tl > 0)
    c1 = [type characterAtIndex:0];
  
  switch (tl) {
    case 0:

    case 4:
      if (c1 == 't') {
        if ([type isEqualToString:@"text"])
          return NSClassFromString(@"WOTextField");
      }
      else if (c1 == 'f') {
        if ([type isEqualToString:@"file"])
          return NSClassFromString(@"WOFileUpload");
      }
      break;
      
    case 5:
      if (c1 == 'i' && [type isEqualToString:@"image"])
        return NSClassFromString(@"WOImageButton");
      else if (c1 == 'r') {
        if ([type isEqualToString:@"radio"])
          return NSClassFromString(@"WORadioButton");
        if ([type isEqualToString:@"reset"])
          return NSClassFromString(@"WOResetButton");
      }
      break;
      
    case 6:
      if (c1 == 's' && [type isEqualToString:@"submit"])
        return NSClassFromString(@"WOSubmitButton");
      else if (c1 == 'h' && [type isEqualToString:@"hidden"])
        return NSClassFromString(@"WOHiddenField");
      else if (c1 == 'b' && [type isEqualToString:@"button"]) {
	[self logWithFormat:@"WARNING: selecting WOSubmitButton for "
	        @"input with type 'button' !"];
        return NSClassFromString(@"WOSubmitButton");
      }
      break;

    case 8:
      if (c1 == 'c' && [type isEqualToString:@"checkbox"])
        return NSClassFromString(@"WOCheckBox");
      else if (c1 == 'p' && [type isEqualToString:@"password"])
        return NSClassFromString(@"WOPasswordField");
      
    default:
      break;
  }
  
  [self logWithFormat:@"WARNING: unknown input type '%@' !", type];
  return NSClassFromString(@"WOTextField");
}

- (Class)classForElement:(id<DOMElement>)_element {
  NSString *nsuri;
  NSString *tag;
  unsigned tl;

  if ((nsuri = [_element namespaceURI]) == nil)
    return Nil;
  
  if (![nsuri isEqualToString:XMLNS_XHTML]) {
    /* check HTML 4 namespace */
    if (![nsuri isEqualToString:XMLNS_HTML40])
      return Nil;
  }
  
  tag = [_element tagName];
  
  if ((tl = [tag length]) == 0)
    return Nil;
  
  switch (tl) {
    case 1:
        if ([tag isEqualToString:@"a"]) {
            if ([_element hasAttribute:@"name" namespaceURI:@"*"])
              return NSClassFromString(@"WOGenericContainer");
            
            return NSClassFromString(@"WOHyperlink");
        }
      break;

    case 3:
      if ([tag isEqualToString:@"img"])
        return NSClassFromString(@"WOImage");
      break;

    case 4:
      if ([tag isEqualToString:@"body"])
        return NSClassFromString(@"WOBody");
#if 0
      if ([tag isEqualToString:@"html"])
        return NSClassFromString(@"WOHtml");
#endif
      if ([tag isEqualToString:@"form"])
        return NSClassFromString(@"WOForm");

      if ([tag isEqualToString:@"meta"]) {
	NSString *val;
	
        val = [_element attribute:@"http-equiv" namespaceURI:XMLNS_XHTML];
	if (val) {
	  val = [[val stringValue] lowercaseString];
	  if ([val hasPrefix:@"refresh"])
	    return NSClassFromString(@"WOMetaRefresh");
	}
      }
      break;
      
    case 5:
      if ([tag isEqualToString:@"input"])
        return [self classForInputElement:_element];
      if ([tag isEqualToString:@"frame"])
        return NSClassFromString(@"WOFrame");
      if ([tag isEqualToString:@"embed"])
        return NSClassFromString(@"WOEmbeddedObject");
      break;
      
    default:
      if ([tag isEqualToString:@"textarea"])
        return NSClassFromString(@"WOText");
      if ([tag isEqualToString:@"iframe"])
        return NSClassFromString(@"WOIFrame");
      
      if ([tag isEqualToString:@"entity"])
        return NSClassFromString(@"WOEntity");
      break;
  }

  if ([_element hasChildNodes])
    return NSClassFromString(@"WOGenericContainer");
  else
    return NSClassFromString(@"WOGenericElement");
}

@end /* WOxHTMLElemBuilder */
