

#include "soundkonverter.h"
#include "options.h"
#include "shareddir.h"
#include "filelist.h"
//#include "filelist_edit.h"
#include "config.h"
#include "config_dialogue.h"
#include "processoptions.h"
#include "convert.h"
#include "protocol.h"
#include "open_files.h"
#include "tools.h"
#include "combobutton.h"
#include "backend_plugins.h"
#include "replaygain_plugins.h"
#include "replaygainscanner.h"
#include "cue_editor.h"
#include "firstrunwizard.h"

#include <qlayout.h>
#include <qprogressbar.h>
#include <qlabel.h>
#include <qfont.h>
#include <qfileinfo.h>
#include <qcombobox.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qfile.h>
#include <qtextedit.h>
#include <qlistview.h>
#include <qtooltip.h>
#include <qdir.h>
#include <qevent.h>
#include <qdragobject.h>
#include <qclipboard.h>
#include <qurl.h>

#include <klocale.h>
#include <kcmdlineargs.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kpushbutton.h>
#include <kpopupmenu.h>
#include <kmenubar.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kedittoolbar.h>
#include <kapplication.h>
#include <kactionclasses.h>
#include <kstddirs.h>
#include <kio/job.h>
#include <kdirlister.h>

#include <fileref.h>
#include <tag.h>
#include <tstring.h>

#include <dcopclient.h>

Config prefs;
ProcessOptions actFile;
Tools tools;
BackendPlugins backendPlugins;
ReplayGainPlugins replayGainPlugins;


soundKonverter::soundKonverter()
    : KMainWindow( 0L, "soundkonverter-mainwindow" ),
      DCOPObject( "soundkonverter-app" )
{
    setCaption("soundKonverter");
    resize(10,10);

    //#ifdef DEBUG
    //system( "echo \"soundKonverter 0.2\n\n\" > ~/debug.txt" );
    //#endif

    instances=0;
    totalTime=0;

    prefs.read();
    tools.calculateFileOpenStrings();

    startAction = new KAction( i18n("&Start conversion"), "run", 0, this, SLOT(startConversion()), actionCollection(), "start" );
    new KAction( i18n("&ReplayGain Tool ..."), "soundkonverter_replaygain", CTRL+Key_R, this, SLOT(showReplayGainScanner()), actionCollection(), "replaygainscanner" );
    new KAction( i18n("C&ue Editor ..."), "kwrite", CTRL+Key_U, this, SLOT(showCueEditor()), actionCollection(), "cueeditor" );
    new KAction( i18n("&Protocol ..."), "view_text", CTRL+Key_P, this, SLOT(showProtocol()), actionCollection(), "protocol" );

    stopAction = new KAction( i18n("S&top after current file is complete"), "stop", CTRL+Key_O, this, SLOT(stopConversion()), actionCollection(), "stop" );
    stopAction->setEnabled(false);
    continueAction = new KAction( i18n("&Continue after current file is complete"), "run", CTRL+Key_T, this, SLOT(continueConversion()), actionCollection(), "continue" );
    continueAction->setEnabled(false);
    killAction = new KAction( i18n("Stop &immediately"), "exit", CTRL+Key_K, this, SLOT(killConversion()), actionCollection(), "kill" );
    killAction->setEnabled(false);
    stopActionMenu = new KActionMenu( i18n("Stop"), "stop", actionCollection(), "stopMenu" );
    stopActionMenu->setDelayed(false);
    stopActionMenu->setEnabled(false);
    stopActionMenu->insert(stopAction);
    stopActionMenu->insert(continueAction);
    stopActionMenu->insert(killAction);

    veryHighPriorityAction = new KToggleAction( i18n("Very hi&gh"), CTRL+Key_1, this, SLOT(priorityChanged()), actionCollection(), "veryhigh" );
    veryHighPriorityAction->setExclusiveGroup("priorityActionMenu");
    highPriorityAction = new KToggleAction( i18n("&High"), CTRL+Key_2, this, SLOT(priorityChanged()), actionCollection(), "high" );
    highPriorityAction->setExclusiveGroup("priorityActionMenu");
    normalPriorityAction = new KToggleAction( i18n("&Normal"), CTRL+Key_3, this, SLOT(priorityChanged()), actionCollection(), "nomal" );
    normalPriorityAction->setExclusiveGroup("priorityActionMenu");
    normalPriorityAction->setChecked(true);
    lowPriorityAction = new KToggleAction( i18n("&Low"), CTRL+Key_4, this, SLOT(priorityChanged()), actionCollection(), "low" );
    lowPriorityAction->setExclusiveGroup("priorityActionMenu");
    veryLowPriorityAction = new KToggleAction( i18n("Very lo&w"), CTRL+Key_5, this, SLOT(priorityChanged()), actionCollection(), "verylow" );
    veryLowPriorityAction->setExclusiveGroup("priorityActionMenu");
    priorityActionMenu = new KActionMenu( i18n("En-/Decoder priority"), "ksysguard", actionCollection(), "priorityMenu" );
    priorityActionMenu->setDelayed(false);
    priorityActionMenu->insert(veryHighPriorityAction);
    priorityActionMenu->insert(highPriorityAction);
    priorityActionMenu->insert(normalPriorityAction);
    priorityActionMenu->insert(lowPriorityAction);
    priorityActionMenu->insert(veryLowPriorityAction);
    priorityChanged();

    new KAction( i18n("A&dd Files ..."), "sound", CTRL+Key_D, this, SLOT(showFileDialogue()), actionCollection(), "add_files" );
    new KAction( i18n("Add &Folder ..."), "folder", CTRL+Key_F, this, SLOT(showDirDialogue()), actionCollection(), "add_folder" );
    KStdAction::quit(this, SLOT(close()), actionCollection());

    //KStdAction::paste(this, SLOT(pasteFiles()), actionCollection());

    KStdAction::preferences(this, SLOT(showConfigDialogue()), actionCollection());

    showToolBarAction = KStdAction::showToolbar(this, SLOT(showToolbar()), actionCollection());
    //KStdAction::showStatusbar(0, 0, actionCollection());
    KStdAction::configureToolbars(this, SLOT(editToolbar()), actionCollection());

    createGUI();


    QWidget *widget = new QWidget(this,"widget");
    setCentralWidget(widget);

    QGridLayout *grid = new QGridLayout( widget, 1, 1, 10, 5, "grid" );

    options = new Options( widget, "options" );
    grid->addWidget( options, 0, 0 );
    connect(this,SIGNAL(resetOptions()),options,SLOT(reset()));
    connect(this,SIGNAL(addFiles(QStringList)),options,SLOT(addFiles(QStringList)));
    connect(this,SIGNAL(quitSoundKonverter()),options,SLOT(saveOptions()));
    connect(options,SIGNAL(timeAdded(float)),this,SLOT(timeAdded(float)));

    QVBoxLayout *notifyTopBox = new QVBoxLayout( -1, "notifyTopBox" );
    grid->addLayout( notifyTopBox, 0, 0 );

    QHBoxLayout *notifyBox = new QHBoxLayout( -1, "notifyBox" );
    notifyTopBox->addLayout( notifyBox );
    notifyTopBox->addStretch();

    notifyBox->addStretch();

    pNewLocalBackends = new KPushButton( " ", widget, "pNewLocalBackends");
    pNewLocalBackends->setPixmap( KGlobal::iconLoader()->loadIcon("kcmsystem",KIcon::Toolbar) );
    QToolTip::add( pNewLocalBackends, i18n("New backends were found. You can now configure them.") );
    pNewLocalBackends->hide();
    notifyBox->addWidget( pNewLocalBackends );
    connect(pNewLocalBackends,SIGNAL(clicked()),this,SLOT(showConfigDialogue()));

    pNewBackends = new KPushButton( " ", widget, "pNewBackends");
    pNewBackends->setPixmap( KGlobal::iconLoader()->loadIcon("kget",KIcon::Toolbar) );
    QToolTip::add( pNewBackends, i18n("New backends were available. You can now download them.") );
    pNewBackends->hide();
    notifyBox->addWidget( pNewBackends );
    connect(pNewBackends,SIGNAL(clicked()),this,SLOT(showConfigDialogue()));

    pNewPlugins = new KPushButton( " ", widget, "pNewPlugins");
    pNewPlugins->setPixmap( KGlobal::iconLoader()->loadIcon("connect_creating",KIcon::Toolbar) );
    QToolTip::add( pNewPlugins, i18n("New plugins are available. You can now download them.") );
    pNewPlugins->hide();
    notifyBox->addWidget( pNewPlugins );
    connect(pNewPlugins,SIGNAL(clicked()),this,SLOT(showConfigDialogue()));

    QHBoxLayout *add_dirBox = new QHBoxLayout( -1, "add_dirBox" );
    grid->addLayout( add_dirBox, 1, 0 );

    cAdd = new ComboButton( widget, "cAdd" );
    cAdd->insertItem( KGlobal::iconLoader()->loadIcon("sound",KIcon::Small), i18n("Add Files ...") );
    cAdd->insertItem( KGlobal::iconLoader()->loadIcon("folder",KIcon::Small), i18n("Add Folder ...") );
    add_dirBox->addWidget( cAdd );
    connect(cAdd,SIGNAL(clicked(int)),this,SLOT(addClicked(int)));

    pScanStatus = new QProgressBar( widget, "pScanStatus" );
    add_dirBox->addWidget( pScanStatus );
    pScanStatus->setMaximumWidth(75);
    pScanStatus->hide();

    add_dirBox->addStretch();

    fileList = new FileList( widget, "fileList" );
    grid->addWidget( fileList, 2, 0 );
    grid->setRowStretch( 2, 1 );
    connect(this,SIGNAL(removeFirstFile()),fileList,SLOT(removeFirstFile()));
    connect(this,SIGNAL(updateFileListLook()),fileList,SLOT(updateLook()));
    connect(this,SIGNAL(readFileList()),fileList,SLOT(readList()));
    connect(this,SIGNAL(saveFileList()),fileList,SLOT(saveList()));
    connect(this,SIGNAL(quitSoundKonverter()),fileList,SLOT(resetList()));
    connect(fileList,SIGNAL(timeRemoved(float)),this,SLOT(timeRemoved(float)));
    connect(fileList,SIGNAL(timeAdded(float)),this,SLOT(timeAdded(float)));
    connect(fileList,SIGNAL(dropped(QDropEvent*)),this,SLOT(fileListDropEvent(QDropEvent*)));

    connect(fileList,SIGNAL(addFiles(QStringList)),options,SLOT(addFiles(QStringList)));
    connect(options,SIGNAL(addFile(QStringList)),fileList,SLOT(addFile(QStringList)));

    QHBoxLayout *cmd_editBox = new QHBoxLayout( -1, "cmd_editBox" );
    grid->addLayout( cmd_editBox, 4, 0 );
    grid->setRowStretch( 4, 0 );

    tCmd = new QTextEdit( widget, "tCmd" );
    tCmd->setReadOnly(true);
    tCmd->setTextFormat(Qt::LogText);
    tCmd->setFixedHeight( 75 );
    cmd_editBox->addWidget( tCmd );

    QHBoxLayout *ctrl_cfgBox = new QHBoxLayout( -1, "ctrl_cfgBox" );
    grid->addLayout( ctrl_cfgBox, 3, 0 );

    sharedDir = new SharedDir( widget, "sharedDir" );
    ctrl_cfgBox->addWidget( sharedDir );
    connect(this,SIGNAL(conversionStarted()),sharedDir,SLOT(conversionStarted()));
    connect(this,SIGNAL(conversionStoped()),sharedDir,SLOT(conversionStoped()));
    connect(this,SIGNAL(quitSoundKonverter()),sharedDir,SLOT(savePaths()));

    pStart = new KPushButton( KGlobal::iconLoader()->loadIcon("run",KIcon::Small), i18n("Start"), widget, "pStart" );
    QToolTip::add( pStart, i18n("Start converting all files in the list above.") );
    ctrl_cfgBox->addWidget( pStart );
    connect(pStart,SIGNAL(clicked()),this,SLOT(startConversion()));

    pStop = new KPushButton( KGlobal::iconLoader()->loadIcon("stop",KIcon::Small), i18n("Stop"), widget, "pStop" );
    pStop->hide();
    pStop->setPopup(stopActionMenu->popupMenu());
    ctrl_cfgBox->addWidget(pStop);

    QGridLayout *statusGrid = new QGridLayout( grid, 1, 2, 5, "statusGrid" );

    pProgress = new QProgressBar( widget, "pProgress" );
    statusGrid->addWidget( pProgress, 0, 0, Qt::AlignVCenter );

    QGridLayout *statusChildGrid = new QGridLayout( statusGrid, 2, 2, 5, "statusChildGrid" );

    QLabel *lSpeedText = new QLabel( i18n("Speed")+":", widget, "lSpeedText" );
    statusChildGrid->addWidget( lSpeedText, 0, 0, Qt::AlignVCenter );

    lSpeed = new QLabel( "0.00x", widget, "lSpeed" );
    lSpeed->setFont( QFont( "Courier" ) );
    statusChildGrid->addWidget( lSpeed, 0, 1, Qt::AlignVCenter | Qt::AlignRight );

    QLabel *lTimeText = new QLabel( i18n("Time remaining")+":", widget, "lTimeText" );
    statusChildGrid->addWidget( lTimeText, 1, 0, Qt::AlignVCenter );

    lTime = new QLabel( "0:00:00", widget, "lTime" );
    lTime->setFont( QFont( "Courier" ) );
    statusChildGrid->addWidget( lTime, 1, 1, Qt::AlignVCenter | Qt::AlignRight );

    convert = new Convert();
    connect(this,SIGNAL(decode()),convert,SLOT(decode()));
    connect(this,SIGNAL(encode()),convert,SLOT(encode()));
    connect(this,SIGNAL(kill()),convert,SLOT(kill()));
    connect(convert,SIGNAL(updateStatus()),this,SLOT(updateStatus()));
    connect(convert,SIGNAL(exited(int)),this,SLOT(conversionExited(int)));
    connect(convert,SIGNAL(sendOutput(QString)),this,SLOT(getOutput(QString)));
    connect(convert,SIGNAL(timeConverted(float)),this,SLOT(timeConverted(float)));

    connect(this,SIGNAL(setPriority(int)),convert,SLOT(priorityChanged(int)));

    protocol = new Protocol(this);
    connect(this,SIGNAL(sendOutput(QStringList)),protocol,SLOT(getOutput(QStringList)));

    dirLister= new KDirLister();
    dirLister->setAutoUpdate(false);
    connect(dirLister,SIGNAL(newItems(const KFileItemList&)),this,SLOT(allFilesListed(const KFileItemList&)));
    connect(dirLister,SIGNAL(percent(int)),this,SLOT(scanStatus(int)));
    //connect(dirLister,SIGNAL(completed()),this,SLOT(scanCompleted()));

    replayGainScanner=0;
    cueEditor=0;

    if( prefs.showToolBar ) {
        toolBar()->show();
        showToolBarAction->setChecked(true);
    }
    else {
        toolBar()->hide();
        showToolBarAction->setChecked(false);
    }

    if( prefs.firstRun ) {
        prefs.readOldConfig();
        firstRunWizard = new FirstRunWizard(this,"firstRunWizard");
        Q_CHECK_PTR(firstRunWizard);
        firstRunWizard->exec();
        delete firstRunWizard;
        prefs.firstRun=false;
        prefs.write();
    }
    else {
        if( prefs.check.backends ) {
            scanForBackends();
        }
        if( prefs.check.online_plugins ) {
            downloadPluginListJob = KIO::file_copy("http://kaligames.de/downloads/soundkonverter/plugins/download.php?version=200",
            locateLocal("data","soundkonverter/pluginlist_new.txt"),-1,true,false,false);
            connect(downloadPluginListJob,SIGNAL(result(KIO::Job*)),this,SLOT(downloadPluginListFinished(KIO::Job*)));
            // TODO temopäre datei in temp erzeugen
        }
        if( prefs.check.online_backends ) {
            downloadBackendListJob = KIO::file_copy("http://kaligames.de/downloads/soundkonverter/backends/download.php?version=0.2",
            locateLocal("data","soundkonverter/backendlist_new.txt"),-1,true,false,false);
            connect(downloadBackendListJob,SIGNAL(result(KIO::Job*)),this,SLOT(downloadBackendListFinished(KIO::Job*)));
            // TODO temopäre datei in temp erzeugen
        }
    }

    //#ifdef DEBUG
    //options->debugInit();
    //#endif
}

soundKonverter::~soundKonverter()
{
    //delete downloadPluginListJob;
    //delete downloadBackendListJob;
    delete replayGainScanner;
    delete cueEditor;
}

void soundKonverter::increaseInstances()
{
    instances++;
}

bool soundKonverter::queryClose()
{
    emit quitSoundKonverter();
    prefs.write();
    return true;
}

void soundKonverter::addClicked(int index)
{
    if(index == 0) {
        showFileDialogue();
    }
    else {
        showDirDialogue();
    }
}

void soundKonverter::showFileDialogue()
{
    pScanStatus->hide();

    KFileDialog *fileDialog;
    if( options->currentFormat() != "wav" ) {
        fileDialog = new KFileDialog( QDir::homeDirPath(), tools.sFileOpenString, this, i18n("Choose files to convert!"), true );
    }
    else {
        fileDialog = new KFileDialog( QDir::homeDirPath(), tools.sFileOpenStringWithoutWave, this, i18n("Choose files to convert!"), true );
    }
    fileDialog->setMode ( KFile::Files | KFile::ExistingOnly | KFile::LocalOnly );
    if( fileDialog->exec() == KDialog::Accepted ) {
        emit addFiles( fileDialog->selectedFiles() );
    }
    emit saveFileList();
}

void soundKonverter::showDirDialogue()
{
    pScanStatus->show();

    QString dirname = KFileDialog::getExistingDirectory( QDir::homeDirPath(), this, i18n("Choose a directory!") );
    if (dirname != NULL)
    {
        dirLister->openURL(dirname);
    }
    emit saveFileList();
}

void soundKonverter::pasteFiles()
{
    QClipboard *cb = QApplication::clipboard();
    QString files=cb->text(QClipboard::Clipboard);
    QStringList list;
    QStringList fileList=QStringList::split('\n',files);
    for( QStringList::Iterator it=fileList.begin(); it!=fileList.end(); ++it ) {
        QUrl url(*it);
        if( url.isLocalFile() ) {
            QFileInfo fileInfo(url.path());
            if( fileInfo.isFile() && tools.supportedFormats.findIndex(tools.fileExtension(url.path())) != -1 ) {
                list.append(url.path());
            }
        }
    }
    emit addFiles(list);
}

void soundKonverter::allFilesListed(const KFileItemList &items)
{
    QString currentFormat=options->currentFormat();
    QString format;
    QStringList files;

    for( KFileItemList::Iterator it=items.begin(); it!=items.end(); ++it )
    {
        QString name=(*it)->url().path();
        QFileInfo fileInfo(name);
        format=tools.fileExtension(name);
        if( fileInfo.isFile() && currentFormat != "wav" && tools.supportedFormats.contains(format) ) {
            files.append(name);
        }
        else if( fileInfo.isFile() && currentFormat == "wav" && tools.supportedFormatsWithoutWave.contains(format) ) {
            files.append(name);
        }
        else if( fileInfo.isDir()) {
            dirLister->openURL(name,true);
        }
    }

    emit addFiles(files);
}

void soundKonverter::scanStatus(int percent)
{
    pScanStatus->setProgress(percent,100);
}

/*void soundKonverter::scanCompleted()
{
    pScanStatus->setProgress(0,0);
    pScanStatus->setEnabled(false);
}*/

// TODO mit Rootpassworteingabe sollte es gehen
void soundKonverter::priorityChanged()
{
    int priority=0;

    if( veryHighPriorityAction->isChecked() ) priority=-19;
    else if( highPriorityAction->isChecked() ) priority=-10;
    else if( lowPriorityAction->isChecked() ) priority=10;
    else if( veryLowPriorityAction->isChecked() ) priority=20;

    emit setPriority(priority);
}

void soundKonverter::startConversion()
{
    bStoped=false;
    bKilled=false;
    pStart->hide();
    startAction->setEnabled(false);
    pStop->show();
    stopActionMenu->setEnabled(true);
    stopAction->setEnabled(true);
    continueAction->setEnabled(false);
    killAction->setEnabled(true);
    emit conversionStarted();
    totalTime-=processedTime;
    processedTime=0;
    pProgress->setTotalSteps((int)(totalTime*100));
    remainingTime.start();
    convertNextFile();
}

void soundKonverter::stopConversion()
{
    bStoped=true;
    stopAction->setEnabled(false);
    continueAction->setEnabled(true);
    getOutput( i18n("Trying to stop...") );
}

void soundKonverter::continueConversion()
{
    bStoped=false;
    stopAction->setEnabled(true);
    continueAction->setEnabled(false);
    getOutput( i18n("Continuing...") );
}

void soundKonverter::killConversion()
{
    bStoped=false;
    bKilled=false;
    emit kill();
}

void soundKonverter::convertNextFile()
{
    QListViewItem *item=fileList->firstFile();
    QString sFile, sOutput, sQuality, sBitrateRange, sMode, sChannels, sSampleRate, sReplayGain, sState, sSharedDir;
    QString temp;
    QFile fFile;
    int i;
    QString params;
    QStringList paramList;

    emit saveFileList();

    while( item != 0 && item->text( fileList->column(i18n("State")) ) == i18n("Error") ) {
        item=item->nextSibling();
    }

    if( item != 0 ) {
        params = item->text( 0 );
        paramList = QStringList::split("$soundkonverter-spacer$",params);
        QStringList::Iterator it=paramList.begin();

        sFile=*it; it++;
        sOutput=*it; it++;
        sQuality=*it; it++;
        sBitrateRange=*it; it++;
        sMode=*it; it++;
        sSampleRate=*it; it++;
        sChannels=*it; it++;
        sReplayGain=*it; it++;
        sState=*it;

        actFile.pItem = item;
        actFile.sFilename = sFile;
        actFile.sFormat = tools.fileExtension(sFile);

        if( actFile.sFormat == "ogg" ||
            actFile.sFormat == "flac"||
            actFile.sFormat == "mpc" ||
            actFile.sFormat == "mp3" ) {
            readTags();
        }
        else {
            resetTags();
        }

        if( sOutput != "-" ) {
            actFile.Convert.bEnabled = true;
            actFile.Convert.sOutputFormat = sOutput;
            actFile.Convert.sOutputFilename = sFile;
                actFile.Convert.sOutputFilename=sharedDir->calcFilename();
                i = actFile.Convert.sOutputFilename.findRev( '.' ) + 1;
                actFile.Convert.sOutputFilename.remove( i, actFile.Convert.sOutputFilename.length() - i );
                actFile.Convert.sOutputFilename.append( actFile.Convert.sOutputFormat );
                fFile.setName( actFile.Convert.sOutputFilename );
                while( fFile.exists() ) {
                    i = actFile.Convert.sOutputFilename.findRev( '.' ) + 1;
                    actFile.Convert.sOutputFilename.remove( i, actFile.Convert.sOutputFilename.length() - i );
                    actFile.Convert.sOutputFilename.append( i18n("new.") );
                    actFile.Convert.sOutputFilename.append( actFile.Convert.sOutputFormat );
                    fFile.setName( actFile.Convert.sOutputFilename );
                }
            if( actFile.Convert.sOutputFormat != "wav" && actFile.sFormat != "wav" ) {
                actFile.Convert.sWaveOutputFilename = sFile;
                    actFile.Convert.sWaveOutputFilename=sharedDir->calcFilename();
                    actFile.Convert.sWaveOutputFilename.append( ".wav" );
                    fFile.setName( actFile.Convert.sWaveOutputFilename );
                    while( fFile.exists() ) {
                        i = actFile.Convert.sWaveOutputFilename.findRev( '.' ) + 1;
                        actFile.Convert.sWaveOutputFilename.remove( i, actFile.Convert.sWaveOutputFilename.length() - i );
                        actFile.Convert.sWaveOutputFilename.append( i18n("new.") );
                        actFile.Convert.sWaveOutputFilename.append( "wav" );
                        fFile.setName( actFile.Convert.sWaveOutputFilename );
                    }
            }
            else if( actFile.Convert.sOutputFormat == "wav" ) {
                actFile.Convert.sWaveOutputFilename = actFile.Convert.sOutputFilename;
            }
            else {
                actFile.Convert.sWaveOutputFilename = actFile.sFilename;
            }

            if( sQuality != "-" ) {
                if( sQuality.contains("kbps") ) {
                    actFile.Convert.Quality.EMode = EBitrate;
                    //sscanf( sQuality, "%i kbps", &actFile.Convert.Quality.iOutputBitrate );
                    sscanf( sQuality, "%i kbps", &actFile.Convert.Quality.iOutputBitrate );
                    actFile.Convert.Quality.sBitrateMode = sMode;
                }
                else {
                    actFile.Convert.Quality.EMode = EQuality;
                    sscanf( sQuality, "%i", &actFile.Convert.Quality.iLevel );
                }
            }
            else {
                actFile.Convert.Quality.EMode = ELossless;
            }

            if( sBitrateRange != "-" ) {
                actFile.Convert.Quality.bRange = true;
                sscanf( sBitrateRange, "%i-%i",
                    &actFile.Convert.Quality.iMinBitrate, &actFile.Convert.Quality.iMaxBitrate );
            }
            else {
                actFile.Convert.Quality.bRange = false;
            }
            actFile.Convert.Quality.sBitrateMode = sMode;
        }
        else {
            actFile.Convert.bEnabled = false;
        }

        if( sSampleRate != "-" ) {
            actFile.SampleRate.bEnabled = true;
            sscanf( sSampleRate, "%i Hz", &actFile.SampleRate.iOutputSampleRate );
        }
        else {
            actFile.SampleRate.bEnabled = false;
        }

        if( sChannels != "-" ) {
            actFile.Channels.bEnabled = true;
            if( sChannels == i18n("Stereo") ) { actFile.Channels.iOutputChannels = 2; }
            else if( sChannels == i18n("Mono") ) { actFile.Channels.iOutputChannels = 1; }
        }
        else {
            actFile.Channels.bEnabled = false;
        }

        if( sReplayGain == i18n("yes") ) {
            actFile.bReplayGain = true;
        }
        else {
            actFile.bReplayGain = false;
        }

        actFile.Time.iRealLength=tools.timeByFile(actFile.sFilename);

        if( actFile.sFormat != "wav" ) {
            actFile.Time.fDec=actFile.Time.iRealLength*prefs.fileFormat(actFile.sFormat).decTime;
        }
        else {
            actFile.Time.fDec=0;
        }
        if( actFile.Convert.sOutputFormat != "wav" ) {
            actFile.Time.fEnc=actFile.Time.iRealLength*prefs.fileFormat(actFile.Convert.sOutputFormat).encTime;
        }
        else {
            actFile.Time.fEnc=0;
        }
        if( actFile.bReplayGain ) {
            actFile.Time.fRep=actFile.Time.iRealLength*prefs.fileFormat(actFile.Convert.sOutputFormat).repTime;
        }
        else {
            actFile.Time.fRep=0;
        }

        if( actFile.sFormat == "wav" ) {
            emit encode();
        }
        else {
            emit decode();
        }
    }
    else {
        emit conversionStoped();
        //processedTime=0;
        pStart->show();
        startAction->setEnabled(true);
        pStop->hide();
        stopActionMenu->setEnabled(false);
        stopAction->setEnabled(false);
        continueAction->setEnabled(false);
        killAction->setEnabled(false);
        lSpeed->setText("0.00x");
        lTime->setText("0:00:00");
        pProgress->setProgress((int)totalTime*1000,(int)totalTime*1000);
        setCaption( i18n("Finished") );
        item=fileList->firstFile();
        while( item != 0 && item->text( fileList->column(i18n("State")) ) == i18n("Error") ) {
            item->setText( fileList->column(i18n("State")), i18n("Waiting...") );
            item=item->nextSibling();
        }
    }
}

void soundKonverter::conversionExited(int status)
{
    int iPercent;
    char cPercent[8];

    if( status == 1 ) {
        if(actFile.pItem!=0) actFile.pItem->setText( fileList->column(i18n("State")), i18n("Error") );
        actFile.pItem=0;
        convertNextFile();
    }
    else if( status == 2 ) {
        bStoped=false;
        emit conversionStoped();
        pStart->show();
        startAction->setEnabled(true);
        pStop->hide();
        stopActionMenu->setEnabled(false);
        stopAction->setEnabled(false);
        continueAction->setEnabled(false);
        killAction->setEnabled(false);
        lSpeed->setText("0.00x");
        lTime->setText("0:00:00");
        pProgress->setProgress( (int)(processedTime*1000), (int)(totalTime*1000) );
        getOutput( i18n("Stoped")+"!" );
        iPercent=(int)(100.0f*processedTime/totalTime);
        sprintf( cPercent, "%i%%", iPercent );
        setCaption( QString(cPercent).append(" ["+i18n("Stoped")+"]") );
        if(actFile.pItem!=0) actFile.pItem->setText( fileList->column(i18n("State")), i18n("Waiting...") );
        actFile.pItem=0;
        QListViewItem *item=fileList->firstFile();
        while( item != 0 && item->text( fileList->column(i18n("State")) ) == i18n("Error") ) {
            item->setText( fileList->column(i18n("State")), i18n("Waiting...") );
            item=item->nextSibling();
        }
    }
    else {
        emit removeFirstFile();

        // TODO write tags - good this way?
        if( actFile.Convert.sOutputFormat == "ogg" ||
            actFile.Convert.sOutputFormat == "flac"||
            actFile.Convert.sOutputFormat == "mp3" ||
            actFile.Convert.sOutputFormat == "mpc" ) {
            writeTags();
        }

        if( !bStoped ) {
            convertNextFile();
        }
        else {
            bStoped=false;
            emit conversionStoped();
            pStart->show();
            startAction->setEnabled(true);
            pStop->hide();
            stopActionMenu->setEnabled(false);
            stopAction->setEnabled(false);
            continueAction->setEnabled(false);
            killAction->setEnabled(false);
            lSpeed->setText("0.00x");
            lTime->setText("0:00:00");
            pProgress->setProgress( (int)(processedTime*1000), (int)(totalTime*1000) );
            getOutput( i18n("Stoped")+"!" );
            iPercent=(int)(100.0f*processedTime/totalTime);
            sprintf( cPercent, "%i%%", iPercent );
            setCaption( QString(cPercent).append(" ["+i18n("Stoped")+"]") );
            QListViewItem *item=fileList->firstFile();
            while( item != 0 && item->text( fileList->column(i18n("State")) ) == i18n("Error") ) {
                item->setText( fileList->column(i18n("State")), i18n("Waiting...") );
                item=item->nextSibling();
            }
        }
    }
}

void soundKonverter::timeAdded(float time)
{
    totalTime+=time;
    pProgress->setTotalSteps( (int)totalTime*1000 );
    //#ifdef DEBUG
    //system( QString("echo \"totalTime: %1").arg(totalTime).append("\ntime: %1\" >> ~/debug.txt").arg(time) );
    //#endif
}

void soundKonverter::timeRemoved(float time)
{
    totalTime-=time;
    pProgress->setTotalSteps( (int)totalTime*1000 );
    //#ifdef DEBUG
    //system( QString("echo \"totalTime: %1").arg(totalTime).append("\ntime: %1\" >> ~/debug.txt").arg(time) );
    //#endif
}

void soundKonverter::timeConverted(float time)
{
    processedTime+=time;
    updateStatus();
}

void soundKonverter::resetTags()
{
    actFile.Tag.sTitle=QString::null;
    actFile.Tag.sArtist=QString::null;
    actFile.Tag.sAlbum=QString::null;
    actFile.Tag.sGenre=QString::null;
    actFile.Tag.sComment=QString::null;
    actFile.Tag.iTrack=0;
    actFile.Tag.iYear=0;
    /*actFile.Tag.sTitle="soundKonverter";
    actFile.Tag.sArtist="Daniel Faust";
    actFile.Tag.sAlbum="KDE";
    actFile.Tag.sGenre="Linux";
    actFile.Tag.sComment="Linux rulez!";
    actFile.Tag.iTrack=1;
    actFile.Tag.iYear=2005;*/
}

void soundKonverter::readTags()
{
    TagLib::FileRef f( QFile::encodeName(actFile.sFilename) );

    if( !f.isNull() && f.tag() )
    {
        TagLib::Tag *tag = f.tag();

        actFile.Tag.sTitle = TStringToQString( tag->title() );
        actFile.Tag.sArtist = TStringToQString( tag->artist() );
        actFile.Tag.sAlbum = TStringToQString( tag->album() );
        actFile.Tag.sGenre = TStringToQString( tag->genre() );
        actFile.Tag.sComment = TStringToQString( tag->comment() );
        actFile.Tag.iTrack = tag->track();
        actFile.Tag.iYear = tag->year();
    }
}

void soundKonverter::writeTags()
{
    TagLib::FileRef f( QFile::encodeName(actFile.Convert.sOutputFilename) );

    if( !f.isNull() )
    {
        TagLib::Tag *tag = f.tag();

        tag->setTitle( QStringToTString( actFile.Tag.sTitle ) );
        tag->setArtist( QStringToTString( actFile.Tag.sArtist ) );
        tag->setAlbum( QStringToTString( actFile.Tag.sAlbum ) );
        tag->setGenre( QStringToTString( actFile.Tag.sGenre ) );
        tag->setComment( QStringToTString( actFile.Tag.sComment ) );
        tag->setTrack( actFile.Tag.iTrack );
        tag->setYear( actFile.Tag.iYear );

        f.save( );
    }
}

void soundKonverter::updateStatus()
{ // TODO Zeit nur bei änderung des fortschrittes neu berechnen (sonst nur herunterzählen)
    char cTimeLeft[16];
    int sec, min, hou, tim;
    float fPercent;
    char cPercent[8];
    int iPercent;

    //#ifdef DEBUG
    //system( QString("echo \"processedTime: %1").arg(processedTime).append("\nactFile.Process.fPercentTime: %1").arg(actFile.Process.fPercentTime).append("\ntotalTime: %1\" >> ~/debug.txt").arg(totalTime) );
    //#endif

    lSpeed->setText( actFile.Process.sSpeed );
    pProgress->setProgress( (int)(processedTime*1000+actFile.Process.fPercentTime*10), (int)(totalTime*1000) );

    tim = 0;
    if(actFile.step == EEncode)
    {
        fPercent = 100 * (processedTime*100+actFile.Process.iPercent*actFile.Time.fEnc) / (totalTime*100);
        actFile.pItem->setText( fileList->column(i18n("State")), i18n("Encoding...")+" "+actFile.Process.sPercent );
    }
    else if(actFile.step == EDecode)
    {
        fPercent = 100 * (processedTime*100+actFile.Process.iPercent*actFile.Time.fDec) / (totalTime*100);
        actFile.pItem->setText( fileList->column(i18n("State")), i18n("Decoding...")+" "+actFile.Process.sPercent );
    }
    else if(actFile.step == EReplayGain)
    {
        fPercent = 100 * (processedTime*100+actFile.Process.iPercent*actFile.Time.fRep) / (totalTime*100);
        actFile.pItem->setText( fileList->column(i18n("State")), i18n("ReplayGain...")+" "+actFile.Process.sPercent );
    }

    if( fPercent > 0.0 && fPercent < 100.0 )
    {
        //tim = (int)( 1 + remainingTime.elapsed() / (fPercent*1000) * (100-fPercent) );
        tim = (int)( 1 + ((100/fPercent)-1) * (remainingTime.elapsed()/1000) );
    }
    sec = tim%60;
    min = ((tim-sec)/60)%60;
    hou = (tim-sec-min*60)/3600;
    sprintf( cTimeLeft, "%d:%02d:%02d", hou, min, sec );
    lTime->setText(cTimeLeft);

//    iPercent=(int)(100.0f*(processedTime*1000.0f+actFile.Process.fPercentTime*10.0f)/(totalTime*1000.0f));
//FIXME    iPercent=pProgress->progress();
//FIXME    if( iPercent > 100 || iPercent < 0 ) iPercent=0;
    iPercent=(int)fPercent;
    sprintf( cPercent, "%i%%", iPercent );
    setCaption(cPercent);
}

void soundKonverter::getOutput(QString string)
{
    cmd.append(string);

    QStringList::Iterator it=cmd.begin();
    if( cmd.count() > 3 ) it+=cmd.count()-3;

    tCmd->clear();

    while( it != cmd.end() )
    {
        tCmd->append(*it);
        it++;
    }

    emit sendOutput(cmd);
}

void soundKonverter::showProtocol()
{
    protocol->exec();
}

void soundKonverter::showConfigDialogue()
{
    int index=-1;

    if( QObject::sender()->name() == QCString("pNewPlugins") ) {
        pNewPlugins->hide();
        index=1;
    }
    else if( QObject::sender()->name() == QCString("pNewBackends") ) {
        pNewBackends->hide();
        index=2;
    }
    else if( QObject::sender()->name() == QCString("pNewLocalBackends") ) {
        pNewLocalBackends->hide();
        index=3;
    }

    ConfigDialogue *dialog = new ConfigDialogue(this,"ConfigDialogue::dialog",index);

    Q_CHECK_PTR(dialog);

    dialog->exec();

    delete dialog;

    tools.calculateFileOpenStrings();
    emit resetOptions();

    emit updateFileListLook();
}

void soundKonverter::openFiles(const QStringList &files)
{
/*    QStringList list;
    QStringList newList;

    QString currentFormat=options->currentFormat();

    for( QStringList::Iterator it=list.begin(); it!=list.end(); ++it ) {
        QFileInfo fileInfo(*it);
        QString format=tools.fileExtension(*it);
        if( fileInfo.isFile() && currentFormat != "wav" && tools.supportedFormats.contains(format) ) {
            newList.append(*it);
        }
        else if( fileInfo.isFile() && currentFormat == "wav" && tools.supportedFormatsWithoutWave.contains(format) ) {
            newList.append(*it);
        }
    }
*/
    if( instances <= 1 || prefs.askForNewOptions )
    {
        OpenFiles *openFiles = new OpenFiles(files,this);

        connect(openFiles,SIGNAL(addFile(QStringList)),fileList,SLOT(addFile(QStringList)));
        connect(openFiles,SIGNAL(timeAdded(float)),this,SLOT(timeAdded(float)));

        Q_CHECK_PTR(openFiles);

        openFiles->exec();

        disconnect(openFiles,SIGNAL(addFile(QStringList)),0,0);
        disconnect(openFiles,SIGNAL(timeAdded(float)),0,0);

        delete openFiles;
    }
    else
    {
        emit addFiles(files);
    }
}

void soundKonverter::openReplayGainFiles(const QStringList &files)
{
    if( replayGainScanner==0 ) {
         replayGainScanner=new ReplayGainScanner(this,"replayGainScanner",false);
         if( replayGainScanner==0 ) {
             return;
         }
         connect(this,SIGNAL(addFilesToReplayGainScanner(QStringList)),replayGainScanner,SLOT(addFiles(QStringList)));
    }
    emit addFilesToReplayGainScanner(files);
    replayGainScanner->show();
}

void soundKonverter::showToolbar()
{
    if( showToolBarAction->isChecked() ) {
        toolBar()->show();
        prefs.showToolBar=true;
    }
    else {
        toolBar()->hide();
        prefs.showToolBar=false;
    }
    prefs.write();
}

void soundKonverter::editToolbar()
{
    saveMainWindowSettings( kapp->config(), "MainWindow" );
    KEditToolbar dlg(actionCollection());
    connect(&dlg,SIGNAL(newToolbarConfig()),this,SLOT(newToolbarConfig()));
    dlg.exec();
}

void soundKonverter::newToolbarConfig()
{
    createGUI();
    applyMainWindowSettings( kapp->config(), "MainWindow" );
}

void soundKonverter::showReplayGainScanner()
{
    if( replayGainScanner==0 ) {
         replayGainScanner=new ReplayGainScanner(this,"replayGainScanner",false);
         if( replayGainScanner==0 ) {
             return;
         }
         connect(this,SIGNAL(addFilesToReplayGainScanner(QStringList)),replayGainScanner,SLOT(addFiles(QStringList)));
    }
    replayGainScanner->show();
}

void soundKonverter::showCueEditor()
{
    if( cueEditor==0 ) {
         cueEditor=new CueEditor(this,"cueEditor",false);
         if( cueEditor==0 ) {
             return;
         }
    }
    cueEditor->show();
}

void soundKonverter::downloadPluginListFinished(KIO::Job* job)
{
    if( job->error() == 0 ) {
        char data[100], dataNew[100];
        int size=6, sizeNew=6;

        QFile file(locateLocal("data","soundkonverter/pluginlist.txt"));
        QFile fileNew(locateLocal("data","soundkonverter/pluginlist_new.txt"));

        if( !file.exists() ) {
            pNewPlugins->show();
            KIO::move(locateLocal("data","soundkonverter/pluginlist_new.txt"),locateLocal("data","soundkonverter/pluginlist.txt"),false);
            return;
        }

        file.open(IO_ReadOnly);
        fileNew.open(IO_ReadOnly);
        while( size > 5 && sizeNew > 5 ) {
            size=file.readLine(data,sizeof(data));
            sizeNew=fileNew.readLine(dataNew,sizeof(dataNew));
            if( QString(data) != QString(dataNew) ) {
                pNewPlugins->show();
                file.close();
                fileNew.close();
                file.remove();
                KIO::move(locateLocal("data","soundkonverter/pluginlist_new.txt"),locateLocal("data","soundkonverter/pluginlist.txt"),false);
                return;
            }
        }
        file.close();
        fileNew.close();
        fileNew.remove();
    }
}

void soundKonverter::downloadBackendListFinished(KIO::Job* job)
{
    if( job->error() == 0 ) {
        char data[100], dataNew[100];
        int size=6, sizeNew=6;

        QFile file(locateLocal("data","soundkonverter/backendlist.txt"));
        QFile fileNew(locateLocal("data","soundkonverter/backendlist_new.txt"));

        if( !file.exists() ) {
            pNewBackends->show();
            KIO::move(locateLocal("data","soundkonverter/backendlist_new.txt"),locateLocal("data","soundkonverter/backendlist.txt"),false);
            return;
        }

        file.open(IO_ReadOnly);
        fileNew.open(IO_ReadOnly);
        while( size > 5 && sizeNew > 5 ) {
            size=file.readLine(data,sizeof(data));
            sizeNew=fileNew.readLine(dataNew,sizeof(dataNew));
            if( QString(data) != QString(dataNew) ) {
                pNewBackends->show();
                file.close();
                fileNew.close();
                file.remove();
                KIO::move(locateLocal("data","soundkonverter/backendlist_new.txt"),locateLocal("data","soundkonverter/backendlist.txt"),false);
                return;
            }
        }
        file.close();
        fileNew.close();
        fileNew.remove();
    }
}

void soundKonverter::scanForBackends()
{
    QStringList list, programs;
    QFile fFile;

    for( BackendPlugins::PluginStructureList::Iterator it=backendPlugins.plugins.begin(); it!=backendPlugins.plugins.end(); ++it )
    {
        if( list.findIndex((*it).enc.bin) == -1 && (*it).enc.enabled ) {
            for( QStringList::Iterator jk=prefs.backend.directories.begin(); jk!=prefs.backend.directories.end(); ++jk )
            {
                fFile.setName((*jk)+"/"+(*it).enc.bin);
                if( fFile.exists() && list.findIndex((*it).enc.bin) == -1 ) {
                    programs.append((*jk)+"/"+(*it).enc.bin);
                    list.append((*it).enc.bin);
                }
            }
        }
        if( list.findIndex((*it).dec.bin) == -1 && (*it).dec.enabled ) {
            for( QStringList::Iterator jk=prefs.backend.directories.begin(); jk!=prefs.backend.directories.end(); ++jk )
            {
                fFile.setName((*jk)+"/"+(*it).dec.bin);
                if( fFile.exists() && list.findIndex((*it).dec.bin) == -1 ) {
                    programs.append((*jk)+"/"+(*it).dec.bin);
                    list.append((*it).dec.bin);
                }
            }
        }
    }
    for( ReplayGainPlugins::PluginStructureList::Iterator it=replayGainPlugins.plugins.begin(); it!=replayGainPlugins.plugins.end(); ++it )
    {
        if( list.findIndex((*it).replaygain.bin) == -1 ) {
            for( QStringList::Iterator jk=prefs.backend.directories.begin(); jk!=prefs.backend.directories.end(); ++jk )
            {
                fFile.setName((*jk)+"/"+(*it).replaygain.bin);
                if( fFile.exists() && list.findIndex((*it).replaygain.bin) == -1 ) {
                    programs.append((*jk)+"/"+(*it).replaygain.bin);
                    list.append((*it).replaygain.bin);
                }
            }
        }
    }

    if( programs != prefs.backend.programs ) {
        pNewLocalBackends->show();
        prefs.backend.programs=programs;
    }
}

void soundKonverter::fileListDropEvent(QDropEvent *e)
{
    QString sFormat;
    QString file;
    QStringList list;
    QStringList files;
    if( QUriDrag::decodeLocalFiles( e, list ) )
    {
        for( QStringList::Iterator it=list.begin(); it!=list.end(); ++it )
        {
            file=QDir::convertSeparators(*it);
            sFormat=tools.fileExtension(file);
            QFileInfo fileInfo(file);
            if( fileInfo.isFile() && tools.supportedFormats.contains(sFormat) )
            {
                files.append(file);
            }
            else if( fileInfo.isDir() )
            {
                pScanStatus->show();
                dirLister->openURL(file);
            }
        }
        emit addFiles(files);
    }
}
