/*
 * libao player support for sox
 * (c) Reuben Thomas <rrt@sc3d.org> 2007
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, write to the Free Software
 * Foundation, Fifth Floor, 51 Franklin Street, Boston, MA 02111-1301,
 * USA.  */

#include "sox_i.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ao/ao.h>

typedef struct ao_priv
{
  int driver_id;
  ao_device *device;
  ao_sample_format format;
  char *buf;
  sox_size_t buf_size;
} *ao_priv_t;

static int startread(UNUSED sox_format_t * ft)
{
  sox_fail("Cannot read from libao driver");
  return SOX_EOF;
}

static int startwrite(sox_format_t * ft)
{
  ao_priv_t ao = (ao_priv_t)ft->priv;

  if (ft->signal.size != SOX_SIZE_16BIT || 
      ft->signal.encoding != SOX_ENCODING_SIGN2)
  {
      sox_report("Forcing to signed 16 bit samples for ao driver");
      ft->signal.size = SOX_SIZE_16BIT;
      ft->signal.encoding = SOX_ENCODING_SIGN2;
  }

  ao->buf_size = sox_globals.bufsiz - (sox_globals.bufsiz % ft->signal.size);
  ao->buf_size *= ft->signal.size;
  ao->buf = xmalloc(ao->buf_size);

  if (!ao->buf)
  {
      sox_fail_errno(ft, SOX_ENOMEM, "Can not allocate memory for ao driver");
      return SOX_EOF;
  }


  ao_initialize();
  if (strcmp(ft->filename,"default") == 0)
  {
      if ((ao->driver_id = ao_default_driver_id()) < 0) {
          sox_fail("Could not find a default ao driver");
          return SOX_EOF;
      }
  }
  else
  {
      if ((ao->driver_id = ao_driver_id(ft->filename)) < 0) {
          sox_fail("Could not find a ao driver %s", ft->filename);
          return SOX_EOF;
      }
  }

  ao->format.bits = ft->signal.size * 8;
  ao->format.rate = ft->signal.rate;
  ao->format.channels = ft->signal.channels;
  ao->format.byte_format = AO_FMT_NATIVE;
  if ((ao->device = ao_open_live(ao->driver_id, &ao->format, NULL)) == NULL) {
    sox_fail("Could not open device: error %d", errno);
    return SOX_EOF;
  }

  return SOX_SUCCESS;
}

static void sox_sw_write_buf(char *buf1, sox_sample_t const * buf2, sox_size_t len, sox_bool swap, sox_size_t * clips)
{
    while (len--)
    {
        uint16_t datum = SOX_SAMPLE_TO_SIGNED_16BIT(*buf2++, *clips);
        if (swap)
            datum = sox_swapw(datum);
        *(uint16_t *)buf1 = datum;
        buf1++; buf1++;
    }
}

static sox_size_t write(sox_format_t *ft, const sox_sample_t *buf, sox_size_t len)
{
  ao_priv_t ao = (ao_priv_t)ft->priv;
  sox_size_t aobuf_size;

  if (len > ao->buf_size / ft->signal.size)
      len = ao->buf_size / ft->signal.size;

  aobuf_size = ft->signal.size * len;

  sox_sw_write_buf((char *)ao->buf, buf, len, ft->signal.reverse_bytes,
                   &(ft->clips));
  if (ao_play(ao->device, (void *)ao->buf, aobuf_size) == 0)
    return 0;

  return len;
}

static int stopwrite(sox_format_t * ft)
{
  ao_priv_t ao = (ao_priv_t)ft->priv;

  free(ao->buf);

  if (ao_close(ao->device) == 0) {
    sox_fail("Error closing libao output");
    return SOX_EOF;
  }
  ao_shutdown();

  return SOX_SUCCESS;
}

/* libao player */
static const char *aonames[] = {
  "ao",
  NULL
};

static sox_format_handler_t sox_ao_format = {
  aonames,
  SOX_FILE_DEVICE | SOX_FILE_NOSTDIO,
  startread,
  sox_format_nothing_read,
  sox_format_nothing,
  startwrite,
  write,
  stopwrite,
  sox_format_nothing_seek
};

const sox_format_handler_t *sox_ao_format_fn(void);

const sox_format_handler_t *sox_ao_format_fn(void)
{
    return &sox_ao_format;
}
