/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*******************************************************************
 * frustum.c : define a pyramid, the field of view of a camera. This 
 * is a mathematical definition to allow visibility computation.
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 *******************************************************************/

#include <math.h>

#include "p3_base.h"
#include "math3d.h"
#include "coordsys.h"
#include "frustum.h"

/*
 * INFO
 *
 *   15-------12
 *   |\       /|
 *   | \     / |
 *   |  3---0  |
 *   |  |   |  |
 *   |  6---9  |
 *   | /     \ |
 *   |/       \|
 *   18-------21
 *
 * plane[ 0] : front plane
 * plane[ 4] : top plane
 * plane[ 8] : bottom plane
 * plane[12] : right plane
 * plane[16] : left plane
 * plane[20] : back plane
 * plane normals are oriented in the exterior of the frustum
 */

/* fov is in degrees... */
void P3_frustum_new (P3_frustum* f, GLfloat fov, GLfloat front, GLfloat back, int surface_width, int surface_height, int ortho) {
  GLfloat l; GLfloat x; GLfloat y; GLfloat ff;
  GLfloat ratio;
  f->position[0] = 0.0;
  f->position[1] = 0.0;
  f->position[2] = 0.0;
  ratio = (GLfloat) (((GLfloat) surface_height) / ((GLfloat) surface_width));
  fov = P3_to_radians(fov);
  /* we use -front and -back cause our camera is looking along axe -z */
  f->points[ 2] = -front;
  f->points[ 5] = -front;
  f->points[ 8] = -front;
  f->points[11] = -front;
  f->points[14] = -back;
  f->points[17] = -back;
  f->points[20] = -back;
  f->points[23] = -back;
  if (ortho == P3_TRUE) {
    l = fov / 75.0;
    ratio = l * ratio;
    f->points[ 0] =    l; f->points[ 1] =  ratio;
    f->points[ 3] =   -l; f->points[ 4] =  ratio;
    f->points[ 6] =   -l; f->points[ 7] = -ratio;
    f->points[ 9] =    l; f->points[10] = -ratio;
    f->points[12] =    l; f->points[13] =  ratio;
    f->points[15] =   -l; f->points[16] =  ratio;
    f->points[18] =   -l; f->points[19] = -ratio;
    f->points[21] =    l; f->points[22] = -ratio;
    f->planes[ 0] =  0.0; f->planes[ 1] =   0.0; f->planes[ 2] =  1.0; f->planes[ 3] = -front;
    f->planes[ 4] =  0.0; f->planes[ 5] =   1.0; f->planes[ 6] =  0.0; f->planes[ 7] =  ratio;
    f->planes[ 8] =  0.0; f->planes[ 9] =  -1.0; f->planes[10] =  0.0; f->planes[11] = -ratio;
    f->planes[12] =  1.0; f->planes[13] =   0.0; f->planes[14] =  0.0; f->planes[15] =  l;
    f->planes[16] = -1.0; f->planes[17] =   0.0; f->planes[18] =  0.0; f->planes[19] = -l;
    f->planes[20] =  0.0; f->planes[21] =   0.0; f->planes[22] = -1.0; f->planes[23] = -back;
  } else {
    l = tan (fov / 2.0);
    y = back * l;    /* y >= 0 */
    x = y / ratio;   /* x >= 0 */
    f->points[12] =  x; f->points[13] =  y;
    f->points[15] = -x; f->points[16] =  y;
    f->points[18] = -x; f->points[19] = -y;
    f->points[21] =  x; f->points[22] = -y;
    f->planes[0] = 0.0; f->planes[1] = 0.0; f->planes[2] = 1.0; f->planes[3] = -front;
    ff = sqrt (y * y + back * back);
    f->planes[4] = 0.0; f->planes[5] = back / ff;      f->planes[ 6] = y / ff;       f->planes[ 7] = 0.0;
    f->planes[8] = 0.0; f->planes[9] = - f->planes[5]; f->planes[10] = f->planes[6]; f->planes[11] = 0.0;
    ff = sqrt (x * x + back * back);
    f->planes[12] = back / ff;       f->planes[13] = 0.0; f->planes[14] = x / ff;        f->planes[15] = 0.0;
    f->planes[16] = - f->planes[12]; f->planes[17] = 0.0; f->planes[18] = f->planes[14]; f->planes[19] = 0.0;
    f->planes[20] =  0.0;            f->planes[21] = 0.0; f->planes[22] = -1.0;          f->planes[23] = -back;
    y = front * l;
    x = y / ratio;
    f->points[0] =  x; f->points[ 1] =  y;
    f->points[3] = -x; f->points[ 4] =  y;
    f->points[6] = -x; f->points[ 7] = -y;
    f->points[9] =  x; f->points[10] = -y;
  }
//  P3_sphere_from_points (f->sphere, f->points, 8);
}

int P3_point_in_frustum (P3_frustum* f, GLfloat p[3]) {
  int i;
  GLfloat d;
  for (i = 0; i < 24; i += 4) {
    d = p[0] * f->planes[i] + p[1] * f->planes[i + 1] + p[2] * f->planes[i + 2] + f->planes[i + 3];
    if (d > 0.0) return P3_FALSE;
  }
  return P3_TRUE;
}

int P3_sphere_in_frustum (P3_frustum* f, GLfloat s[4]) {
  int i;
// TO DO box test 1rst ?
  for (i = 0; i < 24; i += 4) {
    if (s[0] * f->planes[i] + s[1] * f->planes[i + 1] + s[2] * f->planes[i + 2] + f->planes[i + 3] > s[3]) 
      return P3_FALSE; 
  }
  return P3_TRUE;
}

/*
void P3_frustum_dealloc (P3_frustum* f) {
  free(f);
}
*/

P3_frustum* P3_frustum_by_matrix (P3_frustum* r, P3_frustum* f, GLfloat* m) {
  int i;
  GLfloat scaling;
  /* copy */
  memcpy (r->points, f->points, 24 * sizeof (GLfloat));
  memcpy (r->planes, f->planes, 24 * sizeof (GLfloat));
  memcpy (r->position, f->position, 3 * sizeof (GLfloat));
  /* multiply */
  for (i = 0; i < 24; i += 3) { P3_point_by_matrix (r->points + i, m); }
  P3_point_by_matrix (r->position, m); 
  scaling = m[16];
  if (m[17] > scaling) { scaling = m[17]; }
  if (m[18] > scaling) { scaling = m[18]; }
  /* re-compute the normals */
  P3_face_normal (r->planes, r->points, r->points + 3, r->points + 9);
  P3_vector_set_length (r->planes, scaling);
  P3_face_normal (r->planes + 4, r->points + 12, r->points + 15, r->points);
  P3_vector_set_length (r->planes + 4, scaling);
  P3_face_normal (r->planes + 8, r->points + 9, r->points + 6, r->points + 21);
  P3_vector_set_length (r->planes + 8, scaling);
  P3_face_normal (r->planes + 12, r->points + 12, r->points, r->points + 21);
  P3_vector_set_length (r->planes + 12, scaling);
  P3_face_normal (r->planes + 16, r->points + 3, r->points + 15, r->points + 6);
  P3_vector_set_length (r->planes + 16, scaling);
  P3_face_normal (r->planes + 20, r->points + 15, r->points + 12, r->points + 18);
  P3_vector_set_length (r->planes + 20, scaling);
  /* re-compute the constants */
  r->planes[ 3] = -(r->planes[ 0] * r->points[ 0] + r->planes[ 1] * r->points[ 1] + r->planes[ 2] * r->points[ 2]);
  r->planes[ 7] = -(r->planes[ 4] * r->points[ 0] + r->planes[ 5] * r->points[ 1] + r->planes[ 6] * r->points[ 2]);
  r->planes[11] = -(r->planes[ 8] * r->points[ 6] + r->planes[ 9] * r->points[ 7] + r->planes[10] * r->points[ 8]);
  r->planes[15] = -(r->planes[12] * r->points[ 0] + r->planes[13] * r->points[ 1] + r->planes[14] * r->points[ 2]);
  r->planes[19] = -(r->planes[16] * r->points[ 6] + r->planes[17] * r->points[ 7] + r->planes[18] * r->points[ 8]);
  r->planes[23] = -(r->planes[20] * r->points[12] + r->planes[21] * r->points[13] + r->planes[22] * r->points[14]);
  /* sphere */
//  P3_point_by_matrix (r->sphere, m);
//  r->sphere[3] *= scaling;
  return r;
}

P3_frustum* P3_frustum_instance_into (P3_frustum* r, P3_frustum* f, P3_coordsys* oldc, P3_coordsys* newc) {
  GLfloat* matrix;
  GLfloat scalefactor[3] = { 1.0, 1.0, 1.0 };
  GLfloat scaling;
  int i;
  if (r == NULL) {
    /* return a new frustum */
    r = (P3_frustum*) malloc (sizeof (P3_frustum));
  }
  /* clone */
  memcpy (r, f, sizeof (P3_frustum));
  if (newc != oldc) {
    /* change position, points of coordsys */
    if (oldc != NULL) {
      matrix = P3_coordsys_get_root_matrix (oldc);
      for (i = 0; i < 24; i += 3) { P3_point_by_matrix (r->points + i, matrix); }
      scalefactor[0] *= matrix[16];
      scalefactor[1] *= matrix[17];
      scalefactor[2] *= matrix[18];
      /* this previous line worked only for iso scaling of coordsys matrix
       *   for(i = 0; i < 24; i += 4) { P3_vector_by_matrix(r->planes + i, matrix); }
       */
      P3_point_by_matrix (r->position, matrix);
      /* sphere */
//      P3_point_by_matrix (r->sphere, matrix);
    }
    if (newc != NULL) {
      matrix = P3_coordsys_get_inverted_root_matrix (newc);
      for (i = 0; i < 24; i += 3) { P3_point_by_matrix (r->points + i, matrix); }
      scalefactor[0] *= matrix[16];
      scalefactor[1] *= matrix[17];
      scalefactor[2] *= matrix[18];
      /* this previous line worked only for iso scaling of coordsys matrix
       *   for(i = 0; i < 24; i += 4) { P3_vector_by_matrix(r->planes + i, matrix); }
       */
      P3_point_by_matrix (r->position, matrix); 
      /* sphere */
//      P3_point_by_matrix (r->sphere, matrix);
    }
    /* re-compute the normals */
    scaling = scalefactor[0];
    if (scalefactor[1] > scaling) { scaling = scalefactor[1]; }
    if (scalefactor[2] > scaling) { scaling = scalefactor[2]; }
    P3_face_normal (r->planes, r->points, r->points + 3, r->points + 9);
    P3_vector_set_length (r->planes, scaling);
    P3_face_normal (r->planes + 4, r->points + 12, r->points + 15, r->points);
    P3_vector_set_length (r->planes + 4, scaling);
    P3_face_normal (r->planes + 8, r->points + 9, r->points + 6, r->points + 21);
    P3_vector_set_length (r->planes + 8, scaling);
    P3_face_normal (r->planes + 12, r->points + 12, r->points, r->points + 21);
    P3_vector_set_length (r->planes + 12, scaling);
    P3_face_normal (r->planes + 16, r->points + 3, r->points + 15, r->points + 6);
    P3_vector_set_length (r->planes + 16, scaling);
    P3_face_normal (r->planes + 20, r->points + 15, r->points + 12, r->points + 18);
    P3_vector_set_length (r->planes + 20, scaling);
    /* re-compute the constants */
    r->planes[ 3] = -(r->planes[ 0] * r->points[ 0] + r->planes[ 1] * r->points[ 1] + r->planes[ 2] * r->points[ 2]);
    r->planes[ 7] = -(r->planes[ 4] * r->points[ 0] + r->planes[ 5] * r->points[ 1] + r->planes[ 6] * r->points[ 2]);
    r->planes[11] = -(r->planes[ 8] * r->points[ 6] + r->planes[ 9] * r->points[ 7] + r->planes[10] * r->points[ 8]);
    r->planes[15] = -(r->planes[12] * r->points[ 0] + r->planes[13] * r->points[ 1] + r->planes[14] * r->points[ 2]);
    r->planes[19] = -(r->planes[16] * r->points[ 6] + r->planes[17] * r->points[ 7] + r->planes[18] * r->points[ 8]);
    r->planes[23] = -(r->planes[20] * r->points[12] + r->planes[21] * r->points[13] + r->planes[22] * r->points[14]);
    /* sphere */
//    r->sphere[3] *= scaling;
  }
  return r;
}

void P3_frustum_to_box (P3_frustum* frustum, GLfloat* box) {
  GLfloat* ptr = frustum->points;
  int i;
  memcpy (box,     ptr, 3 * sizeof (GLfloat));
  memcpy (box + 3, ptr, 3 * sizeof (GLfloat));
  ptr += 3;
  for (i = 0; i < 7; i++) {
    if      (ptr[0] < box[0]) box[0] = ptr[0];
    else if (ptr[0] > box[3]) box[3] = ptr[0];
    if      (ptr[1] < box[1]) box[1] = ptr[1];
    else if (ptr[1] > box[4]) box[4] = ptr[1];
    if      (ptr[2] < box[2]) box[2] = ptr[2];
    else if (ptr[2] > box[5]) box[5] = ptr[2];
    ptr += 3;
  }
}
