/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * xmesh_raypick.c
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "raypick.h"
#include "mesh.h"


/*============+
 | RAYPICKING |
 +============*/

void P3_xmesh_face_raypick (P3_xmesh* mesh, P3_xface* face, GLfloat* raydata, P3_raypick_data* data, P3_raypickable* parent) {
  GLfloat** ptr = mesh->vertex_coords;
  GLfloat z;
  int r;
  if (face->option & P3_FACE_NON_SOLID) return; 
  if (face->option & P3_FACE_DOUBLE_SIDED && data->option & P3_RAYPICK_CULL_FACE) data->option -= P3_RAYPICK_CULL_FACE;
  if (face->option & P3_FACE_QUAD) {
    /* quad */
    r = P3_quad_raypick (raydata, 
                         ptr[face->v1],
                         ptr[face->v2],
                         ptr[face->v3],
                         ptr[face->v4],
                         face->normal, data->option, &z);
  } else {
    /* triangle */
    r = P3_triangle_raypick (raydata, 
                             ptr[face->v1],
                             ptr[face->v2],
                             ptr[face->v3],
                             face->normal, data->option, &z);
  }
  if (r != P3_FALSE && (data->ret_csys == NULL || fabs (z) < fabs (data->result))) {
    data->result = z;
    data->ret_csys = parent;
    if (r == P3_RAYPICK_DIRECT) {
      memcpy (data->normal, face->normal, 3 * sizeof (GLfloat));
    } else if (r == P3_RAYPICK_INDIRECT) {
      if (face->option & P3_FACE_DOUBLE_SIDED) {
        data->normal[0] = - face->normal[0];
        data->normal[1] = - face->normal[1];
        data->normal[2] = - face->normal[2];
      } else {
        memcpy (data->normal, face->normal, 3 * sizeof (GLfloat));
      }
    }
  }
}

int P3_xmesh_face_raypick_b (P3_xmesh* mesh, P3_xface* face, GLfloat* raydata, P3_raypick_data* data) {
  GLfloat** ptr = mesh->vertex_coords;
  GLfloat z;
  if (face->option & P3_FACE_NON_SOLID) return P3_FALSE;
  if (face->option & P3_FACE_DOUBLE_SIDED && data->option & P3_RAYPICK_CULL_FACE) data->option -= P3_RAYPICK_CULL_FACE;
  if (face->option & P3_FACE_QUAD) {
    /* quad */
    if (P3_quad_raypick (raydata, 
                         ptr[face->v1],
                         ptr[face->v2],
                         ptr[face->v3],
                         ptr[face->v4],
                         face->normal, data->option, &z) != P3_FALSE) return P3_TRUE;
  } else {
    /* triangle */
    if (P3_triangle_raypick (raydata, 
                             ptr[face->v1],
                             ptr[face->v2],
                             ptr[face->v3],
                             face->normal, data->option, &z) != P3_FALSE) return P3_TRUE;
  }
  return P3_FALSE;
}

void P3_xmesh_node_raypick (P3_xmesh* mesh, P3_xnode* node, GLfloat* raydata, P3_raypick_data* data, P3_raypickable* parent) {
  int i;
  if (P3_sphere_raypick (raydata, node->sphere) == P3_TRUE) {
    /* raypick on faces */
    for (i = 0; i < node->nb_faces; i++) 
      P3_xmesh_face_raypick (mesh, node->faces[i], raydata, data, parent);
    /* recurse to children */
    for (i = 0; i < node->nb_child; i++) 
      P3_xmesh_node_raypick (mesh, node->child[i], raydata, data, parent);
  }
}

int P3_xmesh_node_raypick_b (P3_xmesh* mesh, P3_xnode* node, GLfloat* raydata, P3_raypick_data* data) {
  int i;
  if (P3_sphere_raypick (raydata, node->sphere) == P3_TRUE) {
    /* raypick on faces */
    for (i = 0; i < node->nb_faces; i++)
      if (P3_xmesh_face_raypick_b (mesh, node->faces[i], raydata, data) == P3_TRUE) { return P3_TRUE; }
    /* recurse to children */
    for (i = 0; i < node->nb_child; i++)
      if (P3_xmesh_node_raypick_b (mesh, node->child[i], raydata, data) == P3_TRUE) { return P3_TRUE; }
  }
  return P3_FALSE;
}

void P3_xmesh_raypick (P3_xmesh* mesh, GLfloat* raydata, P3_raypick_data* data, P3_raypickable* parent) {
  void* face = mesh->faces;
  void* max = mesh->faces + mesh->faces_size;
  while (face < max) {
    P3_xmesh_face_raypick (mesh, face, raydata, data, parent);
    face += P3_xmesh_face_size (mesh, face);
  }
}

int P3_xmesh_raypick_b (P3_xmesh* mesh, GLfloat* raydata, P3_raypick_data* data) {
  void* face = mesh->faces;
  void* max = mesh->faces + mesh->faces_size;
  while (face < max) {
    if (P3_xmesh_face_raypick_b (mesh, face, raydata, data) == P3_TRUE) return P3_TRUE;
    face += P3_xmesh_face_size (mesh, face);
  }
  return P3_FALSE;
}

void P3_xnode_get_faces (P3_xnode* node, GLfloat* sphere, P3_chunk* chunk) {
  int i;
  if (P3_sphere_distance_sphere (sphere, node->sphere) < 0.0) {
    P3_chunk_add (chunk, node->faces, node->nb_faces * sizeof (void*));
    for (i = 0; i < node->nb_child; i++) {
      P3_xnode_get_faces (node->child[i], sphere, chunk);
    }
  }
}


