/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * xmesh_render.c
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "coordsys.h"
#include "frustum.h"
#include "material.h"
#include "renderer.h"
#include "light.h"
#include "mesh.h"


extern P3_material* current_material;
extern P3_material* light_shader;
extern GLfloat white[4];
extern GLfloat black[4];
extern GLfloat transparency[4];
extern P3_renderer* renderer;


/*===========+
 | RENDERING |
 +===========*/

/*---------+
 | Generic |
 +---------*/

void P3_xmesh_option_activate (int option) {
  if (option & P3_MESH_NEVER_LIT) glDisable (GL_LIGHTING);
  if (option & P3_MESH_NORMALIZE) glEnable  (GL_NORMALIZE);
}

void P3_xmesh_option_inactivate (int option) {
  if (option & P3_MESH_NEVER_LIT) glEnable  (GL_LIGHTING);
  if (option & P3_MESH_NORMALIZE) glDisable (GL_NORMALIZE);
}

void P3_xface_option_activate (int option) {
  if (option & P3_FACE_DOUBLE_SIDED) { 
    glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
    glDisable (GL_CULL_FACE);
  }
}

void P3_xface_option_inactivate (int option) {
  if (option & P3_FACE_DOUBLE_SIDED) { 
    glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
    glEnable (GL_CULL_FACE);
  }
}

static void P3_xmesh_vertex_render (P3_xmesh* mesh, int index, int face_opt) {
  GLfloat* coord = mesh->vertex_coords[index];
  if (renderer->colors != NULL) {
    glColor4fv (renderer->colors[index]);
//    printf ("color %f, %f, %f, %f\n", renderer->colors[index][0], renderer->colors[index][1], renderer->colors[index][2], renderer->colors[index][3]);
  }
  if (mesh->option & P3_MESH_EMISSIVES) glColor4fv (mesh->vertex_emissives[index]);
  if (mesh->option & P3_MESH_TEXCOORDS) glTexCoord2fv (mesh->vertex_texcoords[index]);
  if (face_opt & P3_FACE_SMOOTHLIT) {
    if (mesh->option & P3_MESH_VERTEX_NORMALS)
      glNormal3fv (mesh->vertex_normals[index]);
    else
      glNormal3fv (mesh->vnormals + (coord - mesh->coords));
  }
  glVertex3fv (coord);
}

void P3_xmesh_triangle_render (P3_xmesh* mesh, P3_xface* face) {
  /* face normal */
  if (!(face->option & P3_FACE_SMOOTHLIT)) glNormal3fv (face->normal);
  /* render each vertex */
  P3_xmesh_vertex_render (mesh, face->v1, face->option);
  P3_xmesh_vertex_render (mesh, face->v2, face->option);
  P3_xmesh_vertex_render (mesh, face->v3, face->option);
}

void P3_xmesh_quad_render (P3_xmesh* mesh, P3_xface* face) {
  /* face normal */
  if (!(face->option & P3_FACE_SMOOTHLIT)) glNormal3fv (face->normal);
  /* render each vertex */
  P3_xmesh_vertex_render (mesh, face->v1, face->option);
  P3_xmesh_vertex_render (mesh, face->v2, face->option);
  P3_xmesh_vertex_render (mesh, face->v3, face->option);
  P3_xmesh_vertex_render (mesh, face->v4, face->option);
}

void P3_xmesh_pack_render (P3_xmesh* mesh) {
  P3_double_chain* chain;
  P3_chain* face;
  P3_xpack* pack;
  chain = (P3_double_chain*) (renderer->data->content + renderer->data->nb);
  while (1) {
    pack = (P3_xpack*) chain->data;
    face = (P3_chain*) (renderer->faces->content + chain->link);
    P3_material_activate (pack->material);
    P3_xface_option_activate (pack->option);
    if (pack->option & P3_FACE_TRIANGLE) {
      glBegin (GL_TRIANGLES);
      while (1) {
        P3_xmesh_triangle_render (mesh, face->data);
        if (face->next == -1) break;
        face = (P3_chain*) (renderer->faces->content + face->next);
      }
    } else if (pack->option & P3_FACE_QUAD) {
      glBegin (GL_QUADS);
      while (1) {
        P3_xmesh_quad_render (mesh, face->data);
        if (face->next == -1) break;
        face = (P3_chain*) (renderer->faces->content + face->next);
      }
    }
    glEnd ();
    P3_xface_option_inactivate (pack->option);
    if (chain->next == -1) return;
    chain = (P3_double_chain*) (renderer->data->content + chain->next);
  }
}


/*-------------+
 | CellShading |
 +-------------*/

static GLfloat P3_xmesh_vertex_compute_cell_shade (GLfloat* coord, GLfloat* normal, P3_list* lights, GLfloat shade) {
  P3_light* light;
  GLfloat ptr[3];
  GLfloat tmp;
  int i;
  for (i = 0; i < lights->nb; i++) {
    light = (P3_light*) P3_list_get (lights, i);
    if (light->option & P3_LIGHT_DIRECTIONAL) {
      tmp = - P3_vector_dot_product (normal, light->data);
    } else {
      ptr[0] = light->data[0] - coord[0];
      ptr[1] = light->data[1] - coord[1];
      ptr[2] = light->data[2] - coord[2];
      P3_vector_normalize (ptr);
      tmp = P3_vector_dot_product (normal, ptr);
    }
    if (tmp > 0.0) { shade += tmp; }
  }
  return shade;
}

static void P3_xmesh_vertex_render_cellshaded_smoothlit (P3_xmesh* mesh, int index, int face_opt, GLfloat* shades) {
  GLfloat* coord = mesh->vertex_coords[index];
  int n = coord - mesh->coords;
  GLfloat shade;
  if (renderer->colors != NULL) glColor4fv (renderer->colors[index]);
  if (mesh->option & P3_MESH_EMISSIVES) glColor4fv (mesh->vertex_emissives[index]);
  if (mesh->option & P3_MESH_VERTEX_NORMALS) {
    shade = shades[index];
    glNormal3fv (mesh->vertex_normals[index]);
  } else {
    shade = shades[n / 3];
    glNormal3fv (mesh->vnormals + n);
  }
  if (mesh->option & P3_MESH_TEXCOORDS) {
    glMultiTexCoord2fv (GL_TEXTURE0, mesh->vertex_texcoords[index]);
    glMultiTexCoord2f  (GL_TEXTURE1, shade, shade);
  } else {
    glTexCoord2f (shade, shade);
  }
  glVertex3fv (coord);
}

static void P3_xmesh_vertex_render_cellshaded (P3_xmesh* mesh, int index, int face_opt, GLfloat* shades, GLfloat* fnormal) {
  GLfloat* coord = mesh->vertex_coords[index];
  GLfloat shade;
  shade = P3_xmesh_vertex_compute_cell_shade (coord, fnormal, renderer->top_lights, 0.0);
  shade = P3_xmesh_vertex_compute_cell_shade (coord, fnormal, renderer->c_context->lights, shade);
  if (shade > 1.0) shade = 1.0;
  if (renderer->colors != NULL) glColor4fv (renderer->colors[index]);
  if (mesh->option & P3_MESH_EMISSIVES) glColor4fv (mesh->vertex_emissives[index]);
  if (mesh->option & P3_MESH_TEXCOORDS) {
    glMultiTexCoord2fv (GL_TEXTURE0, mesh->vertex_texcoords[index]);
    glMultiTexCoord2f  (GL_TEXTURE1, shade, shade);
  } else {
    glTexCoord2f (shade, shade);
  }
  if (face_opt & P3_FACE_SMOOTHLIT) {
    if (mesh->option & P3_MESH_VERTEX_NORMALS)
      glNormal3fv (mesh->vertex_normals[index]);
    else
      glNormal3fv (mesh->vnormals + (coord - mesh->coords));
  }
  glVertex3fv (coord);
}

static void P3_xmesh_triangle_render_cellshaded (P3_xmesh* mesh, P3_xface* face, GLfloat* shades) {
  if (face->option & P3_FACE_SMOOTHLIT) {
    P3_xmesh_vertex_render_cellshaded_smoothlit (mesh, face->v1, face->option, shades);
    P3_xmesh_vertex_render_cellshaded_smoothlit (mesh, face->v2, face->option, shades);
    P3_xmesh_vertex_render_cellshaded_smoothlit (mesh, face->v3, face->option, shades);
  } else {
    glNormal3fv (face->normal);
    P3_xmesh_vertex_render_cellshaded (mesh, face->v1, face->option, shades, face->normal);
    P3_xmesh_vertex_render_cellshaded (mesh, face->v2, face->option, shades, face->normal);
    P3_xmesh_vertex_render_cellshaded (mesh, face->v3, face->option, shades, face->normal);
  }
}

static void P3_xmesh_quad_render_cellshaded (P3_xmesh* mesh, P3_xface* face, GLfloat* shades) {
  if (face->option & P3_FACE_SMOOTHLIT) {
    P3_xmesh_vertex_render_cellshaded_smoothlit (mesh, face->v1, face->option, shades);
    P3_xmesh_vertex_render_cellshaded_smoothlit (mesh, face->v2, face->option, shades);
    P3_xmesh_vertex_render_cellshaded_smoothlit (mesh, face->v3, face->option, shades);
    P3_xmesh_vertex_render_cellshaded_smoothlit (mesh, face->v4, face->option, shades);
  } else {
    glNormal3fv (face->normal);
    P3_xmesh_vertex_render_cellshaded (mesh, face->v1, face->option, shades, face->normal);
    P3_xmesh_vertex_render_cellshaded (mesh, face->v2, face->option, shades, face->normal);
    P3_xmesh_vertex_render_cellshaded (mesh, face->v3, face->option, shades, face->normal);
    P3_xmesh_vertex_render_cellshaded (mesh, face->v4, face->option, shades, face->normal);
  }
}

void P3_xmesh_pack_render_cellshaded (P3_xmesh* mesh, GLfloat* shades) {
  P3_double_chain* chain;
  P3_chain* face;
  P3_xpack* pack;
  chain = (P3_double_chain*) (renderer->data->content + renderer->data->nb);
  while (1) {
    pack = (P3_xpack*) chain->data;
    face = (P3_chain*) (renderer->faces->content + chain->link);
    if (pack->material == NULL || pack->material->image == NULL) {
      P3_material_activate (light_shader);
    } else {
      P3_material_activate (pack->material);
      glActiveTexture (GL_TEXTURE1);
      glEnable (GL_TEXTURE_2D);
      glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      if (light_shader->id == 0) P3_material_init (light_shader);
      glBindTexture (GL_TEXTURE_2D, light_shader->id);
    }
    P3_xface_option_activate (pack->option);
    if (pack->option & P3_FACE_TRIANGLE) {
      glBegin (GL_TRIANGLES);
      while (1) {
        P3_xmesh_triangle_render_cellshaded (mesh, face->data, shades);
        if (face->next == -1) break;
        face = (P3_chain*) (renderer->faces->content + face->next);
      }
    } else if (pack->option & P3_FACE_QUAD) {
      glBegin (GL_QUADS);
      while (1) {
        P3_xmesh_quad_render_cellshaded (mesh, face->data, shades);
        if (face->next == -1) break;
        face = (P3_chain*) (renderer->faces->content + face->next);
      }
    }
    glEnd ();
    P3_xface_option_inactivate (pack->option);
    if (pack->material != NULL && pack->material->image != NULL) {
      glDisable (GL_TEXTURE_2D);
      glActiveTexture (GL_TEXTURE0);
    }
    if (chain->next == -1) return;
    chain = (P3_double_chain*) (renderer->data->content + chain->next);
  }
}


/*==========+
 | BATCHING |
 +==========*/

void P3_xpack_batch_face (void* object, P3_xpack* pack, void* face) {
  P3_double_chain* dchain;
  P3_chain* chain;
  int n;
  if (pack->data == -1) {
    /* batch pack */
    n = P3_chunk_register (renderer->data, sizeof (P3_double_chain));
    /* batch object */
    if (pack->option & P3_FACE_ALPHA) {
      /* alpha */
      if (renderer->last_alpha == -1) {
        /* object must be added */
        P3_renderer_batch (renderer->alpha, object, renderer->c_instance, n);
      } else {
        ((P3_chain*) (renderer->data->content + renderer->last_alpha))->next = n;
      }
      renderer->last_alpha = n;
    } else if (pack->option & P3_PACK_SECONDPASS) {
      /* second pass */
      if (renderer->last_secondpass == -1) {
        /* object must be added */
        P3_renderer_batch (renderer->secondpass, object, renderer->c_instance, n);
      } else {
        ((P3_chain*) (renderer->data->content + renderer->last_secondpass))->next = n;
      }
      renderer->last_secondpass = n;
    } else {
      /* opaque */
      if (renderer->last_opaque == -1) {
        /* object must be added */
        P3_renderer_batch (renderer->opaque, object, renderer->c_instance, n);
      } else {
        ((P3_chain*) (renderer->data->content + renderer->last_opaque))->next = n;
      }
      renderer->last_opaque = n;
    }
    dchain = (P3_double_chain*) (renderer->data->content + n);
    dchain->data = pack;
    /* register 1rst face */
    n = P3_chunk_register (renderer->faces, sizeof (P3_chain));
    dchain->link = n;
    pack->data = n;
    chain = (P3_chain*) (renderer->faces->content + n);
    chain->data = face;
  } else {
    n = P3_chunk_register (renderer->faces, sizeof (P3_chain));
    ((P3_chain*) (renderer->faces->content + pack->data))->next = n;
    ((P3_chain*) (renderer->faces->content + n))->data = face;
    pack->data = n;
  }
}

void P3_xmesh_face_batch (P3_xmesh* mesh, P3_xface* face) {
  if (mesh->option & P3_MESH_VERTEX_OPTIONS) {
    if (mesh->vertex_options[face->v1] & P3_VERTEX_INVISIBLE &&
        mesh->vertex_options[face->v2] & P3_VERTEX_INVISIBLE &&
        mesh->vertex_options[face->v3] & P3_VERTEX_INVISIBLE) {
      if (face->option & P3_FACE_TRIANGLE) 
        return;
      else if (mesh->vertex_options[face->v4] & P3_VERTEX_INVISIBLE) return;
    }
    if (mesh->vertex_options[face->v1] & P3_VERTEX_ALPHA ||
        mesh->vertex_options[face->v2] & P3_VERTEX_ALPHA ||
        mesh->vertex_options[face->v3] & P3_VERTEX_ALPHA ||
        (face->option & P3_FACE_QUAD && mesh->vertex_options[face->v3] & P3_VERTEX_ALPHA)) {
      P3_xpack_batch_face (mesh, P3_xpack_get_alpha (face->pack), face);
      return;
    }
  }
  P3_xpack_batch_face (mesh, face->pack, face);
}

void P3_xmesh_node_batch (P3_xmesh* mesh, P3_xnode* node, P3_frustum* frustum) {
  int i;
  /* frustum test */
  if (P3_sphere_in_frustum (frustum, node->sphere) == P3_TRUE) {
    /* batch all faces */
    for (i = 0; i < node->nb_faces; i++) P3_xmesh_face_batch (mesh, node->faces[i]);
    /* recurse */
    for (i = 0; i < node->nb_child; i++) P3_xmesh_node_batch (mesh, node->child[i], frustum);
  }
}

void P3_xmesh_batch_start (P3_instance* inst) {
  renderer->c_instance = inst;
  renderer->last_opaque     = -1;
  renderer->last_secondpass = -1;
  renderer->last_alpha      = -1;
  renderer->pack_start = renderer->data->nb;
  renderer->face_start = renderer->faces->nb;
}

void P3_xmesh_batch_end (void) {
  P3_double_chain* chain;
  P3_double_chain* max;
  P3_xpack* pack;
  /* end pack chain */
  if (renderer->last_opaque != -1)
    ((P3_chain*) (renderer->data->content + renderer->last_opaque))->next = -1;
  if (renderer->last_secondpass != -1)
    ((P3_chain*) (renderer->data->content + renderer->last_secondpass))->next = -1;
  if (renderer->last_alpha != -1)
    ((P3_chain*) (renderer->data->content + renderer->last_alpha))->next = -1;
  /* end face chain */
  chain = (P3_double_chain*) (renderer->data->content + renderer->pack_start);
  max   = (P3_double_chain*) (renderer->data->content + renderer->data->nb);
  while (chain < max) {
    pack = (P3_xpack*) chain->data;
    ((P3_chain*) (renderer->faces->content + pack->data))->next = -1;
    /* re initialize pack */
    pack->data = -1;
    chain++;
  }
}



