/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * Copyright (C) 2003 Bertrand 'blam' LAMY
 *****************************************/

/*========================+
 | GENERIC MESH FUNCTIONS |
 +========================*/

static void PyP3Fx_GetFx (P3_fx* fx, PyObject* args) {
  /* args are: 
   *   fx selection (not parsed by this function)
   *   fx params (alpha or color or None)
   *   [fx duration] (optional, if not given no transition)
   */
  GLfloat color[4];
  PyObject* params = PySequence_Fast_GET_ITEM (args, 1);
  if (PySequence_Size (args) == 2) {
    /* no transition */
    if (params == Py_None) {
      /* restore */
      if (fx->vertex_colors != NULL) fx->color = fx->register_color (fx->obj, white);
      fx->apply = P3_fx_restore;
    } else if (PySequence_Check (params)) {
      PY_TUPLE_FLOAT_TO_ARRAY_4 (color, params);
      /* set warfog colors */
      fx->color = fx->register_color (fx->obj, color);
      fx->apply = P3_fx_set_color;
    } else if (PyFloat_Check (params)) {
      /* set warfog alpha */
      fx->alpha = (GLfloat) PyFloat_AS_DOUBLE (params);
      fx->apply = P3_fx_set_alpha;
    }
  } else {
    /* with transition */
    P3_fx_transition_data* data = P3_fx_transition_data_new (fx);
    fx->data = data;
    fx->duration = (float) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
    fx->list = P3_get_list ();
    if (params == Py_None) {
      /* restore */
      if (fx->vertex_colors != NULL) fx->color = fx->register_color (fx->obj, white);
      fx->apply = P3_fx_transition_restore;
    } else if (PySequence_Check (params)) {
      PY_TUPLE_FLOAT_TO_ARRAY_4 (color, params);
      /* set warfog colors */
      fx->color = fx->register_color (fx->obj, color);
      fx->apply = P3_fx_transition_color;
    } else if (PyFloat_Check (params)) {
      /* set warfog alpha */
      fx->alpha = (GLfloat) PyFloat_AS_DOUBLE (params);
      fx->apply = P3_fx_transition_alpha;
    }
  }
}

static PyObject* PyP3XMesh_GetMaterials (P3_xmesh* a) {
  PyObject* tuple;
  int i;
  tuple = PyTuple_New (a->nb_materials);
  for (i = 0; i < a->nb_materials; i++) {
    if (a->materials[i] == NULL) {
      Py_INCREF (Py_None);
      PyTuple_SET_ITEM (tuple, i, Py_None);
    } else {
      Py_INCREF ((PyObject*) a->materials[i]);
      PyTuple_SET_ITEM (tuple, i, (PyObject*) a->materials[i]);
    }
  }
  return tuple;
}

static void PyP3XMesh_SetMaterials (P3_xmesh* a, PyObject* tuple) {
  int i;
  a->nb_materials = PySequence_Size (tuple);
  if (a->nb_materials > 0) {
    a->materials = (P3_material**) malloc (a->nb_materials * sizeof (P3_material*));
    for (i = 0; i < a->nb_materials; i++) {
      a->materials[i] = (P3_material*) PySequence_Fast_GET_ITEM (tuple, i);
      if ((PyObject*) a->materials[i] == Py_None) {
        a->materials[i] = NULL;
      } else {
        Py_INCREF ((PyObject*) a->materials[i]);
      }
    }
  } else {
    a->materials = NULL;
  }
}

/*======+
 | MESH |
 +======*/

/*---------+
 | Methods |
 +---------*/

static int PyP3Mesh_Init (P3_mesh* a, PyObject* args, PyObject* kwds) {
  P3_mesh_new (a);
  return 0;
}

static void PyP3Mesh_Dealloc (P3_mesh* a) {

  printf ("!!! Mesh %p DEALLOC\n", a);

  PyObject_GC_UnTrack ((PyObject*) a);
  P3_mesh_dealloc (a);
  a->ob_type->tp_free ((PyObject*) a); 
}

static int PyP3Mesh_Traverse (P3_mesh* a, visitproc visit, void* arg) {
  int err;
  int i;
  for (i = 0; i < a->nb_materials; i++) {
    if (a->materials[i] != NULL) {
      err = visit ((PyObject*) a->materials[i], arg);
      if (err) { return err; }
    }
  }
  return 0;
}

static int PyP3Mesh_Clear (P3_mesh* a) {
  int i;

// HACK
printf ("INFO PyP3Mesh_Clear\n");

  for (i = 0; i < a->nb_materials; i++) {
    Py_XDECREF (a->materials[i]);
  }
  free (a->materials);
  a->materials = NULL;
  a->nb_materials = 0;
  return 0;
}

static PyObject* PyP3Mesh_GetState (P3_mesh* a) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  P3_mesh_get_data (a, chunk);
  tuple = PyTuple_New (2);
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
  PyTuple_SET_ITEM (tuple, 1, PyP3XMesh_GetMaterials ((P3_xmesh*) a));
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3Mesh_SetState (P3_mesh* a, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  chunk->content = PyString_AS_STRING (PySequence_Fast_GET_ITEM (args, 0));
  PyP3XMesh_SetMaterials ((P3_xmesh*) a, PySequence_Fast_GET_ITEM (args, 1));
  P3_mesh_set_data (a, chunk);
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

void P3_xmesh_face_force_neighbor (P3_xmesh*, P3_face*, P3_face*);

static PyObject* PyP3Mesh_SetNeighbor (P3_xmesh* a, PyObject* args) {
  P3_xmesh_face_force_neighbor (a, PySequence_Fast_GET_ITEM (args, 0), PySequence_Fast_GET_ITEM (args, 1));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_ToFaces (P3_xmesh* a, PyObject* arg) {
  P3_xmesh_to_faces (a, (P3_world*) arg);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_BuildTree (P3_mesh* a) {
  P3_mesh_build_tree (a);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_OptimizeTree (P3_mesh* a, PyObject* args) {
  GLfloat param;
  int mode;
  mode = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1));
  if (mode == 0) {
    param = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
  }
  P3_mesh_optimize_tree (a, 
                         (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 0)),
                         mode, param);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_BuildDisplayList (P3_mesh* a) {
  P3_mesh_set_xtra1_display_lists (a);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_BuildCellShading (P3_mesh* a, PyObject* args) {
  P3_material* shader;
  PyObject* tuple;
  GLfloat color[4];
  tuple = PySequence_Fast_GET_ITEM (args, 1);
  PY_TUPLE_FLOAT_TO_ARRAY_4 (color, tuple);
  shader = (P3_material*) PySequence_Fast_GET_ITEM (args, 0);
  if ((PyObject*) shader == Py_None) { shader = NULL; }
  P3_mesh_set_xtra1_cell_shading (a, shader, color, (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2)));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_BuildSphere (P3_mesh* a) {
  P3_mesh_set_xtra2_sphere (a);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_BuildFaceList (P3_mesh* a) {
  P3_mesh_set_xtra2_face_list (a);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_InitFx (P3_mesh* a) {
  P3_mesh_check_fx (a);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Mesh_VertexFx (P3_mesh* a, PyObject* args) {
  /* args are: 
   *   fx selection (sphere or cylinderY or layerY or None)
   *   fx params (alpha or color or None)
   *   [fx duration] (optional, if not given no transition)
   */
  PyObject* select = PySequence_Fast_GET_ITEM (args, 0);
  P3_fx fx;
  GLfloat sphere[4];
  int nb;
  fx.obj = a;
  fx.register_color = (register_color_func) P3_xmesh_register_color;
  fx.vertex_warfogs = a->vertex_warfogs;
  fx.vertex_colors  = a->vertex_diffuses;
  fx.vertex_options = a->vertex_options;
  PyP3Fx_GetFx (&fx, args);
  if (select == Py_None) {
    P3_mesh_fx_all (a, &fx);
  } else {
    nb = PySequence_Size (select);
    if (nb == 2) {
      sphere[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (select, 0));
      sphere[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (select, 1));
      P3_mesh_fx_in_layerY (a, &fx, sphere);
    } else if (nb == 3) {
      PY_TUPLE_FLOAT_TO_ARRAY_3 (sphere, select);
      P3_mesh_fx_in_cylinderY (a, &fx, sphere);
    } else {
      PY_TUPLE_FLOAT_TO_ARRAY_4 (sphere, select);
      P3_mesh_fx_in_sphere (a, &fx, sphere);
    }
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyMethodDef PyP3Mesh_Methods[] = {
  { "_getstate",           (PyCFunction) PyP3Mesh_GetState,         METH_NOARGS },
  { "_setstate",           (PyCFunction) PyP3Mesh_SetState,         METH_O },
  { "to_faces",            (PyCFunction) PyP3Mesh_ToFaces,          METH_O },
  { "_build_tree",         (PyCFunction) PyP3Mesh_BuildTree,        METH_NOARGS },
  { "_optimize_tree",      (PyCFunction) PyP3Mesh_OptimizeTree,     METH_VARARGS },
  { "_build_display_list", (PyCFunction) PyP3Mesh_BuildDisplayList, METH_NOARGS },
  { "_build_cell_shading", (PyCFunction) PyP3Mesh_BuildCellShading, METH_VARARGS },
  { "_build_sphere",       (PyCFunction) PyP3Mesh_BuildSphere,      METH_NOARGS },
  { "_build_face_list",    (PyCFunction) PyP3Mesh_BuildFaceList,    METH_NOARGS },
  { "_set_neighbor",       (PyCFunction) PyP3Mesh_SetNeighbor,      METH_VARARGS },
  { "init_fx",             (PyCFunction) PyP3Mesh_InitFx,           METH_NOARGS },
  { "vertex_fx",           (PyCFunction) PyP3Mesh_VertexFx,         METH_VARARGS },
  { NULL, NULL } /* sentinel */
};

/*---------+
 | Get Set |
 +---------*/

static PyObject* PyP3Mesh_GetLit (P3_mesh* a, void* context) {
  if (a->option & P3_MESH_NEVER_LIT) {
    return PyInt_FromLong (0);
  } else {
    return PyInt_FromLong (1);
  } 
}

static int PyP3Mesh_SetLit (P3_mesh* a, PyObject* value, void* context) {
  if (PyObject_IsTrue (value) == 1) {
    a->option &= ~P3_MESH_NEVER_LIT;
  } else {
    a->option |= P3_MESH_NEVER_LIT;
  }
  return 0;
}

static PyObject* PyP3Mesh_GetShadowCast (P3_mesh* a, void* context) {
  if (a->option & P3_MESH_SHADOW_CAST) {
    return PyInt_FromLong (1);
  } else {
    return PyInt_FromLong (0);
  } 
}

static int PyP3Mesh_SetShadowCast (P3_mesh* a, PyObject* value, void* context) {
  if (PyObject_IsTrue (value) == 1) {
    if (a->option & P3_MESH_NEIGHBORS)
      a->option |= P3_MESH_SHADOW_CAST;
    else
      P3_error ("Mesh must have face neighbors to cast shadow");
  } else {
    a->option &= ~P3_MESH_SHADOW_CAST;
  }
  return 0;
}

PY_GET_SET_ON_OPTION (Mesh, P3_mesh*, Normalize, P3_MESH_NORMALIZE)

static PyGetSetDef PyP3Mesh_GetSets[] = {
  { "normalize",   (getter) PyP3Mesh_GetNormalize,  (setter) PyP3Mesh_SetNormalize,  NULL },
  { "shadow_cast", (getter) PyP3Mesh_GetShadowCast, (setter) PyP3Mesh_SetShadowCast, NULL },
  { "lit",         (getter) PyP3Mesh_GetLit,        (setter) PyP3Mesh_SetLit,        NULL },
  { NULL }
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3Mesh_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Mesh",
  sizeof(P3_mesh),
  0,
  (destructor) PyP3Mesh_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3Mesh_Traverse,/* tp_traverse */
  (inquiry) PyP3Mesh_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Mesh_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Mesh_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Mesh_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};


/*==========+
 | MESH LOD |
 +==========*/

/*---------+
 | Get Set |
 +---------*/

static PyObject* PyP3MeshLOD_GetMeshes (P3_mesh_LOD* a, void* context) {
  PyObject* tuple;
  int i;
  tuple = PyTuple_New (a->nb_meshes);
  for (i = 0; i < a->nb_meshes; i++) {
    Py_INCREF ((PyObject*) a->meshes[i]);
    PyTuple_SET_ITEM (tuple, i, (PyObject*) a->meshes[i]);
  }
  return tuple;
}

static int PyP3MeshLOD_SetMeshes (P3_mesh_LOD* a, PyObject* value, void* context) {
  int i;
  for (i = 0; i < a->nb_meshes; i++) Py_DECREF (a->meshes[i]);
  a->nb_meshes = PySequence_Size (value);
  a->meshes = (P3_mesh**) realloc (a->meshes, a->nb_meshes * sizeof (P3_mesh*));
  for (i = 0; i < a->nb_meshes; i++) {
    a->meshes[i] = (P3_mesh*) PySequence_Fast_GET_ITEM (value, i);
    Py_INCREF ((PyObject*) a->meshes[i]);
  }
  return 0;
}

PY_GET_SET_ON_FLOAT (MeshLOD, P3_mesh_LOD*, LODFactor, LOD_factor)

static PyGetSetDef PyP3MeshLOD_GetSets[] = {
  { "shapes",     (getter) PyP3MeshLOD_GetMeshes,    (setter) PyP3MeshLOD_SetMeshes,    NULL },
  { "LOD_factor", (getter) PyP3MeshLOD_GetLODFactor, (setter) PyP3MeshLOD_SetLODFactor, NULL },
  { NULL }
};

/*---------+
 | Methods |
 +---------*/

static int PyP3MeshLOD_Init (P3_mesh_LOD* a, PyObject* args, PyObject* kwds) {
  P3_mesh_LOD_new (a);
  return 0;
}

static void PyP3MeshLOD_Dealloc (P3_mesh_LOD* a) {
  int i;
  PyObject_GC_UnTrack ((PyObject*) a);
  for (i = 0; i < a->nb_meshes; i++) Py_DECREF ((PyObject*) a->meshes[i]);
  P3_mesh_LOD_dealloc (a);
  a->ob_type->tp_free ((PyObject*) a); 
}

static int PyP3MeshLOD_Traverse (P3_mesh_LOD* a, visitproc visit, void* arg) {
  int err;
  int i;
  for (i = 0; i < a->nb_meshes; i++) {
    err = visit ((PyObject*) a->meshes[i], arg);
    if (err) return err;
  }
  return 0;
}

static int PyP3MeshLOD_Clear (P3_mesh_LOD* a) {
  int i;
  for (i = 0; i < a->nb_meshes; i++) Py_DECREF (a->meshes[i]);
  free (a->meshes);
  a->meshes = NULL;
  a->nb_meshes = 0;
  return 0;
}

static PyObject* PyP3MeshLOD_GetState (P3_mesh_LOD* a) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  P3_mesh_LOD_get_data (a, chunk);
  tuple = PyTuple_New (2);
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
  PyTuple_SET_ITEM (tuple, 1, PyP3MeshLOD_GetMeshes (a, NULL));
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3MeshLOD_SetState (P3_mesh_LOD* a, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  chunk->content = PyString_AS_STRING (PySequence_Fast_GET_ITEM (args, 0));
  P3_mesh_LOD_set_data (a, chunk);
  PyP3MeshLOD_SetMeshes (a, PySequence_Fast_GET_ITEM (args, 1), NULL);
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyMethodDef PyP3MeshLOD_Methods[] = {
  { "_getstate", (PyCFunction) PyP3MeshLOD_GetState, METH_NOARGS },
  { "_setstate", (PyCFunction) PyP3MeshLOD_SetState, METH_O },
  { NULL, NULL } /* sentinel */
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3MeshLOD_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._MeshLOD",
  sizeof(P3_mesh_LOD),
  0,
  (destructor) PyP3MeshLOD_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3MeshLOD_Traverse,/* tp_traverse */
  (inquiry) PyP3MeshLOD_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3MeshLOD_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3MeshLOD_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3MeshLOD_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};


/*==================+
 | MESH FX INSTANCE |
 +==================*/

/*---------+
 | Get Set |
 +---------*/

static PyObject* PyP3MeshFXInstance_GetMesh (P3_mesh_fxinstance* a, void* context) {
  if (a->mesh == NULL) {
    Py_INCREF (Py_None);
    return Py_None;
  } else {
    Py_INCREF ((PyObject*) a->mesh);
    return (PyObject*) a->mesh;
  }
}

static int PyP3MeshFXInstance_SetMesh (P3_mesh_fxinstance* a, PyObject* value, void* context) {
  if (value == Py_None) 
    value = NULL;
  else
    Py_INCREF (value);
  P3_mesh_fxinstance_set_mesh (a, (P3_mesh*) value);
  return 0;
}

static PyGetSetDef PyP3MeshFXInstance_GetSets[] = {
  { "shape", (getter) PyP3MeshFXInstance_GetMesh, (setter) PyP3MeshFXInstance_SetMesh, NULL },
  { NULL }
};

/*---------+
 | Methods |
 +---------*/

static int PyP3MeshFXInstance_Init (P3_mesh_fxinstance* a, PyObject* args, PyObject* kwds) {
  P3_mesh_fxinstance_new (a);
  return 0;
}

static void PyP3MeshFXInstance_Dealloc (P3_mesh_fxinstance* a) {
  PyObject_GC_UnTrack ((PyObject*) a);
  Py_DECREF ((PyObject*) a->mesh);
  P3_mesh_fxinstance_dealloc (a);
  a->ob_type->tp_free ((PyObject*) a); 
}

static int PyP3MeshFXInstance_Traverse (P3_mesh_fxinstance* a, visitproc visit, void* arg) {
  int err;
  err = visit ((PyObject*) a->mesh, arg);
  if (err) return err;
  return 0;
}

static int PyP3MeshFXInstance_Clear (P3_mesh_fxinstance* a) {
  Py_DECREF (a->mesh);
  a->mesh = NULL;
  free (a->vertex_options);
  free (a->vertex_warfogs);
  a->vertex_options = NULL;
  a->vertex_warfogs = NULL;
  return 0;
}

static PyObject* PyP3MeshFXInstance_VertexFx (P3_mesh_fxinstance* a, PyObject* args) {
  /* args are: 
   *   fx selection (sphere or cylinderY or layerY or None)
   *   fx params (alpha or color or None)
   *   [fx duration] (optional, if not given no transition)
   */
  PyObject* select = PySequence_Fast_GET_ITEM (args, 0);
  P3_fx fx;
  GLfloat sphere[4];
  int nb;
  fx.obj = a;
  fx.register_color = (register_color_func) P3_mesh_fxinstance_register_color;
  fx.vertex_warfogs = a->vertex_warfogs;
  fx.vertex_colors  = a->mesh->vertex_diffuses;
  fx.vertex_options = a->vertex_options;
  PyP3Fx_GetFx (&fx, args);
  if (select == Py_None) {
    P3_mesh_fx_all (a->mesh, &fx);
  } else {
    nb = PySequence_Size (select);
    if (nb == 2) {
      sphere[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (select, 0));
      sphere[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (select, 1));
      P3_mesh_fx_in_layerY (a->mesh, &fx, sphere);
    } else if (nb == 3) {
      PY_TUPLE_FLOAT_TO_ARRAY_3 (sphere, select);
      P3_mesh_fx_in_cylinderY (a->mesh, &fx, sphere);
    } else {
      PY_TUPLE_FLOAT_TO_ARRAY_4 (sphere, select);
      P3_mesh_fx_in_sphere (a->mesh, &fx, sphere);
    }
  }
  Py_INCREF (Py_None);
  return Py_None;
}

/*
static PyObject* PyP3MeshLOD_GetState (P3_mesh_LOD* a) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  P3_mesh_LOD_get_data (a, chunk);
  tuple = PyTuple_New (2);
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
  PyTuple_SET_ITEM (tuple, 1, PyP3MeshLOD_GetMeshes (a, NULL));
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3MeshLOD_SetState (P3_mesh_LOD* a, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  chunk->content = PyString_AS_STRING (PySequence_Fast_GET_ITEM (args, 0));
  P3_mesh_LOD_set_data (a, chunk);
  PyP3MeshLOD_SetMeshes (a, PySequence_Fast_GET_ITEM (args, 1), NULL);
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}
*/
static PyMethodDef PyP3MeshFXInstance_Methods[] = {
//  { "_getstate", (PyCFunction) PyP3MeshLOD_GetState, METH_NOARGS },
//  { "_setstate", (PyCFunction) PyP3MeshLOD_SetState, METH_O },
  { "vertex_fx", (PyCFunction) PyP3MeshFXInstance_VertexFx, METH_VARARGS },
  { NULL, NULL } /* sentinel */
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3MeshFXInstance_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._MeshFXInstance",
  sizeof(P3_mesh_fxinstance),
  0,
  (destructor) PyP3MeshFXInstance_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3MeshFXInstance_Traverse,/* tp_traverse */
  (inquiry) PyP3MeshFXInstance_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3MeshFXInstance_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3MeshFXInstance_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3MeshFXInstance_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};

