/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * Copyright (C) 2003 Bertrand 'blam' LAMY
 *****************************************/

/*=======+
 | WORLD |
 +=======*/

/*---------+
 | Methods |
 +---------*/

static PyObject* PyP3World_GetBox (P3_world* w) {
  PyObject* o;
  P3_point* p;
  int edited = P3_FALSE;
  GLfloat box[6] = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
  P3_object_get_box ((P3_any_object*) w, NULL, box, &edited);
  o = PyTuple_New (2);
  p = (P3_point*) PyP3Point_Type.tp_alloc (&PyP3Point_Type, 0);
  p->parent = (P3_coordsys*) w;
  Py_INCREF ((PyObject*) w);
  memcpy (p->coord, box, 3 * sizeof (GLfloat));
  PyTuple_SET_ITEM (o, 0, (PyObject*) p);
  p = (P3_point*) PyP3Point_Type.tp_alloc (&PyP3Point_Type, 0);
  p->parent = (P3_coordsys*) w;
  Py_INCREF ((PyObject*) w);
  memcpy (p->coord, box + 3, 3 * sizeof (GLfloat));
  PyTuple_SET_ITEM (o, 1, (PyObject*) p);
  return o;
}

static PyObject* PyP3World_GetDimensions (P3_world* w) {
  PyObject* tuple;
  int edited = P3_FALSE;
  GLfloat box[6] = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
  P3_object_get_box ((P3_any_object*) w, NULL, box, &edited);
  tuple = PyTuple_New (3);
  PyTuple_SET_ITEM (tuple, 0, PyFloat_FromDouble ((double) box[3] - box[0]));
  PyTuple_SET_ITEM (tuple, 1, PyFloat_FromDouble ((double) box[4] - box[1]));
  PyTuple_SET_ITEM (tuple, 2, PyFloat_FromDouble ((double) box[5] - box[2]));
  return tuple;
}

static PyObject* PyP3World_SetDimensions (P3_world* w, PyObject* args) {
  GLfloat dim[3];
  dim[0] = PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 0));
  dim[1] = PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1));
  dim[2] = PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
  P3_world_set_dimensions (w, dim);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3World_GetRaypickContext (P3_world* w, PyObject* args) {
  PyP3RaypickContext* rc; 
  P3_coordsys* csys;
  GLfloat* coord;
  GLfloat sphere[4];
//  PyObject* o;
//  PyObject* p;
  /* args are: (point, radius, [raypick_context]) */
  if (PySequence_Size (args) == 3) {
    rc = (PyP3RaypickContext*) PySequence_Fast_GET_ITEM (args, 2);
  } else {
    rc = (PyP3RaypickContext*) PyP3RaypickContext_Type.tp_alloc (&PyP3RaypickContext_Type, 0);
    rc->chunk = P3_chunk_new ();
  }
  PyP3_GetPosition (PySequence_Fast_GET_ITEM (args, 0), &coord, &csys);
  memcpy (sphere, coord, 3 * sizeof (GLfloat));
/*
  o = PySequence_Fast_GET_ITEM (args, 0);
  p = PyObject_GetAttrString (o, "x");
  sphere[0] = (GLfloat) PyFloat_AS_DOUBLE (p);
  Py_DECREF (p);
  p = PyObject_GetAttrString (o, "y");
  sphere[1] = (GLfloat) PyFloat_AS_DOUBLE (p);
  Py_DECREF (p);
  p = PyObject_GetAttrString (o, "z");
  sphere[2] = (GLfloat) PyFloat_AS_DOUBLE (p);
  Py_DECREF (p);
  p = PyObject_GetAttrString (o, "parent");
  if (p != Py_None) P3_point_by_matrix (sphere, P3_coordsys_get_root_matrix ((P3_coordsys*) p));
  Py_DECREF (p);
*/
  sphere[3] = PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1));
  if (csys != NULL) P3_point_by_matrix (sphere, P3_coordsys_get_root_matrix (csys));
  P3_get_raypick_context ((P3_any_object*) w, NULL, sphere, sphere, rc->chunk);
  return (PyObject*) rc;
}

static PyObject* PyP3World_BuildMesh (P3_world* w, PyObject* args) {
  P3_xmesh_from_world ((P3_xmesh*) PySequence_Fast_GET_ITEM (args, 0), w, 
    P3_to_radians ((GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1))),
    (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 2)));
  Py_INCREF (Py_None);
  return Py_None;
}

/*
static PyObject* PyP3World_BuildMorph (P3_world* w) {
  P3_morph_data* data;
  data = (P3_morph_data*) PyObject_CallMethod (P3Module, "new_morph", NULL, NULL);
    if (PyErr_Occurred() != NULL) { PyErr_Print(); }
  return (PyObject*) P3_morph_data_from_world (data, w);
}
*/

static int PyP3World_Init (P3_world* a, PyObject* args, PyObject* kwds) {
  P3_world_new (a);
  return 0;
}

static void PyP3World_Dealloc (P3_world* a) {
  PyObject_GC_UnTrack ((PyObject*) a);
  Py_XDECREF (a->shape);
  Py_XDECREF (a->parent);
  Py_XDECREF (a->children);
  Py_XDECREF (a->atmosphere);
  a->ob_type->tp_free ((PyObject*) a); 
}

static int PyP3World_Traverse (P3_world* a, visitproc visit, void* arg) {
  int err;
  if (a->shape != NULL) {
    err = visit ((PyObject*) a->shape, arg);
    if (err) { return err; }
  }
  if (a->parent != NULL) {
    err = visit ((PyObject*) a->parent, arg);
    if (err) { return err; }
  }
  if (a->children != NULL) {
    err = visit ((PyObject*) a->children, arg);
    if (err) { return err; }
  }
  if (a->atmosphere != NULL) {
    err = visit ((PyObject*) a->atmosphere, arg);
    if (err) { return err; }
  }
  return 0;
}

static int PyP3World_Clear (P3_world* a) {
  Py_XDECREF (a->shape);
  a->shape = NULL;
  Py_XDECREF (a->parent);
  a->parent = NULL;
  Py_XDECREF (a->children);
  a->children = NULL;
  Py_XDECREF (a->atmosphere);
  a->atmosphere = NULL;
  return 0;
}

static PyObject* PyP3World_GetState (P3_world* a) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  P3_world_get_data (a, chunk);
  tuple = PyTuple_New (5);
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
// HACK because previous object saved their parent
  Py_INCREF (Py_None);
  PyTuple_SET_ITEM (tuple, 1, Py_None);
  if (a->shape == NULL) {
    Py_INCREF (Py_None);
    PyTuple_SET_ITEM (tuple, 2, Py_None);
  } else {
    Py_INCREF ((PyObject*) a->shape);
    PyTuple_SET_ITEM (tuple, 2, (PyObject*) a->shape);
  }
  if (a->children == NULL) {
    Py_INCREF (Py_None);
    PyTuple_SET_ITEM (tuple, 3, Py_None);
  } else {
    Py_INCREF ((PyObject*) a->children);
    PyTuple_SET_ITEM (tuple, 3, (PyObject*) a->children);
  }
  if (a->atmosphere == NULL) {
    Py_INCREF (Py_None);
    PyTuple_SET_ITEM (tuple, 4, Py_None);
  } else {
    Py_INCREF ((PyObject*) a->atmosphere);
    PyTuple_SET_ITEM (tuple, 4, (PyObject*) a->atmosphere);
  }
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3World_SetState (P3_world* a, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* o;
  int i;
  o = PySequence_Fast_GET_ITEM (args, 0);
  chunk->content = PyString_AS_STRING (o);
  P3_world_set_data (a, chunk);
  a->shape = (P3_any_object*) PySequence_Fast_GET_ITEM (args, 2);
  if ((PyObject*) a->shape == Py_None) {
    a->shape = NULL;
  } else {
    Py_INCREF ((PyObject*) a->shape);
  }
  a->children = (P3_children) PySequence_Fast_GET_ITEM (args, 3);
  if ((PyObject*) a->children == Py_None) {
    a->children = NULL;
  } else {
    Py_INCREF ((PyObject*) a->children);
  }
  a->atmosphere = (P3_atmosphere*) PySequence_Fast_GET_ITEM (args, 4);
  if ((PyObject*) a->atmosphere == Py_None) {
    a->atmosphere = NULL;
  } else {
    Py_INCREF ((PyObject*) a->atmosphere);
  }
  for (i = 0; i < P3_children_size (a->children); i++) {
    o = (PyObject*) P3_children_get (a->children, i);
    PyObject_SetAttrString (o, "_parent", (PyObject*) a);
  }
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyMethodDef PyP3World_Methods[] = {

//  PYP3_COORDSYS_FUNCS,

  { "_getstate",      (PyCFunction) PyP3World_GetState,          METH_NOARGS },
  { "_setstate",      (PyCFunction) PyP3World_SetState,          METH_O },
  { "set_dimensions", (PyCFunction) PyP3World_SetDimensions,     METH_VARARGS },
  { "get_dimensions", (PyCFunction) PyP3World_GetDimensions,     METH_NOARGS },
  { "get_box",        (PyCFunction) PyP3World_GetBox,            METH_NOARGS },
  { "_to_shape",      (PyCFunction) PyP3World_BuildMesh,         METH_VARARGS },
//  { "_to_morph",      (PyCFunction) PyP3World_BuildMorph,        METH_NOARGS },
  { "RaypickContext", (PyCFunction) PyP3World_GetRaypickContext, METH_VARARGS },
  { NULL, NULL } /* sentinel */
};

/*---------+
 | Get Set |
 +---------*/

PY_GET_SET_ON_OBJECT (World, P3_world*, Atmosphere, atmosphere, P3_atmosphere*)
PY_GET_SET_ON_OBJECT (World, P3_world*, Children, children, PyObject*)

static PyGetSetDef PyP3World_GetSets[] = {
//  PYP3_CHILD_GETSETS,
//  PYP3_SOLID_GETSETS,
//  PYP3_COORDSYS_GETSETS,
//  PYP3_VISIBLE_GETSETS,

  { "children",   (getter) PyP3World_GetChildren,   (setter) PyP3World_SetChildren,   NULL },
  { "atmosphere", (getter) PyP3World_GetAtmosphere, (setter) PyP3World_SetAtmosphere, NULL },

//  { "shape",      (getter) PyP3Volume_GetShape,     (setter) PyP3Volume_SetShape,     NULL },

  { NULL }
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3World_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._World",
  sizeof(P3_world),
  0,
  (destructor) PyP3World_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3World_Traverse,/* tp_traverse */
  (inquiry) PyP3World_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3World_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3World_GetSets,/* tp_getset */
  &PyP3Volume_Type,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3World_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};

