/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * sprite.c
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "util.h"
#include "renderer.h"
#include "material.h"
#include "coordsys.h"
#include "math3d.h"
#include "sprite.h"


extern P3_renderer* renderer;
extern GLfloat      white[4];


static GLfloat sprite_matrix[16] = { 1.0, 0.0, 0.0, 0.0, 
                                     0.0, 1.0, 0.0, 0.0, 
                                     0.0, 0.0, 1.0, 0.0, 
                                     0.0, 0.0, 0.0, 1.0 };
static GLfloat cylinder_matrix[16] = { 1.0, 0.0, 0.0, 0.0, 
                                       0.0, 1.0, 0.0, 0.0, 
                                       0.0, 0.0, 1.0, 0.0, 
                                       0.0, 0.0, 0.0, 1.0 };


/*===============*
 * SPRITE SPHERE *
 *===============*/

P3_class P3_class_sprite = { 
  P3_ID_SPRITE,
  (batch_func)     P3_sprite_batch,
  (render_func)    P3_sprite_render,
  (shadow_func)    0,
  (raypick_func)   0,
  (raypick_b_func) 0,
};

P3_sprite* P3_sprite_new (P3_sprite* s) {
  if (s == NULL) {
    s = (P3_sprite*) malloc (sizeof (P3_sprite));
  }
  s->class = &P3_class_sprite;
  s->parent = NULL;
  s->option = 0;
  s->position[0] = 0.0;
  s->position[1] = 0.0;
  s->position[2] = 0.0;
  s->color[0] = 1.0;
  s->color[1] = 1.0;
  s->color[2] = 1.0;
  s->color[3] = 1.0;
  s->material = NULL;
  s->w = 0.5;
  s->h = 0.5;
  return s;
}

void P3_sprite_batch (P3_sprite* s, P3_instance* inst) {
  if (s->option & P3_OBJECT_HIDDEN) return;
  if (s->option & P3_SPRITE_RECEIVE_SHADOW) {
    if (s->option & P3_SPRITE_ALPHA) {
      P3_renderer_add_alpha (s, NULL);
    } else {
      P3_renderer_add (s, NULL);
    }
  } else {
    P3_renderer_batch (renderer->specials, s, NULL, -1);
  }
}

void P3_sprite_render (P3_sprite* s, P3_instance* inst) {
  GLfloat* a; GLfloat* b;
  /* compute render matrix */
  b = ((P3_instance*) s->parent)->render_matrix;
  a = s->position;
  sprite_matrix[12] = a[0] * b[0] + a[1] * b[4] + a[2] * b[ 8] + b[12];
  sprite_matrix[13] = a[0] * b[1] + a[1] * b[5] + a[2] * b[ 9] + b[13];
  sprite_matrix[14] = a[0] * b[2] + a[1] * b[6] + a[2] * b[10] + b[14];
  P3_material_activate (s->material);
  glLoadMatrixf (sprite_matrix);
  glDisable (GL_CULL_FACE);
  if (s->option & P3_SPRITE_NEVER_LIT) { glDisable (GL_LIGHTING); } else { glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE); }
  glColor4fv (s->color);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 0.0); 
  glVertex3f (- s->w, - s->h, 0.0);
  glTexCoord2f (1.0, 0.0); 
  glVertex3f (  s->w, - s->h, 0.0);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (  s->w,   s->h, 0.0);
  glTexCoord2f (0.0, 1.0); 
  glVertex3f (- s->w,   s->h, 0.0);
  glEnd ();
  glEnable (GL_CULL_FACE);
  if (s->option & P3_SPRITE_NEVER_LIT) { glEnable (GL_LIGHTING); } else { glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE); }
}

void P3_sprite_compute_alpha (P3_sprite* s) {
  s->option &= ~P3_SPRITE_ALPHA;
  if (s->material != NULL && (s->material->option & P3_MATERIAL_ALPHA || s->material->option & P3_MATERIAL_MASK)) { s->option |= P3_SPRITE_ALPHA; }
  if (s->color[3] < 1.0) { s->option |= P3_SPRITE_ALPHA; }
}

void P3_sprite_get_data (P3_sprite* s, P3_chunk* chunk) {
  P3_chunk_save_int (chunk, s->option);
  P3_chunk_save (chunk, s->position, 3 * sizeof (GLfloat));
  P3_chunk_save (chunk, s->color, 4 * sizeof (GLfloat));
  P3_chunk_save_float (chunk, s->w);
  P3_chunk_save_float (chunk, s->h);
}

void P3_sprite_set_data (P3_sprite* s, P3_chunk* chunk) {
  s->class = &P3_class_sprite;
  s->option = P3_chunk_load_int (chunk);
  P3_chunk_load (chunk, s->position, 3 * sizeof (GLfloat));
  P3_chunk_load (chunk, s->color, 4 * sizeof (GLfloat));
  s->w = P3_chunk_load_float (chunk);
  s->h = P3_chunk_load_float (chunk);
}


/*=================*
 * SPRITE CYLINDER *
 *=================*/

P3_class P3_class_cylinder = { 
  P3_ID_CYLINDER,
  (batch_func)     P3_sprite_batch,
  (render_func)    P3_cylinder_render,
  (shadow_func)    0,
  (raypick_func)   0,
  (raypick_b_func) 0,
};

P3_cylinder* P3_cylinder_new (P3_cylinder* s) {
  if (s == NULL) { s = (P3_cylinder*) malloc (sizeof (P3_cylinder)); }
  s->class = &P3_class_cylinder;
  s->parent = NULL;
  s->option = 0;
  s->position[0] = 0.0;
  s->position[1] = 0.0;
  s->position[2] = 0.0;
  s->direction[0] = 0.0;
  s->direction[1] = 0.0;
  s->direction[2] = 1.0;
  s->color[0] = 1.0;
  s->color[1] = 1.0;
  s->color[2] = 1.0;
  s->color[3] = 1.0;
  s->material = NULL;
  s->w = 0.5;
  s->h = 0.5;
  return s;
}

void P3_cylinder_render (P3_cylinder* c, P3_instance* inst) {
  GLfloat x; GLfloat y;
  GLfloat f;
  GLfloat* a; GLfloat* m;
  /* compute render matrix */
  m = ((P3_instance*) c->parent)->render_matrix;
  a = c->direction;
  cylinder_matrix[ 8] = a[0] * m[0] + a[1] * m[4] + a[2] * m[ 8];
  cylinder_matrix[ 9] = a[0] * m[1] + a[1] * m[5] + a[2] * m[ 9];
  cylinder_matrix[10] = a[0] * m[2] + a[1] * m[6] + a[2] * m[10];
  a = c->position;
  cylinder_matrix[12] = a[0] * m[0] + a[1] * m[4] + a[2] * m[ 8] + m[12];
  cylinder_matrix[13] = a[0] * m[1] + a[1] * m[5] + a[2] * m[ 9] + m[13];
  cylinder_matrix[14] = a[0] * m[2] + a[1] * m[6] + a[2] * m[10] + m[14];
  if (cylinder_matrix[10] == 0.0) {
    x = cylinder_matrix[8];
    y = cylinder_matrix[9];
  } else {
    f = cylinder_matrix[14] / cylinder_matrix[10];
    x = cylinder_matrix[12] - f * cylinder_matrix[8];
    y = cylinder_matrix[13] - f * cylinder_matrix[9];
    /* sign of x, y ? */
//    if (y < 0) { 
//      y = - y;
//      x = - x; 
//    }
  }
  if (x == 0.0 && y == 0.0) {
    cylinder_matrix[4] = 0.0;
    cylinder_matrix[5] = 1.0;
  } else {
    f = (GLfloat) 1.0 / sqrt(x * x + y * y);
    cylinder_matrix[5] = - x * f;
    cylinder_matrix[4] =   y * f;
  }
  cylinder_matrix[0] =   cylinder_matrix[5] * cylinder_matrix[10] - cylinder_matrix[6] * cylinder_matrix[9];
  cylinder_matrix[1] = - cylinder_matrix[4] * cylinder_matrix[10] + cylinder_matrix[6] * cylinder_matrix[8];
  cylinder_matrix[2] =   cylinder_matrix[4] * cylinder_matrix[ 9] - cylinder_matrix[5] * cylinder_matrix[8];
  /* render */
  P3_material_activate (c->material);
  glLoadMatrixf (cylinder_matrix);
  glDisable (GL_CULL_FACE);
  if (c->option & P3_SPRITE_NEVER_LIT) { glDisable (GL_LIGHTING); } else { glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE); }
  glColor4fv (c->color);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 0.0);
  glVertex3f (0.0, - c->h, - c->w);
  glTexCoord2f (1.0, 0.0);
  glVertex3f (0.0,   c->h, - c->w);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (0.0,   c->h,   c->w);
  glTexCoord2f (0.0, 1.0);
  glVertex3f (0.0, - c->h,   c->w);
  glEnd ();
  glEnable (GL_CULL_FACE);
  if (c->option & P3_SPRITE_NEVER_LIT) { glEnable (GL_LIGHTING); } else { glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE); }
}

void P3_cylinder_get_data (P3_cylinder* c, P3_chunk* chunk) {
  P3_chunk_save_int (chunk, c->option);
  P3_chunk_save (chunk, c->position, 3 * sizeof (GLfloat));
  P3_chunk_save (chunk, c->color, 4 * sizeof (GLfloat));
  P3_chunk_save_float (chunk, c->w);
  P3_chunk_save_float (chunk, c->h);
  P3_chunk_save (chunk, c->direction, 3 * sizeof (GLfloat));
}

void P3_cylinder_set_data (P3_cylinder* c, P3_chunk* chunk) {
  c->class = &P3_class_cylinder;
  c->option = P3_chunk_load_int (chunk);
  P3_chunk_load (chunk, c->position, 3 * sizeof (GLfloat));
  P3_chunk_load (chunk, c->color, 4 * sizeof (GLfloat));
  c->w = P3_chunk_load_float (chunk);
  c->h = P3_chunk_load_float (chunk);
  P3_chunk_load (chunk, c->direction, 3 * sizeof (GLfloat));
}


/*=======*
 * BONUS *
 *=======*/

P3_class P3_class_bonus = { 
  P3_ID_BONUS,
  (batch_func)     P3_bonus_batch,
  (render_func)    P3_bonus_render,
  (shadow_func)    0,
  (raypick_func)   0,
  (raypick_b_func) 0,
};

P3_bonus* P3_bonus_new (P3_bonus* b) {
  if (b == NULL) {
    b = (P3_bonus*) malloc (sizeof (P3_bonus));
  }
  b->class = &P3_class_bonus;
  b->option = 0;
  b->material = NULL;
  b->halo = NULL;
  b->angle = 0.0;
  return b;
}

void P3_bonus_batch (P3_bonus* b, P3_instance* csys) {
  if (!(b->option & P3_BONUS_BATCHED)) {
    /* make the bonus rotate */
    b->option |= P3_BONUS_BATCHED;
    b->angle += 4.0;
    if (b->angle > 360.0) { b->angle = 0.0; }
  }
  P3_renderer_add_alpha (b, csys);
}

void P3_bonus_render (P3_bonus* b, P3_instance* inst) {
  GLfloat* m;
  b->option &= ~P3_BONUS_BATCHED;
  glDisable (GL_CULL_FACE);
  if (b->option & P3_SPRITE_NEVER_LIT) { glDisable (GL_LIGHTING); }
  /* draw halo */
  m = inst->render_matrix;
  sprite_matrix[12] = m[12];
  sprite_matrix[13] = m[13] + 1.0;
  sprite_matrix[14] = m[14];
  glLoadMatrixf (sprite_matrix);
  P3_material_activate (b->halo);
  glColor4fv (b->color);
  glDisable (GL_LIGHTING);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 1.0);
  glVertex3f (- 1.0, - 1.0, 0.0);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (- 1.0,   1.0, 0.0);
  glTexCoord2f (1.0, 0.0);
  glVertex3f (  1.0,   1.0, 0.0);
  glTexCoord2f (0.0, 0.0);
  glVertex3f (  1.0, - 1.0, 0.0);
  glEnd ();
  glEnable (GL_LIGHTING);
  /* draw 2D square */
  glRotatef (b->angle, 0.0, 1.0, 0.0);
  glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
  P3_material_activate (b->material);
  glNormal3f (0.0, 0.0, 1.0);
  glBegin (GL_QUADS);
  glTexCoord2f (0.0, 1.0);
  glVertex3f (- 0.5, -0.5, 0.0);
  glTexCoord2f (1.0, 1.0);
  glVertex3f (  0.5, -0.5, 0.0);
  glTexCoord2f (1.0, 0.0);
  glVertex3f (  0.5,  0.5, 0.0);
  glTexCoord2f (0.0, 0.0);
  glVertex3f (- 0.5,  0.5, 0.0);
  glEnd ();
  glEnable (GL_CULL_FACE);
  glLightModeli (GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
  if (b->option & P3_SPRITE_NEVER_LIT) { glEnable (GL_LIGHTING); }
}

void P3_bonus_get_data (P3_bonus* b, P3_chunk* chunk) {
  P3_chunk_save_int (chunk, b->option);
  P3_chunk_save (chunk, b->color, 4 * sizeof (GLfloat));
}

void P3_bonus_set_data (P3_bonus* b, P3_chunk* chunk) {
  b->option = P3_chunk_load_int (chunk);
  P3_chunk_load (chunk, b->color, 4 * sizeof (GLfloat));
  b->class = &P3_class_bonus;
  b->angle = 0.0;
}


/*===========*
 * PARTICLES *
 *===========*/

// TO DO cylinder/direction

P3_class P3_class_particles = { 
  P3_ID_PARTICLES,
  (batch_func)     P3_particles_batch,
  (render_func)    P3_particles_render,
  (shadow_func)    0,
  (raypick_func)   0,
  (raypick_b_func) 0,
};

P3_particles* P3_particles_new (P3_particles* s) {
  if (s == NULL) {
    s = (P3_particles*) malloc (sizeof (P3_particles));
  }
  P3_coordsys_initialize ((P3_coordsys*) s);
  s->class = &P3_class_particles;
  s->option = 0;
  s->parent = NULL;
  s->material = NULL;
  s->nb = 0;
  s->max_nb = 0;
  s->particles = NULL;
  s->particle_size = 11;
  s->nb_colors = 0;
  s->fading_colors = NULL;
  s->nb_sizes = 1;
  s->sizes = (GLfloat*) malloc (2 * sizeof (GLfloat));
  s->sizes[0] = 1.0;
  s->sizes[1] = 1.0;
  s->generator = 0;
  s->coordsys = NULL;
  s->delta_time = 0.0f;
  return s;
}

void P3_particles_init (P3_particles* p) {
  p->particle_size = 11;
  if (p->option & P3_PARTICLES_COLOREDS)   { p->particle_size += 4; }
  if (p->option & P3_PARTICLES_MULTI_SIZE) { p->particle_size += 2; }
  if (p->option & P3_PARTICLES_CYLINDER)   { p->particle_size += 3; }
  p->particles = (GLfloat*) realloc (p->particles, p->max_nb * p->particle_size * sizeof (GLfloat));
}

void P3_particles_free_data (P3_particles* p) {
  free (p->particles);
  free (p->fading_colors);
  free (p->sizes);
}

void P3_particles_advance (P3_particles* p, GLfloat proportion) {
  GLfloat* particle;
  GLfloat decay;
  int i;
  decay = - 0.05 * proportion;
  particle = p->particles;
  i = 0;
  while (i < p->nb) {
    particle[0] += decay;
    if (particle[0] < 0.0) {
      /* particle is dead */
      if (p->generator == 0 || p->generator (p, i) == P3_TRUE) {
        /* remove the particle */
        (p->nb)--;
        memcpy (p->particles + i     * p->particle_size, 
                p->particles + p->nb * p->particle_size, 
                p->particle_size * sizeof (GLfloat));
      } else {
        particle += p->particle_size;
        i++;
      }
    } else {
      /* advance position */
      /* speed += acceleration * proportion */
      particle[5] += (particle[ 8] * proportion);
      particle[6] += (particle[ 9] * proportion);
      particle[7] += (particle[10] * proportion);
      /* position += speed * proportion */
      particle[2] += (particle[5] * proportion);
      particle[3] += (particle[6] * proportion);
      particle[4] += (particle[7] * proportion);
      /* advance color */
      if (p->nb_colors > 0 && p->option & P3_PARTICLES_COLOREDS) {
        P3_particles_get_fading_color (p, particle[0], particle[1], particle + 11);
      }
      /* advance size */
      if (p->option & P3_PARTICLES_MULTI_SIZE) {
        if (p->option & P3_PARTICLES_COLOREDS) {
          P3_particles_get_size (p, particle[0], particle[1], particle + 15);
        } else {
          P3_particles_get_size (p, particle[0], particle[1], particle + 11);
        }
      }
      /* advance direction */
      if (p->option & P3_PARTICLES_CYLINDER) {
// TO DO

      }
      particle += p->particle_size;
      i++;
    }
  }
}

void P3_particles_advance_time (P3_particles* p, GLfloat proportion) {
  p->delta_time += proportion;
}

void P3_particles_batch (P3_particles* p, P3_instance* inst) {
  if (p->option & P3_OBJECT_HIDDEN) return;
// TO DO usefull ?
//  P3_multiply_matrix (p->render_matrix, renderer->c_camera->render_matrix, P3_coordsys_get_root_matrix ((P3_coordsys*) p));

// TO DO
  P3_particles_advance (p, p->delta_time);
  p->delta_time = 0.0;
  if (p->option & P3_SPRITE_RECEIVE_SHADOW) {
    if (p->option & P3_SPRITE_ALPHA) {
      P3_renderer_add_alpha (p, (P3_instance*) p->coordsys);
    } else {
      P3_renderer_add (p, (P3_instance*) p->coordsys);
    }
  } else {
    P3_renderer_batch (renderer->specials, p, (P3_instance*) p->coordsys, -1);
  }
}

void P3_particles_render (P3_particles* p, P3_instance* inst) {
  GLfloat* particle;
  GLfloat posi[3];
  GLfloat* m;
  GLfloat w = 1.0;
  GLfloat h = 1.0;
  int i;
  P3_material_activate (p->material);
  glDisable (GL_CULL_FACE);
  if (p->option & P3_SPRITE_NEVER_LIT) glDisable (GL_LIGHTING);
  if (!(p->option & P3_PARTICLES_MULTI_SIZE)) {
    w = p->sizes[0];
    h = p->sizes[1];
  }
  glLoadIdentity ();
  if (p->coordsys == NULL) { 
    m = renderer->c_camera->render_matrix;
  } else {
    m = ((P3_instance*) p->coordsys)->render_matrix;
  }
  particle = p->particles;
  if (p->option & P3_SPRITE_COLORED) { 
    glColor4fv (p->fading_colors); 
  }
  glBegin (GL_QUADS);
  for (i = 0; i < p->nb; i++) {
    if (p->option & P3_PARTICLES_COLOREDS) {
      glColor4fv (particle + 11);
    }
    if (p->option & P3_PARTICLES_MULTI_SIZE) {
      if (p->option & P3_PARTICLES_COLOREDS) {
        w = particle[15];
        h = particle[16];
      } else {
        w = particle[11];
        h = particle[12];
      }
    }
    posi[0] = particle[2] * m[0] + particle[3] * m[4] + particle[4] * m[ 8] + m[12];
    posi[1] = particle[2] * m[1] + particle[3] * m[5] + particle[4] * m[ 9] + m[13];
    posi[2] = particle[2] * m[2] + particle[3] * m[6] + particle[4] * m[10] + m[14];
    glTexCoord2f (0.0, 0.0);
    glVertex3f (posi[0] - w, posi[1] - h, posi[2]);
    glTexCoord2f (1.0, 0.0);
    glVertex3f (posi[0] + w, posi[1] - h, posi[2]);
    glTexCoord2f (1.0, 1.0);
    glVertex3f (posi[0] + w, posi[1] + h, posi[2]);
    glTexCoord2f (0.0, 1.0);
    glVertex3f (posi[0] - w, posi[1] + h, posi[2]);
    particle += p->particle_size;
  }
  glEnd ();
  if (p->option & P3_SPRITE_NEVER_LIT) { glEnable (GL_LIGHTING); }
  glEnable (GL_CULL_FACE);
}

void P3_particles_compute_alpha (P3_particles* s) {
  s->option &= ~P3_SPRITE_ALPHA;
  if (s->material != NULL && (s->material->option & P3_MATERIAL_ALPHA || s->material->option & P3_MATERIAL_MASK)) s->option |= P3_SPRITE_ALPHA;
  if (s->option & P3_SPRITE_COLORED && s->fading_colors[3] < 1.0) s->option |= P3_SPRITE_ALPHA;
  if (s->option & P3_PARTICLES_COLOREDS) s->option |= P3_SPRITE_ALPHA;
}

void P3_particles_get_fading_color (P3_particles* f, GLfloat life, GLfloat max_life, GLfloat* returned_color) {
  if (life <= 0.0) {
    memcpy (returned_color, f->fading_colors + 4 * (f->nb_colors - 1), 4 * sizeof (GLfloat));
  } else if (life >= max_life) {
    memcpy (returned_color, f->fading_colors, 4 * sizeof (GLfloat));
  } else {
    int i;
    GLfloat f1; GLfloat f2;
    GLfloat* c1; GLfloat* c2;
    life = (1.0 - (life / max_life)) * (f->nb_colors - 1);
    i = (int) life;
    c1 = f->fading_colors + 4 * i;
    c2 = f->fading_colors + 4 * (i + 1);
    f2 = life - i;
    f1 = 1.0 - f2;
    returned_color[0] = c1[0] * f1 + c2[0] * f2;
    returned_color[1] = c1[1] * f1 + c2[1] * f2;
    returned_color[2] = c1[2] * f1 + c2[2] * f2;
    returned_color[3] = c1[3] * f1 + c2[3] * f2;
  }
}

void P3_particles_get_size (P3_particles* f, GLfloat life, GLfloat max_life, GLfloat* returned_size) {
  if (life <= 0.0) {
    memcpy (returned_size, f->sizes + 2 * (f->nb_sizes - 1), 2 * sizeof (GLfloat));
  } else if (life >= max_life) {
    memcpy (returned_size, f->sizes, 2 * sizeof (GLfloat));
  } else {
    int i;
    GLfloat f1; GLfloat f2;
    GLfloat* c1; GLfloat* c2;
    life = (1.0 - (life / max_life)) * (f->nb_sizes - 1);
    i = (int) life;
    c1 = f->sizes + 2 * i;
    c2 = f->sizes + 2 * (i + 1);
    f2 = life - i;
    f1 = 1.0 - f2;
    returned_size[0] = c1[0] * f1 + c2[0] * f2;
    returned_size[1] = c1[1] * f1 + c2[1] * f2;
  }
}

GLfloat* P3_particles_generate (P3_particles* p, int index, GLfloat life) {
  GLfloat* particle;
  particle = p->particles + index * p->particle_size;
  particle[0] = life;
  particle[1] = life;
  if (p->parent == NULL) {
    memcpy (particle + 2, p->m + 12, 3 * sizeof (GLfloat));
  } else {
    P3_point_by_matrix_copy (particle + 2, p->m + 12, P3_coordsys_get_root_matrix (p->parent));
  }
  if (p->coordsys != NULL) {
    P3_point_by_matrix (particle + 2, P3_coordsys_get_inverted_root_matrix (p->coordsys));
  }
  if (p->option & P3_PARTICLES_COLOREDS) {
    memcpy (particle + 11, p->fading_colors, 4 * sizeof (GLfloat));
  }
  if (p->option & P3_PARTICLES_MULTI_SIZE) {
    if (p->option & P3_PARTICLES_COLOREDS) {
      memcpy (particle + 15, p->sizes, 2 * sizeof (GLfloat));
    } else {
      memcpy (particle + 11, p->sizes, 2 * sizeof (GLfloat));
    }
  }
  if (p->nb <= index) { p->nb = index + 1; }
  return p->particles + p->particle_size * index;
}

void P3_particles_get_data (P3_particles* p, P3_chunk* chunk) {
  P3_chunk_save (chunk, p->m, 19 * sizeof (GLfloat));
  P3_chunk_save_int (chunk, p->option);
  P3_chunk_save_int (chunk, p->max_nb);
  P3_chunk_save_int (chunk, p->nb_colors);
  P3_chunk_save (chunk, p->fading_colors, p->nb_colors * 4 * sizeof (GLfloat));
  P3_chunk_save_int (chunk, p->nb_sizes);
  P3_chunk_save (chunk, p->sizes, p->nb_sizes * 2 * sizeof (GLfloat));
  if (p->generator == 0) {
    P3_chunk_add_char (chunk, '\0');
  } else {
    P3_chunk_add_char (chunk, '\1');
  }
}

void P3_particles_set_data (P3_particles* p, P3_chunk* chunk) {
  p->class = &P3_class_particles;
  p->validity = P3_COORDSYS_INVALID;
  p->parent = NULL;
  P3_chunk_load (chunk, p->m, 19 * sizeof (GLfloat));
  p->option = P3_chunk_load_int (chunk);
  p->max_nb = P3_chunk_load_int (chunk);
  p->nb_colors = P3_chunk_load_int (chunk);
  if (p->nb_colors > 0) {
    p->fading_colors = (GLfloat*) malloc (p->nb_colors * 4 * sizeof (GLfloat));
    P3_chunk_load (chunk, p->fading_colors, p->nb_colors * 4 * sizeof (GLfloat));
  } else {
    p->fading_colors = NULL;
  }
  p->nb_sizes = P3_chunk_load_int (chunk);
  if (p->nb_sizes > 0) {
    p->sizes = (GLfloat*) malloc (p->nb_sizes * 2 * sizeof (GLfloat));
    P3_chunk_load (chunk, p->sizes, p->nb_sizes * 2 * sizeof (GLfloat));
  } else {
    p->sizes = NULL;
  }
  P3_particles_init (p);
}
