/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * util.c : usefull structs for C
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 **********************************************/

#include <SDL/SDL.h>

#include "p3_base.h"
#include "util.h"

extern P3_list* chunks;


/*======*
 * LIST *
 *======*/

P3_list* P3_list_new (int size) {
  P3_list* l;
  l = (P3_list*) malloc (sizeof (P3_list));
  l->max = size;
  l->nb = 0;
  if (size <= 0) {
    l->content = NULL;
  } else {
    l->content = (void*) malloc (l->max * sizeof (void*));
  }
  return l;
}

void P3_list_dealloc (P3_list* l) {
  free(l->content);
  free(l);
}

static void P3_list_size_up (P3_list* l) {
  /* double the size */
  if (l->max == 0) {
    l->max = 1;
  } else {
    l->max = l->max << 1;
  }
  l->content = (void**) realloc (l->content, l->max * sizeof (void*));
}

static void P3_list_size_down (P3_list* l) {
  l->max = l->max >> 1;
  if (l->max == 0) {
    free (l->content);
    l->content = NULL;
  } else {
    l->content = (void**) realloc (l->content, l->max * sizeof (void*));
  }
}

void P3_list_add (P3_list* l, void* c) {
  if (l->nb >= l->max) {
    P3_list_size_up (l);
  }
  l->content[l->nb] = c;
  (l->nb)++;
}

void P3_list_remove (P3_list* l, int i) {
  l->content[i] = l->content[--(l->nb)];
  l->content[l->nb] = NULL;
  if (l->nb < (l->max >> 1)) {
    P3_list_size_down (l);
  }
}

int P3_list_find (P3_list* l, void* ptr) {
  int i;
  for (i = 0; i < l->nb; i++) {
    if (l->content[i] == ptr) {
      return i;
    }
  }
  return -1;
}

void P3_list_update (P3_list* l, void* c) {
  if (P3_list_find (l, c) == -1)
    P3_list_add (l, c);
}

int P3_list_remove_named (P3_list* l, void* ptr) {
  int i;
  for (i = 0; i < l->nb; i++) {
    if (l->content[i] == ptr) {
      P3_list_remove (l, i);
      return P3_TRUE;
    }
  }
  return P3_FALSE;
}

P3_list* P3_list_clone (P3_list* l1, P3_list* l2) {
  if (l1 == NULL) {
    l1 = (P3_list*) malloc (sizeof (P3_list));
    l1->content = NULL;
  }
  l1->max = l2->max;
  l1->nb = l2->nb;
  l1->content = realloc (l1->content, l1->max * sizeof (void*));
  memcpy (l1->content, l2->content, l2->nb * sizeof (void*));
  return l1;
}


/*=======*
 * CHUNK *
 *=======*/

P3_chunk* P3_chunk_new (void) {
  P3_chunk* chunk;
  chunk = (P3_chunk*) malloc (sizeof (P3_chunk));
  chunk->nb = 0;
  chunk->max = 0;
  chunk->content = NULL;
  return chunk;
}

void P3_chunk_dealloc (P3_chunk* chunk) {
  free (chunk->content);
  free (chunk);
}

static void P3_chunk_size_up (P3_chunk* chunk, int size) {
  chunk->max = (chunk->nb + size) << 1;
  chunk->content = (void*) realloc (chunk->content, chunk->max);
}

int P3_chunk_register (P3_chunk* chunk, int size) {
  int i;
  if (chunk->max < chunk->nb + size) {
    P3_chunk_size_up (chunk, size);
  }
  i = chunk->nb;
  chunk->nb += size;
  return i;
}

void P3_chunk_add (P3_chunk* chunk, void* ptr, int size) {
  if (chunk->max < chunk->nb + size) {
    P3_chunk_size_up (chunk, size);
  }
  memcpy (chunk->content + chunk->nb, ptr, size);
  chunk->nb += size;
}

void P3_chunk_add_char (P3_chunk* chunk, char c) {
  if (chunk->max < chunk->nb + sizeof (char)) {
    P3_chunk_size_up (chunk, sizeof (char));
  }
  *((char*) (chunk->content + chunk->nb)) = c;
  chunk->nb += sizeof (char);
}

void P3_chunk_add_int (P3_chunk* chunk, int i) {
  if (chunk->max < chunk->nb + sizeof (int)) {
    P3_chunk_size_up (chunk, sizeof (int));
  }
  *((int*) (chunk->content + chunk->nb)) = i;
  chunk->nb += sizeof (int);
}

void P3_chunk_add_float (P3_chunk* chunk, GLfloat f) {
  if (chunk->max < chunk->nb + sizeof (GLfloat)) {
    P3_chunk_size_up (chunk, sizeof (GLfloat));
  }
  *((GLfloat*) (chunk->content + chunk->nb)) = f;
  chunk->nb += sizeof (GLfloat);
}

void P3_chunk_add_ptr (P3_chunk* chunk, void* ptr) {
  if (chunk->max < chunk->nb + sizeof (void*)) {
    P3_chunk_size_up (chunk, sizeof (void*));
  }
  *((void**) (chunk->content + chunk->nb)) = ptr;
  chunk->nb += sizeof (void*);
}

void P3_chunk_get (P3_chunk* chunk, void* ptr, int size) {
  memcpy (ptr, chunk->content + chunk->nb, size);
  chunk->nb += size;
}

char P3_chunk_get_char (P3_chunk* chunk) {
  char c;
  c = *((char*) (chunk->content + chunk->nb));
  chunk->nb += sizeof (char);
  return c;
}

int P3_chunk_get_int (P3_chunk* chunk) {
  int i;
  i = *((int*) (chunk->content + chunk->nb));
  chunk->nb += sizeof (int);
  return i;
}

GLfloat P3_chunk_get_float (P3_chunk* chunk) {
  GLfloat f;
  f = *((GLfloat*) (chunk->content + chunk->nb));
  chunk->nb += sizeof (GLfloat);
  return f;
}

void* P3_chunk_get_ptr (P3_chunk* chunk) {
  void* ptr;
  ptr = *((void**) (chunk->content + chunk->nb));
  chunk->nb += sizeof (void*);
  return ptr;
}

void P3_chunk_save_int (P3_chunk* chunk, int i) {
  if (chunk->max < chunk->nb + sizeof (int)) {
    P3_chunk_size_up (chunk, sizeof (int));
  }
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  SDL_Swap32 (i);
#endif
  *((int*) (chunk->content + chunk->nb)) = i;
  chunk->nb += sizeof (int);
}

void P3_chunk_save_float (P3_chunk* chunk, GLfloat f) {
  if (chunk->max < chunk->nb + sizeof (GLfloat)) {
    P3_chunk_size_up (chunk, sizeof (GLfloat));
  }
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  SDL_Swap32 (f);
#endif
  *((GLfloat*) (chunk->content + chunk->nb)) = f;
  chunk->nb += sizeof (GLfloat);
}

int P3_chunk_load_int (P3_chunk* chunk) {
  int i;
  i = *((int*) (chunk->content + chunk->nb));
  chunk->nb += sizeof (int);
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  SDL_Swap32 (i);
#endif
  return i;
}

GLfloat P3_chunk_load_float (P3_chunk* chunk) {
  GLfloat f;
  f = *((GLfloat*) (chunk->content + chunk->nb));
  chunk->nb += sizeof (GLfloat);
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  SDL_Swap32 (f);
#endif
  return f;
}

void P3_chunk_save (P3_chunk* chunk, void* ptr, int size) {
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  int i;
#endif
  if (chunk->max < chunk->nb + size) {
    P3_chunk_size_up (chunk, size);
  }
  memcpy (chunk->content + chunk->nb, ptr, size);
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  for (i = 0; i < size; i++) {
    SDL_Swap32 (chunk->content + chunk->nb + i);
    i += 4;
  }
#endif
  chunk->nb += size;
}

void P3_chunk_load (P3_chunk* chunk, void* ptr, int size) {
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  int i;
#endif
  memcpy (ptr, chunk->content + chunk->nb, size);
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  for (i = 0; i < size; i++) {
    SDL_Swap32 (ptr + i);
    i += 4;
  }
#endif
  chunk->nb += size;
}


/*=====================*
 * CHUNK/LIST RECYCLER *
 *=====================*/

P3_chunk* P3_get_chunk (void) {

//  printf ("nb chunks %i\n", chunks->nb);

  if (chunks->nb == 0)
    return P3_chunk_new ();
  else
    return chunks->content[--(chunks->nb)];
}

P3_list* P3_get_list (void) {
  P3_chunk* chunk;
  if (chunks->nb == 0) {
    return P3_list_new (0);
  } else {
    chunk = chunks->content[--(chunks->nb)];
    /* convert chunk to list */
    chunk->max = (int) (chunk->max / sizeof (void*));
    return (P3_list*) chunk;
  }
}

void P3_drop_chunk (P3_chunk* chunk) {
  chunk->nb = 0;
  P3_list_add (chunks, chunk);
}

void P3_drop_list (P3_list* list) {
  list->nb = 0;
  /* convert list to chunk */
  list->max *= sizeof (void*);
  P3_list_add (chunks, list);
}


/* - string - */

#if 0
char* P3_strcat (char* a, char* b) {
  /* A better strcat that return a new string (doesn't matter buffer overwriting) */
  int i; 
  char* r;
  if (a == NULL) { return strdup (b); }
  if (b == NULL) { return strdup (a); }
  i = strlen (a);
  r = (char*) malloc ((i + strlen (b) + 1) * sizeof (char));
  /* don't copy the final \0 */
  memcpy (r, a, i * sizeof (char));
  strcpy (r + i, b);
  return r;
}
#endif

char* P3_filename_extension (char* str) {
  char* r;
  r = strrchr (str, '.');
  if (r == NULL) {
    return NULL;
  } else {
    return (r + 1);
  }
}


// --- DEPRECATED --- !!!

#if 0

/* - chain - */

P3_chain* P3_chain_append (P3_chain* chain, void* data) {
// TO DO optimizable with a trashcan ?
  P3_chain* c;
  c = (P3_chain*) malloc (sizeof (P3_chain));
  c->data = data;
  c->next = NULL;
  c->prev = chain;
  if (chain != NULL) { chain->next = c; }
  return c;
}

void P3_chain_dealloc_nexts (P3_chain* chain) {
  P3_chain* c = chain;
  while (c != NULL) {
    chain = c->next;
    free (c);
    c = chain;
  }
}

#endif
