# Soya 3D
# Copyright (C) 2001-2002 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import soya, soya.math3d as math3d, soya.soya3d as soya3d
import _soya
import sys, os, os.path, weakref, cPickle as pickle, copy, copy_reg

def Image(filename):
  """Returns an image object, given its FILENAME."""
  if filename is None: return None
  if not os.path.exists(os.path.join(Image.PATH, filename)): raise IOError, "No such file or directory: '%s'" % filename
  return _soya._Image(os.path.join(Image.PATH, filename))

Image.PATH = ""
Image.get = Image.load = Image

class Material(soya.SavedInAPath, soya._CObj, _soya._Material):
  """Material

A material regroups all the property of a surface, like colors, shininess and
even textures.

Interesting properties are:

- diffuse: the diffuse color.

- specular: the specular color; this one is used for shiny part of the surface.

- separate_specular: set this to true to enable separate specular; this usually
  results in a more shiny specular effect.

- shininess: the shininess ranges from 0.0 to 128.0; 0.0 is the most metallic / shiny
  and 128.0 is the most plastic.

- tex_filename: the texture filename (not the complete filename; model.Image.PATH
  is prepended to it).

- additive_blending: set it to true for additive blending. For semi-transparent surfaces
  (=alpha blended) only. Usefull for special effect (Ray, Sprite,...).

See tutorial lesson 003.
"""
  
  _alls = weakref.WeakValueDictionary() # Required by SavedInAPath
  def __init__(self, filename = ""):
    """Material(filename = "") -> Material

Creates a new Material with the given FILENAME."""
    _soya._Material.__init__(self)
    soya.SavedInAPath.__init__(self, filename)
    self._tex_filename = None
    
  def get_tex_filename(self): return self._tex_filename
  
  def set_tex_filename(self, filename):
    self._tex_filename = filename
    self.image = Image(filename)
    
  def __repr__(self): return "<Material %s>" % self.filename
  
  tex_filename = property(get_tex_filename, set_tex_filename)
  
_soya.get_material = Material.get



class Shape(soya.SavedInAPath, soya._CObj, _soya._Mesh): # I prefer "Shape" to "Mesh" :-)
  """Shape

A Shape is a compiled and optimized set of faces"""
  
  _alls = weakref.WeakValueDictionary() # Required by SavedInAPath
  def __init__(self):
    """DO NOT call this directly!!! Use World.shapify()!"""

    # blam hack
    self._filename=""
    
    soya.SavedInAPath.__init__(self, "")
    _soya._Mesh.__init__(self)
    
  def __repr__(self): return "<Shape %s>" % self.filename
  
  def to_world(self):
    """Shape.to_world() -> World

Convert this shape into a World.

It first attempts to read the World from the World directory (soya3d.World.PATH)
and, if it fails, it tries to uncompile the Shape. In the second case, the result
may not be identical to the World used to create the Shape!"""
    if self.filename:
      try:
        return soya3d.World.get(self.filename)
      except IOError: pass
      
    world = soya3d.World()
    self.to_faces(world)
    return world
    
  def get(klass, name): return Shape._alls.get(name) or Shape.load(name)
  get = classmethod(get)
  
  def load(klass, name):
    filename       = os.path.join(Shape.PATH       , name + ".data")
    world_filename = os.path.join(soya3d.World.PATH, name + ".data")
    
    if os.path.exists(filename) and (not soya3d.World._alls.has_key(name)) and ((not os.path.exists(world_filename)) or (os.path.getmtime(filename) >= os.path.getmtime(world_filename))):
      return pickle.load(open(filename, "rb"))
    
    print "Compiling world %s to shape..." % name
    
    shape = soya3d.World.get(name).shapify()

    try: shape.save()
    except:
      sys.excepthook(*sys.exc_info())
      print "WARNNG : can't save compiled shape %s!" % filename
    return shape
  load = classmethod(load)
  
  availables = soya3d.World.availables
  


  
class ShapeLOD(soya.SavedInAPath, soya._CObj, _soya._MeshLOD):
  """ShapeLOD

A Shape with multiple LOD (levels of detail). The transition between LOD levels
is not smoothed. ShapeLOD.shapes is a list that must contain the Shape for
each LOD level. First shape is the most accurate one."""
  
  _alls = weakref.WeakValueDictionary() # Required by SavedInAPath
  def __init__(self, shapes = []):
    soya.SavedInAPath.__init__(self, "")
    _soya._MeshLOD.__init__(self)
    if shapes: self.shapes = shapes
    
  def __repr__(self): return "<ShapeLOD %s>" % self.filename

# TO DO load/save doesn't seem to work...
  
  def get(klass, name): return ShapeLOD._alls.get(name) or ShapeLOD.load(name)
  get = classmethod(get)
  
  def load(klass, name):
    filename = os.path.join(Shape.PATH, name + ".data")
    
    if os.path.exists(filename):
      return pickle.load(open(filename, "rb"))
    
  load = classmethod(load)
  
#  availables = soya3d..availables



class ShapeFXInstance(soya.SavedInAPath, soya._CObj, _soya._MeshFXInstance):
  """ShapeFXInstance

A Shape instance used to apply vertex FX"""
  
  def __init__(self, shape = None):
    _soya._MeshFXInstance.__init__(self)
    if shape: self.shape = shape
    
  def __repr__(self): return "<ShapeFXInstance %s>" % self.shape
  



## class MorphShape(soya.SavedInAPath, soya._CObj, _soya._MorphData): 
##   """MorphShape

## A MorphShape is a Shape with morphable faces / animations."""
  
##   def __init__(self):
##     """DO NOT call this directly!!! Use World.shapify() with the specific shapify_args!"""
##     soya.SavedInAPath.__init__(self, "")
##     _soya._MorphData.__init__(self)
    
##     import warnings
##     warnings.warn("MorphShape are deprecated and might be removed later; use rather Cal3D!", DeprecationWarning, stacklevel = 2)
    
##   def __repr__(self): return "<Morph data %s>" % self.filename
  
##   def to_world(self):
##     """Shape.to_world() -> World

## Convert this shape into a World.

## It first attempts to read the World from the World directory (soya3d.World.PATH)
## and, if it fails, it tries to uncompile the Shape. In the second case, the result
## may not be identical to the World used to create the Shape!"""
##     if self.filename:
##       try:
##         return soya3d.World.get(self.filename)
##       except IOError: pass
      
##     world = soya3d.World()
##     self.to_faces(world)
##     return world
  
##   _alls      = Shape._alls
##   get        = Shape.get
##   load       = Shape.load
##   availables = Shape.availables
##   def save(self, filename = None):
##     soya.SavedInAPath.save(self, filename or os.path.join(Shape.PATH, self._filename))
  


class Bonus(soya.SavedInAPath, soya._CObj, _soya._Bonus):
  """Bonus"""
  
  def __init__(self):
    soya.SavedInAPath.__init__(self, "")
    _soya._Bonus.__init__(self)
    
  def __repr__(self):
    return "<Bonus>"
  
  _alls      = Shape._alls
  get        = Shape.get
  load       = Shape.load
  availables = Shape.availables
  def save(self, filename = None):
    soya.SavedInAPath.save(self, filename or os.path.join(Shape.PATH, self._filename))
    


class Vertex(math3d.Point):
  """Vertex

A Vertex is a subclass of Point, which is used for building Faces.

It has additionnal properties:

- color: the vertex color

- tex_x, tex_y: the texture coordinates (sometime called U and V).

"""
  def __init__(self, parent = None, x = 0.0, y = 0.0, z = 0.0, tex_x = 0.0, tex_y = 0.0, color = None):
    """Vertex(parent = None, x = 0.0, y = 0.0, z = 0.0, tex_x = 0.0, tex_y = 0.0, color = None)

Creates a new Vertex in coordinate systems PARENT, at position X, Y, Z, with texture
coordinates TEX_X and TEX_Y, and the given COLOR."""
    math3d.Point.__init__(self, parent, x, y, z)
    self.tex_x = tex_x
    self.tex_y = tex_y
    self.color = color

  def __getstate__(self):
    state = self.__dict__.copy()
    state["_cdata"] = math3d.Point.__getstate__(self)
    return state
  
  def __setstate__(self, state):
    if state.has_key("_cdata"):
      math3d.Point.__setstate__(self, state["_cdata"])
      del state["_cdata"]
    else:
      math3d.Point.__setstate__(self, state)
      del state["x"]
      del state["y"]
      del state["z"]
      del state["parent"]
      
    self.__dict__ = state
  
  def __eq__(a, b): return a is b    
  __hash__ = object.__hash__
  
  def render(self, coordSyst = None):
    """Vertex.render(coordSyst = None)

Render this Vertex (by calling the OpenGL glVertex* function).
If COORDSYST is given, the Vertex coordinates are converted in this
coordinate system before rendering."""
    _soya.glTexCoord2f(self.tex_x, self.tex_y)
    if self.color: _soya.glColor4f(*self.color)
    
    if coordSyst:
      coord = self % coordSyst
      _soya.glVertex3f(coord.x, coord.y, coord.z)
    else:
      _soya.glVertex3f(self.x, self.y, self.z)
      
  def __repr__(self):
    if self.parent: return "<Vertex %f %f %f in %s>" % (self.x, self.y, self.z, self.parent)
    return "<Vertex %f %f %f>" % (self.x, self.y, self.z)
  
  def __deepcopy__(self, memo):
    # Do not clone the parent nor the face (except if already cloned before, in the memo) !
    import copy
    
    red = self.__reduce__()
    
    clone = red[1][0]()
    memo[id(self)] = clone
    
    state = copy.copy(red[2])
    
    if self.parent and (not id(self.parent) in memo.keys()):
      state["parent"] = None
      _cdata = state["_cdata"]
      state["_cdata"] = (None, _cdata[1], _cdata[2], _cdata[3])
      
    if hasattr(self, "face") and (not id(self.face) in memo.keys()): del state["face"]

    s2 = copy.deepcopy(state, memo)
    
    if hasattr(clone, "__setstate__"): clone.__setstate__(s2)
    else:                              clone.__dict__ = s2
    
    return clone
  
_soya.new_vertex = Vertex
#copy_reg.pickle(Vertex, soya.math3d.soya_generic_reduce, soya._soya_reconstructor)



class Face(soya._CObj, _soya._Face):
  """Face

A Face is a polygon composed of several Vertices.

According to the number of Vertices, the result differs:
- 1 point => Plot
- 2       => Line
- 3       => Triangle
- 4       => Quad

The vertices are understood to be coplannar.

Interesting properties are:

- vertices: the list of Vertices
- material: the material used to draw the face's surface
- double_sided: true if you want to see both sides of the Face. Default
  is false.
- solid: true to enable the use of this Face for raypicking. Default: true.
- lit: true to enable lighting on the Face. Default: true.

The following options are used when compiling the Face into a Shape,
but does not affect the rendering of the Face itself:

- static_lit: true to enable static lighting (faster). Default: true.
- smooth_lit: true to. Default: false.

"""
  colored = 0
  
  def __init__(self, parent = None, vertices = None, material = None):
    """Face(parent = None, vertices = None, material = None) -> Face

Creates a new Face in World PARENT, with the given list of VERTICES,
and the given Material."""
    _soya._Face.__init__(self)
    self.vertices     = vertices or []
    self.double_sided = 0
    self.material     = material
    self._parent      = None
    self.normal       = None
    self.smooth_lit   = 0
    self.lit          = 1
    self.solid        = 1
    self.static_lit   = 1
    
    for vertex in self.vertices: vertex.face = self
    
    if parent: parent.add(self)
    
  def get_parent(self): return self._parent
  def set_parent(self, parent):
    if self._parent: self._parent.remove(self)
    if parent: parent.add(self)
  parent = property(get_parent, set_parent)
  
  def get_root(self): return self.parent and self.parent.get_root()
  
  def insert(self, index, vertex):
    """Face.insert(index, vertex)

Inserts the given VERTEX to the Face at position INDEX."""
    vertex.face = self
    self.vertices.insert(index, vertex)
  def append(self, vertex):
    """Face.append(vertex)

Appends the given VERTEX to the Face."""
    vertex.face = self
    self.vertices.append(vertex)
  add = append
  
  def compute_normal(self):
    """Face.compute_normal()

Computes the normal vector of the Face."""
    if self.vertices and len(self.vertices) > 2:
      if self.normal:
        _soya.points_normal(self.vertices[0], self.vertices[1], self.vertices[2], self.normal)
      else:
        self.normal = _soya.points_normal(self.vertices[0], self.vertices[1], self.vertices[2])
        
  def is_colored(self):
    """is_colored() -> boolean

Returns true if the Face is colored, i.e. at least one of its Vertices is colored."""
    for vertex in self.vertices:
      if vertex.color: return 1
    return 0
  
  def is_morphable(self):
    """is_morphable() -> boolean

Returns true if the Face is morphable, i.e. at least two of its Vertices are not defined in the same coordinate system."""
    if not self.vertices: return 0
    parent = self.vertices[0].parent
    for vertex in self.vertices:
      if not vertex.parent is parent: return 1
    return 0
  
  def is_alpha(self):
    """is_alpha() -> boolean

Returns true if the Face is alpha-blended."""
    return (self.material and self.material.is_alpha()) or self.has_alpha_vertex()
  def has_alpha_vertex(self):
    """has_alpha_vertex() -> boolean

Returns true if the Face has at least one alpha blended Vertex."""
    for vertex in self.vertices:
      if vertex.color and vertex.color[3] < 1.0: return 1
    return 0
  
  def batch(self):
    return (self.is_alpha(), None)
  
  def render(self):
    if not self.vertices: return

    _soya.activate_material(self.material)
    #if self.material: self.material.activate()
    #else: _soya.glDisable(_soya.GL_TEXTURE_2D)
    
    if not self.lit: _soya.glDisable(_soya.GL_LIGHTING)
    
    if self.double_sided:
      _soya.glLightModeli(_soya.GL_LIGHT_MODEL_TWO_SIDE, _soya.GL_TRUE);
      _soya.glDisable(_soya.GL_CULL_FACE);
      
    self.compute_normal()
    if self.normal: _soya.glNormal3f(self.normal.x, self.normal.y, self.normal.z)
    
    i = len(self.vertices)
    if   i == 1: _soya.glBegin(_soya.GL_POINTS)
    elif i == 2: _soya.glBegin(_soya.GL_LINES)
    elif i == 3: _soya.glBegin(_soya.GL_TRIANGLES)
    elif i == 4: _soya.glBegin(_soya.GL_QUADS)
    else:        _soya.glBegin(_soya.GL_POLYGON)
    
    for vertex in self.vertices: vertex.render(self.parent)
    
    _soya.glEnd()
    
    if self.double_sided:
      _soya.glLightModeli(_soya.GL_LIGHT_MODEL_TWO_SIDE, _soya.GL_FALSE);
      _soya.glEnable(_soya.GL_CULL_FACE);
      
    if not self.lit: _soya.glEnable(_soya.GL_LIGHTING)
    
    #if self.material: self.material.inactivate()
    #else: _soya.glEnable(_soya.GL_TEXTURE_2D)

  def __iter__(self): return iter(self.vertices)
  def __getitem__(self, index): return self.vertices[index]
  def __contains__(self, vertex): return item in self.vertices
  
  def __repr__(self):
    i = len(self.vertices)
    if   i == 1: r = "<Point"
    elif i == 2: r = "<Line"
    elif i == 3: r = "<Triangle"
    elif i == 4: r = "<Quad"
    else:        r = "<Polygon"
    
    if self.material: r = r + ", material %s" % self.material.filename
    
    return r + ">"
  
  def begin_round(self): pass
  def advance_time(self, proportion): pass
  def end_round(self): pass

  def __getstate__(self):
    if self.material and self.material.filename:
      state = self.__dict__.copy()
      state["material"] = self.material.filename
      return state
    else: return self.__dict__
    
  def __setstate__(self, state):
    self.__dict__ = state
    if type(self.material) is str:
      self.material = Material.get(self.material)
      
_soya.new_face = Face

WHITE       = (1.0, 1.0, 1.0, 1.0)
TRANSPARENT = (1.0, 1.0, 1.0, 0.0)



