# Soya 3D tutorial
# Copyright (C) 2001-2002 Jean-Baptiste LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Lesson 011: Animations

# This lesson is the first about animations.


import soya, soya.soya3d as soya3d, soya.animation as animation, soya.model as model, soya.cube as cube, soya.sphere as sphere
import os, os.path, sys

soya.init()

scene = soya3d.World()

# Creates a cube and a sphere

cube   = soya3d.Volume(scene, cube.Cube    ().shapify()); cube  .set_xyz(1.0, -0.5, 0.0)
sphere = soya3d.Volume(scene, sphere.Sphere().shapify()); sphere.set_xyz(-1.5, 0.5, -3.0)

# Creates an animation object

anim = animation.Animation()

# Adds the current state of the scene at time 0.0 of our new animation

anim.insert_state(0.0, scene)

# You can remove a state as following:
#del anim[0.0]

# Moves the sphere and adds the new state at time 1.0
# Any movement / rotation can be used, or any combination of them.

sphere.x -= 2
anim.insert_state(1.0, scene)

# Rotates the cube and adds the new state at time 2.0

cube.rotate_lateral(60.0)
anim.insert_state(2.0, scene)

# Moves the cube and the sphere, and adds the new state at time 3.0

sphere.z += 4
sphere.x += 2
cube.y   += 2
anim.insert_state(3.0, scene)

# This animation should be cyclic -- For a cyclic animation, the cyclic_lap attribute
# is the duration between the last state (here, 3.0) and the first one (here, 0.0).
# If the animation is not cyclic, cyclic_lap should be 0.0

anim.cyclic_lap = 1.0


# Creates a light and a camera

light = soya3d.Light(scene)
light.set_xyz(0.0, 0.2, 1.0)

camera = soya3d.Camera(scene)
camera.z = 8.0
soya.set_root_widget(camera)


# Returns back to the 0.0 state -- you CAN use set_to_state with time values different
# from the one you have entered previously -- in this case, an interpolation occurs.

anim.set_to_state(0.0, scene)

# Notice that our animation can be used to animate the scene, but the SAME animation
# object can also works with any deepcopy of scene as well (actually the "_anim_id"
# attributes of 3D object (cube, sphere here) are used to track them; this attribute
# is automatically created and copied as needed). This is usefull e.g. if you want to
# animate 2 identical monsters in a game.
#
# For example, the following is OK :
#
#import copy
#scene2 = copy.deepcopy(scene)
#soya.cube.Cube(scene2)
#anim.set_to_state(0.0, scene2)
#
# As it has been added after animation has been created, the second cube won't be animated.


# Plays the animation, renders and waits

import time

while 1:
  for i in range(30):
    
    # Set a new state.
    # The first_time and last_time attributes are the lower and higher time available
    # for the animation: here, 0.0 and 4.0
    # (4.0 is 3.0, the last state we have entered, + 1.0, the cyclic_lap)
    
    anim.set_to_state(anim.first_time + (i / 30.0) * (anim.last_time - anim.first_time), scene)
    
    soya.render()
    time.sleep(0.1)
