# Soya 3D tutorial
# Copyright (C) 2001-2002 Bertrand 'blam!' LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# -------------------------------
# Lesson 108: Writing text (Font)
# -------------------------------
#
# This lesson may require the URW (free) font package.
#
# There 2 kind of Font in Soya3D: Raster Font and Texture Font
#
# Raster Font
# -----------
#  These fonts are rendered with the glBitmap function which implies:
#  - each character of the font has the same size (width and height)
#
# Texture Font
# ------------
#  These fonts are rendered like textured quads:
#  - each character has its own width

import sys, os, os.path, time

import soya
import soya.model
import soya.widget


soya.init()


# Creating a new font
# -------------------
#   There are 2 ways: from an image or from a font file (like .ttf, .pfb
#   any format supported by FreeType2)

# Raster Font from an image
filename = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data", "font_8.tga")
image = soya.model.Image(filename)
# Size of each character in pixels
width  = 8
height = 12
# 33 is '!', this is the first character drawn in the image
first_character = 33
font = soya.FontRaster(image, width, height, first_character)

# Texture Font from an image
filename = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data", "font_bookman_26.tga")
image = soya.model.Image(filename)
first_character = 33
# the image must contain each character limited by a vertical line of 1 color
# which is the following one (see font_bookman_26.tga)
limit_color = (255, 0, 0, 255)
font = soya.FontTexture(image, first_character, limit_color)

# Font from a font file
filename = "/usr/share/texmf/fonts/type1/urw/courier/ucrb8a.pfb"
# Size of the font you want (if the font is scalable)
width  = 23
height = 23
first_character = 33
last_character  = 255
font = soya.FontRaster (filename, width, height, first_character, last_character)
font = soya.FontTexture(filename, width, height, first_character, last_character)


# Saving and loading Font
# -----------------------

# Now that the Font is created, you can save it by using:
filename = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data", "fonts", "font_tutorial")
font.save(filename)

# And to load a saved Font
font = soya.load_font(filename)
# And for Raster Font:
#   font = soya.font_raster_load (filename)


# The basic of Widget
# -------------------
#   Before drawing text, we must create a Widget to display the text. With
#   3D objects, you use a Camera as Widget, here we will create a new class
#   of Widget.

class HelloWorld(soya.widget.Widget):
  def __init__(self):
    soya.widget.Widget.__init__(self)

  def render(self):

    # This is the only method of interest for now

    # Clear the screen with a green color
    soya.glClearWithColor(0.0, 1.0, 0.0, 1.0)

    # Change the color of the text to pure red
    soya.glColor4f(1.0, 0.0, 0.0, 1.0)

    # There are 2 functions to draw text:

    # draw_2D
    x = 100
    y = 0
    # x and y represent the upper-left corner of the text
    font.draw_2D("Hello World!", x, y)
        
    # draw_area
    x = 100
    y = 40
    width  = 300
    height = 200
    # The text will automatically be cut to be adjusted to the defined box
    # If the box is too small to contains all the text, a part of the text
    # will be hidden
    align = 1
    # align can be:
    # - 0 = left alignment
    # - 1 = center
    font.draw_area("This text is rendered by\n-the Soya3D-\nengine", x, y, width, height, align)
    # Notice that '\n' character is allowed
    
    # If the Font is a Texture Font, you can add 2 more optionnal arguments
    # to the function draw_2D:
    scale  = (1.0, 1.5)
    # This will scale all the character
    color1 = (1.0, 1.0, 0.0, 1.0)
    color2 = (0.5, 0.5, 1.0, 1.0)
    # Will create a gradient
    font.draw_2D("Soya3D is so coool!!!\nI'm very happy to use it", 150, 200, scale, color1, color2)
    # Notice that you can also use only scale or the 2 colors

    # the function get_print_size will return the size needed to print the text
    (width, height) = font.get_print_size("this is a very very long text that will need to be clipped")
    # if you want to force the text to return to a new line after a certain width,
    # you can pass the width as second argument:
    (width, height, new_text) = font.get_print_size("this is a very very long text that will need to be clipped", 200)
    # new_text is a new string which includes \n characters well positioned so that
    # it can be directly drawed by a draw_2D function:
    font.draw_2D(new_text, 10, 300)


# Create a new HelloWorld Widget and tell soya to render it
soya.set_root_widget(HelloWorld())


while(1):
  time.sleep(0.1)
  soya.render()


