# Soya 3D tutorial
# Copyright (C) 2001-2002 Jean-Baptiste LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Lesson 112: Raypicking : the labyrinth

# In this lesson, you creates a mouse-controlled sphere in a labyrinth.
# Raypicking is used for wall detection.

import soya, soya.soya3d as soya3d, soya.model as model, soya.sphere as sphere, soya.widget as widget, soya.idler as idler
from soya.math3d import Point, Vector
import os, os.path, sys

soya.init()

scene = soya3d.World()


# Defines the texture, material and world directories.

data_dir = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data")

model.Image   .PATH = os.path.join(data_dir, "images")
model.Material.PATH = os.path.join(data_dir, "materials")
soya3d.World  .PATH = os.path.join(data_dir, "worlds")
model.Shape   .PATH = os.path.join(data_dir, "shapes")


# Loads the labyrinth's walls
labyrinth = soya3d.Volume(scene, model.Shape.get("labyrinth"))
labyrinth.y = 1.5


# Creates the sphere

class Perso(soya3d.Volume):
  def __init__(self, parent):
    self.mouse_x = self.mouse_y = 0.0
    
    # Creates a Sphere
    
    s = sphere.Sphere()
    
    # Disable raypicking on all the Faces of the Sphere.
    # Raypick should occurs only on wall !
    
    for face in s: face.solid = 0
    
    soya3d.Volume.__init__(self, parent, s.shapify())
    
  def begin_round(self):
    # Processes the events
    for event in soya.process_event():
      if event[0] == soya.MOUSEMOTION:
        # For mouse motion event, save the mouse coordinates.
        
        self.mouse_x = event[1]
        self.mouse_y = event[2]
        
    if self.mouse_x:
        # First compute the movement's Vector.
        
        mouse = Point(
          scene,
          (float(self.mouse_x) / camera.get_screen_width () - 0.5) *  10.0,
          (float(self.mouse_y) / camera.get_screen_height() - 0.5) * -10.0,
          self.z,
          )
        move = self >> mouse # "a >> b" means "the Vector from a to b"
        
        # Avoid VERY long moves, when the mouse is going crazy !
        
        if move.length() > 0.3: move.set_length(0.3)
        
        # If the move is OK, performs it
        
        if self.check_move(move): self += move # "a += v" means "translate a with Vector v"
        else:
          
          # If the move is not possible, we try to perform only the horizontal part
          # of the move. Then we try the vertical one.
          
          # This allows the sphere to "slide" on wall.
          
          horizontal_move = move.copy() # Clones the move Vector.
          horizontal_move.y = 0.0
          if self.check_move(horizontal_move): self += horizontal_move
          else:
            vertical_move = move.copy()
            vertical_move.x = 0.0
            if self.check_move(vertical_move): self += vertical_move
            
  def check_move(self, move):
    """Checks if the given MOVE (a Vector) is possible or not.
It relies on raypicking for that."""
    
    # Computes the new position of the sphere, if the move is realized.
    
    new_pos = self + move
    
    # With raypicking, we check if there is something at the new position.
    # Soya 3D provides 2 raypicking functions: raypick and raypick_b ("b"
    # stands for boolean). Both take the same arguments.
    
    # raypick returns a tuple (impact_position, normal), or None if there is no
    # impact. impact_position is the position of the impact (a Point); the object
    # hit (the "obstacle") is impact_position.parent. normal is the normal
    # of the face hit; it may be used for reflection effect (see the source of
    # soya.laser).
    # raypick_b returns true if something is hit. It is usually faster than raypick.
    
    # Here, the boolean form is the one we need, since we don't care the position
    # of the impact.
    
    # The arguments for raypick / raypick_b are :
    # - origin: the origin of the ray (a position: a Point, a Volume, a World,...)
    # - direction: the direction of the ray (a Vector)
    # - max_lenght: the maximum length of the ray (use -1.0 for no distance limit)
    # - flags: 2 flags are available; you may combine them (=> flag value: 3):
    #          - 1: cull face: for face with only one side visible (Face.double_sided = 0)
    #               don't check hit on the invisible side.
    #          - 2: half-line: Perform raypicking in only one sense (the one of the
    #               direction Vector). By default, raypicking is performed in both
    #               senses of the given direction (full-line).
    
    # For speeding up, raypick has 2 optional arguments, a Point and a Vector.
    # If given, these Point and Vector will be returned in the tuple, instead of
    # creating new objects.
    
    # Here we check in 4 directions; you may use only 2 directions (faster) or try 8.
    
    return not (
      self.get_root().raypick_b(new_pos, Vector(scene, 1.0,  0.0, 0.0), 1.0, 0) or
      self.get_root().raypick_b(new_pos, Vector(scene, 1.0,  1.0, 0.0), 1.0, 0) or
      self.get_root().raypick_b(new_pos, Vector(scene, 1.0, -1.0, 0.0), 1.0, 0) or
      self.get_root().raypick_b(new_pos, Vector(scene, 0.0,  1.0, 0.0), 1.0, 0)
      )
    
perso = Perso(scene)
perso.x = -0.2

light = soya3d.Light(scene)
light.set_xyz(0.0, 0.2, 2.0)

camera = soya3d.Camera(scene)
camera.set_xyz(0.0, 0.0, 7.0)

soya.set_root_widget(widget.Group())
soya.root_widget.add(camera)
soya.root_widget.add(widget.FPSLabel())


# Hide the mouse cursor

soya.cursor_set_visible(0)


# Main loop

idler.Idler(scene).idle()
  
      
# TODO (left as an exercice):
# Turn this tutorial lesson into a full labyrinth game!

# Add the ability to perform a partial move when a full move is not  possible
# (e.g. if moving 0.2 step left is not possible because of a wall, get the distance
# from the wall and make a move of that distance, e.g. move 0.1 step left so as the
# sphere will touch the wall)
# Hint: use raypick instead of raypick_b.
