# Soya 3D tutorial
# Copyright (C) 2001-2003 Bertrand 'blam!' LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# ------------------
# Lesson 119: Portal
# ------------------

# A portal is a 2D rectangle used to link 2 worlds just as if the world
# beyond was seen through an open window.
# Portals are usefull to compute visibility and avoid renderering the
# world beyond.


import os, os.path, sys, time

import soya
import soya.model
import soya.soya3d
import soya.cube


soya.init()


# Create a world
scene = soya.soya3d.World()


# Some Materials
soya.model.Image.PATH = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data", "images")
m1 = soya.model.Material()
m1.tex_filename = "block2.tga"
m2 = soya.model.Material()
m2.tex_filename = "metal1.tga"


w = soya.soya3d.World()

# Create world 1
w1 = soya.soya3d.World(scene)
w1.atmosphere = soya.soya3d.Atmosphere()
w1.atmosphere.bg_color = (0.0, 0.0, 1.0, 1.0)
w1.atmosphere.ambient  = (0.0, 0.5, 0.0, 1.0)
w1.atmosphere.fog_color = (0.0, 0.0, 1.0, 1.0)
w1.atmosphere.fog_start = 10.0
w1.atmosphere.fog_end = 50.0
w1.atmosphere.fog = 1
w1.atmosphere.fog_type = 0
w1.set_xyz( 2.0, 0.0, 0.0)
w1.set_shape(soya.cube.Cube(w, m1).shapify())

# Create world 2
w2 = soya.soya3d.World(scene)
w2.atmosphere = soya.soya3d.Atmosphere()
w2.atmosphere.bg_color  = (1.0, 0.0, 0.0, 1.0)
w2.atmosphere.ambient   = (0.5, 0.5, 0.0, 1.0)
w2.atmosphere.skyplane  = 1
w2.atmosphere.sky_color = (1.0, 1.0, 0.0, 0.0)
w2.atmosphere.cloud = soya.model.Material()
w2.atmosphere.cloud.tex_filename = 'cloud.tga'
w2.set_xyz(-2.0, 0.0, 0.0)
w2.set_shape(soya.cube.Cube(w, m2).shapify())

# Notice that the 2 worlds we have created doesn't have the same atmosphere

w = None


# Add a light in world 2. Notice that this light will not enlight the objects
# that are in world 1 unless the light has attr top_level set to 1.
light = soya.soya3d.Light(w2)
light.ambient = (1.0, 1.0, 0.0, 1.0)
light.top_level = 0
light.set_xyz(0.0, 2.0, 0.0)


# Portal creation
# ---------------

# Create a portal that link world 1 to world 2
portal = soya.soya3d.Portal(w1)
#   Attr beyond is the world seen through the portal
portal.beyond = w2
#   To change the size of the portal you must scale it (the z scale factor
#   has no meaning)
portal.scale(4.0, 4.0, 1.0)
portal.rotate_lateral(20.0)
portal.set_xyz(0.0, 0.0, 0.0)
#   Attr bound_atm must be set to 1 if the 2 worlds linked doesn't have the
#   same atmosphere (else set the value to 0).
portal.bound_atm = 1
#   Setting use_clip_plane to 1 will affect object rendered in world beyond,
#   this means only the part of the objects that are visible through the portal
#   2D rectangle will be rendered.
portal.use_clip_plane = 1

# Create a portal that link world 2 to world 1
portal2 = soya.soya3d.Portal(w2)
portal2.beyond = w1
portal2.scale(4.0, 4.0, 1.0)
portal2.rotate_lateral(20.0 + 180.0)
portal2.set_xyz(4.0, 0.0, 0.0)
portal2.bound_atm = 1
portal2.use_clip_plane = 1



# Add a camera and a loop to render
camera = soya.soya3d.Camera(scene)
camera.to_render = w1
camera.set_xyz(5.0, -2.0, 1.0)
camera.look_at(soya.soya3d.Point(scene, 0.0, 0.0, 0.0))

roow = soya.widget.Group()
roow.add(camera)

label = soya.widget.Label(roow, " - Portal demo -\n Left click: forward, Right click: backward")
label.resize_style = soya.widget.WIDGET_RESIZE_MAXIMIZE

soya.set_root_widget(roow)


def get_portals(world, list):
  for e in world.children:
    if isinstance (e, soya.soya3d.World):
      get_portal(e, list)
    elif isinstance (e, soya.soya3d.Portal):
      list.append (e)


# See below about the 2 ways to manage camera and portals
method = 0
if method == 1:
  scene.remove(w1)
  scene.remove(w2)
  w1.add(camera)
  camera.to_render = None


avance = 0

while(1):
  time.sleep(0.1)
  soya.render()

# Control:
#   - mouse move to look where you want
#   - left  click to go forward
#   - right click to go backward
#   - any key to exit

  for event in soya.process_event():
    if event[0] == soya.MOUSEMOTION:
      a = event[3]
      b = event[4]
      if a > 5: a = 5
      if b > 5: b = 5
      camera.turn_lateral  (-0.5 * a)
      camera.turn_vertical (-0.5 * b)
    elif event[0] == soya.KEYDOWN:
      sys.exit()
    elif event[0] == soya.MOUSEBUTTONDOWN:
      if (event[1] == 3):
        avance = -1
      else:
        avance = 1
    elif event[0] == soya.MOUSEBUTTONUP:
      avance = 0

  if avance == 1:
    camera.shift (0.0, 0.0, -0.5)
  if avance == -1:
    camera.shift (0.0, 0.0, 0.5)

  portals = []
  if method == 0:
    get_portals (camera.to_render, portals)
    for p in portals:

# portal.is_beyond(soya.soya3d.Point) => return if the point is beyond
# the portal, here it return true if the camera has gone through a portal.
# In that case, we set the to_render attribute of the camera to the world
# beyond => camera has now leaved this world and is now in the world beyond.
    
      if (p.is_beyond (soya.soya3d.Point(camera, 0.0, 0.0, 0.0))):
        camera.to_render = p.beyond

# There is another way to do it if you don't want to use the to_render property
# of the camera. Notice that the previous hierarchy of objects was:
#
#          ___Scene___
#         /     |     \
#   world1   camera    world2
#
# Camera was not a child of world1 nor world2. The other method correspond to
# the following hierarchy:
#
#   world1      world2
#     |
#   camera
#
# Set method to 1 in the lines above to test it (you won't see the difference ;).

  else:
    get_portals (camera.get_root(), portals)
    for p in portals:
      if (p.is_beyond (soya.soya3d.Point(camera, 0.0, 0.0, 0.0))):
        p.beyond.add(camera)




