# Soya 3D tutorial
# Copyright (C) 2003 Bertrand 'blam!' LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# -------------------
# Lesson 125: Shadows
# -------------------
#
# Shape dynamic shadows

import os, os.path, sys, time, math, random

import soya
import soya.model
import soya.widget
import soya.soya3d
import soya.cube


soya.init()


# Create a world
scene = soya.soya3d.World()

# Create a cube Shape
# Notice a few things about Shape and shdows:
#   - double sided faces are considered as normal faces, that is to
#       say there is one side of the face that doesn't cast shadow
#   - avoid Shape with face where one edge can have multiple face neighbors
shape = soya.cube.Cube(None, None)


# Shadows
# -------
# all is in the shapify_args
shape.shapify_args = ('normal', { 'shadowcast': 1 })

# There are 3 conditions for a Shape to cast shadow:
#   - shape.shadow_cast must be true (that is set by the shapify_args but
#       you can turn off Shape shadows with shape.shadows = 0)
#   - shadows must be enabled in soya engine (it is by default unless
#       a it fails to open an OpenGL context with stencil buffer)
#       You can switch shadow rendering with:
#            soya.set_shadow (0)
#            soya.set_shadow (1)
#   - a Light must lighten the Shape and light.shadow_cast must be true
#       (it is true by default)
#
# All objects will receive the shadows except a few ones:
#   - watercube
#   - sprite and particles

scene.shape = shape.shapify()


# Add a landscaped floor to receive the shadows
# This code is copied from lesson-120. That's just to show
# that shadows are casted on whatever object, not only plane
world = soya.soya3d.World()
W = 20
H = 20
heights = []
j = 0
while (j < H):
  i = 0
  while (i < W):
    heights.append(random.random() * 0.6 - 2.0)
    i += 1
  j += 1
j = 0
while (j < H - 1):
  i = 0
  while (i < W - 1):
    f = soya.model.Face(
      world,
      [soya.model.Vertex(world, float(i - 10.0),     heights[i + j * W],       float(j - 10.0)),
       soya.model.Vertex(world, float(i - 10.0),     heights[i + (j + 1) * W], float(j - 10.0 + 1)),
       soya.model.Vertex(world, float(i - 10.0 + 1), heights[i + 1 + j * W],   float(j - 10.0))
      ],
      None)
    f.smooth_lit = 1
    f = soya.model.Face(
      world,
      [soya.model.Vertex(world, float(i - 10.0 + 1), heights[i + 1 + j * W],       float(j - 10.0)),
       soya.model.Vertex(world, float(i - 10.0),     heights[i + (j + 1) * W],     float(j - 10.0 + 1)),
       soya.model.Vertex(world, float(i - 10.0 + 1), heights[i + 1 + (j + 1) * W], float(j - 10.0 + 1))
      ],
      None)
    f.smooth_lit = 1
    i += 1
  j += 1

v = soya.soya3d.Volume(scene)
v.shape = world.shapify()


# Add a light
light = soya.soya3d.Light(scene)
light.set_xyz(0.0, 1.0, 0.0)


# Add a camera and a loop to render
roow = soya.widget.Group()
camera = soya.soya3d.Camera(scene)
camera.set_xyz(3.0, 0.0, 3.0)
camera.look_at(scene)
roow.add(camera)

label = soya.widget.Label(roow, " - Shadow demo -")
label.resize_style = soya.widget.WIDGET_RESIZE_MAXIMIZE

soya.set_root_widget(roow)


angle = 0.0
radius = 2.0

while(1):
  time.sleep(0.1)

# rotate the light
  angle += 0.1
  if angle > 180.0: angle -= 180.0
  light.x = math.cos(angle) * radius
  light.z = math.sin(angle) * radius

  for event in soya.process_event():

    if event[0] == soya.KEYDOWN:
      sys.exit()

  soya.render()


# Footnote:
# Shadows implementation use the Shadow Volume technic
# Notice that there is no visual bug when the camera is
# inside the shadow ;)
