# Soya 3D
# Copyright (C) 2001-2002 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os
import _soya, soya
    
WIDGET_RESIZE_MAXIMIZE = ('maximize', )
default_font = soya.load_font(os.path.join(soya.DATADIR, "font_time_22.data"))

# ----------------------------------------

class Widget(object):
  def __init__(self, master = None):
    # these attributes are named (left,top) and not (x,y) to avoid confusion with
    # the (x,y,z) attributes of 3D objects
    self.left = 0
    self.top  = 0
    self.width  = 0
    self.height = 0
    if not hasattr(self, 'resize_style'): self.resize_style = None
    # this attribute is not called parent for avoiding confusion with 3D objects parent
    # attribute and also because Camera is a 3D object and a Widget
    if master: master.add(self)
    else: self.master = None
  def resize(self, parent_left, parent_top, parent_width, parent_height):
    # this method can be rewritten for each widget instance or you can use the standard
    # resize style statement
    if self.resize_style:
      for e in self.resize_style:
        if type(e) == str:
          if   e == 'maximize':
            self.left   = parent_left
            self.top    = parent_top
            self.width  = parent_width
            self.height = parent_height
          elif e == 'parent left':
            self.left = parent_left
          elif e == 'parent top':
            self.left = parent_top
          elif e == 'maximize width':
            self.width = parent_width - self.left + parent_left
          elif e == 'maximize height':
            self.height = parent_height - self.top + parent_top
          elif e == 'center x':
            self.left = int((parent_width - self.width) * 0.5)
          elif e == 'center y':
            self.top = int((parent_height - self.height) * 0.5)
        else:
          if   e[0] == 'percent left':
            self.left = parent_left + int(parent_width * e[1])
          elif e[0] == 'percent top':
            self.top = parent_top + int(parent_height * e[1])
          elif e[0] == 'percent width':
            self.width = int(parent_width * e[1])
          elif e[0] == 'percent height':
            self.height = int(parent_height * e[1])
          elif e[0] == 'margin left':
            self.left  += e[1]
            self.width -= e[1]
          elif e[0] == 'margin top':
            self.top    += e[1]
            self.height -= e[1]
          elif e[0] == 'margin right':
            self.width -= e[1]
          elif e[0] == 'margin bottom':
            self.height -= e[1]
          elif e[0] == 'max width':
            if self.width > e[1]: self.width = e[1]
          elif e[0] == 'max height':
            if self.height > e[1]: self.height = e[1]
          elif e[0] == 'keep ratio':
            # ratio is width / height
            if self.height * e[1] < self.width:
              self.width = int(self.height * e[1])
            else:
              self.height = int(self.width / e[1])
      
  def begin_round(self): pass
  def advance_time(self, proportion): pass
  def end_round(self): pass

  
# ----------------------------------------

class Group(Widget):
  def __init__(self, master = None):
    self.children = []
    self.resize_style = WIDGET_RESIZE_MAXIMIZE
    Widget.__init__(self, master)
    self.visible = 1
  def render(self):
    if self.visible:
      for widget in self.children:
        widget.render()
  def resize(self, parent_left, parent_top, parent_width, parent_height):
    Widget.resize(self, parent_left, parent_top, parent_width, parent_height)
    for widget in self.children: widget.resize(parent_left, parent_top, parent_width, parent_height)
  def add(self, widget):
    self.children.append(widget)
    widget.resize(self.left, self.top, self.width, self.height)
    widget.master = self
  def insert(self, index, widget):
    self.children.insert(index, widget)
    widget.resize(self.left, self.top, self.width, self.height)
    widget.master = self
  def remove(self, widget):
    self.children.remove(widget)
    widget.master = None
  def begin_round(self):
    for widget in self.children: widget.begin_round()      
  def advance_time(self, proportion):
    for widget in self.children: widget.advance_time(proportion)      
  def end_round(self):
    for widget in self.children: widget.end_round()      
  
class Clearer(Widget):
  def __init__(self, master = None, color = (0.0, 0.0, 0.0, 1.0)):
    Widget.__init__(self)
    self.color = color
    if master: master.add(self)
  def render(self):
    soya.glClearWithColor(*self.color)


# ----------------------------------------

class Label(Widget):
  def __init__(self, master = None, text = "", align = 0, color = (1.0, 1.0, 1.0, 1.0), font = default_font):
    self.text  = text
    self.color = color
    self.font  = font
    # align: 0 left, 1 center
    self.align  = align
    Widget.__init__(self, master)
  def render(self):
    if self.text:
      _soya.glColor4f(*self.color)
      self.font.draw_area(self.text, self.left, self.top, self.width, self.height, self.align)
      
# ----------------------------------------

class Image(Widget):
  def __init__(self, master = None, material = None, color = (1.0, 1.0, 1.0, 1.0)):
    self.tex_mode = 0
    self.material = material
    self.tex_top    = 0.0
    self.tex_left   = 0.0
    self.tex_bottom = 1.0
    self.tex_right  = 1.0
    self.color = color
    # mode 0 : normal
    # mode 1 : texcoords follow the height and width of the widget
    Widget.__init__(self, master)
    
  def render(self):
    _soya.activate_material(self.material)
    if (self.material and self.material.is_alpha()) or self.color[3] < 1.0: _soya.glEnable(_soya.GL_BLEND)
    _soya.glColor4f(*self.color)
    _soya.glBegin(_soya.GL_QUADS)
    _soya.glTexCoord2f(self.tex_left, self.tex_top)
    _soya.glVertex2i(self.left, self.top)
    _soya.glTexCoord2f(self.tex_left, self.tex_bottom)
    _soya.glVertex2i(self.left, self.top + self.height)
    _soya.glTexCoord2f(self.tex_right, self.tex_bottom)
    _soya.glVertex2i(self.left + self.width, self.top + self.height)
    _soya.glTexCoord2f(self.tex_right, self.tex_top)
    _soya.glVertex2i(self.left + self.width, self.top)
    _soya.glEnd()
    _soya.activate_material(None)
    if (self.material and self.material.is_alpha()) or self.color[3] < 1.0: _soya.glDisable(_soya.GL_BLEND)
  def resize(self, parent_left, parent_top, parent_width, parent_height):
    Widget.resize(self, parent_left, parent_top, parent_width, parent_height)
    if self.material and self.tex_mode == 1:
      self.tex_right  = float(self.width)  / self.material.width  + self.tex_left
      self.tex_bottom = float(self.height) / self.material.height + self.tex_top


# ----------------------------------------

class FPSLabel(Widget):
  """FPSLabel

A label that shows the FPS.
It works ONLY with soya.idler !!!"""
  
  def resize(self, parent_left, parent_top, parent_width, parent_height):
    self.x = parent_left + parent_width  - 155
    self.y = parent_top  + parent_height - 37
    
  def render(self):
    import soya.idler as idler
    
    _soya.glColor4f(1.0, 1.0, 1.0, 1.0)
    
    if idler.IDLER:
      default_font.draw_area("%.1f FPS" % idler.IDLER.fps, self.x, self.y, 150, 30, 1)
    else:
      default_font.draw_area("??? FPS", self.x, self.y, 150, 30, 1)


# ----------------------------------------

class BannerElement:
  
  def __init__(self, text = "", color = (1.0, 1.0, 1.0, 1.0), font = default_font):
    self.text = text.encode("latin")
    self.color = color
    self.font = font

  def compute_size(self, width = None):
    if width == None:
      t = self.font.get_print_size(self.text)
    else:
      t = self.font.get_print_size(self.text, width)
    self.width  = t[0]
    self.height = t[1]


class Banner(Widget):

  def __init__(self, master = None, elements = []):
    self.valid = 0
    self.set_elements(elements)
    self.resize_style = WIDGET_RESIZE_MAXIMIZE
    Widget.__init__(self, master)
    self.loop = 1
    self.total = 0
    self.speed = 4.0
    self.visible = 1

  def set_elements(self, elements, separator = None):
    if separator:
      for e in elements:
        self.elements.append(separator)
        self.elements.append(e)
    else:
      self.elements = elements
    self.init()

  def add_elements_from_strings(self, strings = [], attribs = {}, separator = None):
    try:    color = attribs['color']
    except: color = (1.0, 1.0, 1.0, 1.0)
    try:    font  = attribs['font']
    except: font  = default_font
    if type(separator) == str:
      separator = BannerElement(separator, color, font)
    for s in strings:
      if separator: self.elements.append(separator)
      self.elements.append(BannerElement(s, color, font))
    self.init()
      
  def add_elements_from_tuples(self, tuples = [], key_attribs = {}, value_attribs = {}, separator = None):
    try:    k_color = key_attribs['color']
    except: k_color = (1.0, 1.0, 1.0, 1.0)
    try:    k_font  = key_attribs['font']
    except: k_font  = default_font
    try:    v_color = value_attribs['color']
    except: v_color = (1.0, 1.0, 1.0, 1.0)
    try:    v_font  = value_attribs['font']
    except: v_font  = default_font
    if type(separator) == str:
      separator = BannerElement(separator, v_color, v_font)
    for t in tuples:
      if separator: self.elements.append(separator)
      self.elements.append(BannerElement(t[0], k_color, k_font))
      self.elements.append(BannerElement(t[1], v_color, v_font))
    self.init()
    
  def advance_time(self, proportion):
    self._position += proportion * self.speed
    self.position = int(self._position)
    if self.position >= self.total:
      if self.loop:
        self._position -= self.total
#        self.position -= self.total
      else:
        if self.master: self.master.remove(self)


class HorizontalBanner(Banner):

  def __init__(self, master = None, elements = []):
    Banner.__init__(self, master, elements)
    self.position = - self.width
    self._position = self.position
    self.top = 0
    self.height = 0

  def init(self):
    for e in self.elements:
      e.compute_size()
      if e.font.height > self.height: self.height = e.font.height
      self.total += e.width
      
  def render(self):
    if self.visible:
      n = 0
      i = 0
      while (i < len(self.elements)):
        e = self.elements[i]
        n += e.width
        if n >= self.position:
          # element e is after position
          _soya.glColor4f(*e.color)
          e.font.draw_2D(e.text, n - e.width - self.position, self.top)
          while (n - self.position < self.width):
            i += 1
            if i >= len(self.elements):
              if self.loop:
                i = 0
              else:
                break
            e = self.elements[i]
            _soya.glColor4f(*e.color)
            e.font.draw_2D(e.text, n - self.position, self.top)
            n += e.width
          break
        i += 1
      _soya.glColor4f(1.0, 1.0, 1.0, 1.0)
      self.valid = 1

  def resize(self, parent_left, parent_top, parent_width, parent_height):
    Widget.resize(self, parent_left, parent_top, parent_width, parent_height)
    if self.valid == 0:
      self.position = - self.width
      self._position = -self.width


class VerticalBanner(Banner):

  def __init__(self, master = None, elements = []):
    Banner.__init__(self, master, elements)
    self.position = - self.height
    self._position = self.position
    self.alpha_zone = 150
    self.center_text = 1
    
  def init(self):
    self.total = 0
    for e in self.elements:
      e.compute_size(self.width)
      self.total += e.height
      
  def render(self):

    def render_element(e, y):

      def color_for_y(y, color):
        return (e.color[0], e.color[1], e.color[2], )
      
      if self.alpha_zone == 0:
        _soya.glColor4f(*e.color)
        e.font.draw_area(e.text, self.left, y, self.width, self.height, self.center_text)
      else:
        if y < self.alpha_zone + self.top:
          _soya.glColor4f(e.color[0], e.color[1], e.color[2], float(y - self.top) / self.alpha_zone)
        elif y > self.top + self.height - self.alpha_zone - e.height:
          _soya.glColor4f(e.color[0], e.color[1], e.color[2], float(self.height - y - e.height + self.top) / self.alpha_zone)
        else:
          _soya.glColor4f(*e.color)
        e.font.draw_area(e.text, self.left, y, self.width, self.height, self.center_text)

    if self.visible:
      n = self.top
      i = 0
      while (i < len(self.elements)):
        e = self.elements[i]
        o = n - self.position
        n += e.height
        if n >= self.position:
          # element e is after position
          render_element(e, o)
          while (n - self.position < self.height + self.top):
            i += 1
            if i >= len(self.elements):
              if self.loop:
                i = 0
              else:
                break
            e = self.elements[i]
            render_element(e, n - self.position)
            n += e.height
          break
        i += 1
      _soya.glColor4f(1.0, 1.0, 1.0, 1.0)
      self.valid = 1

  def resize(self, parent_left, parent_top, parent_width, parent_height):
    Widget.resize(self, parent_left, parent_top, parent_width, parent_height)
    if self.valid == 0:
      self.position = - self.height
      self._position = - self.height
    self.init()


# ----------------------------------------

class Choice:

  def __init__(self, label = '', action = None, value = None, range = None, incr = None):
    self.label  = label.encode('latin')
    self.action = action
    self.value  = value
    self.range  = range
    self.incr   = incr

  def get_label(self):
    if (self.value != None):
      if isinstance(self.value, unicode):
        return '%s - %s' % (self.label, self.value.encode('latin'))
      else:
        return '%s - %s' % (self.label, self.value)
    else: return self.label

  def increment(self):
    if (self.incr):
      self.value = self.value + self.incr
      if (self.range):
        if (self.value < self.range[0]): self.value = self.range[0]
        if (self.value > self.range[1]): self.value = self.range[1]
    else:
      nb = len(self.range)
      i = 0
      while (i < nb):
        if (self.range[i] == self.value):
          i = i + 1
          if (i >= nb): i = 0
          self.value = self.range[i]
          break
        i = i + 1

  def decrement(self):
    if (self.incr):
      self.value = self.value - self.incr
      if (self.range):
        if (self.value < self.range[0]): self.value = self.range[0]
        if (self.value > self.range[1]): self.value = self.range[1]
    else:
      nb = len(self.range)
      i = 0
      while (i < nb):
        if (self.range[i] == self.value):
          i = i - 1
          if (i < 0): i = nb - 1
          self.value = self.range[i]
          break
        i = i + 1    

  def mouse_click(self, button):
    if self.value == None:
      if button == soya.BUTTON_RIGHT or button == soya.BUTTON_LEFT:
        if self.action: self.action()
    else:
      if button == soya.BUTTON_RIGHT or button == 5:
        self.decrement()
      elif button == soya.BUTTON_LEFT or button == 4:
        self.increment()

  def key_down(self, key_id):
    if self.value == None:
      if (key_id == soya.K_RETURN or key_id == soya.K_KP_ENTER or key_id == soya.K_SPACE):
        if self.action: self.action()
    else:
      if (key_id == soya.K_RIGHT or key_id == soya.K_PLUS or key_id == soya.K_KP_PLUS or key_id == soya.K_RETURN or key_id == soya.K_KP_ENTER or key_id == soya.K_SPACE):
        self.increment()
      elif (key_id == soya.K_LEFT or key_id == soya.K_MINUS or key_id == soya.K_KP_MINUS):
        self.decrement()


class ChoiceInput:

  def __init__(self, label = '', value = ''):
    self.label  = label.encode('latin')
    self.value  = value
    self.grab   = 0

  def get_label(self):
    if (self.value != None):
      return '%s - %s' % (self.label, self.value)
    else: return self.label

  def mouse_click(self, button): pass

  def key_down(self, key_id):
    if (key_id == soya.K_DELETE or key_id == soya.K_BACKSPACE or key_id == soya.K_CLEAR):
      self.value = self.value[:-1]
    elif (key_id == soya.K_KP0 or key_id == soya.K_0): self.value += '0'
    elif (key_id == soya.K_KP1 or key_id == soya.K_1): self.value += '1'
    elif (key_id == soya.K_KP2 or key_id == soya.K_2): self.value += '2'
    elif (key_id == soya.K_KP3 or key_id == soya.K_3): self.value += '3'
    elif (key_id == soya.K_KP4 or key_id == soya.K_4): self.value += '4'
    elif (key_id == soya.K_KP5 or key_id == soya.K_5): self.value += '5'
    elif (key_id == soya.K_KP6 or key_id == soya.K_6): self.value += '6'
    elif (key_id == soya.K_KP7 or key_id == soya.K_7): self.value += '7'
    elif (key_id == soya.K_KP8 or key_id == soya.K_8): self.value += '8'
    elif (key_id == soya.K_KP9 or key_id == soya.K_9): self.value += '9'
    elif (key_id == soya.K_KP_PERIOD or key_id == soya.K_PERIOD): self.value += '.'
    else:
      if soya.get_mod() & (soya.KMOD_SHIFT | soya.KMOD_CAPS):
        if   (key_id >= soya.K_a and key_id <= soya.K_z):
          self.value += chr(key_id - 32)
        elif (key_id == soya.K_SEMICOLON): self.value += '.'
      else:
        if   (key_id >= soya.K_a and key_id <= soya.K_z):
          self.value += chr(key_id)
        elif (key_id >= 0 and key_id < 256):
          self.value += chr(key_id)


class ChoiceList(Widget):

  def __init__(self, master = None, choices = [], font = default_font, color = (1.0, 0.5, 0.5, 1.0), highlight = (1.0, 1.0, 0.0, 1.0), cancel = None):
    self.x_percent = 0.0
    self.y_percent = 0.0
    self.w_percent = 1.0
    self.h_percent = 1.0
    Widget.__init__(self, master)
    self.choices = choices
    self.font = font
    self.color = color
    self.highlight = highlight
    self.selected = 0
    self.visible = 1
    if master: master.add(self)
    if cancel != None: self.cancel = self.choices[cancel]
    else: self.cancel = None

  def process_event(self, event):
    if (event[0] == soya.KEYDOWN):
      self.key_down(event[1])
    elif (event[0] == soya.MOUSEMOTION):
      self.mouse_move(event[1], event[2])
    elif (event[0] == soya.MOUSEBUTTONDOWN):
      self.choices[self.selected].mouse_click(event[1])
    elif (event[0] == soya.JOYAXISMOTION):
      if event[1] == 0:
        if event[2] < 0:
          self.key_down(soya.K_LEFT)
        elif event[2] > 0:
          self.key_down(soya.K_RIGHT)
      elif event[1] == 1:
        if event[2] < 0:
          self.key_down(soya.K_UP)
        elif event[2] > 0:
          self.key_down(soya.K_DOWN)
    elif (event[0] == soya.JOYBUTTONDOWN):
      self.key_down(soya.K_RETURN)
  
  def key_down(self, key_id):
    if key_id == soya.K_DOWN:
      self.selected = self.selected + 1
      if (self.selected >= len(self.choices)): self.selected = 0
    elif key_id == soya.K_UP:
      self.selected = self.selected - 1
      if (self.selected < 0): self.selected = len(self.choices) - 1
    elif key_id == soya.K_ESCAPE :
      if self.cancel:
        self.cancel.key_down(soya.K_RETURN)
    else: self.choices[self.selected].key_down(key_id)
    
  def mouse_move(self, x, y):
    if (x >= self.left and x <= self.left + self.width):
      i = 0
      nb = len(self.choices)
      h1 = int (self.height / nb)
      h2 = int (h1 * 0.5)
      t = int (self.top + h1 * 0.5)
      while (i < nb):
        if (y >= t - h2 and y <= t + h2):
          self.selected = i
          break
        t = t + h1
        i = i + 1
        
  def render(self):
    if self.visible:
      nb = len (self.choices)
      hi = self.height / nb
      h = int (self.top + hi * 0.5) - 18
      i = 0
      while (i < nb):
        if (i == self.selected):
          _soya.glColor4f(*self.highlight)
        else:
          _soya.glColor4f(*self.color)
        _soya.glDisable(_soya.GL_LIGHTING)
        self.font.draw_area(self.choices[i].get_label(), self.left, h, self.width, h + hi, 1)
        h = h + hi
        i = i + 1


# ----------------------------------------

class Selector(Widget):

  def __init__(self, master = None, choices = [], title = None, font = default_font, color = (1.0, 0.5, 0.5, 1.0), highlight = (1.0, 1.0, 0.0, 1.0), arrows = None):
    self.x_percent = 0.0
    self.y_percent = 0.0
    self.w_percent = 1.0
    self.h_percent = 1.0
    self.arrow_size = 32
    self.choices = choices
    self.title = title.encode('latin')
    self.nb_displayed = 5
    self.font = font
    self.arrows = arrows
    Widget.__init__(self, master)
    self.selected  = 0
    self.display_from = 0
    self.selected_arrow = 0
    self.color = color
    self.highlight = highlight
    self.visible = 1
    self.enabled = 0
    self.scrolling = 0
    
  def resize(self, parent_left, parent_top, parent_width, parent_height):
    Widget.resize(self, parent_left, parent_top, parent_width, parent_height)
    a = self.height - self.font.height
    if self.arrows: a -= 2 * self.arrow_size
    self.nb_displayed = a / self.font.height
    if not self.nb_displayed & 1: self.nb_displayed -= 1
    if self.arrows:
      self._h = int(self.top + (self.height - self.font.height * (self.nb_displayed + 1) - 2 * self.arrow_size) * 0.5)
    else:
      self._h = int(self.top + (self.height - self.font.height * (self.nb_displayed + 1)) * 0.5)

  def get_choice_at(self, y):
    h = self._h
    if y < h: return -4
    if self.title:
      h += self.font.height
      if (y < h):
        return -1
    h += self.arrow_size
    if (y < h): return -2
    else:
      i = 0
      while(i < self.nb_displayed):
        h += self.font.height
        if (y < h): return self.display_from + i
        i += 1
      h += self.arrow_size
      if (y < h): return -3
    return -4
  
  def scroll_up(self):
    if (self.display_from > 0): self.display_from -= 1
    
  def scroll_down(self):
    if (self.display_from < len(self.choices) - self.nb_displayed): self.display_from += 1
    
  def scroll(self):
    if (self.scrolling > 0):
      if (self.selected + 1 < len(self.choices)): self.selected += 1
      if (self.selected - self.display_from > (self.nb_displayed - 1) * 0.5):
        self.scroll_down()
    elif (self.scrolling < 0):
      if (self.selected > 0): self.selected -= 1
      if (self.selected - self.display_from < (self.nb_displayed - 1) * 0.5):
        self.scroll_up()

  def select(self, index):
    if (index >= 0 and index < len(self.choices)):
      self.selected = index
      self.display_from = int(index - (self.nb_displayed - 1) * 0.5)
      if   (self.display_from < 0): self.display_from = 0
      elif (self.display_from > len(self.choices) - self.nb_displayed): self.display_from = len(self.choices) - self.nb_displayed
      
  def process_event(self, event):
    if (event[0] == soya.KEYDOWN):
      if (event[1] == soya.K_DOWN):
        self.scrolling = 1
#        if (self.selected + 1 < len(self.choices)): self.selected += 1
#        if (self.selected - self.display_from > (self.nb_displayed - 1) * 0.5):
#          self.scroll_down()
      elif (event[1] == soya.K_UP):
        self.scrolling = -1
#        if (self.selected > 0): self.selected -= 1
#        if (self.selected - self.display_from < (self.nb_displayed - 1) * 0.5):
#          self.scroll_up()
    elif (event[0] == soya.KEYUP):
      self.scrolling = 0
    elif (event[0] == soya.MOUSEMOTION):
      i = self.get_choice_at(event[2])
      if   (i == -2): self.selected_arrow = 1
      elif (i == -3): self.selected_arrow = 2
      else:           self.selected_arrow = 0
    elif (event[0] == soya.MOUSEBUTTONDOWN):
      if   (event[1] == 4): self.scroll_up()
      elif (event[1] == 5): self.scroll_down()
      else:
        i = self.get_choice_at(event[3])
        if (i >= 0): self.select(i)
        elif (i == -2): self.scroll_up()
        elif (i == -3): self.scroll_down()
    elif (event[0] == soya.JOYAXISMOTION):
      if (event[1] == 1):
        if (event[2] == 0):
          self.scrolling = 0
        elif (event[2] > 0):
          self.scrolling = 1
#          if (self.selected + 1 < len(self.choices)): self.selected += 1
#          if (self.selected - self.display_from > (self.nb_displayed - 1) * 0.5):
#            self.scroll_down()
        elif (event[2] < 0):
          self.scrolling = -1
#          if (self.selected > 0): self.selected -= 1
#          if (self.selected - self.display_from < (self.nb_displayed - 1) * 0.5):
#            self.scroll_up()
    self.scroll()

  def draw_arrow(self, top, yoffset):
    x = int((self.width - self.left - self.arrow_size) * 0.5)
    soya.glTexCoord2f(0.0, yoffset)
    soya.glVertex2i(x, top)
    soya.glTexCoord2f(0.0, yoffset + 0.5)
    soya.glVertex2i(x, top + self.arrow_size)
    soya.glTexCoord2f(1.0, yoffset + 0.5)
    soya.glVertex2i(x + self.arrow_size, top + self.arrow_size)
    soya.glTexCoord2f(1.0, yoffset)
    soya.glVertex2i(x + self.arrow_size, top)
    
  def render(self):
    if self.visible:

      if (self.scrolling > 0):
        self.scrolling += 1
        if (self.scrolling > 8):
          self.scrolling = 1
          self.scroll()
      elif (self.scrolling < 0):
        self.scrolling -= 1
        if (self.scrolling < -8):
          self.scrolling = -1
          self.scroll()

      h = self._h
      if self.title:
        if self.enabled:
          soya.glColor4f(*self.highlight)
        else:
          soya.glColor4f(*self.color)
        self.font.draw_area(self.title, self.left, h, self.width, self.font.height, 1)
        h += self.font.height

      if self.arrows:
        self.arrows.activate()
        if self.arrows.is_alpha(): _soya.glEnable(_soya.GL_BLEND)
        soya.glBegin(soya.GL_QUADS)
        if self.display_from > 0:
          if self.selected_arrow == 1:
            soya.glColor4f(*self.highlight)
          else:
            soya.glColor4f(*self.color)
          self.draw_arrow(h, 0)
        h += self.arrow_size
        if self.display_from < len(self.choices) - self.nb_displayed:
          if self.selected_arrow == 2:
            soya.glColor4f(*self.highlight)
          else:
            soya.glColor4f(*self.color)
          self.draw_arrow(h + self.font.height * self.nb_displayed, 0.5)
        soya.glEnd()
        _soya.activate_material(None)
        if self.arrows.is_alpha(): _soya.glDisable(_soya.GL_BLEND)

      i = 0
      while(i < self.nb_displayed):
        index = i + self.display_from
        c = self.choices[index]
        if (self.selected == index): 
          soya.glColor4f(*self.highlight)
        else:
          soya.glColor4f(*self.color)
        self.font.draw_area(c, self.left, h, self.width, h + self.font.height, 1)
        h += self.font.height
        if (index == len(self.choices) - 1): break
        i += 1





## # TO DO
## class OptimizedLabel(Widget):
##   def __init__(self, master = None, text = "", align = 0, color = (1.0, 1.0, 1.0, 1.0), font = default_font):
##     self._text = text
##     self._color = color
##     # font must be a TextureFont or it crashes (you MUST NOT use a RasterFont)
##     self.font = font
##     # align: 0 left, 1 center
##     self.align = align
##     self.call_list = -1
##     self.valid = 0
##     Widget.__init__(self, master)

## # Cause troubles with cyclic references
    
## #  def __del__(self):
## #    _soya.glDeleteList(self.call_list)
## #    self.call_list = -1
## #    self.valid = 0
##   def render(self):
##     if self._text:
##       if self.valid == 0:
##         if self.call_list == -1: self.call_list = _soya.glGenList()
##         _soya.glNewList(self.call_list)
##         _soya.glColor4f(*self._color)
##         self.height = self.font.draw_area(self._text, 0, 0, self.width, -1, self.align)
##         _soya.glEndList()
##         self.valid = 1
      
##       self.font.set_position(self.left, self.top)
##       _soya.glCallList(self.call_list)
##       self.font.set_position()
##   def resize(self, parent_left, parent_top, parent_width, parent_height):
##     self.width = parent_width
##     self.valid = 0
    
##   def get_text(self): return self._text
##   def set_text(self, text):
##     self._text = text
##     self.valid = 0
##   text = property(get_text, set_text)
  
##   def get_color(self): return self._color
##   def set_color(self, color):
##     self._color = color
##     self.valid = 0
##   color = property(get_color, set_color)
