///###////////////////////////////////////////////////////////////////////////
//
// Burton Computer Corporation
// http://www.burton-computer.com
// http://www.cooldevtools.com
// $Id: File.cc 103 2005-03-24 22:03:21Z brian $
//
// Copyright (C) 2000 Burton Computer Corporation
// ALL RIGHTS RESERVED
//
// This program is open source software; you can redistribute it
// and/or modify it under the terms of the Q Public License (QPL)
// version 1.0. Use of this software in whole or in part, including
// linking it (modified or unmodified) into other programs is
// subject to the terms of the QPL.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// Q Public License for more details.
//
// You should have received a copy of the Q Public License
// along with this program; see the file LICENSE.txt.  If not, visit
// the Burton Computer Corporation or CoolDevTools web site
// QPL pages at:
//
//    http://www.burton-computer.com/qpl.html
//    http://www.cooldevtools.com/qpl.html
//

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdexcept>
#include "File.h"

const char SEPARATOR_CHAR('/');
const string SEPARATOR("/");

File::File()
{
}

File::File(const string &path)
{
  setPath(path);
}

File::File(const File &other)
  : m_path(other.m_path)
{
}

File::File(const File &other,
           const string &name)
  : m_path(other.m_path)
{
  addComponent(name);
}

File::~File()
{
}

File &File::operator=(const File &other)
{
  m_path = other.m_path;
  return *this;
}

void File::addComponent(const string &name)
{
  if (m_path != SEPARATOR) {
    m_path += SEPARATOR;
  }
  string::size_type last = name.length() - 1;
  if (last >= 1 && name[last] == SEPARATOR_CHAR) {
    m_path.append(name, 0, last);
  } else {
    m_path += name;
  }
}

void File::setPath(const string &path)
{
  string::size_type last = path.length() - 1;
  if (last >= 1 && path[last] == SEPARATOR_CHAR) {
    m_path.assign(path, 0, last);
  } else {
    m_path = path;
  }
}

bool File::exists() const
{
  struct stat buf;
  return stat(m_path.c_str(), &buf) == 0;
}

bool File::isFile() const
{
  struct stat buf;
  return stat(m_path.c_str(), &buf) == 0 && S_ISREG(buf.st_mode);
}

bool File::isDirectory() const
{
  struct stat buf;
  return stat(m_path.c_str(), &buf) == 0 && S_ISDIR(buf.st_mode);
}

unsigned long File::getSize() const
{
  struct stat buf;
  return stat(m_path.c_str(), &buf) == 0 ? buf.st_size : 0;
}

File File::parent() const
{
  string::size_type pos = m_path.rfind(SEPARATOR);
  if (pos == string::npos) {
    return File("..");
  } else {
    return File(m_path.substr(0, pos));
  }
}

void File::makeDirectory(int mode) const
{
  if (mkdir(m_path.c_str(), (mode_t)mode) != 0) {
    throw runtime_error(strerror(errno));
  }
}

void File::rename(const string &new_path)
{
  if (::rename(m_path.c_str(), new_path.c_str()) != 0) {
    throw runtime_error(strerror(errno));
  }
  m_path = new_path;
}

void File::remove()
{
  if (unlink(m_path.c_str()) != 0) {
    throw runtime_error(strerror(errno));
  }
}

File File::getHomeDir()
{
  char *home = getenv("HOME");
  if (home == NULL) {
    return File();
  } else {
    return File(home);
  }
}
