///###////////////////////////////////////////////////////////////////////////
//
// Burton Computer Corporation
// http://www.burton-computer.com
// http://www.cooldevtools.com
// $Id: LRUPtrCache.h 80 2004-11-07 04:01:42Z brian $
//
// Copyright (C) 2000 Burton Computer Corporation
// ALL RIGHTS RESERVED
//
// This program is open source software; you can redistribute it
// and/or modify it under the terms of the Q Public License (QPL)
// version 1.0. Use of this software in whole or in part, including
// linking it (modified or unmodified) into other programs is
// subject to the terms of the QPL.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// Q Public License for more details.
//
// You should have received a copy of the Q Public License
// along with this program; see the file LICENSE.txt.  If not, visit
// the Burton Computer Corporation or CoolDevTools web site
// QPL pages at:
//
//    http://www.burton-computer.com/qpl.html
//    http://www.cooldevtools.com/qpl.html
//

#ifndef _LRUPtrCache_h
#define _LRUPtrCache_h

#include <map>
#include <list>
#include <cassert>
#include "NewPtr.h"

using namespace std;

template<class KeyType, class ValueType> class LRUPtrCache
{
  struct Node {
    Node()
      : value(0), next(this), prev(this)
    {
    }

    Node(const KeyType &_key, ValueType *_value, bool _isLocked)
      : key(_key), value(_value), isLocked(_isLocked), next(this), prev(this)
    {
    }

    ~Node()
    {
      delete value;
    }

    void setValue(ValueType *new_value)
    {
      if (new_value != value) {
        delete value;
        value = new_value;
      }
    }

    void linkAfter(Node *other)
    {
      prev->next = next;
      next->prev = prev;
      next = other->next;
      prev = other;
      next->prev = this;
      prev->next = this;
    }

    void unlink()
    {
      prev->next = next;
      next->prev = prev;
      prev = this;
      next = this;
    }

    KeyType key;
    ValueType *value;
    bool isLocked;
    Node *next, *prev;
  };

  typedef map<KeyType,Node *> MapType;
  typedef typename MapType::iterator MapIteratorType;

public:
  class Iterator
  {
    friend class LRUPtrCache;

    Iterator(MapType *index,
             MapIteratorType iter)
      : m_index(index), m_iterator(iter)
    {
    }

  public:
    bool atEnd()
    {
      return m_iterator == m_index->end();
    }

    bool next()
    {
      ++m_iterator;
      return !atEnd();
    }

    const KeyType &key()
    {
      assert(!atEnd());
      return m_iterator->first;
    }

    const ValueType *value()
    {
      assert(!atEnd());
      return m_iterator->second->value;
    }

    bool isLocked()
    {
      assert(!atEnd());
      return m_iterator->second->isLocked;
    }

  private:
    MapType *m_index;
    MapIteratorType m_iterator;
  };

public:
  LRUPtrCache(int max_size)
    : m_maxSize(max_size), m_disposableCount(0)
  {
  }

  ~LRUPtrCache()
  {
    clear();
  }

  ValueType *get(const KeyType &key)
  {
    MapIteratorType i = m_index.find(key);
    if (i == m_index.end()) {
      return 0;
    }

    Node *node = i->second;

    if (!node->isLocked) {
      node->linkAfter(&m_disposableList);
    }

    assert(node->isLocked || node == m_disposableList.next);
    return node->value;
  }

  void put(const KeyType &key,
           ValueType *value,
           bool is_locked)
  {
    MapIteratorType i = m_index.find(key);
    if (i == m_index.end()) {
      addNewNode(key, value, is_locked);
    } else {
      Node *node = i->second;
      setLocked(node, is_locked);
      node->setValue(value);
    }
  }

  void dump()
  {
    cerr << "locked: " << lockedCount() << ": ";
    cerr << "locked: ";
    dump(&m_lockedList);
    cerr << "disposable: " << unlockedCount() << ": ";
    dump(&m_disposableList);
  }

  void clear()
  {
    clear(&m_disposableList);
    clear(&m_lockedList);
    m_index.clear();
  }

  Iterator first()
  {
    return Iterator(&m_index, m_index.begin());
  }

  Iterator from(const KeyType &key)
  {
    return Iterator(&m_index, m_index.find(key));
  }

  void unlock(const KeyType &key)
  {
    setLocked(key, false);
  }

  void lock(const KeyType &key)
  {
    setLocked(key, true);
  }

  int size()
  {
    return m_index.size();
  }

  int maxSize()
  {
    return m_maxSize;
  }

  int lockedCount()
  {
    return size() - m_disposableCount;
  }

  int unlockedCount()
  {
    return m_disposableCount;
  }

private:
  void addNewNode(const KeyType &key,
                  ValueType *value,
                  bool is_locked)
  {
    if (!is_locked) {
      eliminateOldObjects();
    }

    Node *node = new Node(key, value, is_locked);
    if (is_locked) {
      node->linkAfter(&m_lockedList);
    } else {
      node->linkAfter(&m_disposableList);
      m_disposableCount += 1;
    }
    m_index.insert(make_pair(key, node));
  }

  void setLocked(const KeyType &key,
                 bool is_locked)
  {
    MapIteratorType i = m_index.find(key);
    if (i != m_index.end()) {
      setLocked(i->second, is_locked);
    }
  }

  void setLocked(Node *node,
                 bool is_locked)
  {
    if (!is_locked) {
      eliminateOldObjects();
    }
    if (is_locked) {
      if (!node->isLocked) {
        m_disposableCount -= 1;
        node->linkAfter(&m_lockedList);
        node->isLocked = true;
      }
    } else {
      if (node->isLocked) {
        m_disposableCount += 1;
        node->linkAfter(&m_disposableList);
        node->isLocked = false;
      }
    }
  }

  void eliminateOldObjects()
  {
    while (m_disposableCount >= m_maxSize) {
      Node *oldest = m_disposableList.prev;
      oldest->unlink();
      m_disposableCount -= 1;
      m_index.erase(oldest->key);
      delete oldest;
    }
  }

  void dump(Node *node)
  {
    for (Node *n = node->next; n != node; n = n->next) {
      dump(*n);
    }
    cerr << endl;
  }

  void dump(Node &node)
  {
    cerr << "Node(" << node.key << "," << node.value << "," << node.isLocked << "," << ")";
  }

  void clear(Node *node)
  {
    Node *current = node->next;
    while (current != node) {
      Node *next = current->next;
      current->unlink();
      delete current;
      current = next;
    }
  }

private:
  int m_maxSize;
  Node m_disposableList;
  Node m_lockedList;
  MapType m_index;
  int m_disposableCount;
};

#endif
