-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure up_wf_store (Node   : in     STree.SyntaxNode;
                       Scope  : in     Dictionary.Scopes;
                       EStack : in out ExpStack.ExpStackType) is
   TypeInfo, ExpResult : Exp_Record;
   Sym                 : Dictionary.Symbol;
   FieldIdentNode      : STree.SyntaxNode;
   FieldIdent          : LexTokenManager.Lex_String;
   FieldSymbol         : Dictionary.Symbol;
   ErrorFound          : Boolean := False;

   function BranchesFound (StartNode, EndNode : STree.SyntaxNode) return Boolean
   --# global in STree.Table;
   is
      NextNode : STree.SyntaxNode;
      result   : Boolean;
   begin
      result   := False;
      NextNode := Parent_Node (Current_Node => StartNode);
      while NextNode /= EndNode loop
         if Next_Sibling (NextNode) /= STree.NullNode then
            result := True;
            exit;
         end if;
         NextNode := Parent_Node (Current_Node => NextNode);
      end loop;
      return result;
   end BranchesFound;

begin
   ExpStack.Pop (ExpResult, EStack);
   ExpStack.Pop (TypeInfo, EStack);
   Sym := TypeInfo.Other_Symbol;

   if Dictionary.IsArrayTypeMark (TypeInfo.Type_Symbol, Scope) then
      if TypeInfo.Param_Count = Dictionary.GetNumberOfDimensions (TypeInfo.Type_Symbol) then
         --right number of index expressions so just check type check needed
         if not Dictionary.CompatibleTypes
           (Scope,
            Dictionary.GetArrayComponent (TypeInfo.Type_Symbol),
            ExpResult.Type_Symbol) then
            ErrorFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 38,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Next_Sibling (Child_Node (Node))),
               Id_Str    => LexTokenManager.Null_String);
         end if;
      else
         --insufficient index expressions
         TypeInfo := UnknownSymbolRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 93,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Dictionary.GetSimpleName (Sym));
      end if;

   else
      --  must be record multiple field name check
      if Syntax_Node_Type (Node => Child_Node (Child_Node (Node))) = SPSymbols.store_list then
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 324,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      else
         FieldIdentNode := Last_Child_Of (Start_Node => Node);
         if Syntax_Node_Type (Node => FieldIdentNode) = SPSymbols.identifier then
            if BranchesFound (FieldIdentNode, Child_Node (Node)) then
               ErrorFound := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 102,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => FieldIdentNode),
                  Id_Str    => Dictionary.GetSimpleName (TypeInfo.Other_Symbol));
            else
               FieldIdent  := Node_Lex_String (Node => FieldIdentNode);
               FieldSymbol :=
                 Dictionary.LookupSelectedItem
                 (Prefix   => TypeInfo.Type_Symbol,
                  Selector => FieldIdent,
                  Scope    => Scope,
                  Context  => Dictionary.ProofContext);
               if FieldSymbol = Dictionary.NullSymbol or else not Dictionary.IsRecordComponent (FieldSymbol) then
                  --no such field
                  ErrorFound := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 8,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => FieldIdentNode),
                     Id_Str    => FieldIdent);
               else --field name exists so type check of result assigned to it required
                  if Dictionary.CompatibleTypes (Scope, Dictionary.GetType (FieldSymbol), ExpResult.Type_Symbol) then
                     STree.Set_Node_Lex_String (Sym  => FieldSymbol,
                                                Node => FieldIdentNode);
                  else
                     ErrorFound := True;
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 38,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => Next_Sibling (Child_Node (Node))),
                        Id_Str    => LexTokenManager.Null_String);
                  end if;
               end if;
            end if;
         else --identifier not found
            ErrorFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 102,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => FieldIdentNode),
               Id_Str    => Dictionary.GetSimpleName (TypeInfo.Other_Symbol));
         end if;
      end if;
   end if;
   TypeInfo.Errors_In_Expression := ErrorFound or TypeInfo.Errors_In_Expression or ExpResult.Errors_In_Expression;
   ExpStack.Push (TypeInfo, EStack);

end up_wf_store;
