-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Debug;
with SPrint;
with E_Strings;

separate (Sem.CompUnit)
procedure WalkAnnotationExpression
  (Exp_Node       : in     STree.SyntaxNode;
   Scope          : in     Dictionary.Scopes;
   Type_Context   : in     Dictionary.Symbol;
   Context        : in     Anno_Tilde_Context;
   Result         :    out Exp_Record;
   Component_Data : in out ComponentManager.ComponentData) is

   Next_Node, Last_Node, Local_Node : STree.SyntaxNode;
   Node_Type                        : SPSymbols.SPSymbol;
   L_Heap                           : Lists.List_Heap;
   Val                              : Maths.Value;
   E_Stack                          : ExpStack.ExpStackType;
   T_Stack                          : TypeContextStack.TStackType;
   Sym                              : Dictionary.Symbol;
   Ref_Var                          : SeqAlgebra.Seq;
   L_Scope                          : Dictionary.Scopes;
   String_Value                     : LexTokenManager.Lex_String;

   --------------------------------------------------------------
   -- Procedures for debugging Expression syntax and tree walking
   -- These are hidden with "derives ;" so as not to pollute the
   -- annotations
   --------------------------------------------------------------

   procedure Dump_Syntax_Tree
   --# derives ;
   is
      --# hide Dump_Syntax_Tree;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPrint.Dump_Syntax_Tree (Exp_Node, 0);
      end if;
   end Dump_Syntax_Tree;

   procedure Dump_Down_Node
   --# derives ;
   is
      --# hide Dump_Down_Node;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPARK_IO.Put_Line
           (SPARK_IO.Standard_Output,
            "WalkAnnotationExpression DOWN encounters node" &
              ExaminerConstants.RefType'Image (STree.NodeToRef (Last_Node)) &
              ' ' &
              SPSymbols.SPSymbol'Image (Syntax_Node_Type (Node => Last_Node)),
            0);
      end if;
   end Dump_Down_Node;

   procedure Dump_Up_Node
   --# derives ;
   is
      --# hide Dump_Up_Node;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPARK_IO.Put_Line
           (SPARK_IO.Standard_Output,
            "WalkAnnotationExpression UP   encounters node" &
              ExaminerConstants.RefType'Image (STree.NodeToRef (Last_Node)) &
              ' ' &
              SPSymbols.SPSymbol'Image (Syntax_Node_Type (Node => Last_Node)),
            0);
      end if;
   end Dump_Up_Node;

   procedure Dump_Result
   --# derives ;
   is
      --# hide Dump_Result;
   begin
      if CommandLineData.Content.Debug.Expressions then
         if Result.Is_Constant then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, "WalkAnnotationExpression constant result is ", 0);
            E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                                E_Str => Maths.ValueToString (Result.Value));
         else
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "WalkAnnotationExpression result is not constant", 0);
         end if;
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "WalkExpression Result is: ", 0);
         Put_Exp_Record (Result);
      end if;
   end Dump_Result;

begin -- WalkAnnotationExpression

   Dump_Syntax_Tree;

   ExpStack.Init (E_Stack);
   AggregateStack.Init;
   Lists.Init (L_Heap);
   TypeContextStack.Init (T_Stack);
   TypeContextStack.Push (Type_Context, T_Stack);

   L_Scope := Scope;
   --  scope may change locally in loops but will always be back to
   --  original scope on exit from procedure.  In all calls below
   --  L_Scope replaces Scope.

   SeqAlgebra.CreateSeq (TheHeap => TheHeap,
                         S       => Ref_Var);

   Next_Node := Exp_Node;
   loop -----------------------------------------------------down loop----
      Last_Node := Next_Node;
      Node_Type := Syntax_Node_Type (Node => Last_Node);

      Dump_Down_Node;

      case Node_Type is
         when SPSymbols.character_literal =>
            -- ASSUME Last_Node = character_literal
            GetCharacterLiteral (Last_Node, Val);
            ExpStack.Push
              (Exp_Record'(Type_Symbol             => Dictionary.GetPredefinedCharacterType,
                           Other_Symbol            => Dictionary.NullSymbol,
                           Stream_Symbol           => Dictionary.NullSymbol,
                           Tagged_Parameter_Symbol => Dictionary.NullSymbol,
                           Variable_Symbol         => Dictionary.NullSymbol,
                           Param_Count             => 0,
                           Param_List              => Lists.Null_List,
                           Sort                    => Type_Result,
                           Arg_List_Found          => False,
                           Is_AVariable            => False,
                           Is_An_Entire_Variable   => False,
                           Errors_In_Expression    => False,
                           Has_Operators           => False,
                           Is_Constant             => True,
                           Is_Static               => True,
                           Is_ARange               => False,
                           String_Value            => LexTokenManager.Null_String,
                           Value                   => Val,
                           Range_RHS               => Maths.NoValue),
               E_Stack);
            Next_Node := STree.NullNode;
         when SPSymbols.string_literal =>
            -- ASSUME Last_Node = string_literal
            GetStringLiteralLength (Last_Node, Val);
            String_Value := Node_Lex_String (Last_Node);

            --# accept F, 41, "Stable expression here OK";
            if CommandLineData.Content.Debug.Expressions then
               Debug.Print_Lex_Str ("WalkAnnotationExpression pushing string literal with value: ", String_Value);
            end if;
            --# end accept;

            ExpStack.Push
              (Exp_Record'(Type_Symbol             => Dictionary.GetPredefinedStringType,
                           Other_Symbol            => Dictionary.NullSymbol,
                           Stream_Symbol           => Dictionary.NullSymbol,
                           Tagged_Parameter_Symbol => Dictionary.NullSymbol,
                           Variable_Symbol         => Dictionary.NullSymbol,
                           Param_Count             => 0,
                           Param_List              => Lists.Null_List,
                           Sort                    => Type_Result,
                           Arg_List_Found          => False,
                           Is_AVariable            => False,
                           Is_An_Entire_Variable   => False,
                           Errors_In_Expression    => False,
                           Has_Operators           => False,
                           Is_Constant             => True,
                           Is_Static               => (CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83),
                           Is_ARange               => False,
                           String_Value            => String_Value,
                           Value                   => Maths.NoValue,
                           Range_RHS               => Val),
               E_Stack);
            Next_Node := STree.NullNode;
         when SPSymbols.numeric_literal =>
            -- ASSUME Last_Node = numeric_literal
            Local_Node := Child_Node (Current_Node => Child_Node (Current_Node => Last_Node));
            GetLiteralValue (Local_Node, Val);
            -- ASSUME Local_Node = integer_number OR real_number OR based_integer OR based_real
            if Syntax_Node_Type (Node => Local_Node) = SPSymbols.real_number then
               -- ASSUME Local_Node = real_number
               ExpStack.Push
                 (Exp_Record'(Type_Symbol             => Dictionary.GetUniversalRealType,
                              Other_Symbol            => Dictionary.NullSymbol,
                              Stream_Symbol           => Dictionary.NullSymbol,
                              Tagged_Parameter_Symbol => Dictionary.NullSymbol,
                              Variable_Symbol         => Dictionary.NullSymbol,
                              Param_Count             => 0,
                              Param_List              => Lists.Null_List,
                              Sort                    => Type_Result,
                              Arg_List_Found          => False,
                              Is_AVariable            => False,
                              Is_An_Entire_Variable   => False,
                              Errors_In_Expression    => False,
                              Has_Operators           => False,
                              Is_Constant             => True,
                              Is_Static               => True,
                              Is_ARange               => False,
                              String_Value            => LexTokenManager.Null_String,
                              Value                   => Val,
                              Range_RHS               => Maths.NoValue),
                  E_Stack);
            elsif Syntax_Node_Type (Node => Local_Node) = SPSymbols.integer_number
              or else Syntax_Node_Type (Node => Local_Node) = SPSymbols.based_integer
              or else Syntax_Node_Type (Node => Local_Node) = SPSymbols.based_real then
               -- ASSUME Local_Node = integer_number OR based_integer OR based_real
               ExpStack.Push
                 (Exp_Record'(Type_Symbol             => Dictionary.GetUniversalIntegerType,
                              Other_Symbol            => Dictionary.NullSymbol,
                              Stream_Symbol           => Dictionary.NullSymbol,
                              Tagged_Parameter_Symbol => Dictionary.NullSymbol,
                              Variable_Symbol         => Dictionary.NullSymbol,
                              Param_Count             => 0,
                              Param_List              => Lists.Null_List,
                              Sort                    => Type_Result,
                              Arg_List_Found          => False,
                              Is_AVariable            => False,
                              Is_An_Entire_Variable   => False,
                              Errors_In_Expression    => False,
                              Has_Operators           => False,
                              Is_Constant             => True,
                              Is_Static               => True,
                              Is_ARange               => False,
                              String_Value            => LexTokenManager.Null_String,
                              Value                   => Val,
                              Range_RHS               => Maths.NoValue),
                  E_Stack);
            else
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     =>
                    "Expect Local_Node = integer_number OR real_number OR based_integer OR based_real in WalkAnnotationExpression");
            end if;
            Next_Node := STree.NullNode;
         when SPSymbols.annotation_selector =>
            -- ASSUME Last_Node = annotation_selector
            Next_Node := STree.NullNode;
         when SPSymbols.annotation_simple_name =>
            -- ASSUME Last_Node = annotation_simple_name
            if Syntax_Node_Type (Node => Parent_Node (Current_Node => Last_Node)) =
              SPSymbols.annotation_named_argument_association then
               -- do not look at identifier in this case
               Next_Node := STree.NullNode;
            else
               Next_Node := Child_Node (Current_Node => Last_Node);
            end if;
         when SPSymbols.identifier =>
            -- ASSUME Last_Node = identifier
            wf_identifier
              (Node         => Last_Node,
               Scope        => L_Scope,
               EStack       => E_Stack,
               RefVar       => Ref_Var,
               Context      => Context,
               IsAnnotation => True);
            Next_Node := STree.NullNode;
         when SPSymbols.annotation_name_argument_list =>
            -- ASSUME Last_Node = annotation_name_argument_list
            down_wf_name_argument_list
              (Node         => Last_Node,
               Scope        => L_Scope,
               EStack       => E_Stack,
               HeapParam    => L_Heap,
               Next_Node    => Next_Node,
               IsAnnotation => True);
         when SPSymbols.annotation_aggregate =>
            -- ASSUME Last_Node = annotation_aggregate
            down_wf_aggregate
              (Node         => Last_Node,
               Scope        => L_Scope,
               Next_Node    => Next_Node,
               EStack       => E_Stack,
               HeapParam    => L_Heap,
               IsAnnotation => True);
         when SPSymbols.annotation_aggregate_choice_rep =>
            -- ASSUME Last_Node = annotation_aggregate_choice_rep
            down_wf_aggregate_choice_rep
              (Node         => Last_Node,
               Scope        => L_Scope,
               EStack       => E_Stack,
               HeapParam    => L_Heap,
               Next_Node    => Next_Node,
               IsAnnotation => True);
         when SPSymbols.record_component_selector_name =>
            -- ASSUME Last_Node = record_component_selector_name
            Wf_Record_Component_Selector_Name (Node       => Last_Node,
                                               Scope      => Scope,
                                               E_Stack    => E_Stack,
                                               Heap_Param => L_Heap);
            Next_Node := STree.NullNode;
         when SPSymbols.annotation_aggregate_or_expression =>
            -- ASSUME Last_Node = annotation_aggregate_or_expression
            down_wf_aggregate_or_expression (Node         => Last_Node,
                                             EStack       => E_Stack,
                                             IsAnnotation => True,
                                             Next_Node    => Next_Node);
         when SPSymbols.annotation_attribute_designator =>
            -- ASSUME Last_Node = annotation_attribute_designator
            TypeContextStack.Push (AttributeDesignatorTypeFromContext (Last_Node, False, E_Stack, T_Stack), T_Stack);
            Next_Node := Child_Node (Current_Node => Last_Node);
         when SPSymbols.annotation_range_constraint =>
            -- ASSUME Last_Node = annotation_range_constraint
            TypeContextStack.Push (RangeConstraintTypeFromContext (Last_Node, E_Stack, T_Stack), T_Stack);
            Next_Node := Child_Node (Current_Node => Last_Node);
         when SPSymbols.annotation_simple_expression =>
            -- ASSUME Last_Node = annotation_simple_expression
            TypeContextStack.Push (SimpleExpressionTypeFromContext (Last_Node, T_Stack), T_Stack);
            Next_Node := Child_Node (Current_Node => Last_Node);
         when SPSymbols.annotation_expression =>
            -- ASSUME Last_Node = annotation_expression
            ExpressionTypeFromContext (Exp_Node         => Last_Node,
                                       E_Stack          => E_Stack,
                                       T_Stack          => T_Stack,
                                       New_Context_Type => Sym);
            TypeContextStack.Push (Sym, T_Stack);
            Next_Node := Child_Node (Current_Node => Last_Node);
         when SPSymbols.annotation_primary =>
            -- ASSUME Last_Node = annotation_primary
            TypeContextStack.Push (PrimaryTypeFromContext (Last_Node, T_Stack), T_Stack);
            Next_Node := Child_Node (Current_Node => Last_Node);
         when SPSymbols.store =>
            -- ASSUME Last_Node = store
            down_wf_store (Last_Node, L_Scope, E_Stack,
                           --to get
                           Next_Node);
         when SPSymbols.store_list =>
            -- ASSUME Last_Node = store_list
            down_wf_store_list (Last_Node, L_Scope, E_Stack,
                                --to get
                                Next_Node);
         when SPSymbols.quantified_expression =>
            -- ASSUME Last_Node = quantified_expression
            down_wf_quantifier (Last_Node,
                                --using and to get
                                E_Stack, L_Scope,
                                --to get
                                Next_Node);
         when others =>
            if Node_Type in SPSymbols.SPNonTerminal then
               Next_Node := Child_Node (Current_Node => Last_Node);
            else
               Next_Node := STree.NullNode;
            end if;
      end case;

      -------------------------------------------------up loop----------
      if Next_Node = STree.NullNode then
         loop
            Next_Node := Next_Sibling (Current_Node => Last_Node);
            exit when Next_Node /= STree.NullNode; -- new branch to right

            Next_Node := Parent_Node (Current_Node => Last_Node);
            Last_Node := Next_Node;

            Dump_Up_Node;

            case Syntax_Node_Type (Node => Last_Node) is
               when SPSymbols.annotation_expression =>
                  -- ASSUME Last_Node = annotation_expression
                  Wf_Expression (Node    => Last_Node,
                                 Scope   => L_Scope,
                                 E_Stack => E_Stack,
                                 T_Stack => T_Stack);
                  TypeContextStack.Pop (T_Stack);
               when SPSymbols.annotation_expression_rep1 |
                 SPSymbols.annotation_expression_rep2 |
                 SPSymbols.annotation_expression_rep3 |
                 SPSymbols.annotation_expression_rep4 |
                 SPSymbols.annotation_expression_rep5 |
                 SPSymbols.annotation_expression_rep6 |
                 SPSymbols.annotation_expression_rep7 =>
                  -- ASSUME Last_Node = annotation_expression_rep1 OR annotation_expression_rep2 OR annotation_expression_rep3 OR
                  --                    annotation_expression_rep4 OR annotation_expression_rep5 OR annotation_expression_rep6 OR
                  --                    annotation_expression_rep7
                  Wf_Expression (Node    => Last_Node,
                                 Scope   => L_Scope,
                                 E_Stack => E_Stack,
                                 T_Stack => T_Stack);
               when SPSymbols.annotation_simple_expression =>
                  -- ASSUME Last_Node = annotation_simple_expression
                  Wf_Simple_Expression
                    (Node                    => Last_Node,
                     Scope                   => L_Scope,
                     E_Stack                 => E_Stack,
                     T_Stack                 => T_Stack,
                     Context_Requires_Static => False);
                  TypeContextStack.Pop (T_Stack);
               when SPSymbols.annotation_simple_expression_opt =>
                  -- ASSUME Last_Node = annotation_simple_expression_opt
                  Wf_Simple_Expression_Opt (Node    => Last_Node,
                                            Scope   => L_Scope,
                                            E_Stack => E_Stack,
                                            T_Stack => T_Stack);
               when SPSymbols.annotation_term =>
                  -- ASSUME Last_Node = annotation_term
                  Wf_Term
                    (Node                    => Last_Node,
                     Scope                   => L_Scope,
                     E_Stack                 => E_Stack,
                     T_Stack                 => T_Stack,
                     Context_Requires_Static => False);
               when SPSymbols.annotation_factor =>
                  -- ASSUME Last_Node = annotation_factor
                  Wf_Factor (Node    => Last_Node,
                             Scope   => L_Scope,
                             E_Stack => E_Stack,
                             T_Stack => T_Stack);
               when SPSymbols.annotation_relation =>
                  -- ASSUME Last_Node = annotation_relation
                  Wf_Relation (Node    => Last_Node,
                               Scope   => L_Scope,
                               E_Stack => E_Stack,
                               T_Stack => T_Stack);
               when SPSymbols.annotation_range_constraint =>
                  -- ASSUME Last_Node = annotation_range_constraint
                  TypeContextStack.Pop (T_Stack);
               when SPSymbols.annotation_arange =>
                  -- ASSUME Last_Node = annotation_arange
                  wf_arange (Node         => Last_Node,
                             Scope        => L_Scope,
                             EStack       => E_Stack,
                             IsAnnotation => True);
               when SPSymbols.annotation_selected_component =>
                  -- ASSUME Last_Node = annotation_selected_component
                  Wf_Selected_Component
                    (Node           => Last_Node,
                     Scope          => L_Scope,
                     E_Stack        => E_Stack,
                     Ref_Var        => Ref_Var,
                     Component_Data => Component_Data,
                     Context        => Context);
               when SPSymbols.annotation_attribute =>
                  -- ASSUME Last_Node = annotation_attribute
                  wf_attribute (E_Stack);
               when SPSymbols.annotation_attribute_designator =>
                  -- ASSUME Last_Node = annotation_attribute_designator
                  wf_attribute_designator
                    (Node         => Last_Node,
                     Scope        => L_Scope,
                     EStack       => E_Stack,
                     RefVar       => Ref_Var,
                     IsAnnotation => True);
                  TypeContextStack.Pop (T_Stack);
               when SPSymbols.annotation_primary =>
                  -- ASSUME Last_Node = annotation_primary
                  wf_primary
                    (Node          => Last_Node,
                     Scope         => L_Scope,
                     EStack        => E_Stack,
                     RefVar        => Ref_Var,
                     IsAnnotation  => True,
                     ComponentData => Component_Data);
                  TypeContextStack.Pop (T_Stack);
               when SPSymbols.annotation_positional_argument_association =>
                  -- ASSUME Last_Node = annotation_positional_argument_association
                  wf_positional_argument_association
                    (Node          => Last_Node,
                     Scope         => L_Scope,
                     EStack        => E_Stack,
                     IsAnnotation  => True,
                     RefVar        => Ref_Var,
                     ComponentData => Component_Data);
               when SPSymbols.annotation_named_argument_association =>
                  -- ASSUME Last_Node = annotation_named_argument_association
                  wf_named_argument_association
                    (Node         => Last_Node,
                     Scope        => L_Scope,
                     EStack       => E_Stack,
                     HeapParam    => L_Heap,
                     IsAnnotation => True);
               when SPSymbols.annotation_name_argument_list =>
                  -- ASSUME Last_Node = annotation_name_argument_list
                  up_wf_name_argument_list (Node         => Last_Node,
                                            EStack       => E_Stack,
                                            HeapParam    => L_Heap,
                                            IsAnnotation => True);
               when SPSymbols.annotation_ancestor_part =>
                  -- ASSUME Last_Node = annotation_ancestor_part
                  wf_ancestor_part (Node         => Last_Node,
                                    Scope        => Scope,
                                    EStack       => E_Stack,
                                    HeapParam    => L_Heap,
                                    IsAnnotation => True);
               when SPSymbols.annotation_aggregate_choice =>
                  -- ASSUME Last_Node = annotation_aggregate_choice
                  up_wf_aggregate_choice (Node   => Last_Node,
                                          Scope  => L_Scope,
                                          EStack => E_Stack);
               when SPSymbols.annotation_named_association_rep =>
                  -- ASSUME Last_Node = annotation_named_association_rep
                  up_wf_named_association_rep
                    (Node         => Last_Node,
                     Scope        => L_Scope,
                     EStack       => E_Stack,
                     HeapParam    => L_Heap,
                     IsAnnotation => True);
               when SPSymbols.annotation_named_record_component_association =>
                  -- ASSUME Last_Node = annotation_named_record_component_association
                  up_wf_named_record_component_association
                    (Node      => Last_Node,
                     Scope     => Scope,
                     EStack    => E_Stack,
                     HeapParam => L_Heap);
               when SPSymbols.annotation_positional_association | SPSymbols.annotation_record_component_association =>
                  -- ASSUME Last_Node = annotation_positional_association OR annotation_record_component_association
                  up_wf_positional_association (Node   => Last_Node,
                                                EStack => E_Stack);
               when SPSymbols.annotation_aggregate_or_expression =>
                  -- ASSUME Last_Node = annotation_aggregate_or_expression
                  up_wf_aggregate_or_expression (Node         => Last_Node,
                                                 Scope        => L_Scope,
                                                 EStack       => E_Stack,
                                                 IsAnnotation => True);
               when SPSymbols.annotation_positional_record_component_association =>
                  -- ASSUME Last_Node = annotation_positional_record_component_association
                  wf_positional_record_component_association
                    (Node         => Last_Node,
                     Scope        => Scope,
                     EStack       => E_Stack,
                     IsAnnotation => True);
               when SPSymbols.annotation_component_association =>
                  -- ASSUME Last_Node = annotation_component_association
                  up_wf_component_association (Node         => Last_Node,
                                               Scope        => L_Scope,
                                               EStack       => E_Stack,
                                               IsAnnotation => True);
               when SPSymbols.annotation_aggregate | SPSymbols.annotation_extension_aggregate =>
                  -- ASSUME Last_Node = annotation_aggregate OR annotation_extension_aggregate
                  up_wf_aggregate (Node         => Last_Node,
                                   Scope        => L_Scope,
                                   EStack       => E_Stack,
                                   IsAnnotation => True);
               when SPSymbols.annotation_qualified_expression =>
                  -- ASSUME Last_Node = annotation_qualified_expression
                  Wf_Qualified_Expression (Node    => Last_Node,
                                           Scope   => L_Scope,
                                           E_Stack => E_Stack);
               when SPSymbols.store_list =>
                  -- ASSUME Last_Node = store_list
                  Wf_Store_List (Node    => Last_Node,
                                 Scope   => L_Scope,
                                 E_Stack => E_Stack);
               when SPSymbols.store =>
                  -- ASSUME Last_Node = store
                  up_wf_store (Last_Node, L_Scope, E_Stack);
               when SPSymbols.quantified_expression =>
                  -- ASSUME Last_Node = quantified_expression
                  up_wf_quantifier (Last_Node, E_Stack,
                                    --to get
                                    L_Scope);
               when others =>
                  null;
            end case;

            exit when Next_Node = Exp_Node; -- got back to top
         end loop; -- up
      end if;

      exit when Next_Node = Exp_Node; -- met start point on way up

   end loop; -- down

   if not ExpStack.HasOneEntry (E_Stack) then
      SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Expression_Stack_Corrupt,
                                Msg     => "in WalkAnnotationExpression");
   end if;

   if not (TypeContextStack.HasOneEntry (T_Stack) and then TypeContextStack.Top (T_Stack) = Type_Context) then
      SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Type_Context_Stack_Corrupt,
                                Msg     => "in WalkAnnotationExpression");
   end if;

   --# accept Flow, 10, E_Stack, "Expected ineffective assignment";
   ExpStack.Pop (Result, E_Stack);
   --# end accept;

   Dump_Result;

   SeqAlgebra.DisposeOfSeq (TheHeap => TheHeap,
                            S       => Ref_Var);

end WalkAnnotationExpression;  -- AggregateStack will appear later
