-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.WalkStatements)

--------------------------------------------------------------------
--  VariableUpdateHistory
--
--  Implementation Notes:
--    The History_T ADT is implmented as a list of pairs using atoms
--    from a given Heap.HeapRecord.  See heap.ads.
--------------------------------------------------------------------
package body VariableUpdateHistory is
   --------------------------------------------------------------------
   --  CreateHistory
   --
   --  Implementation Notes:
   --    Assumes that TheHeap is initialised using Heap.Initialize.
   --    Sets the A and B values of the initial atom to 0.  The A and B
   --    pointers are set to 0 by Heap.CreateAtom.
   --------------------------------------------------------------------
   procedure CreateHistory (TheHeap : in out Heap.HeapRecord;
                            History :    out History_T) is
      Atom : Heap.Atom;
   begin
      Heap.CreateAtom (TheHeap, Atom);
      Heap.UpdateAValue (TheHeap, Atom, 0);
      Heap.UpdateBValue (TheHeap, Atom, 0);
      History := History_T (Atom);
   end CreateHistory;

   --------------------------------------------------------------------
   --  DisposeOfHistory
   --
   --  Implementation Notes:
   --    The last atom in the list has a null A pointer.
   --------------------------------------------------------------------
   procedure DisposeOfHistory (TheHeap : in out Heap.HeapRecord;
                               History : in     History_T) is
      Atom     : Heap.Atom;
      NextAtom : Heap.Atom;
   begin
      Atom := Heap.Atom (History);

      loop
         NextAtom := Heap.APointer (TheHeap, Atom);
         Heap.DisposeOfAtom (TheHeap, Atom);
         Atom := NextAtom;
         exit when Heap.IsNullPointer (NextAtom);
      end loop;
   end DisposeOfHistory;

   --------------------------------------------------------------------
   --  AddUpdate
   --
   --  Implementation Notes:
   --    A linear search is used to locate the specified Variable.
   --    If the end of the list is reached without finding the Variable
   --    a new atom is created and added to the end of the list.
   --    The list cannot contain duplicate variables.
   --------------------------------------------------------------------
   procedure AddUpdate
     (TheHeap  : in out Heap.HeapRecord;
      History  : in out History_T;
      Variable : in     Natural;
      Node     : in     STree.SyntaxNode) is
      Atom    : Heap.Atom;
      NewAtom : Heap.Atom;
      Value   : Natural;
   begin
      Atom  := Heap.Atom (History);
      Value := Heap.AValue (TheHeap, Atom);

      while Value /= Variable and not Heap.IsNullPointer (Heap.APointer (TheHeap, Atom)) loop
         Atom  := Heap.APointer (TheHeap, Atom);
         Value := Heap.AValue (TheHeap, Atom);
      end loop;

      if Value /= Variable then
         -- Variable not found, add a new atom to the end of the list.
         Heap.CreateAtom (TheHeap, NewAtom);
         Heap.UpdateAValue (TheHeap, NewAtom, Variable);
         Heap.UpdateBValue (TheHeap, NewAtom, Natural (STree.NodeToRef (Node)));
         Heap.UpdateAPointer (TheHeap, Atom, NewAtom);
      else
         -- The variable is already in the history, update the previous node.
         Heap.UpdateBValue (TheHeap, Atom, Natural (STree.NodeToRef (Node)));
      end if;
      --# accept F, 31, History, "History is logically updated but is represented by a pointer." &
      --#        F, 50, History, TheHeap, "History is logically dependent on the contents of TheHeap." &
      --#        F, 50, History, Variable, "History is logically dependent on Variable." &
      --#        F, 50, History, Node, "History is logically dependent on Node.";
   end AddUpdate;

   --------------------------------------------------------------------
   --  GetLastUpdate
   --
   --  Implementation Notes:
   --    A linear search is used to locate the specified Variable.
   --    If the end of the list is reached without finding the Variable
   --    the Result is set to STree.NullNode
   --------------------------------------------------------------------
   function GetLastUpdate
     (TheHeap  : Heap.HeapRecord;
      History  : History_T;
      Variable : Natural)
     return     STree.SyntaxNode
   is
      Result : STree.SyntaxNode;
      Atom   : Heap.Atom;
      Value  : Natural;
   begin
      Atom  := Heap.Atom (History);
      Value := Heap.AValue (TheHeap, Atom);
      while Value /= Variable and not Heap.IsNullPointer (Heap.APointer (TheHeap, Atom)) loop
         Atom  := Heap.APointer (TheHeap, Atom);
         Value := Heap.AValue (TheHeap, Atom);
      end loop;

      if Value /= Variable then
         -- No previous update, return NullNode.
         Result := STree.NullNode;
      else
         Result := STree.RefToNode (ExaminerConstants.RefType (Heap.BValue (TheHeap, Atom)));
      end if;
      return Result;
   end GetLastUpdate;

end VariableUpdateHistory;
