-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.wf_package_body.wf_refine)
procedure wf_clause (Node  : in STree.SyntaxNode;
                     Scope : in Dictionary.Scopes) is
   type Modes is record
      Value   : Dictionary.Modes;
      IsValid : Boolean;
   end record;

   SubjectSym      : Dictionary.Symbol;
   SubjectNode     : STree.SyntaxNode;
   SubjectLoc      : LexTokenManager.Token_Position;
   ConstituentNode : STree.SyntaxNode;

   -----------------------------------------------------------------

   procedure DoSubject (SubjectNode : in     STree.SyntaxNode;
                        Scope       : in     Dictionary.Scopes;
                        SubjectSym  :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubjectNode &
   --#         STree.Table,
   --#         SubjectSym                 from Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         SubjectNode;
   is
      VarSym : Dictionary.Symbol;
      VarPos : LexTokenManager.Token_Position;
      VarStr : LexTokenManager.Lex_String;
   begin -- DoSubject
      VarStr := Node_Lex_String (Node => SubjectNode);
      VarPos := Node_Position (Node => SubjectNode);
      VarSym :=
        Dictionary.LookupItem
        (Name              => VarStr,
         Scope             => Dictionary.VisibleScope (Dictionary.GetRegion (Scope)),
         Context           => Dictionary.ProofContext,
         Full_Package_Name => False);
      if VarSym = Dictionary.NullSymbol then --not declared or visible
         ErrorHandler.Semantic_Error
           (Err_Num   => 136,
            Reference => ErrorHandler.No_Reference,
            Position  => VarPos,
            Id_Str    => VarStr);
         SubjectSym := Dictionary.NullSymbol;
      elsif Dictionary.IsOwnVariable (VarSym) then     -- possibly ok, it's an own var
         if not Dictionary.GetOwnVariableProtected (VarSym) then -- cannot refine protected state
            if not Dictionary.IsVirtualElement (VarSym) then -- cannot refine virtual elements
               if Dictionary.IsDeclared (VarSym)
                 or else     -- declared as concrete
                 Dictionary.OwnVariableHasConstituents (VarSym) then -- or already refined away so raise error 73
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 73,
                     Reference => ErrorHandler.No_Reference,
                     Position  => VarPos,
                     Id_Str    => VarStr);
                  SubjectSym := Dictionary.NullSymbol;
               elsif Dictionary.OwnVariableHasType (VarSym, Scope) then
                  -- type announced so refinement valid only if type is abstract proof type
                  if Dictionary.IsDeclared (Dictionary.GetType (VarSym)) then
                     -- type is concrete type so refinement definitely illegal
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 73,
                        Reference => ErrorHandler.No_Reference,
                        Position  => VarPos,
                        Id_Str    => VarStr);
                     SubjectSym := Dictionary.NullSymbol;
                  else
                     -- type not declared so we don't know whether problem is illegal refinement
                     -- or missing abstract type declaration.  Need special error message here
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 273,
                        Reference => ErrorHandler.No_Reference,
                        Position  => VarPos,
                        Id_Str    => VarStr);
                     SubjectSym := Dictionary.NullSymbol;
                  end if;
               else  -- this is the valid case
                  STree.Set_Node_Lex_String (Sym  => VarSym,
                                             Node => SubjectNode);
                  SubjectSym := VarSym;
               end if;
            else -- it's a virtual element which cannot be refined
               ErrorHandler.Semantic_Error
                 (Err_Num   => 858,
                  Reference => ErrorHandler.No_Reference,
                  Position  => VarPos,
                  Id_Str    => VarStr);
               SubjectSym := Dictionary.NullSymbol;
            end if;
         else -- it's a protected own variable which cannot be refined
            ErrorHandler.Semantic_Error
              (Err_Num   => 859,
               Reference => ErrorHandler.No_Reference,
               Position  => VarPos,
               Id_Str    => VarStr);
            SubjectSym := Dictionary.NullSymbol;
         end if;
      else -- it's not an own variable so it can't be refined
         ErrorHandler.Semantic_Error (Err_Num   => 74,
                                      Reference => ErrorHandler.No_Reference,
                                      Position  => VarPos,
                                      Id_Str    => VarStr);
         SubjectSym := Dictionary.NullSymbol;
      end if;
   end DoSubject;

   -----------------------------------------------------------------

   procedure DoConstituentList
     (ConstituentListNode : in STree.SyntaxNode;
      Scope               : in Dictionary.Scopes;
      SubjectSym          : in Dictionary.Symbol;
      SubjectLoc          : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ConstituentListNode,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         SubjectSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ConstituentListNode,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubjectLoc,
   --#                                         SubjectSym;
   is
      It              : STree.Iterator;
      ConstituentNode : STree.SyntaxNode;
      Mode            : Modes;

      -----------------------------------------------------------------

      procedure DoConstituent
        (ConstituentNode : in STree.SyntaxNode;
         Scope           : in Dictionary.Scopes;
         SubjectSym      : in Dictionary.Symbol;
         Mode            : in Modes;
         SubjectLoc      : in LexTokenManager.Token_Position)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Dictionary.Dict            from *,
      --#                                         CommandLineData.Content,
      --#                                         ConstituentNode,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         LexTokenManager.State,
      --#                                         Mode,
      --#                                         Scope,
      --#                                         STree.Table,
      --#                                         SubjectSym &
      --#         ErrorHandler.Error_Context from *,
      --#                                         CommandLineData.Content,
      --#                                         ConstituentNode,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Mode,
      --#                                         Scope,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         SubjectSym &
      --#         SPARK_IO.File_Sys          from *,
      --#                                         CommandLineData.Content,
      --#                                         ConstituentNode,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Mode,
      --#                                         Scope,
      --#                                         STree.Table,
      --#                                         SubjectLoc,
      --#                                         SubjectSym &
      --#         STree.Table                from *,
      --#                                         CommandLineData.Content,
      --#                                         ConstituentNode,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         SubjectSym;
      is
         FirstNode  : STree.SyntaxNode;
         SecondNode : STree.SyntaxNode;

         ----------------------------------------------------------------

         procedure DoConcreteConstituent
           (VarNode    : in STree.SyntaxNode;
            Scope      : in Dictionary.Scopes;
            SubjectSym : in Dictionary.Symbol;
            SubjectLoc : in LexTokenManager.Token_Position)
         --# global in     CommandLineData.Content;
         --#        in     ContextManager.Ops.Unit_Stack;
         --#        in     LexTokenManager.State;
         --#        in     Mode;
         --#        in     STree.Table;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out SPARK_IO.File_Sys;
         --# derives Dictionary.Dict            from *,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         LexTokenManager.State,
         --#                                         Mode,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         SubjectSym,
         --#                                         VarNode &
         --#         ErrorHandler.Error_Context from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         LexTokenManager.State,
         --#                                         Scope,
         --#                                         SPARK_IO.File_Sys,
         --#                                         STree.Table,
         --#                                         VarNode &
         --#         SPARK_IO.File_Sys          from *,
         --#                                         CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Mode,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         SubjectLoc,
         --#                                         SubjectSym,
         --#                                         VarNode;
         is
            VarPos : LexTokenManager.Token_Position;
            VarStr : LexTokenManager.Lex_String;
         begin --DoConcreteConstituent
            VarStr := Node_Lex_String (Node => VarNode);
            VarPos := Node_Position (Node => VarNode);
            if Dictionary.IsDefined (Name              => VarStr,
                                     Scope             => Scope,
                                     Context           => Dictionary.ProofContext,
                                     Full_Package_Name => False) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => VarPos,
                  Id_Str    => VarStr);
            elsif Mode.IsValid then
               Dictionary.AddConstituent
                 (Name                 => VarStr,
                  Subject              => SubjectSym,
                  Mode                 => Mode.Value,
                  SubjectReference     => Dictionary.Location'(Start_Position => SubjectLoc,
                                                               End_Position   => SubjectLoc),
                  Comp_Unit            => ContextManager.Ops.Current_Unit,
                  ConstituentReference => Dictionary.Location'(Start_Position => VarPos,
                                                               End_Position   => VarPos));
            end if;
         end DoConcreteConstituent;

         -----------------------------------------------------------------

         procedure DoEmbeddedConstituent
           (PackDSimpleNameNode : in STree.SyntaxNode;
            VarIdentNode        : in STree.SyntaxNode;
            Scope               : in Dictionary.Scopes;
            SubjectSym          : in Dictionary.Symbol;
            SubjectLoc          : in LexTokenManager.Token_Position)
         --# global in     CommandLineData.Content;
         --#        in     ContextManager.Ops.Unit_Stack;
         --#        in     LexTokenManager.State;
         --#        in     Mode;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out SPARK_IO.File_Sys;
         --#        in out STree.Table;
         --# derives Dictionary.Dict            from *,
         --#                                         CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         LexTokenManager.State,
         --#                                         Mode,
         --#                                         PackDSimpleNameNode,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         SubjectSym,
         --#                                         VarIdentNode &
         --#         ErrorHandler.Error_Context from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         LexTokenManager.State,
         --#                                         PackDSimpleNameNode,
         --#                                         Scope,
         --#                                         SPARK_IO.File_Sys,
         --#                                         STree.Table,
         --#                                         VarIdentNode &
         --#         SPARK_IO.File_Sys          from *,
         --#                                         CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Mode,
         --#                                         PackDSimpleNameNode,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         SubjectLoc,
         --#                                         SubjectSym,
         --#                                         VarIdentNode &
         --#         STree.Table                from *,
         --#                                         Dictionary.Dict,
         --#                                         LexTokenManager.State,
         --#                                         PackDSimpleNameNode,
         --#                                         Scope;
         is
            PackSym  : Dictionary.Symbol;
            PackPos  : LexTokenManager.Token_Position;
            PackStr  : LexTokenManager.Lex_String;
            PackNode : STree.SyntaxNode;
            VarSym   : Dictionary.Symbol;
            VarPos   : LexTokenManager.Token_Position;
            VarStr   : LexTokenManager.Lex_String;
            VarNode  : STree.SyntaxNode;
            OkToAdd  : Boolean;
         begin
            OkToAdd := False;

            PackNode := Child_Node (PackDSimpleNameNode);
            PackStr  := Node_Lex_String (Node => PackNode);
            PackPos  := Node_Position (Node => PackNode);

            VarNode := VarIdentNode;
            VarStr  := Node_Lex_String (Node => VarNode);
            VarPos  := Node_Position (Node => VarNode);

            PackSym := Dictionary.LookupItem (Name              => PackStr,
                                              Scope             => Scope,
                                              Context           => Dictionary.ProofContext,
                                              Full_Package_Name => False);
            if PackSym = Dictionary.NullSymbol then
               OkToAdd := True;
            elsif Dictionary.IsPackage (PackSym) then
               if Dictionary.GetContext (PackSym) = Dictionary.ProgramContext then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 75,
                     Reference => ErrorHandler.No_Reference,
                     Position  => PackPos,
                     Id_Str    => PackStr);
               else
                  STree.Set_Node_Lex_String (Sym  => PackSym,
                                             Node => PackNode);
                  VarSym :=
                    Dictionary.LookupSelectedItem
                    (Prefix   => PackSym,
                     Selector => VarStr,
                     Scope    => Scope,
                     Context  => Dictionary.ProofContext);
                  if VarSym = Dictionary.NullSymbol then
                     OkToAdd := True;
                  else
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 10,
                        Reference => ErrorHandler.No_Reference,
                        Position  => VarPos,
                        Id_Str    => VarStr);
                  end if;
               end if;
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 76,
                  Reference => ErrorHandler.No_Reference,
                  Position  => PackPos,
                  Id_Str    => PackStr);
            end if;

            if OkToAdd and Mode.IsValid then
               VarNode := VarIdentNode;
               VarStr  := Node_Lex_String (Node => VarNode);
               VarPos  := Node_Position (Node => VarNode);
               Dictionary.AddEmbeddedConstituent
                 (PackageName          => PackStr,
                  VariableName         => VarStr,
                  Subject              => SubjectSym,
                  Mode                 => Mode.Value,
                  SubjectReference     => Dictionary.Location'(Start_Position => SubjectLoc,
                                                               End_Position   => SubjectLoc),
                  Comp_Unit            => ContextManager.Ops.Current_Unit,
                  ConstituentReference => Dictionary.Location'(Start_Position => VarPos,
                                                               End_Position   => VarPos));
            end if;
         end DoEmbeddedConstituent;

         ----------------------------------------------------------------

         procedure DoChildConstituent
           (PackDSimpleNameNode : in STree.SyntaxNode;
            VarIdentNode        : in STree.SyntaxNode;
            Scope               : in Dictionary.Scopes;
            SubjectSym          : in Dictionary.Symbol;
            SubjectLoc          : in LexTokenManager.Token_Position)
         --# global in     CommandLineData.Content;
         --#        in     ContextManager.Ops.Unit_Stack;
         --#        in     LexTokenManager.State;
         --#        in     Mode;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out SPARK_IO.File_Sys;
         --#        in out STree.Table;
         --# derives Dictionary.Dict            from *,
         --#                                         CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         LexTokenManager.State,
         --#                                         Mode,
         --#                                         PackDSimpleNameNode,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         SubjectSym,
         --#                                         VarIdentNode &
         --#         ErrorHandler.Error_Context from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         LexTokenManager.State,
         --#                                         Mode,
         --#                                         PackDSimpleNameNode,
         --#                                         Scope,
         --#                                         SPARK_IO.File_Sys,
         --#                                         STree.Table,
         --#                                         SubjectSym,
         --#                                         VarIdentNode &
         --#         SPARK_IO.File_Sys          from *,
         --#                                         CommandLineData.Content,
         --#                                         ContextManager.Ops.Unit_Stack,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Mode,
         --#                                         PackDSimpleNameNode,
         --#                                         Scope,
         --#                                         STree.Table,
         --#                                         SubjectLoc,
         --#                                         SubjectSym,
         --#                                         VarIdentNode &
         --#         STree.Table                from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         LexTokenManager.State,
         --#                                         PackDSimpleNameNode,
         --#                                         Scope,
         --#                                         SubjectSym,
         --#                                         VarIdentNode;
         is
            PackSym  : Dictionary.Symbol;
            PackPos  : LexTokenManager.Token_Position;
            PackStr  : LexTokenManager.Lex_String;
            PackNode : STree.SyntaxNode;
            VarSym   : Dictionary.Symbol;
            VarPos   : LexTokenManager.Token_Position;
            VarStr   : LexTokenManager.Lex_String;
            Owner    : Dictionary.Symbol;
            ErrNum   : Natural;
         begin
            PackNode := Last_Child_Of (Start_Node => PackDSimpleNameNode);
            PackStr  := Node_Lex_String (Node => PackNode);
            PackPos  := Node_Position (Node => PackNode);

            PackSym :=
              Dictionary.LookupItem (Name              => PackStr,
                                     Scope             => Dictionary.GlobalScope,
                                     Context           => Dictionary.ProofContext,
                                     Full_Package_Name => False);
            loop
               if PackSym = Dictionary.NullSymbol then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 137,
                     Reference => ErrorHandler.No_Reference,
                     Position  => PackPos,
                     Id_Str    => PackStr);
                  exit;
               end if;

               if not Dictionary.IsPackage (PackSym) then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 18,
                     Reference => ErrorHandler.No_Reference,
                     Position  => PackPos,
                     Id_Str    => PackStr);
                  PackSym := Dictionary.NullSymbol;
                  exit;
               end if;
               STree.Set_Node_Lex_String (Sym  => PackSym,
                                          Node => PackNode);
               PackNode := Next_Sibling (Parent_Node (Current_Node => PackNode));

               exit when PackNode = VarIdentNode;

               PackStr := Node_Lex_String (Node => PackNode);
               PackPos := Node_Position (Node => PackNode);
               PackSym :=
                 Dictionary.LookupSelectedItem
                 (Prefix   => PackSym,
                  Selector => PackStr,
                  Scope    => Dictionary.GlobalScope,
                  Context  => Dictionary.ProofContext);
            end loop;

            if PackSym /= Dictionary.NullSymbol then
               Owner := Dictionary.GetRegion (Scope);
               if Dictionary.GetPackageOwner (PackSym) /= Owner then
                  -- package not owned by subject package
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 619,
                     Reference => ErrorHandler.No_Reference,
                     Position  => PackPos,
                     Id_Str    => PackStr);
               else
                  VarStr := Node_Lex_String (Node => VarIdentNode);
                  VarPos := Node_Position (Node => VarIdentNode);
                  VarSym :=
                    Dictionary.LookupSelectedItem
                    (Prefix   => PackSym,
                     Selector => VarStr,
                     Scope    => Scope,
                     Context  => Dictionary.ProofContext);
                  if VarSym = Dictionary.NullSymbol then
                     -- constituent own variable not found
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 138,
                        Reference => ErrorHandler.No_Reference,
                        Position  => VarPos,
                        Id_Str    => VarStr);
                  elsif not Dictionary.IsOwnVariable (VarSym) then
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 74,
                        Reference => ErrorHandler.No_Reference,
                        Position  => VarPos,
                        Id_Str    => VarStr);
                  elsif Dictionary.IsRefinementConstituent (Owner, VarSym) then
                     -- already declared as constituent
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 10,
                        Reference => ErrorHandler.No_Reference,
                        Position  => VarPos,
                        Id_Str    => VarStr);
                  else -- valid constituent
                     if Dictionary.OwnVariableIsInitialized (SubjectSym) /= Dictionary.OwnVariableIsInitialized (VarSym) then
                        -- subject and constituent have different initialization
                        if Dictionary.OwnVariableIsInitialized (SubjectSym) then
                           -- if subject is initialized and constituent is not then
                           -- it is only an error if the constituent is not a stream var
                           if Dictionary.GetOwnVariableOrConstituentMode (VarSym) = Dictionary.DefaultMode then
                              ErrorHandler.Semantic_Error
                                (Err_Num   => 622,
                                 Reference => ErrorHandler.No_Reference,
                                 Position  => VarPos,
                                 Id_Str    => VarStr);
                           else
                              STree.Set_Node_Lex_String (Sym  => VarSym,
                                                         Node => VarIdentNode);
                           end if;
                        else
                           ErrorHandler.Semantic_Error
                             (Err_Num   => 623,
                              Reference => ErrorHandler.No_Reference,
                              Position  => VarPos,
                              Id_Str    => VarStr);
                        end if;
                     else
                        STree.Set_Node_Lex_String (Sym  => VarSym,
                                                   Node => VarIdentNode);
                     end if;

                     -- constituent must have same mode as own variable it being
                     -- associated with
                     if Mode.Value /= Dictionary.GetOwnVariableMode (VarSym) then
                        if Dictionary.GetOwnVariableMode (VarSym) = Dictionary.InMode then
                           ErrNum := 705; -- must be IN
                        elsif Dictionary.GetOwnVariableMode (VarSym) = Dictionary.OutMode then
                           ErrNum := 706; -- must be OUT
                        else
                           ErrNum := 707; -- no mode permitted
                        end if;
                        ErrorHandler.Semantic_Error
                          (Err_Num   => ErrNum,
                           Reference => ErrorHandler.No_Reference,
                           Position  => VarPos,
                           Id_Str    => VarStr);
                     end if;
                     -- end of mode checks

                     if Mode.IsValid then
                        Dictionary.AddChildConstituent
                          (Variable             => VarSym,
                           Subject              => SubjectSym,
                           Mode                 => Mode.Value,
                           SubjectReference     => Dictionary.Location'(Start_Position => SubjectLoc,
                                                                        End_Position   => SubjectLoc),
                           Comp_Unit            => ContextManager.Ops.Current_Unit,
                           ConstituentReference => Dictionary.Location'(Start_Position => VarPos,
                                                                        End_Position   => VarPos));
                     end if;
                  end if;
               end if;
            end if;
         end DoChildConstituent;

         ----------------------------------------------------------------

      begin --DoConstituent
         FirstNode  := Child_Node (Child_Node (ConstituentNode));
         SecondNode := Next_Sibling (FirstNode);
         if SecondNode = STree.NullNode then
            DoConcreteConstituent (FirstNode, Scope, SubjectSym, SubjectLoc);
         elsif Syntax_Node_Type (Node => Child_Node (FirstNode)) = SPSymbols.identifier then
            DoEmbeddedConstituent (FirstNode, SecondNode, Scope, SubjectSym, SubjectLoc);
         elsif CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83 then
            DoChildConstituent (FirstNode, SecondNode, Scope, SubjectSym, SubjectLoc);
         else
            ErrorHandler.Semantic_Error
              (Err_Num   => 156,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => ConstituentNode),
               Id_Str    => LexTokenManager.Null_String);
         end if;
      end DoConstituent;

      -----------------------------------------------------------------

      procedure CheckModifier (Node       : in     STree.SyntaxNode;
                               SubjectSym : in     Dictionary.Symbol;
                               Mode       :    out Modes)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         SubjectSym &
      --#         Mode                       from Dictionary.Dict,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         SubjectSym;
      is
         ModifierNode, ErrNode, ModeNode : STree.SyntaxNode;
         SubjectMode                     : Dictionary.Modes;

      begin --CheckMode
            -- NodeType (Node)is  own_variable_modifier

         SubjectMode := Dictionary.GetOwnVariableMode (SubjectSym);

         Mode := Modes'(Dictionary.DefaultMode, False); -- default, invalid answers

         ModifierNode := Child_Node (Node);
         ErrNode      := Next_Sibling (Node); -- can't point at modes

         case Syntax_Node_Type (Node => ModifierNode) is
            when SPSymbols.mode =>
               ModeNode := Child_Node (ModifierNode);
               if ModeNode /= STree.NullNode then -- mode is present
                  case Syntax_Node_Type (Node => ModeNode) is
                     when SPSymbols.in_mode =>
                        -- IN is legal if subject is IN or subject if Default
                        if SubjectMode = Dictionary.DefaultMode or else SubjectMode = Dictionary.InMode then
                           Mode := Modes'(Dictionary.InMode, True);
                        else
                           ErrorHandler.Semantic_Error
                             (Err_Num   => 701,
                              Reference => ErrorHandler.No_Reference,
                              Position  => Node_Position (Node => ErrNode),
                              Id_Str    => Dictionary.GetSimpleName (SubjectSym));
                        end if;

                     when SPSymbols.out_mode =>
                        -- OUT is legal if subject is OUT or subject if Default
                        if SubjectMode = Dictionary.DefaultMode or else SubjectMode = Dictionary.OutMode then
                           Mode := Modes'(Dictionary.OutMode, True);
                        else
                           ErrorHandler.Semantic_Error
                             (Err_Num   => 701,
                              Reference => ErrorHandler.No_Reference,
                              Position  => Node_Position (Node => ErrNode),
                              Id_Str    => Dictionary.GetSimpleName (SubjectSym));
                        end if;

                     when SPSymbols.inout_mode =>
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 700,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Node => ErrNode),
                           Id_Str    => LexTokenManager.Null_String);

                     when others =>
                        null; -- can't happen
                  end case;
               else -- no mode supplied so wff for default mode case

                  -- Default is legal only if subject is Default
                  if SubjectMode = Dictionary.DefaultMode then
                     Mode := Modes'(Dictionary.DefaultMode, True);
                  else
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 701,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => ErrNode),
                        Id_Str    => Dictionary.GetSimpleName (SubjectSym));
                  end if;
               end if;
            when SPSymbols.protected_modifier =>
               -- Refinement constituents cannot be protected.
               ErrorHandler.Semantic_Error
                 (Err_Num   => 859,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ErrNode),
                  Id_Str    => Dictionary.GetSimpleName (SubjectSym));

            when others =>
               null;
         end case;
      end CheckModifier;

      ---------------------------------------------

   begin --DoConstituentList
      It :=
        Find_First_Node
        (Node_Kind    => SPSymbols.own_variable_modifier,
         From_Root    => ConstituentListNode,
         In_Direction => STree.Down);

      while not STree.IsNull (It) loop
         ConstituentNode := Get_Node (It => It);
         CheckModifier (ConstituentNode, SubjectSym,
                        --to get
                        Mode);

         DoConstituent (Next_Sibling (ConstituentNode), Scope, SubjectSym, Mode, SubjectLoc);
         It := STree.NextNode (It);
      end loop;
   end DoConstituentList;

   -----------------------------------------------------------------

begin -- wf_refinement_clause

   SubjectNode     := Child_Node (Node);
   SubjectLoc      := Node_Position (Node => SubjectNode);
   ConstituentNode := Next_Sibling (SubjectNode);

   if ConstituentNode = STree.NullNode then
      -- There is _no_ constituent node at all.  This used to be
      -- rejected by the gammar alone, but this resulted in a poor
      -- error message, and it's a common "beginner" mistake.
      -- The grammar now accepts this, so we can detect and reject it
      -- here with a much better message.
      ErrorHandler.Semantic_Error
        (Err_Num   => 115,
         Reference => ErrorHandler.No_Reference,
         Position  => SubjectLoc,
         Id_Str    => LexTokenManager.Null_String);
   else
      DoSubject (SubjectNode, Scope,
                 --to get
                 SubjectSym);
      if SubjectSym /= Dictionary.NullSymbol then
         DoConstituentList (ConstituentNode, Scope, SubjectSym, SubjectLoc);
      end if;
   end if;

end wf_clause;
