-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure wf_package_body (Node  : in     STree.SyntaxNode;
                           Scope : in out Dictionary.Scopes) is
   Ident_Str                                                : LexTokenManager.Lex_String;
   Sym                                                      : Dictionary.Symbol;
   Ident_Node, With_Node, Ref_Node, Next_Node, Grand_Parent : STree.SyntaxNode;
   Spec_Found, Ok_To_Add_Body, Body_Is_Hidden               : Boolean;
   Pack_Scope                                               : Dictionary.Scopes;

   ----------------------------------------------

   procedure FindPackage
     (Ident_Node : in out STree.SyntaxNode;
      Ident_Str  : in out LexTokenManager.Lex_String;
      Scope      : in     Dictionary.Scopes;
      Found      :    out Boolean;
      ThePackage :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Found,
   --#         Ident_Node,
   --#         Ident_Str,
   --#         STree.Table,
   --#         ThePackage                 from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         Ident_Str,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         Ident_Str,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      Sym, ParentSym : Dictionary.Symbol;
      Ok             : Boolean;
   begin
      Sym := Dictionary.LookupImmediateScope (Name    => Ident_Str,
                                              Scope   => Scope,
                                              Context => Dictionary.ProgramContext);

      Ok := Sym /= Dictionary.NullSymbol and then Dictionary.IsPackage (Sym);
      if not Ok then
         ErrorHandler.Semantic_Error
           (Err_Num   => 11,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
         --there is no spec, must create one for Dict to add body to later
         Dictionary.AddPackage
           (Name          => Ident_Str,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                  End_Position   => Node_Position (Node => Ident_Node)),
            Scope         => Scope,
            ThePackage    => Sym);
      else
         STree.Set_Node_Lex_String (Sym  => Sym,
                                    Node => Ident_Node);
      end if;
      if Syntax_Node_Type (Node => Next_Sibling (Parent_Node (Current_Node => Ident_Node))) = SPSymbols.identifier then
         -- child package form
         if CommandLineData.Content.Language_Profile = CommandLineData.SPARK83 then
            ErrorHandler.Semantic_Error
              (Err_Num   => 610,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Next_Sibling (Parent_Node (Current_Node => Ident_Node))),
               Id_Str    => LexTokenManager.Null_String);
         elsif Ok then
            loop -- to handle multiple prefixes
               Ident_Node := Next_Sibling (Parent_Node (Current_Node => Ident_Node));
               Ident_Str  := Node_Lex_String (Node => Ident_Node);
               ParentSym  := Sym;
               Sym        :=
                 Dictionary.LookupSelectedItem
                 (Prefix   => ParentSym,
                  Selector => Ident_Str,
                  Scope    => Scope,
                  Context  => Dictionary.ProofContext);
               if Sym = Dictionary.NullSymbol or else not Dictionary.IsPackage (Sym) then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 11,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
                  --there is no spec, must create one for Dict to add body to later
                  Dictionary.AddChildPackage
                    (TheParent     => ParentSym,
                     Sort          => Dictionary.Public,
                     Name          => Ident_Str,
                     Comp_Unit     => ContextManager.Ops.Current_Unit,
                     Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                           End_Position   => Node_Position (Node => Ident_Node)),
                     Scope         => Scope,
                     ThePackage    => Sym);
                  Ok := False;
                  exit;
               end if;
               STree.Set_Node_Lex_String (Sym  => Sym,
                                          Node => Ident_Node);
               exit when Syntax_Node_Type (Node => Next_Sibling (Parent_Node (Current_Node => Ident_Node))) /=
                 SPSymbols.identifier;
               -- when no more identifier (s) to right
            end loop;
         end if;
      end if;

      Found      := Ok;
      ThePackage := Sym;
   end FindPackage;

   ----------------------------------------------

   --    check that all own variables of private children (and their public
   --    descendents) have appeared as refinement constituents
   procedure CheckOwnedPackages (Owner : in Dictionary.Symbol;
                                 Node  : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Owner,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      OwnedPackages : Dictionary.Iterator;
      OwnVars       : Dictionary.Iterator;
      PackSym       : Dictionary.Symbol;
      VarSym        : Dictionary.Symbol;

      function GetErrorPos (Node : STree.SyntaxNode) return LexTokenManager.Token_Position
      --# global in STree.Table;
      is -- Node is package_body
         ErrNode : STree.SyntaxNode;
      begin
         ErrNode := Next_Sibling (Child_Node (Node));
         if Syntax_Node_Type (Node => ErrNode) = SPSymbols.refinement_definition then  -- report at last constituent
            ErrNode := Child_Node (Child_Node (ErrNode));
            if Syntax_Node_Type (Node => ErrNode) /= SPSymbols.refinement_clause then
               ErrNode := Next_Sibling (Next_Sibling (ErrNode));
            end if;
            -- ErrNode is now a refinement_clause
            ErrNode := Next_Sibling (Child_Node (Next_Sibling (Child_Node (ErrNode))));
            if Syntax_Node_Type (Node => ErrNode) /= SPSymbols.entire_variable then
               ErrNode := Next_Sibling (ErrNode);
            end if;
         else -- no refinement definition - report at package name
            ErrNode := Last_Child_Of (Start_Node => Node);
         end if;

         return Node_Position (Node => ErrNode);
      end GetErrorPos;

   begin
      OwnedPackages := Dictionary.FirstOwnedPackage (Owner);
      while not Dictionary.IsNullIterator (OwnedPackages) loop
         PackSym := Dictionary.CurrentSymbol (OwnedPackages);
         OwnVars := Dictionary.FirstOwnVariable (PackSym);

         while not Dictionary.IsNullIterator (OwnVars) loop
            VarSym := Dictionary.CurrentSymbol (OwnVars);
            if not Dictionary.IsRefinementConstituent (Owner, VarSym) then
               -- missing own variable
               ErrorHandler.Semantic_Error_Sym
                 (Err_Num   => 621,
                  Reference => ErrorHandler.No_Reference,
                  Position  => GetErrorPos (Node),
                  Sym       => VarSym,
                  Scope     => Dictionary.GlobalScope);
            end if;
            OwnVars := Dictionary.NextSymbol (OwnVars);
         end loop;

         OwnedPackages := Dictionary.NextSymbol (OwnedPackages);
      end loop;
   end CheckOwnedPackages;

   ----------------------------------------------

   procedure wf_refine (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

begin -- wf_package_body

   -- ASSUME Node = package_body
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.package_body,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = package_body in Wf_Package_Body");
   Ident_Node := Last_Child_Of (Start_Node => Node);
   -- ASSUME Ident_Node = identifier
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Ident_Node = identifier in Wf_Package_Body");
   Ident_Str    := Node_Lex_String (Node => Ident_Node);
   Grand_Parent := Parent_Node (Current_Node => Parent_Node (Current_Node => Node));
   -- ASSUME Grand_Parent = abody OR subunit OR secondary_unit
   if Syntax_Node_Type (Node => Grand_Parent) = SPSymbols.abody then
      -- ASSUME Grand_Parent = abody
      With_Node := STree.NullNode;
   elsif Syntax_Node_Type (Node => Grand_Parent) = SPSymbols.subunit
     or else Syntax_Node_Type (Node => Grand_Parent) = SPSymbols.secondary_unit then
      -- ASSUME Grand_Parent = subunit OR secondary_unit
      if Syntax_Node_Type (Node => Grand_Parent) = SPSymbols.subunit then
         With_Node := Parent_Node (Current_Node => Grand_Parent);
      else
         With_Node := Grand_Parent;
      end if;
      -- ASSUME With_Node = secondary_unit
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => With_Node) = SPSymbols.secondary_unit,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect with_Node = secondary_unit in Wf_Package_Body");
      With_Node := Child_Node (Current_Node => Parent_Node (Current_Node => With_Node));
      -- ASSUME With_Node = secondary_unit OR context_clause
      if Syntax_Node_Type (Node => With_Node) = SPSymbols.secondary_unit then
         -- ASSUME With_Node = secondary_unit
         With_Node := STree.NullNode;
      elsif Syntax_Node_Type (Node => With_Node) /= SPSymbols.context_clause then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect With_Node = secondary_unit OR context_clause in Wf_Package_Body");
      end if;
   else
      With_Node := STree.NullNode;
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Grand_Parent = abody OR subunit OR secondary_unit in Wf_Package_Body");
   end if;
   -- ASSUME With_Node = context_clause OR NULL

   Ref_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Node));
   if Syntax_Node_Type (Node => Ref_Node) = SPSymbols.package_implementation then
      Ref_Node := STree.NullNode;
   end if;
   -- ASSUME Ref_Node = refinement_definition OR NULL

   Ok_To_Add_Body := False;
   FindPackage (Ident_Node, Ident_Str, Scope,
                -- to get
                Spec_Found, Sym);

   --# assert True;
   if not Spec_Found then
      Ok_To_Add_Body := True;
   elsif Syntax_Node_Type (Node => Grand_Parent) = SPSymbols.abody and then Dictionary.HasBodyStub (Sym) then
      ErrorHandler.Semantic_Error
        (Err_Num   => 17,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Ident_Node),
         Id_Str    => Ident_Str);
   elsif Syntax_Node_Type (Node => Grand_Parent) = SPSymbols.subunit then
      -- additional if clause to ensure extra package body subunits reported
      if not Dictionary.HasBodyStub (Sym) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 15,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
      elsif Dictionary.HasBody (Sym) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 16,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
      else
         Ok_To_Add_Body := True;
      end if;
   elsif Dictionary.HasBody (Sym) then
      ErrorHandler.Semantic_Error
        (Err_Num   => 16,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Ident_Node),
         Id_Str    => Ident_Str);
   else -- no errors found
      CheckPackageNeedsBody (Ident_Node, Sym);
      Ok_To_Add_Body := True;
   end if;

   Next_Node := Child_Node (Current_Node => Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Node)));
   -- ASSUME Next_Node = pragma_rep OR hidden_part
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Next_Node) = SPSymbols.pragma_rep
        or else Syntax_Node_Type (Node => Next_Node) = SPSymbols.hidden_part,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Next_Node = pragma_rep OR hidden_part in Wf_Package_Body");
   Body_Is_Hidden := Syntax_Node_Type (Node => Next_Node) = SPSymbols.hidden_part;

   --# assert True;
   if Ok_To_Add_Body then
      Dictionary.AddBody
        (CompilationUnit => Sym,
         Comp_Unit       => ContextManager.Ops.Current_Unit,
         TheBody         => Dictionary.Location'(Start_Position => Node_Position (Node => Ident_Node),
                                                 End_Position   => Node_Position (Node => Ident_Node)),
         Hidden          => Body_Is_Hidden);
   end if;
   Pack_Scope := Dictionary.LocalScope (Sym);

   --# assert True;
   if Syntax_Node_Type (Node => With_Node) = SPSymbols.context_clause then
      -- ASSUME With_Node = context_clause
      Wf_Context_Clause (With_Node, Sym, Pack_Scope);
   elsif With_Node /= STree.NullNode then
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect With_Node = context_clause OR NULL in Wf_Package_Body");
   end if;

   --# assert True;
   if Syntax_Node_Type (Node => Ref_Node) = SPSymbols.refinement_definition then
      -- ASSUME Ref_Node = refinement_definition
      Dictionary.AddRefinementDefinition
        (Sym,
         Dictionary.Location'(Start_Position => Node_Position (Node => Ref_Node),
                              End_Position   => Node_Position (Node => Ref_Node)));
      wf_refine (Ref_Node, Pack_Scope);
   elsif Ref_Node /= STree.NullNode then
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Ref_Node = refinement_definition OR NULL in Wf_Package_Body");
   end if;

   --# assert True;
   if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83 then
      CheckOwnedPackages (Sym, Node);
   end if;

   --# assert True;
   if Body_Is_Hidden then
      ErrorHandler.Hidden_Text
        (Position => Node_Position (Node => Next_Node),
         Unit_Str => Ident_Str,
         Unit_Typ => SPSymbols.package_implementation);
   end if;

   Scope := Pack_Scope;

end wf_package_body;
