-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with LexTokenLists;

separate (SparkLex)
procedure Lex
  (Prog_Text    : in     SPARK_IO.File_Type;
   Allow_Dollar : in     Boolean;
   Token        :    out SPSymbols.SPTerminal;
   Lex_Val      :    out LexTokenManager.Lex_Value;
   Punct_Token  :    out Boolean) is
   Start_Line                      : LexTokenManager.Line_Numbers;
   Start_Posn, End_Posn            : E_Strings.Lengths;
   Lex_Str                         : LexTokenManager.Lex_String;
   Next_Token                      : SPSymbols.SPTerminal;
   Ch                              : Character;
   Err_Val                         : LexTokenManager.Lex_Value;
   Hidden_Part, End_Hide_Found     : Boolean;
   Hide_Designator, End_Designator : LexTokenLists.Lists;

   procedure GetIdent (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.Curr_Line;
   --# derives LineManager.Curr_Line,
   --#         Token                 from LineManager.Curr_Line;
      is separate;

   procedure GetNumber (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.Curr_Line;
   --# derives LineManager.Curr_Line,
   --#         Token                 from LineManager.Curr_Line;
      is separate;

   procedure HyphIntro (Token : out SPSymbols.SPTerminal)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LineManager.Curr_Line;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         LineManager.Curr_Line,
   --#                                         SPARK_IO.File_Sys &
   --#         LineManager.Curr_Line,
   --#         Token                      from CommandLineData.Content,
   --#                                         LineManager.Curr_Line;
      is separate;

   procedure GetString (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.Curr_Line;
   --# derives LineManager.Curr_Line,
   --#         Token                 from LineManager.Curr_Line;
      is separate;

   procedure ApostIntro (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.Curr_Line;
   --# derives LineManager.Curr_Line,
   --#         Token                 from LineManager.Curr_Line;
      is separate;

   procedure LTIntro (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.Curr_Line;
   --# derives LineManager.Curr_Line,
   --#         Token                 from LineManager.Curr_Line;
      is separate;

   procedure NextLex (Token : out SPSymbols.SPTerminal)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LineManager.Curr_Line;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         LineManager.Curr_Line,
   --#                                         SPARK_IO.File_Sys &
   --#         LineManager.Curr_Line,
   --#         Token                      from CommandLineData.Content,
   --#                                         LineManager.Curr_Line;
   --
   -- The Ada Lexis allows the class of a token to be determined by the
   -- first character in the text string representing it. Given the first
   -- character of the string representing the token is at the current position
   -- in the line buffer, LineManager.Curr_Line, Next_Lex determines the class of the token and
   -- calls a procedure to recognise that class of token.
   -- On exit Token is set to a value representing the token.
      is separate;

   -- Modified list of reserved words which are punctuation tokens
   -- this is a knock on effect from change in syntax of proof contexts
   function Is_Punct_Token (Token : SPSymbols.SPTerminal) return Boolean is
      Result : Boolean;
   begin
      case Token is
         when SPSymbols.left_paren       |
           SPSymbols.right_paren      |
           SPSymbols.comma            |
           SPSymbols.colon            |
           SPSymbols.semicolon        |
           SPSymbols.becomes          |
           SPSymbols.double_dot       |
           SPSymbols.point            |
           SPSymbols.apostrophe       |
           SPSymbols.vertical_bar     |
           SPSymbols.arrow            |
           SPSymbols.annotation_start |
           SPSymbols.annotation_end   |
           SPSymbols.proof_context    |
           SPSymbols.RWabort          |
           SPSymbols.RWaccess         |
           SPSymbols.RWall            |
           SPSymbols.RWarray          |
           SPSymbols.RWassert         |
           SPSymbols.RWat             |
           SPSymbols.RWbegin          |
           SPSymbols.RWbody           |
           SPSymbols.RWcase           |
           SPSymbols.RWcheck          |
           SPSymbols.RWconstant       |
           SPSymbols.RWdeclare        |
           SPSymbols.RWdelta          |
           SPSymbols.RWderives        |
           SPSymbols.RWdigits         |
           SPSymbols.RWdo             |
           SPSymbols.RWelse           |
           SPSymbols.RWelsif          |
           SPSymbols.RWend            |
           SPSymbols.RWentry          |
           SPSymbols.RWexception      |
           SPSymbols.RWfor            |
           SPSymbols.RWfrom           |
           SPSymbols.RWfunction       |
           SPSymbols.RWgeneric        |
           SPSymbols.RWglobal         |
           SPSymbols.RWgoto           |
           SPSymbols.RWhide           |
           SPSymbols.RWif             |
           SPSymbols.RWin             |
           SPSymbols.RWinherit        |
           SPSymbols.RWinitializes    |
           SPSymbols.RWis             |
           SPSymbols.RWlimited        |
           SPSymbols.RWloop           |
           SPSymbols.RWnew            |
           SPSymbols.RWnotin          |
           SPSymbols.RWof             |
           SPSymbols.RWothers         |
           SPSymbols.RWout            |
           SPSymbols.RWown            |
           SPSymbols.RWpackage        |
           SPSymbols.RWpost           |
           SPSymbols.RWpragma         |
           SPSymbols.RWpre            |
           SPSymbols.RWprivate        |
           SPSymbols.RWprocedure      |
           SPSymbols.RWraise          |
           SPSymbols.RWrange          |
           SPSymbols.RWrecord         |
           SPSymbols.RWrenames        |
           SPSymbols.RWreturn         |
           SPSymbols.RWreverse        |
           SPSymbols.RWselect         |
           SPSymbols.RWseparate       |
           SPSymbols.RWsubtype        |
           SPSymbols.RWtask           |
           SPSymbols.RWterminate      |
           SPSymbols.RWthen           |
           SPSymbols.RWtype           |
           SPSymbols.RWuse            |
           SPSymbols.RWwhen           |
           SPSymbols.RWwhile          |
           SPSymbols.RWwith           |
           -- SPARK95 reserved words
           SPSymbols.RWabstract       |
           SPSymbols.RWrequeue        |
           SPSymbols.RWprotected      |
           SPSymbols.RWuntil          |
           -- SPARK2005 reserved words
           SPSymbols.RWinterface      |
           SPSymbols.RWsynchronized   =>
            Result := True;
         when others =>
            Result := False;
      end case;
      return Result;
   end Is_Punct_Token;

   procedure Check_Following_Token (Possible_Follower : in SPSymbols.SPSymbol;
                                    Transformed_Token : in SPSymbols.SPSymbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Prog_Text;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out LineManager.Curr_Line;
   --#        in out Next_Token;
   --#        in out SPARK_IO.File_Sys;
   --#           out Lex_Str;
   --# derives ErrorHandler.Error_Context,
   --#         LexTokenManager.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         LineManager.Curr_Line,
   --#                                         Prog_Text,
   --#                                         SPARK_IO.File_Sys &
   --#         Lex_Str                    from  &
   --#         LineManager.Curr_Line      from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Possible_Follower,
   --#                                         Prog_Text,
   --#                                         SPARK_IO.File_Sys &
   --#         Next_Token                 from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         LineManager.Curr_Line,
   --#                                         Possible_Follower,
   --#                                         Prog_Text,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Transformed_Token;
   is
      Following_Token : SPSymbols.SPTerminal;
   begin
      LineManager.Next_Sig_Char (Prog_Text);
      LineManager.Record_Curr_Pos;
      GetIdent (Token => Following_Token);
      if Following_Token = SPSymbols.identifier then
         Check_Reserved
           (Start_Pos  => LineManager.Curr_Line.Last_Token_Pos,
            End_Pos    => LineManager.Curr_Line.Curr_Pos - 1,
            Look_Ahead => True,
            Token      => Following_Token);
         if Following_Token = Possible_Follower then
            Next_Token := Transformed_Token;
         else
            LineManager.Reset_Curr_Pos;
         end if;
      else
         LineManager.Reset_Curr_Pos;
      end if;
      Lex_Str := LexTokenManager.Null_String;
   end Check_Following_Token;

   ---------------------

   procedure Insert_String_Literal
     (Line_In            : in     Line;
      Start_Pos, End_Pos : in     Line_Lengths;
      Lex_Str            :    out LexTokenManager.Lex_String)
   --# global in out LexTokenManager.State;
   --# derives LexTokenManager.State,
   --#         Lex_Str               from End_Pos,
   --#                                    LexTokenManager.State,
   --#                                    Line_In,
   --#                                    Start_Pos;

   is
      Modified_Line : E_Strings.T := E_Strings.Empty_String;
      Ch            : Character;
      Skip_Next     : Boolean     := False;
   begin
      -- copy leading quote
      E_Strings.Append_Char (E_Str => Modified_Line,
                             Ch    => Line_In (Start_Pos));

      -- copy character up to closing quote eliminating doubled quotes
      for I in E_Strings.Positions range Start_Pos + 1 .. End_Pos - 1 loop
         if Skip_Next then
            Skip_Next := False;
         else
            Ch := Line_In (I);
            E_Strings.Append_Char (E_Str => Modified_Line,
                                   Ch    => Ch);
            Skip_Next := Ch = '"';
         end if;
      end loop;

      -- copy closing quote
      E_Strings.Append_Char (E_Str => Modified_Line,
                             Ch    => Line_In (End_Pos));
      -- put in string table
      LexTokenManager.Insert_Examiner_String (Str     => Modified_Line,
                                              Lex_Str => Lex_Str);
   end Insert_String_Literal;

   ---------------------

   procedure Insert_Lex_String
     (The_Line  : in     Line;
      Start_Pos : in     Line_Lengths;
      End_Pos   : in     Line_Lengths;
      Lex_Str   :    out LexTokenManager.Lex_String)
   --# global in out LexTokenManager.State;
   --# derives LexTokenManager.State,
   --#         Lex_Str               from End_Pos,
   --#                                    LexTokenManager.State,
   --#                                    Start_Pos,
   --#                                    The_Line;
   is
      E_Str : E_Strings.T := E_Strings.Empty_String;
   begin
      for I in Line_Lengths range Start_Pos .. End_Pos loop
         E_Strings.Append_Char (E_Str => E_Str,
                                Ch    => The_Line (I));
      end loop;
      LexTokenManager.Insert_Examiner_String (Str     => E_Str,
                                              Lex_Str => Lex_Str);
   end Insert_Lex_String;

begin -- Lex

   --# accept W, 169, LineManager.Curr_Line.Anno_Context, "Direct updates here OK";
   Hidden_Part     := False;
   Hide_Designator := LexTokenLists.Null_List;
   loop
      --# assert True; -- for RTC generation
      LineManager.Next_Sig_Char (Prog_Text);
      Start_Line := LineManager.Curr_Line.Line_No;
      Start_Posn := LineManager.Curr_Line.Curr_Pos;
      NextLex (Token => Next_Token);
      End_Posn := LineManager.Curr_Line.Curr_Pos - 1;

      --# assert True; -- for RTC generation

      case Next_Token is
         when SPSymbols.identifier =>
            Check_Reserved (Start_Pos  => Start_Posn,
                            End_Pos    => End_Posn,
                            Look_Ahead => False,
                            Token      => Next_Token);
            if Next_Token = SPSymbols.identifier then

               if LineManager.Curr_Line.Conts (Start_Posn) = '$' then
                  --# accept F, 41, "Stable expression here OK";
                  if Allow_Dollar then
                     null;
                  else
                     -- Identifier starting with a $, which is illegal in
                     -- this case.  Issue a lexical error and then treat
                     -- as an identifier starting at the next character.
                     ErrorHandler.Lex_Error
                       ("Illegal token",
                        "Token ignored",
                        LexTokenManager.Lex_Value'(Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                                                                Start_Pos     => Start_Posn),
                                                   Token_Str => LexTokenManager.Null_String));
                     Start_Posn := Start_Posn + 1;
                  end if;
                  --# end accept;
               end if;
               Insert_Lex_String
                 (The_Line  => LineManager.Curr_Line.Conts,
                  Start_Pos => Start_Posn,
                  End_Pos   => End_Posn,
                  Lex_Str   => Lex_Str);
               if (LineManager.Curr_Line.Context = In_Annotation) then
                  LineManager.Curr_Line.Anno_Context := Mid_Annotation;
               end if;

            elsif Next_Token = SPSymbols.RWand then
               Check_Following_Token (Possible_Follower => SPSymbols.RWthen,
                                      Transformed_Token => SPSymbols.RWandthen);
            elsif Next_Token = SPSymbols.RWor then
               Check_Following_Token (Possible_Follower => SPSymbols.RWelse,
                                      Transformed_Token => SPSymbols.RWorelse);
            elsif Next_Token = SPSymbols.RWnot then
               Check_Following_Token (Possible_Follower => SPSymbols.RWin,
                                      Transformed_Token => SPSymbols.RWnotin);
            elsif Next_Token = SPSymbols.RWfor then
               Check_Following_Token (Possible_Follower => SPSymbols.RWall,
                                      Transformed_Token => SPSymbols.RWforall);
               if Next_Token = SPSymbols.RWfor then
                  Check_Following_Token (Possible_Follower => SPSymbols.RWsome,
                                         Transformed_Token => SPSymbols.RWforsome);
               end if;
            else
               Lex_Str := LexTokenManager.Null_String;
            end if;

         when SPSymbols.string_literal =>
            Insert_String_Literal
              (Line_In   => LineManager.Curr_Line.Conts,
               Start_Pos => Start_Posn,
               End_Pos   => End_Posn,
               Lex_Str   => Lex_Str);
         when SPSymbols.character_literal | SPSymbols.integer_number | SPSymbols.real_number | SPSymbols.based_integer =>
            Insert_Lex_String
              (The_Line  => LineManager.Curr_Line.Conts,
               Start_Pos => Start_Posn,
               End_Pos   => End_Posn,
               Lex_Str   => Lex_Str);
         when SPSymbols.annotation_start | SPSymbols.proof_context =>
            if LineManager.Curr_Line.Context = In_Annotation then
               Next_Token := SPSymbols.annotation_continuation;
            else
               LineManager.Set_Context (In_Annotation);
            end if;

            if (LineManager.Curr_Line.Anno_Context = Ended_Annotation) then
               LineManager.Curr_Line.Anno_Context := Start_Annotation;
            end if;

            Lex_Str := LexTokenManager.Null_String;

         when SPSymbols.annotation_end =>
            LineManager.Set_Context (In_Ada);
            Lex_Str := LexTokenManager.Null_String;

         when SPSymbols.hide_directive =>
            -- skip over hide
            LineManager.Next_Sig_Char (Prog_Text);

            --# accept F, 10, Next_Token, "Skipping so ineffective assignment";
            NextLex (Token => Next_Token);
            --# end accept;

            -- now get designator
            LineManager.Next_Sig_Char (Prog_Text);
            Start_Line := LineManager.Curr_Line.Line_No;
            Start_Posn := LineManager.Curr_Line.Curr_Pos;
            NextLex (Token => Next_Token);
            End_Posn := LineManager.Curr_Line.Curr_Pos - 1;
            if Next_Token = SPSymbols.identifier then
               Check_Reserved (Start_Pos  => Start_Posn,
                               End_Pos    => End_Posn,
                               Look_Ahead => False,
                               Token      => Next_Token);
            end if;

            --# assert True; -- for RTC generation

            if Next_Token /= SPSymbols.identifier then
               -- ???? Report Error ???? --
               Err_Val :=
                 LexTokenManager.Lex_Value'
                 (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                               Start_Pos     => Start_Posn),
                  Token_Str => LexTokenManager.Null_String);
               ErrorHandler.Lex_Error ("Hide designator missing", "Hide directive ignored", Err_Val);
               LineManager.Reset_Curr_Pos;
               Next_Token := SPSymbols.comment;
            else
               Hidden_Part     := True;
               Hide_Designator := LexTokenLists.Null_List;
               Insert_Lex_String
                 (The_Line  => LineManager.Curr_Line.Conts,
                  Start_Pos => Start_Posn,
                  End_Pos   => End_Posn,
                  Lex_Str   => Lex_Str);
               LexTokenLists.Append (Hide_Designator, Lex_Str);

               loop
                  --# assert True; -- for RTC generation
                  LineManager.Next_Sig_Char (Prog_Text);
                  LineManager.Inspect_Char (Ch);
                  exit when Ch /= '.';
                  LineManager.Lookahead_Char (Ch);
                  LineManager.Reject_Lookahead;
                  exit when Ch = '.';   -- stop if ..
                  LineManager.Accept_Char; -- absorb dot
                  LineManager.Next_Sig_Char (Prog_Text);
                  Start_Line := LineManager.Curr_Line.Line_No;
                  Start_Posn := LineManager.Curr_Line.Curr_Pos;
                  NextLex (Token => Next_Token);
                  End_Posn := LineManager.Curr_Line.Curr_Pos - 1;
                  if Next_Token = SPSymbols.identifier then
                     Check_Reserved (Start_Pos  => Start_Posn,
                                     End_Pos    => End_Posn,
                                     Look_Ahead => False,
                                     Token      => Next_Token);
                  end if;
                  if Next_Token /= SPSymbols.identifier then
                     -- ???? Report Error ???? --
                     Err_Val :=
                       LexTokenManager.Lex_Value'
                       (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                                     Start_Pos     => Start_Posn),
                        Token_Str => LexTokenManager.Null_String);
                     ErrorHandler.Lex_Error ("Hide designator incomplete", "Hide directive ignored", Err_Val);
                     LineManager.Reset_Curr_Pos;
                     Next_Token  := SPSymbols.comment;
                     Hidden_Part := False;
                     exit;
                  end if;
                  Insert_Lex_String
                    (The_Line  => LineManager.Curr_Line.Conts,
                     Start_Pos => Start_Posn,
                     End_Pos   => End_Posn,
                     Lex_Str   => Lex_Str);
                  LexTokenLists.Append (Hide_Designator, Lex_Str);
               end loop;
            end if;
            Lex_Str := LexTokenManager.Null_String;

         when SPSymbols.apostrophe =>
            LineManager.Next_Sig_Char (Prog_Text);  -- Check for attribute designator.
            LineManager.Inspect_Char (Ch);
            if Ada.Characters.Handling.Is_Letter (Ch) then
               Start_Line := LineManager.Curr_Line.Line_No;
               Start_Posn := LineManager.Curr_Line.Curr_Pos;
               GetIdent (Token => Next_Token);
               End_Posn := LineManager.Curr_Line.Curr_Pos - 1;
               Insert_Lex_String
                 (The_Line  => LineManager.Curr_Line.Conts,
                  Start_Pos => Start_Posn,
                  End_Pos   => End_Posn,
                  Lex_Str   => Lex_Str);
               if Next_Token = SPSymbols.identifier then
                  Next_Token := SPSymbols.attribute_ident;
               else
                  -- ???? Report Error ???? --
                  Err_Val :=
                    LexTokenManager.Lex_Value'
                    (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                                  Start_Pos     => Start_Posn),
                     Token_Str => Lex_Str);
                  ErrorHandler.Lex_Error ("Attribute identifier expected", "Attribute identifier assumed", Err_Val);
                  Next_Token := SPSymbols.attribute_ident; -- Error recovery.
               end if;
            else
               Lex_Str := LexTokenManager.Null_String;
            end if;

         when SPSymbols.illegal_id =>
            Next_Token := SPSymbols.identifier;   -- Error recovery.
            Insert_Lex_String
              (The_Line  => LineManager.Curr_Line.Conts,
               Start_Pos => Start_Posn,
               End_Pos   => End_Posn,
               Lex_Str   => Lex_Str);
            -- ???? Report Error ???? --
            Err_Val :=
              LexTokenManager.Lex_Value'
              (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                            Start_Pos     => Start_Posn),
               Token_Str => Lex_Str);
            ErrorHandler.Lex_Error ("Illegal identifier", "Identifier assumed", Err_Val);

         when SPSymbols.illegal_number =>
            Next_Token := SPSymbols.integer_number;
            Insert_Lex_String
              (The_Line  => LineManager.Curr_Line.Conts,
               Start_Pos => Start_Posn,
               End_Pos   => End_Posn,
               Lex_Str   => Lex_Str);
            -- ???? Report Error ???? --
            Err_Val :=
              LexTokenManager.Lex_Value'
              (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                            Start_Pos     => Start_Posn),
               Token_Str => Lex_Str);
            if LineManager.Curr_Line.Conts (Start_Posn) = '0' then
               ErrorHandler.Lex_Error
                 ("Illegal number - possible mis-use of '0' instead of 'O' as first letter of identifier",
                  "Number assumed",
                  Err_Val);
            else
               ErrorHandler.Lex_Error ("Illegal number", "Number assumed", Err_Val);
            end if;

         when SPSymbols.based_real =>
            Insert_Lex_String
              (The_Line  => LineManager.Curr_Line.Conts,
               Start_Pos => Start_Posn,
               End_Pos   => End_Posn,
               Lex_Str   => Lex_Str);
            -- ???? Report Error ???? --
            Err_Val :=
              LexTokenManager.Lex_Value'
              (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                            Start_Pos     => Start_Posn),
               Token_Str => Lex_Str);
            ErrorHandler.Lex_Error ("Based real numbers are not allowed", "Number assumed", Err_Val);

         when SPSymbols.unterminated_string =>
            Next_Token := SPSymbols.string_literal;  -- Error recovery.
            Insert_Lex_String
              (The_Line  => LineManager.Curr_Line.Conts,
               Start_Pos => Start_Posn,
               End_Pos   => End_Posn,
               Lex_Str   => Lex_Str);
            -- ???? Report Error ???? --
            Err_Val :=
              LexTokenManager.Lex_Value'
              (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                            Start_Pos     => Start_Posn),
               Token_Str => Lex_Str);
            ErrorHandler.Lex_Error ("Unterminated string", "String assumed", Err_Val);

         when SPSymbols.illegal_comment =>
            Lex_Str    := LexTokenManager.Null_String;
            Next_Token := SPSymbols.comment;  -- Error recovery.

            -- ???? Report Error ???? --
            Err_Val :=
              LexTokenManager.Lex_Value'
              (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                            Start_Pos     => Start_Posn),
               Token_Str => LexTokenManager.Null_String);
            ErrorHandler.Lex_Error ("Illegal character in comment", "Illegal character ignored", Err_Val);

         when SPSymbols.illegal_token =>
            Lex_Str := LexTokenManager.Null_String;
            -- ???? Report Error ???? --
            Err_Val :=
              LexTokenManager.Lex_Value'
              (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                            Start_Pos     => Start_Posn),
               Token_Str => LexTokenManager.Null_String);
            ErrorHandler.Lex_Error ("Illegal token", "Token ignored", Err_Val);

         when SPSymbols.LONGLINE =>
            Lex_Str := LexTokenManager.Null_String;
            -- ???? Give Error Message ???? --
            Err_Val :=
              LexTokenManager.Lex_Value'
              (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                            Start_Pos     => End_Posn),
               Token_Str => LexTokenManager.Null_String);
            ErrorHandler.Lex_Error ("Line too long", "Ignoring excess characters", Err_Val);

         when SPSymbols.semicolon =>
            Lex_Str := LexTokenManager.Null_String;
            if (LineManager.Curr_Line.Context = In_Annotation) then
               LineManager.Curr_Line.Anno_Context := Ended_Annotation;
            end if;

         when others =>
            Lex_Str := LexTokenManager.Null_String;
      end case;

      --# assert True; -- for RTC generation

      exit when Hidden_Part or
        (Next_Token /= SPSymbols.comment and
           Next_Token /= SPSymbols.annotation_continuation and
           Next_Token /= SPSymbols.LONGLINE and
           Next_Token /= SPSymbols.illegal_token);
   end loop;

   --# assert True; -- for RTC generation

   if Hidden_Part then
      End_Hide_Found := False;

      loop
         --# assert True; -- for RTC generation
         LineManager.Next_Sig_Char (Prog_Text);
         Start_Line := LineManager.Curr_Line.Line_No;
         Start_Posn := LineManager.Curr_Line.Curr_Pos;
         NextLex (Token => Next_Token);
         End_Posn := LineManager.Curr_Line.Curr_Pos - 1;

         if Next_Token = SPSymbols.SPEND then
            End_Hide_Found := True;
         elsif Next_Token = SPSymbols.identifier then
            Check_Reserved (Start_Pos  => Start_Posn,
                            End_Pos    => End_Posn,
                            Look_Ahead => False,
                            Token      => Next_Token);
         end if;
         if Next_Token = SPSymbols.RWend then
            LineManager.Next_Sig_Char (Prog_Text);
            Start_Line := LineManager.Curr_Line.Line_No;
            Start_Posn := LineManager.Curr_Line.Curr_Pos;
            NextLex (Token => Next_Token);
            End_Posn := LineManager.Curr_Line.Curr_Pos - 1;

            if Next_Token = SPSymbols.identifier then
               End_Designator := LexTokenLists.Null_List;
               Insert_Lex_String
                 (The_Line  => LineManager.Curr_Line.Conts,
                  Start_Pos => Start_Posn,
                  End_Pos   => End_Posn,
                  Lex_Str   => Lex_Str);
               LexTokenLists.Append (End_Designator, Lex_Str);

               -- Process remainder of dotted name (if any)
               loop
                  --# assert True; -- for RTC generation
                  LineManager.Next_Sig_Char (Prog_Text);
                  LineManager.Inspect_Char (Ch);
                  exit when Ch /= '.';
                  LineManager.Lookahead_Char (Ch);
                  LineManager.Reject_Lookahead;
                  exit when Ch = '.';   -- stop if '..'
                  LineManager.Accept_Char; -- absorb dot
                  LineManager.Next_Sig_Char (Prog_Text);
                  Start_Posn := LineManager.Curr_Line.Curr_Pos;
                  NextLex (Token => Next_Token); -- get expected identifier
                  End_Posn := LineManager.Curr_Line.Curr_Pos - 1;
                  exit when Next_Token /= SPSymbols.identifier;
                  Insert_Lex_String
                    (The_Line  => LineManager.Curr_Line.Conts,
                     Start_Pos => Start_Posn,
                     End_Pos   => End_Posn,
                     Lex_Str   => Lex_Str);
                  LexTokenLists.Append (End_Designator, Lex_Str);
               end loop;

               if Next_Token = SPSymbols.identifier then
                  End_Hide_Found := LexTokenLists.Eq_Unit (First_Item => Hide_Designator,
                                                           Second     => End_Designator);
               end if;
            end if;
         end if;

         --# assert True; -- for RTC generation

         if End_Hide_Found then
            Hidden_Part := False;
            Next_Token  := SPSymbols.hide_directive;
            Lex_Str     := LexTokenManager.Null_String;
         end if;
         exit when not Hidden_Part;
      end loop;
   end if;

   --# assert True; -- for RTC generation

   Token       := Next_Token;
   Lex_Val     :=
     LexTokenManager.Lex_Value'
     (Position  => LexTokenManager.Token_Position'(Start_Line_No => Start_Line,
                                                   Start_Pos     => Start_Posn),
      Token_Str => Lex_Str);
   Punct_Token := Is_Punct_Token (Token => Next_Token);

end Lex;
