-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Statistics;
with SystemErrors;

package body STree is

   type NodeValue is record
      NodeType            : SPSymbols.SPSymbol;
      Position            : LexTokenManager.Token_Position;
      Parent, Next, Child : SyntaxNode;
      Token_Str           : LexTokenManager.Lex_String;
      -- case ??? is
      --   when ??? =>
      Ref_Str : LexTokenManager.Lex_String;
      --   when ??? =>
      Ref_Sym : Dictionary.Symbol;
      -- end case;
   end record;

   type SyntaxTreeContents is array (SyntaxNode) of NodeValue;

   type TableStructure is record
      FreeList       : SyntaxNode;
      TopUsed        : SyntaxNode;
      CurrSyntaxNode : SyntaxNode;
      Contents       : SyntaxTreeContents;
   end record;

   Table : TableStructure;

   ------------------------------------------------------------------------

   procedure RetrieveCurrentRoot (Root : out SyntaxNode) is
   begin
      Root                 := Table.CurrSyntaxNode;
      Table.CurrSyntaxNode := NullNode;
   end RetrieveCurrentRoot;

   ------------------------------------------------------------------------

   function NodeToRef (Node : SyntaxNode) return ExaminerConstants.RefType is
   begin
      return ExaminerConstants.RefType (Node);
   end NodeToRef;

   ------------------------------------------------------------------------

   function RefToNode (Ref : ExaminerConstants.RefType) return SyntaxNode is
   begin
      return SyntaxNode (Ref);
   end RefToNode;

   ------------------------------------------------------------------------

   procedure AllocateNode (Node : out SyntaxNode)
   --# global in out Table;
   --# derives Node,
   --#         Table from Table;
   is
      NewNode : SyntaxNode;
   begin
      if Table.FreeList /= NullNode then
         NewNode := Table.FreeList;
         -- Child field used as free list pointer.
         Table.FreeList := Table.Contents (Table.FreeList).Child;
      elsif Table.TopUsed /= SyntaxTreeContents'Last then
         Table.TopUsed := Table.TopUsed + 1;
         NewNode       := Table.TopUsed;
      else
         NewNode := NullNode;
      end if;

      if NewNode = NullNode then
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Syntax_Tree_Overflow,
                                   Msg     => "");
      else
         Table.Contents (NewNode) :=
           NodeValue'
           (NodeType  => SPSymbols.SPEND,
            Position  => LexTokenManager.Null_Token_Position,
            Parent    => NullNode,
            Next      => NullNode,
            Child     => NullNode,
            Token_Str => LexTokenManager.Null_String,
            Ref_Str   => LexTokenManager.Null_String,
            Ref_Sym   => Dictionary.NullSymbol);
      end if;

      Node := NewNode;
   end AllocateNode;

   ------------------------------------------------------------------------

   procedure NewProduction (Production : in     SPSymbols.SPNonTerminal;
                            Node       :    out SyntaxNode) is
      CurrNode : SyntaxNode;
   begin
      AllocateNode (CurrNode);
      Table.Contents (CurrNode).NodeType := Production;
      Table.CurrSyntaxNode               := CurrNode;
      Node                               := CurrNode;
   end NewProduction;

   ------------------------------------------------------------------------

   procedure NewTerminal (Terminal    : in     SPSymbols.SPTerminal;
                          TerminalVal : in     LexTokenManager.Lex_Value;
                          Node        :    out SyntaxNode) is
      CurrNode : SyntaxNode;
   begin
      AllocateNode (CurrNode);
      Table.Contents (CurrNode).NodeType  := Terminal;
      Table.Contents (CurrNode).Position  := TerminalVal.Position;
      Table.Contents (CurrNode).Token_Str := TerminalVal.Token_Str;
      Table.Contents (CurrNode).Ref_Str   := TerminalVal.Token_Str;
      Node                                := CurrNode;
   end NewTerminal;

   ------------------------------------------------------------------------

   procedure AddDerivative (Child_Node : in SyntaxNode) is
      CurrNode    : SyntaxNode;
      LastChild   : SyntaxNode;
      ThePosition : LexTokenManager.Token_Position;
   begin
      CurrNode  := Table.CurrSyntaxNode;
      LastChild := Table.Contents (CurrNode).Child;
      if LastChild = NullNode then
         Table.Contents (CurrNode).Child    := Child_Node;
         ThePosition                        := Table.Contents (Child_Node).Position;
         Table.Contents (CurrNode).Position := ThePosition;
      else
         loop
            exit when Table.Contents (LastChild).Next = NullNode;
            LastChild := Table.Contents (LastChild).Next;
         end loop;
         if Table.Contents (CurrNode).Position = LexTokenManager.Null_Token_Position then
            ThePosition                        := Table.Contents (Child_Node).Position;
            Table.Contents (CurrNode).Position := ThePosition;
         end if;
         Table.Contents (LastChild).Next := Child_Node;
      end if;
      Table.Contents (Child_Node).Parent := CurrNode;
   end AddDerivative;

   ------------------------------------------------------------------------

   procedure AddChildNode (Node         : in SyntaxNode;
                           ChildNode    : in SyntaxNode;
                           LinkToParent : in Boolean) is
      LastChild : SyntaxNode;
   begin
      LastChild := Table.Contents (Node).Child;
      if LastChild = NullNode then
         Table.Contents (Node).Child := ChildNode;
      else
         loop
            exit when Table.Contents (LastChild).Next = NullNode;
            LastChild := Table.Contents (LastChild).Next;
         end loop;
         Table.Contents (LastChild).Next := ChildNode;
      end if;
      if LinkToParent then
         Table.Contents (ChildNode).Parent := Node;
      end if;
   end AddChildNode;

   ------------------------------------------------------------------------

   function Child_Node (Current_Node : SyntaxNode) return SyntaxNode is
   begin
      return Table.Contents (Current_Node).Child;
   end Child_Node;

   ------------------------------------------------------------------------

   function Next_Sibling (Current_Node : SyntaxNode) return SyntaxNode is
   begin
      return Table.Contents (Current_Node).Next;
   end Next_Sibling;

   ------------------------------------------------------------------------

   function Parent_Node (Current_Node : SyntaxNode) return SyntaxNode is
   begin
      return Table.Contents (Current_Node).Parent;
   end Parent_Node;

   ------------------------------------------------------------------------

   function Syntax_Node_Type (Node : SyntaxNode) return SPSymbols.SPSymbol is
   begin
      return Table.Contents (Node).NodeType;
   end Syntax_Node_Type;

   ------------------------------------------------------------------------

   function Node_Position (Node : SyntaxNode) return LexTokenManager.Token_Position is
      Position : LexTokenManager.Token_Position;
   begin
      if Node = NullNode then
         Position := LexTokenManager.Null_Token_Position;
      else
         Position := Table.Contents (Node).Position;
      end if;
      return Position;
   end Node_Position;

   ------------------------------------------------------------------------

   procedure Set_Node_Lex_String (Sym  : in Dictionary.Symbol;
                                  Node : in SyntaxNode) is
      Str : LexTokenManager.Lex_String;
   begin
      Str := Dictionary.GetSimpleName (Item => Sym);
      SystemErrors.RT_Assert
        (C       => Table.Contents (Node).NodeType in SPSymbols.SPTerminal
           and then LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => Str,
            Lex_Str2 => Table.Contents (Node).Ref_Str) =
           LexTokenManager.Str_Eq,
         Sys_Err => SystemErrors.Other_Internal_Error,
         Msg     => "STree.Set_Node_Lex_String : Program Error");
      Table.Contents (Node).Ref_Str := Str;
   end Set_Node_Lex_String;

   ------------------------------------------------------------------------

   function Node_Lex_String (Node : SyntaxNode) return LexTokenManager.Lex_String is
      Lex_Str : LexTokenManager.Lex_String;
   begin
      if Node /= NullNode and then (Table.Contents (Node).NodeType in SPSymbols.SPTerminal) then
         Lex_Str := Table.Contents (Node).Ref_Str;
      else
         Lex_Str := LexTokenManager.Null_String;
      end if;
      return Lex_Str;
   end Node_Lex_String;

   ------------------------------------------------------------------------

   function Node_Token_String (Node : SyntaxNode) return LexTokenManager.Lex_String is
   begin
      return Table.Contents (Node).Token_Str;
   end Node_Token_String;

   ------------------------------------------------------------------------

   procedure CollectNode (Node : in SyntaxNode)
   --# global in out Table;
   --# derives Table from *,
   --#                    Node;
   is
      TheNodeValue : NodeValue;
   begin
      TheNodeValue                := Table.Contents (NullNode);
      Table.Contents (Node)       := TheNodeValue;
      Table.Contents (Node).Child := Table.FreeList;
      Table.FreeList              := Node;
   end CollectNode;

   ----------------------------------------------------------------------

   procedure DeleteSyntaxTree (Root          : in SyntaxNode;
                               KeepConstants : in Boolean) is
      CurrNode, The_Next_Node, TempNode : SyntaxNode;
   begin
      CurrNode := Root;
      loop
         The_Next_Node := Child_Node (CurrNode);

         --  Subprogram constraints are needed by the VCG later, so we
         --  don't return such nodes to the free list. Similarly,
         --  constant declarations are needed by the Declarations
         --  package for proof rule generation. Furthermore, we also
         --  need to protect generic_actual_part to preserve
         --  expressions used to initialize generic formal objects. We
         --  need also to keep inherit_clause to calculate the
         --  dependency closure.
         while The_Next_Node /= NullNode
           and then ((Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.procedure_constraint) or
                       (Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.function_constraint) or
                       (KeepConstants and Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.constant_declaration) or
                       (Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.generic_actual_part) or
                       (Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.inherit_clause)) loop
            Table.Contents (The_Next_Node).Parent := NullNode;
            TempNode                              := Next_Sibling (The_Next_Node);
            Table.Contents (The_Next_Node).Next   := NullNode;
            The_Next_Node                         := TempNode;
         end loop;

         if The_Next_Node = NullNode then
            loop
               The_Next_Node := Next_Sibling (CurrNode);
               while The_Next_Node /= NullNode
                 and then ((Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.procedure_constraint) or
                             (Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.function_constraint) or
                             (KeepConstants and Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.constant_declaration) or
                             (Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.generic_actual_part) or
                             (Syntax_Node_Type (Node => The_Next_Node) = SPSymbols.inherit_clause)) loop
                  Table.Contents (The_Next_Node).Parent := NullNode;
                  TempNode                              := Next_Sibling (The_Next_Node);
                  Table.Contents (The_Next_Node).Next   := NullNode;
                  The_Next_Node                         := TempNode;
               end loop;
               if The_Next_Node /= NullNode then
                  CollectNode (CurrNode);
                  exit;
               end if;
               The_Next_Node := Parent_Node (Current_Node => CurrNode);
               CollectNode (CurrNode);
               exit when The_Next_Node = NullNode;
               CurrNode := The_Next_Node;
            end loop;
         end if;
         exit when The_Next_Node = NullNode;
         CurrNode := The_Next_Node;
      end loop;
   end DeleteSyntaxTree;

   -----------------------------------------------------------------------

   function TraverseAcross (RootNode : SyntaxNode;
                            CurrNode : SyntaxNode) return SyntaxNode
   --# global in Table;
   -- Traverses across the tree in a pre-order fashion.
   is
      The_Next_Node : SyntaxNode;
   begin
      The_Next_Node := CurrNode;
      while The_Next_Node /= NullNode loop
         if The_Next_Node = RootNode then
            The_Next_Node := NullNode;
            exit;
         end if;
         if Next_Sibling (The_Next_Node) /= NullNode then
            The_Next_Node := Next_Sibling (The_Next_Node);
            exit;
         end if;
         The_Next_Node := Parent_Node (Current_Node => The_Next_Node);
      end loop;
      return The_Next_Node;
   end TraverseAcross;

   -----------------------------------------------------------------------

   function Traverse (RootNode : SyntaxNode;
                      CurrNode : SyntaxNode) return SyntaxNode
   --# global in Table;
   -- Traverses the tree in a pre-order fashion.
   is
      The_Next_Node : SyntaxNode;
   begin
      if Child_Node (CurrNode) /= NullNode then
         -- Depth first
         The_Next_Node := Child_Node (CurrNode);
      else
         -- breadth next
         The_Next_Node := TraverseAcross (RootNode, CurrNode);
      end if;
      return The_Next_Node;
   end Traverse;

   -----------------------------------------------------------------------

   function Last_Child_Of (Start_Node : SyntaxNode) return SyntaxNode is
      Last_Valid_Node_Found, The_Next_Node : SyntaxNode;
   begin
      Last_Valid_Node_Found := Start_Node;
      The_Next_Node         := Start_Node;
      while The_Next_Node /= NullNode loop
         Last_Valid_Node_Found := The_Next_Node;
         The_Next_Node         := Child_Node (The_Next_Node);
      end loop;

      return Last_Valid_Node_Found;

   end Last_Child_Of;

   --------------------------------------------------------------

   function Last_Sibling_Of (Start_Node : SyntaxNode) return SyntaxNode is
      Last_Valid_Node_Found, The_Next_Node : SyntaxNode;
   begin
      Last_Valid_Node_Found := Start_Node;
      The_Next_Node         := Start_Node;
      while The_Next_Node /= NullNode loop
         Last_Valid_Node_Found := The_Next_Node;
         The_Next_Node         := Next_Sibling (The_Next_Node);
      end loop;

      return Last_Valid_Node_Found;

   end Last_Sibling_Of;

   --------------------------------------------------------------------

   procedure AddNodeSymbol (Node : in SyntaxNode;
                            Sym  : in Dictionary.Symbol) is
   begin
      Table.Contents (Node).Ref_Sym := Sym;
   end AddNodeSymbol;

   --------------------------------------------------------------------

   function NodeSymbol (Node : SyntaxNode) return Dictionary.Symbol is
      Sym : Dictionary.Symbol;
   begin
      Sym := Table.Contents (Node).Ref_Sym;
      return Sym;
   end NodeSymbol;

   -------------------------------------------------------

   procedure ReportUsage is
   begin
      -- TopUsed is peak usage, because freelist is used up before allocating
      -- a new cell at the top of the table
      Statistics.SetTableUsage (Statistics.SyntaxTree, Integer (Table.TopUsed));
   end ReportUsage;

   -------------------------------------------------------

   function NextNodeType (It : Iterator) return Iterator
   --# global in Table;
   is
      Node   : SyntaxNode;
      NextIt : Iterator;
   begin
      Node := It.Current;
      while Node /= NullNode loop
         --# accept Flow, 41, "Expected stable expression";
         if It.SearchDirection = Down then -- expect stable expression
            if Syntax_Node_Type (Node => Node) = It.SearchNodeType then
               -- This node was returned in the search so ignore all nodes
               -- below it.
               Node := TraverseAcross (It.Root, Node);
            else
               -- Get the next pre-order node
               Node := Traverse (RootNode => It.Root,
                                 CurrNode => Node);
            end if;
         else
            Node := Parent_Node (Current_Node => Node);
         end if;
         --# end accept;
         exit when Syntax_Node_Type (Node => Node) = It.SearchNodeType;
      end loop;

      if Node = NullNode then
         NextIt := NullIterator;
      else
         NextIt :=
           Iterator'
           (TheSearchKind   => It.TheSearchKind,
            SearchNodeType  => It.SearchNodeType,
            SearchDirection => It.SearchDirection,
            Current         => Node,
            Root            => It.Root);
      end if;
      return NextIt;
   end NextNodeType;

   -------------------------------------------------------

   function IsBranchNode (Node : SyntaxNode) return Boolean
   --# global in Table;
   is
   begin
      return Node /= NullNode
        and then
        -- has more than one child
        Child_Node (Node) /= NullNode
        and then Next_Sibling (Child_Node (Node)) /= NullNode;
   end IsBranchNode;

   -------------------------------------------------------

   function NextBranch (It : Iterator) return Iterator
   --# global in Table;
   is
      Node   : SyntaxNode;
      NextIt : Iterator;
   begin
      Node := It.Current;
      while Node /= NullNode loop
         --# accept Flow, 41, "Expected stable expression";
         if It.SearchDirection = Down then -- expect stable expression
            Node := Traverse (RootNode => It.Root,
                              CurrNode => Node);
         else
            Node := Parent_Node (Current_Node => Node);
         end if;
         --# end accept;
         exit when IsBranchNode (Node);
      end loop;

      if Node = NullNode then
         NextIt := NullIterator;
      else
         NextIt :=
           Iterator'
           (TheSearchKind   => It.TheSearchKind,
            SearchNodeType  => It.SearchNodeType,
            SearchDirection => It.SearchDirection,
            Current         => Node,
            Root            => It.Root);
      end if;
      return NextIt;
   end NextBranch;

   -------------------------------------------------------

   function IsFormalParameterNode (Node     : SyntaxNode;
                                   RootNode : SyntaxNode) return Boolean
   --# global in Table;
   is
      MyNode : SyntaxNode;
      Result : Boolean;
   begin
      if Syntax_Node_Type (Node => Node) /= SPSymbols.identifier
        or else Syntax_Node_Type (Node => Parent_Node (Current_Node => Node)) /= SPSymbols.simple_name then
         Result := False;
      else
         Result := True;
         MyNode := Node;
         while MyNode /= RootNode loop
            if Syntax_Node_Type (Node => MyNode) = SPSymbols.expression then
               Result := False;
               exit;
            end if;
            MyNode := Parent_Node (Current_Node => MyNode);
         end loop;
      end if;
      return Result;
   end IsFormalParameterNode;

   -------------------------------------------------------

   function NextFormalParameter (It : Iterator) return Iterator
   --# global in Table;
   is
      Node   : SyntaxNode;
      NextIt : Iterator;
   begin
      Node := It.Current;
      while Node /= NullNode loop
         Node := Traverse (RootNode => It.Root,
                           CurrNode => Node);
         exit when IsFormalParameterNode (Node, It.Root);
      end loop;

      if Node = NullNode then
         NextIt := NullIterator;
      else
         NextIt :=
           Iterator'
           (TheSearchKind   => It.TheSearchKind,
            SearchNodeType  => It.SearchNodeType,
            SearchDirection => It.SearchDirection,
            Current         => Node,
            Root            => It.Root);
      end if;
      return NextIt;
   end NextFormalParameter;

   -------------------------------------------------------

   function NextNode (It : Iterator) return Iterator is
      NextIt : Iterator;
   begin
      case It.TheSearchKind is
         when Undefined =>
            NextIt := NullIterator;
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Syntax_Tree_Walk_Error,
                                      Msg     => "in STree.NextNode");
         when NodeTypeSearch =>
            NextIt := NextNodeType (It);
         when BranchSearch =>
            NextIt := NextBranch (It);
         when FormalParameterSearch =>
            NextIt := NextFormalParameter (It);
      end case;
      return NextIt;
   end NextNode;

   -------------------------------------------------------

   function Find_First_Node
     (Node_Kind    : SPSymbols.SPSymbol;
      From_Root    : SyntaxNode;
      In_Direction : TraverseDirection)
     return         Iterator
   is
      It : Iterator;
   begin
      It :=
        Iterator'
        (TheSearchKind   => NodeTypeSearch,
         SearchNodeType  => Node_Kind,
         SearchDirection => In_Direction,
         Current         => From_Root,
         Root            => From_Root);

      if Syntax_Node_Type (Node => From_Root) /= It.SearchNodeType then
         It := NextNode (It);
      end if;
      return It;
   end Find_First_Node;

   -------------------------------------------------------

   function Find_First_Branch_Node (From_Root    : SyntaxNode;
                                    In_Direction : TraverseDirection) return Iterator is
      It : Iterator;
   begin
      It :=
        Iterator'
        (TheSearchKind   => BranchSearch,
         SearchNodeType  => SPSymbols.SPEND,
         SearchDirection => In_Direction,
         Current         => From_Root,
         Root            => From_Root);

      if not IsBranchNode (From_Root) then
         It := NextNode (It);
      end if;
      return It;
   end Find_First_Branch_Node;

   -------------------------------------------------------

   function FindFirstFormalParameterNode (FromRoot : SyntaxNode) return Iterator is
      It : Iterator;
   begin
      if Syntax_Node_Type (Node => FromRoot) /= SPSymbols.named_argument_association then
         It := NullIterator;
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Syntax_Tree_Walk_Error,
                                   Msg     => "in STree.FindFormalParameterNode");
      else
         It := Find_First_Node (Node_Kind    => SPSymbols.identifier,
                                From_Root    => FromRoot,
                                In_Direction => Down);
      end if;

      return Iterator'
        (TheSearchKind   => FormalParameterSearch,
         SearchNodeType  => SPSymbols.SPEND,
         SearchDirection => Down,
         Current         => It.Current,
         Root            => FromRoot);
   end FindFirstFormalParameterNode;

   -------------------------------------------------------

   function Get_Node (It : Iterator) return SyntaxNode is
   begin
      return It.Current;
   end Get_Node;

   -------------------------------------------------------

   function IsNull (It : Iterator) return Boolean is
   begin
      return It = NullIterator;
   end IsNull;

   -------------------------------------------------------

   function FindLastItemInDependencyRelation (Node : SyntaxNode) return SyntaxNode is separate;

   -------------------------------------------------------

   function FindLastActualParameterNode (FromRoot : SyntaxNode) return SyntaxNode is
      LastFormal : SyntaxNode;
      LastActual : SyntaxNode;
      It         : Iterator;
   begin
      LastFormal := NullNode;
      if Syntax_Node_Type (Node => FromRoot) /= SPSymbols.named_argument_association then
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Syntax_Tree_Walk_Error,
                                   Msg     => "in STree.FindLastActualParameterNode");
      else
         -- Get the first formal
         It := FindFirstFormalParameterNode (FromRoot => FromRoot);
         -- find the last formal
         while not IsNull (It) loop
            LastFormal := Get_Node (It => It);
            It         := NextNode (It);
         end loop;
      end if;

      if LastFormal = NullNode then
         LastActual := NullNode;
      else
         -- work out the last actual
         -- simple_name => expression
         LastActual :=
           Next_Sibling
           (Get_Node (It => Find_First_Node (Node_Kind    => SPSymbols.simple_name,
                                             From_Root    => LastFormal,
                                             In_Direction => Up)));
      end if;
      return LastActual;
   end FindLastActualParameterNode;

   -------------------------------------------------------

   function ExpressionFromPositionalArgumentAssociation (Node : SyntaxNode) return SyntaxNode is
      Result : SyntaxNode;
   begin
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.positional_argument_association,
         Sys_Err => SystemErrors.Precondition_Failure,
         Msg     => "in function ExpressionFromPositionalArgumentAssociation");
      Result := Child_Node (Node);
      if Syntax_Node_Type (Node => Result) /= SPSymbols.expression
        and then Syntax_Node_Type (Node => Result) /= SPSymbols.annotation_expression then
         Result := Next_Sibling (Result);
      end if;
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Result) = SPSymbols.expression
           or else Syntax_Node_Type (Node => Result) = SPSymbols.annotation_expression,
         Sys_Err => SystemErrors.Postcondition_Failure,
         Msg     => "in function ExpressionFromPositionalArgumentAssociation");
      return Result;
   end ExpressionFromPositionalArgumentAssociation;

   -------------------------------------------------------

   function ExpressionFromNamedArgumentAssociation (Node : SyntaxNode) return SyntaxNode is
      Result : SyntaxNode;
   begin
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.named_argument_association,
         Sys_Err => SystemErrors.Precondition_Failure,
         Msg     => "in function ExpressionFromNamedArgumentAssociation");
      Result := Child_Node (Node);
      if Syntax_Node_Type (Node => Result) /= SPSymbols.simple_name
        and then Syntax_Node_Type (Node => Result) /= SPSymbols.annotation_simple_name then
         Result := Next_Sibling (Result);
      end if;
      -- skip over parameter name to get expression
      Result := Next_Sibling (Result);
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Result) = SPSymbols.expression
           or else Syntax_Node_Type (Node => Result) = SPSymbols.annotation_expression,
         Sys_Err => SystemErrors.Postcondition_Failure,
         Msg     => "in function ExpressionFromNamedArgumentAssociation");
      return Result;
   end ExpressionFromNamedArgumentAssociation;

   -------------------------------------------------------

   function LoopParameterSpecFromEndOfLoop (Node : SyntaxNode) return SyntaxNode is
      Result : SyntaxNode;
   begin
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.end_of_loop,
         Sys_Err => SystemErrors.Precondition_Failure,
         Msg     => "in function LoopParameterSpecFromEndOfLoop");

      Result := Child_Node (Parent_Node (Current_Node => Node));   -- simple_name or loop_statement_opt
      if Syntax_Node_Type (Node => Result) = SPSymbols.simple_name then
         Result := Next_Sibling (Result);       -- loop_statement_opt
      end if;
      Result := Child_Node (Child_Node (Result));

      SystemErrors.RT_Assert
        (C       => Result = NullNode
           or else Syntax_Node_Type (Node => Result) = SPSymbols.loop_parameter_specification
           or else Syntax_Node_Type (Node => Result) = SPSymbols.condition,
         Sys_Err => SystemErrors.Postcondition_Failure,
         Msg     => "in function LoopParameterSpecFromEndOfLoop");
      return Result;
   end LoopParameterSpecFromEndOfLoop;

   -------------------------------------------------------

   function IdentifierHasTildeSuffix (Node : SyntaxNode) return Boolean is
   begin
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.identifier,
         Sys_Err => SystemErrors.Precondition_Failure,
         Msg     => "in function IdentifierHasTildeSuffix");
      return Syntax_Node_Type (Node => Next_Sibling (Node)) = SPSymbols.tilde;
   end IdentifierHasTildeSuffix;

   -------------------------------------------------------

   function IdentifierHasPercentSuffix (Node : SyntaxNode) return Boolean is
   begin
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.identifier,
         Sys_Err => SystemErrors.Precondition_Failure,
         Msg     => "in function IdentifierHasPercentSuffix");
      return Syntax_Node_Type (Node => Next_Sibling (Node)) = SPSymbols.percent;
   end IdentifierHasPercentSuffix;

   -------------------------------------------------------

begin
   Table.CurrSyntaxNode := NullNode;
   --# accept Flow, 23, Table.Contents, "Init partial but effective";
   Table.Contents (NullNode) := -- Expect flow error on 1st write to array
     NodeValue'
     (NodeType  => SPSymbols.SPEND,
      Position  => LexTokenManager.Null_Token_Position,
      Parent    => NullNode,
      Next      => NullNode,
      Child     => NullNode,
      Token_Str => LexTokenManager.Null_String,
      Ref_Str   => LexTokenManager.Null_String,
      Ref_Sym   => Dictionary.NullSymbol);
   --# end accept;
   Table.FreeList := NullNode;
   Table.TopUsed  := NullNode;
   --# accept Flow, 602, Table, Table.Contents, "Init partial but effective";
end STree; --  Init. is partial but effective.  Expect 1 warning.
