-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Output 'fatal' error messages                                               --
--Note that the Process procedure does not return to point of call. Instead   --
--it raises an exception which is trapped by the main program, which causes   --
--the program to stop politely.                                               --
--                                                                            --
--                                                                            --
--------------------------------------------------------------------------------

with CommandLine;
with POGS_Exceptions;
with SPARK_IO;

package body FatalErrors is
   --# hide FatalErrors;
   -- hidden to mask the use of exceptions and the non-initialization of
   -- FatalErrors.State

   procedure Process (Error   : in Error_Type;
                      Message : in E_Strings.T) is
      T : E_Strings.T;
   begin
      case Error is
         when Could_Not_Open_Input_File =>
            T := E_Strings.Copy_String (Str => "Could not open expected input file " & "- unexpected file system error");

         when Could_Not_Create_Report_File =>
            E_Strings.Put_String
              (File  => SPARK_IO.Standard_Output,
               E_Str => E_Strings.Copy_String (Str => "Could not create report file "));
            E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                                E_Str => Message);
            T := E_Strings.Copy_String (Str => "Check you have write permission for directory");

         when Data_Structure_Inconsistency =>
            T := E_Strings.Copy_String (Str => "Internal file table inconsistent.");

         when VC_Data_Structure_Inconsistency =>
            T := E_Strings.Copy_String (Str => "Internal VC table inconsistent.");

         when Expected_Directory_Missing =>
            E_Strings.Put_String (File  => SPARK_IO.Standard_Output,
                                  E_Str => E_Strings.Copy_String (Str => "Directory "));
            E_Strings.Put_String (File  => SPARK_IO.Standard_Output,
                                  E_Str => Message);
            T := E_Strings.Copy_String (Str => " not found or not readable.");

         when File_Heap_Full =>
            T := E_Strings.Copy_String (Str => "Internal file table too small.");

         when VC_Heap_Full =>
            T := E_Strings.Copy_String (Str => "Internal VC table too small.");

         when Invalid_Command_Line =>
            T := E_Strings.Copy_String (Str => "Usage: ");
            E_Strings.Append_Examiner_String (E_Str1 => T,
                                              E_Str2 => CommandLine.Usage_String);

         when Problem_Reading_File =>
            T := E_Strings.Copy_String (Str => "Could not read from input file " & "- unexpected file system error");

         when Problem_Creating_Temp_File =>
            T := E_Strings.Copy_String (Str => "Could not create temporary file.");

         when Subprogram_Totals_Inconsistent =>
            T := E_Strings.Copy_String (Str => "Overall subprogram summary counts inconsistent.");

      end case;

      E_Strings.Put_Line (File  => SPARK_IO.Standard_Output,
                          E_Str => T);

      raise POGS_Exceptions.Usage_Error;
   end Process;

end FatalErrors;
