#include <stdio.h>
#include <gtk/gtk.h>

#include "specimen.h"
#include "patch.h"
#include "gui.h"

/* parameter codes */
typedef enum param_s {
     ATTACK,
     DECAY,
     SUSTAIN,
     RELEASE,
} param_t;

/* track which parameter's envelope we're setting */
static adsr_t cur_param = ADSR_VOLUME;

/* widgets */
static GtkWidget *window;
static GtkWidget *on_check;
static GtkWidget *a_spin;
static GtkWidget *d_spin;
static GtkWidget *s_spin;
static GtkWidget *r_spin;

/* adjustments */
static GtkAdjustment *a_adj;
static GtkAdjustment *d_adj;
static GtkAdjustment *s_adj;
static GtkAdjustment *r_adj;

static void sensitize(gboolean val)
{
     gtk_widget_set_sensitive(a_spin, val);
     gtk_widget_set_sensitive(d_spin, val);
     gtk_widget_set_sensitive(s_spin, val);
     gtk_widget_set_sensitive(r_spin, val);
}
static int update_interface()
{
     int cp;
     gdouble a, d, s, r;
     int on;

     if ((cp = get_current_patch()) < 0)
	  return -1;

     a = (gdouble) patch_attack_get(cp, cur_param);
     d = (gdouble) patch_decay_get(cp, cur_param);
     s = (gdouble) patch_sustain_get(cp, cur_param);
     r = (gdouble) patch_release_get(cp, cur_param);
     on = patch_envelope_get(cp, cur_param);

     if (!on) {
	  sensitize(FALSE);
     } else {
	  sensitize(TRUE);
     }

     gtk_spin_button_set_value(GTK_SPIN_BUTTON(a_spin), a);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(d_spin), d);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(s_spin), s);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(r_spin), r);
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(on_check), (on? TRUE: FALSE));

     return 0;
}

static void cb_close()
{
     gtk_widget_hide(window);
}

static void cb_set(GtkWidget *widget, param_t param)
{     
     int cp, err;
     float val;

     if ((cp = get_current_patch()) < 0)
	  return;

     switch (param) {
     case ATTACK:
	  val = (float) gtk_spin_button_get_value(GTK_SPIN_BUTTON(a_spin));

	  if ((err=patch_attack_set(cp, cur_param, val)) < 0)
	       errmsg("Failed to set attack for patch %d (%s)\n", cp, patch_strerror(err));
	  break;
     case DECAY:
	  val = (float) gtk_spin_button_get_value(GTK_SPIN_BUTTON(d_spin));

	  if ((err=patch_decay_set(cp, cur_param, val)) < 0)
	       errmsg("Failed to set decay for patch %d (%s)\n", cp, patch_strerror(err));
	  break;
     case SUSTAIN:
	  val = (float) gtk_spin_button_get_value(GTK_SPIN_BUTTON(s_spin));

	  if ((err=patch_sustain_set(cp, cur_param, val)) < 0)
	       errmsg("Failed to set sustain for patch %d (%s)\n", cp, patch_strerror(err));
	  break;
    case RELEASE:
	  val = (float) gtk_spin_button_get_value(GTK_SPIN_BUTTON(r_spin));

	  if ((err=patch_release_set(cp, cur_param, val)) < 0)
	       errmsg("Failed to set release for patch %d (%s)\n", cp, patch_strerror(err));
	  break;
     }

     return;
}

static void cb_param(GtkWidget *widget, adsr_t param)
{
	  cur_param = param;
	  update_interface();
}

static void cb_on(GtkWidget *widget, gpointer data)
{
     int val;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     val = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(on_check));
     patch_envelope_set(cp, cur_param, val);
     update_interface();
}

void adsr_settings_show()
{
     if (update_interface() == 0)
	  gtk_widget_show(window);
}

void adsr_settings_init(GtkWidget *parent)
{
     GtkWidget *vbox, *hbox;
     GtkWidget *button;
     GtkWidget *label;
     GtkWidget *sep;
     GtkWidget *menu;
     GtkWidget *opt;
     GtkWidget *item;

     /* main window */
     window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(window), "ADSR");
     gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
     gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(parent));
     gtk_window_set_modal(GTK_WINDOW(window), TRUE);
     g_signal_connect(GTK_WINDOW(window), "delete-event", G_CALLBACK(cb_close), NULL);

     /* vbox */
     vbox = gtk_vbox_new(FALSE, SPACING);
     gtk_container_add(GTK_CONTAINER(window), vbox);
     gtk_container_set_border_width(GTK_CONTAINER(window), SPACING);
     gtk_widget_show(vbox);

     /* hbox */
     hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
     gtk_widget_show(hbox);

     /* parameter button menu */
     menu = gtk_menu_new();

     /* volume */
     item = gtk_menu_item_new_with_label("Volume");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
     g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(cb_param),
		      (gpointer)ADSR_VOLUME);
     gtk_widget_show(item);

     /* panning */
     item = gtk_menu_item_new_with_label("Panning");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
     g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(cb_param),
		      (gpointer)ADSR_PANNING);
     gtk_widget_show(item);

     /* cutoff freq */
     item = gtk_menu_item_new_with_label("Cutoff");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
     g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(cb_param),
		      (gpointer)ADSR_CUTOFF);
     gtk_widget_show(item);

     /* resonance */
     item = gtk_menu_item_new_with_label("Resonance");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
     g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(cb_param),
		      (gpointer)ADSR_RESONANCE);
     gtk_widget_show(item);

     /* paramter optionmenu */
     opt = gtk_option_menu_new();
     gtk_box_pack_start(GTK_BOX(hbox), opt, FALSE, FALSE, 0);
     gtk_option_menu_set_menu(GTK_OPTION_MENU(opt), menu);
     gtk_widget_show(opt);

     /* on check box */
     on_check = gtk_check_button_new_with_label("On");
     gtk_box_pack_end(GTK_BOX(hbox), on_check, FALSE, FALSE, 0);
     g_signal_connect(G_OBJECT(on_check), "toggled", G_CALLBACK(cb_on), NULL);
     gtk_widget_show(on_check);

     /* hbox */
     hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
     gtk_widget_show(hbox);

     /**********/
     /* attack */
     /**********/
     label = gtk_label_new("Attack:");
     gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* adjustment */
     a_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, 0.0, 3.0, 0.1, 0.0, 0.0);
     g_signal_connect(G_OBJECT(a_adj), "value-changed", G_CALLBACK(cb_set), (gpointer)ATTACK);

     /* spin */
     a_spin = gtk_spin_button_new(a_adj, 0, 1);
     gtk_box_pack_end(GTK_BOX(hbox), a_spin, FALSE, FALSE, 0);
     gtk_widget_show(a_spin);

     /* hbox */
     hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
     gtk_widget_show(hbox);

     /*********/
     /* decay */
     /*********/
     label = gtk_label_new("Decay:");
     gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* adjustment */
     d_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, 0.0, 3.0, 0.1, 0.0, 0.0);
     g_signal_connect(G_OBJECT(d_adj), "value-changed", G_CALLBACK(cb_set), (gpointer)DECAY);

     /* spin */
     d_spin = gtk_spin_button_new(d_adj, 0, 1);
     gtk_box_pack_end(GTK_BOX(hbox), d_spin, FALSE, FALSE, 0);
     gtk_widget_show(d_spin);

     /* hbox */
     hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
     gtk_widget_show(hbox);

     /***********/
     /* sustain */
     /***********/
     label = gtk_label_new("Sustain:");
     gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* adjustment */
     s_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, 0.0, 1.0, 0.1, 0.0, 0.0);
     g_signal_connect(G_OBJECT(s_adj), "value-changed", G_CALLBACK(cb_set), (gpointer)SUSTAIN);

     /* spin */
     s_spin = gtk_spin_button_new(s_adj, 0, 1);
     gtk_box_pack_end(GTK_BOX(hbox), s_spin, FALSE, FALSE, 0);
     gtk_widget_show(s_spin);

     /* hbox */
     hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
     gtk_widget_show(hbox);

     /***********/
     /* release */
     /***********/
     label = gtk_label_new("Release:");
     gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* adjustment */
     r_adj = (GtkAdjustment *)gtk_adjustment_new(0.0, 0.0, 3.0, 0.1, 0.0, 0.0);
     g_signal_connect(G_OBJECT(r_adj), "value-changed", G_CALLBACK(cb_set), (gpointer)RELEASE);

     /* spin */
     r_spin = gtk_spin_button_new(r_adj, 0, 1);
     gtk_box_pack_end(GTK_BOX(hbox), r_spin, FALSE, FALSE, 0);
     gtk_widget_show(r_spin);

     /* hbox */
     hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
     gtk_widget_show(hbox);

     /**************/
     sep = gtk_hseparator_new();
     gtk_box_pack_start(GTK_BOX(vbox), sep, FALSE, FALSE, 0);
     gtk_widget_show(sep);
     
     /* hbox */
     hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
     gtk_widget_show(hbox);

     /* close button */
     button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
     g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(cb_close), NULL);
     gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
     gtk_widget_show(button);
}
