/* iparray.h - header file for the IPv4 sparse array module
**
** This data structure represents an array of bytes indexed by IPv4
** addresses - approximately equivalent to char[256][256][256][256].
** That would take up 4 GB, which is too much, but the sparse array
** only allocates the parts of the array which are actually in use.
** In practice that works out to between 150 and 300 bytes per IP
** address.  Access speed is still very fast.  Also, it can represent
** subnets with a single entry, without having to allocate all
** the actual addresses.
**
**
** Copyright  2004 by Jef Poskanzer <jef@acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/

#ifndef _IPARRAY_H_
#define _IPARRAY_H_


/* This represents an IPv4 address (if w is 32) or network. */
typedef struct {
    unsigned char a, b, c, d;
    unsigned char w;
    } octets;


/* Opaque type. */
typedef void* iparray;


/* Creates a new iparray.  Returns 0 on failure. */
extern iparray iparray_new( void );
 
/* Clears an iparray. */
extern void iparray_clear( iparray ipa );
  
/* Gets rid of an iparray. */
extern void iparray_delete( iparray ipa );

/* Terminates the iparray package.  Call after you have already
** iparray_deleted all your iparrays.
*/
extern void iparray_fini( void );
   
/* Parses a string IPv4 address into octets.  This should be a "dotted quad"
** address, e.g. "127.0.0.1".  It can optionally be followed by a subnet
** indicator such as "/24".  Subnets not on byte boundaries *are* supported.
** Returns 1 on success, 0 on failure.
*/
extern int iparray_parse_octets( char* str, octets* ip );

/* Increments the value for a IP address in an iparray.  Returns 1 on
** success, 0 on failure.  If the address is a network (w != 32) all
** values within that net get incremented.
*/
extern int iparray_incr( iparray ipa, const octets ip );

/* Fetches the value for a IP address in an iparray.  If the address
** is a network (w != 32), you'll get the sum of all the values within
** that net.
*/
extern int iparray_get( const iparray ipa, const octets ip );

#endif /* _IPARRAY_H_ */
