/****************************************************************************
*      SPLAT: An RF Signal Propagation Loss and Terrain Analysis Tool       *
*			  Last update: 21-Dec-2006			    *
*****************************************************************************
*	     Project started in 1997 by John A. Magliacane, KD2BD 	    *
*****************************************************************************
*									    *
*     Extensively modified by J. D. McDonald in Jan. 2004 to include        *
*    the Longley-Rice propagation model using C++ code from NTIA/ITS.	    *
*									    *
*              See: http://flattop.its.bldrdoc.gov/itm.html                 *
*									    *
*****************************************************************************
*                                                                           *
* This program is free software; you can redistribute it and/or modify it   *
* under the terms of the GNU General Public License as published by the     *
* Free Software Foundation; either version 2 of the License or any later    *
* version.								    *
*									    *
* This program is distributed in the hope that it will useful, but WITHOUT  *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
* for more details.							    *
*									    *
*****************************************************************************
 g++ -Wall -O3 -s -lm -lbz2 -fomit-frame-pointer itm.cpp splat.cpp -o splat 
*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <bzlib.h>
#include <unistd.h>
#include "fontdata.h"
#include "smallfont.h"

#define GAMMA 2.5
#define MAXSLOTS 9
#define BZBUFFER 65536

#if MAXSLOTS==4
#define ARRAYSIZE 4950
#endif

#if MAXSLOTS==9
#define ARRAYSIZE 10870
#endif

#if MAXSLOTS==16
#define ARRAYSIZE 19240
#endif

#if MAXSLOTS==25
#define ARRAYSIZE 30025
#endif

char 	string[255], sdf_path[255], opened=0, *splat_version={"1.2.0"};

double	TWOPI=6.283185307179586, HALFPI=1.570796326794896,
	PI=3.141592653589793, deg2rad=1.74532925199e-02,
	EARTHRADIUS=20902230.97, METERS_PER_MILE=1609.344,
	METERS_PER_FOOT=0.3048, KM_PER_MILE=1.609344, earthradius,
	max_range=0.0;

int	min_north=90, max_north=-90, min_west=360, max_west=-1,
	max_elevation=-32768, min_elevation=32768, bzerror, maxdB=230;

unsigned char got_elevation_pattern=0, got_azimuth_pattern=0, metric=0;

struct site {	double lat;
		double lon;
		float alt;
		char name[50];
	    } 	site;

struct path {	double lat[ARRAYSIZE];
		double lon[ARRAYSIZE];
		double elevation[ARRAYSIZE];
		double distance[ARRAYSIZE];
		int length;
	    }	path;

struct dem {	int min_north;
		int max_north;
		int min_west;
		int max_west;
		int max_el;
		int min_el;
		short data[1200][1200];
		unsigned char mask[1200][1200];
           }	dem[MAXSLOTS];

struct LR {	double eps_dielect; 
		double sgm_conductivity; 
		double eno_ns_surfref;
		double frq_mhz; 
		double conf; 
		double rel;
		int radio_climate;  
		int pol;
		float antenna_pattern[361][1001];
          }	LR;

double elev_l[ARRAYSIZE+10];

void point_to_point(double elev[], double tht_m, double rht_m,
	  double eps_dielect, double sgm_conductivity, double eno_ns_surfref,
	  double frq_mhz, int radio_climate, int pol, double conf,
	  double rel, double &dbloss, char *strmode, int &errnum);

double arccos(double x, double y)
{
	/* This function implements the arc cosine function,
	   returning a value between 0 and TWOPI. */

	double result=0.0;

	if (y>0.0)
		result=acos(x/y);

	if (y<0.0)
		result=PI+acos(x/y);

	return result;
}

int ReduceAngle(double angle)
{
	/* This function normalizes the argument to
	   an integer angle between 0 and 180 degrees */

	double temp;

	temp=acos(cos(angle*deg2rad));

	return (int)rint(temp/deg2rad);
}

char *dec2dms(double decimal)
{
	/* Converts decimal degrees to degrees, minutes, seconds,
	   (DMS) and returns the result as a character string. */

	char	sign;
	int	degrees, minutes, seconds;
	double	a, b, c, d;

	if (decimal<0.0)
	{
		decimal=-decimal;
		sign=-1;
	}

	else
		sign=1;

	a=floor(decimal);
	b=60.0*(decimal-a);
	c=floor(b);
	d=60.0*(b-c);

	degrees=(int)a;
	minutes=(int)c;
	seconds=(int)d;

	if (seconds<0)
		seconds=0;

	if (seconds>59)
		seconds=59;

	string[0]=0;
	sprintf(string,"%d%c %d\' %d\"", degrees*sign, 176, minutes, seconds);
	return (string);
}

int OrMask(double lat, double lon, int value)
{
	/* Lines, text, markings, and coverage areas are stored in a
	   mask that is combined with topology data when topographic
	   maps are generated by SPLAT!.  This function sets bits in
	   the mask based on the latitude and longitude of the area
	   pointed to. */

	int	x, y, indx, minlat, minlon;
	char	found;

	minlat=(int)floor(lat);
	minlon=(int)floor(lon);

	for (indx=0, found=0; indx<MAXSLOTS && found==0;)
		if (minlat==dem[indx].min_north && minlon==dem[indx].min_west)
			found=1;
		else
			indx++;

	if (found)
	{
		x=(int)(1199.0*(lat-floor(lat)));
		y=(int)(1199.0*(lon-floor(lon)));

		dem[indx].mask[x][y]|=value;

		return (dem[indx].mask[x][y]);
	}

	else
		return -1;
}

int GetMask(double lat, double lon)
{
	/* This function returns the mask bits based on the latitude
	   and longitude given. */

	return (OrMask(lat,lon,0));
}

double GetElevation(struct site location)
{
	/* This function returns the elevation (in feet) of any location
	   represented by the digital elevation model data in memory.
	   Function returns -5000.0 for locations not found in memory. */

	char	found;
	int	x, y, indx, minlat, minlon;
	double	elevation;

	elevation=-5000.0;

	minlat=(int)floor(location.lat);
	minlon=(int)floor(location.lon);

	x=(int)(1199.0*(location.lat-floor(location.lat)));
	y=(int)(1199.0*(location.lon-floor(location.lon)));

	for (indx=0, found=0; indx<MAXSLOTS && found==0; indx++)
	{
		if (minlat==dem[indx].min_north && minlon==dem[indx].min_west)
		{

			elevation=3.28084*dem[indx].data[x][y];
			found=1;
		}
	}
	
	return elevation;
}

int AddElevation(double lat, double lon, double height)
{
	/* This function adds a user-defined terrain feature
	   (in meters AGL) to the digital elevation model data
	   in memory.  Does nothing and returns 0 for locations
	   not found in memory. */

	char	found;
	int	x, y, indx, minlat, minlon;

	minlat=(int)floor(lat);
	minlon=(int)floor(lon);

	x=(int)(1199.0*(lat-floor(lat)));
	y=(int)(1199.0*(lon-floor(lon)));

	for (indx=0, found=0; indx<MAXSLOTS && found==0; indx++)
	{
		if (minlat==dem[indx].min_north && minlon==dem[indx].min_west)
		{

			dem[indx].data[x][y]+=(short)rint(height);
			found=1;
		}
	}
	
	return found;
}

double Distance(struct site site1, struct site site2)
{
	/* This function returns the great circle distance
	   in miles between any two site locations. */

	double	lat1, lon1, lat2, lon2, distance;

	lat1=site1.lat*deg2rad;
	lon1=site1.lon*deg2rad;
	lat2=site2.lat*deg2rad;
	lon2=site2.lon*deg2rad;

	distance=3959.0*acos(sin(lat1)*sin(lat2)+cos(lat1)*cos(lat2)*cos((lon1)-(lon2)));

	return distance;
}

double Azimuth(struct site source, struct site destination)
{
	/* This function returns the azimuth (in degrees) to the
	   destination as seen from the location of the source. */

	double	dest_lat, dest_lon, src_lat, src_lon,
		beta, azimuth, diff, num, den, fraction;

	dest_lat=destination.lat*deg2rad;
	dest_lon=destination.lon*deg2rad;

	src_lat=source.lat*deg2rad;
	src_lon=source.lon*deg2rad;
		
	/* Calculate Surface Distance */

	beta=acos(sin(src_lat)*sin(dest_lat)+cos(src_lat)*cos(dest_lat)*cos(src_lon-dest_lon));

	/* Calculate Azimuth */

	num=sin(dest_lat)-(sin(src_lat)*cos(beta));
	den=cos(src_lat)*sin(beta);
	fraction=num/den;

	/* Trap potential problems in acos() due to rounding */

	if (fraction>=1.0)
		fraction=1.0;

	if (fraction<=-1.0)
		fraction=-1.0;

	/* Calculate azimuth */

	azimuth=acos(fraction);

	/* Reference it to True North */

	diff=dest_lon-src_lon;

	if (diff<=-PI)
		diff+=TWOPI;

	if (diff>=PI)
		diff-=TWOPI;

	if (diff>0.0)
		azimuth=TWOPI-azimuth;

	return (azimuth/deg2rad);		
}

double ElevationAngle(struct site source, struct site destination)
{
	/* This function returns the angle of elevation (in degrees)
	   of the destination as seen from the source location.
	   A positive result represents an angle of elevation (uptilt),
	   while a negative result represents an angle of depression
	   (downtilt), as referenced to a normal to the center of
	   the earth. */
	   
	register double a, b, dx;

	a=GetElevation(destination)+destination.alt+earthradius;
	b=GetElevation(source)+source.alt+earthradius;

 	dx=5280.0*Distance(source,destination);

	/* Apply the Law of Cosines */

	return ((180.0*(acos(((b*b)+(dx*dx)-(a*a))/(2.0*b*dx)))/PI)-90.0);
}

void ReadPath(struct site source, struct site destination)
{
	/* This function generates a sequence of latitude and
	   longitude positions between source and destination
	   locations along a great circle path, and stores
	   elevation and distance information for points
	   along that path in the "path" structure. */

	int	c;
	double	azimuth, distance, lat1, lon1, beta, den, num,
		lat2, lon2, total_distance, x, y, path_length,
		increment;
	struct	site tempsite;

	lat1=source.lat*deg2rad;
	lon1=source.lon*deg2rad;

	lat2=destination.lat*deg2rad;
	lon2=destination.lon*deg2rad;

	azimuth=Azimuth(source,destination)*deg2rad;

	total_distance=Distance(source,destination);

	x=68755.0*acos(cos(lon1-lon2));		/* 1200 samples per degree */
	y=68755.0*acos(cos(lat1-lat2));		/* 68755 samples per radian */

	path_length=sqrt((x*x)+(y*y));		/* Total number of samples */

	increment=total_distance/path_length;	/* Miles per sample */

	for (distance=0, c=0; distance<=total_distance; distance+=increment)
	{
		beta=distance/3959.0;
		lat2=asin(sin(lat1)*cos(beta)+cos(azimuth)*sin(beta)*cos(lat1));
		num=cos(beta)-(sin(lat1)*sin(lat2));
		den=cos(lat1)*cos(lat2);

		if (azimuth==0.0 && (beta>HALFPI-lat1))
			lon2=lon1+PI;

		else if (azimuth==HALFPI && (beta>HALFPI+lat1))
			lon2=lon1+PI;

		else if (fabs(num/den)>1.0)
			lon2=lon1;

		else
		{
			if ((PI-azimuth)>=0.0)
				lon2=lon1-arccos(num,den);
			else
				lon2=lon1+arccos(num,den);
		}
	
		while (lon2<0.0)
			lon2+=TWOPI;

		while (lon2>TWOPI)
			lon2-=TWOPI;
 
		lat2=lat2/deg2rad;
		lon2=lon2/deg2rad;

		if (c<ARRAYSIZE)
		{
			path.lat[c]=lat2;
			path.lon[c]=lon2;
			tempsite.lat=lat2;
			tempsite.lon=lon2;
			path.elevation[c]=GetElevation(tempsite);
			path.distance[c]=distance;
			c++;
		}
	}

	/* Make sure exact destination point is recorded at path.length-1 */

	if (c<ARRAYSIZE)
	{
		path.lat[c]=destination.lat;
		path.lon[c]=destination.lon;
		path.elevation[c]=GetElevation(destination);
		path.distance[c]=total_distance;
		c++;
	}

	if (c<ARRAYSIZE)
		path.length=c;
	else
		path.length=ARRAYSIZE-1;
}

double ElevationAngle2(struct site source, struct site destination, double er)
{
	/* This function returns the angle of elevation (in degrees)
	   of the destination as seen from the source location, UNLESS
	   the path between the sites is obstructed, in which case, the
	   elevation angle to the first obstruction is returned instead.
	   "er" represents the earth radius. */

	int	x;
	char	block=0;
	double	source_alt, destination_alt, cos_xmtr_angle,
		cos_test_angle, test_alt, elevation, distance,
		source_alt2, first_obstruction_angle=0.0;
	struct	path temp;

	temp=path;

	ReadPath(source,destination);

	distance=5280.0*Distance(source,destination);
	source_alt=er+source.alt+GetElevation(source);
	destination_alt=er+destination.alt+GetElevation(destination);
	source_alt2=source_alt*source_alt;

	/* Calculate the cosine of the elevation angle of the
	   destination (receiver) as seen by the source (transmitter). */

	cos_xmtr_angle=((source_alt2)+(distance*distance)-(destination_alt*destination_alt))/(2.0*source_alt*distance);

	/* Test all points in between source and destination locations to
	   see if the angle to a topographic feature generates a higher
	   elevation angle than that produced by the destination.  Begin
	   at the source since we're interested in identifying the FIRST
	   obstruction along the path between source and destination. */
 
	for (x=2, block=0; x<path.length && block==0; x++)
	{
		distance=5280.0*path.distance[x];

		test_alt=earthradius+path.elevation[x];

		cos_test_angle=((source_alt2)+(distance*distance)-(test_alt*test_alt))/(2.0*source_alt*distance);

		/* Compare these two angles to determine if
		   an obstruction exists.  Since we're comparing
		   the cosines of these angles rather than
		   the angles themselves, the sense of the
		   following "if" statement is reversed from
		   what it would be if the angles themselves
		   were compared. */

		if (cos_xmtr_angle>cos_test_angle)
		{
			block=1;
			first_obstruction_angle=((acos(cos_test_angle))/deg2rad)-90.0;
		}
	}

	if (block)
		elevation=first_obstruction_angle;

	else
		elevation=((acos(cos_xmtr_angle))/deg2rad)-90.0;

	path=temp;

	return elevation;
}

double AverageTerrain(struct site source, double azimuthx, double start_distance, double end_distance)
{
	/* This function returns the average terrain calculated in
	   the direction of "azimuth" (degrees) between "start_distance"
	   and "end_distance" (miles) from the source location.  If
	   the terrain is all water (non-critical error), -5000.0 is
	   returned.  If not enough SDF data has been loaded into
	   memory to complete the survey (critical error), then
	   -9999.0 is returned. */
 
	int	c, samples, endpoint;
	double	beta, lat1, lon1, lat2, lon2, num, den, azimuth, terrain=0.0;
	struct	site destination;

	lat1=source.lat*deg2rad;
	lon1=source.lon*deg2rad;

	/* Generate a path of elevations between the source
	   location and the remote location provided. */

	beta=end_distance/3959.0;

	azimuth=deg2rad*azimuthx;

	lat2=asin(sin(lat1)*cos(beta)+cos(azimuth)*sin(beta)*cos(lat1));
	num=cos(beta)-(sin(lat1)*sin(lat2));
	den=cos(lat1)*cos(lat2);

	if (azimuth==0.0 && (beta>HALFPI-lat1))
		lon2=lon1+PI;

	else if (azimuth==HALFPI && (beta>HALFPI+lat1))
		lon2=lon1+PI;

	else if (fabs(num/den)>1.0)
		lon2=lon1;

	else
	{
		if ((PI-azimuth)>=0.0)
			lon2=lon1-arccos(num,den);
		else
			lon2=lon1+arccos(num,den);
	}
	
	while (lon2<0.0)
		lon2+=TWOPI;

	while (lon2>TWOPI)
		lon2-=TWOPI;
 
	lat2=lat2/deg2rad;
	lon2=lon2/deg2rad;

	destination.lat=lat2;
	destination.lon=lon2;

	/* If SDF data is missing for the endpoint of
	   the radial, then the average terrain cannot
	   be accurately calculated.  Return -9999.0 */

	if (GetElevation(destination)<-4999.0)
		return (-9999.0);
	else
	{
		ReadPath(source,destination);

		endpoint=path.length;

		/* Shrink the length of the radial if the
		   outermost portion is not over U.S. land. */

		for (c=endpoint-1; c>=0 && path.elevation[c]==0.0; c--);

		endpoint=c+1;

		for (c=0, samples=0; c<endpoint; c++)
		{
			if (path.distance[c]>=start_distance)
			{
				terrain+=path.elevation[c];
				samples++;
			}
		}

		if (samples==0)
			terrain=-5000.0;  /* No land */
		else
			terrain=(terrain/(double)samples);

		return terrain;
	}
}

double haat(struct site antenna)
{
	/* This function returns the antenna's Height Above Average
	   Terrain (HAAT) based on FCC Part 73.313(d).  If a critical
	   error occurs, such as a lack of SDF data to complete the
	   survey, -5000.0 is returned. */

	int	azi, c;
	char	error=0;
	double	terrain, avg_terrain, haat, sum=0.0;

	/* Calculate the average terrain between 2 and 10 miles
	   from the antenna site at azimuths of 0, 45, 90, 135,
	   180, 225, 270, and 315 degrees. */

	for (c=0, azi=0; azi<=315 && error==0; azi+=45)
	{
		terrain=AverageTerrain(antenna, (double)azi, 2.0, 10.0);

		if (terrain<-9998.0)  /* SDF data is missing */
			error=1;

		if (terrain>-4999.0)  /* It's land, not water */
		{
			sum+=terrain;  /* Sum of averages */
			c++;
		}
	}

	if (error)
		return -5000.0;
	else
	{
		avg_terrain=(sum/(double)c);
		haat=(antenna.alt+GetElevation(antenna))-avg_terrain;
		return haat;
	}
}

float LonDiff(float lon1, float lon2)
{
	/* This function returns the short path longitudinal
	   difference between longitude1 and longitude2 
	   as an angle between -180.0 and +180.0 degrees.
	   If lon1 is west of lon2, the result is positive.
	   If lon1 is east of lon2, the result is negative. */

	float diff;

	diff=lon1-lon2;

	if (diff<=-180.0)
		diff+=360.0;

	if (diff>=180.0)
		diff-=360.0;

	return diff;
}

void PlaceMarker(struct site location)
{
	/* This function places text and marker data in the mask array
	   for illustration on topographic maps generated by SPLAT!.
	   By default, SPLAT! centers text information BELOW the marker,
	   but may move it above, to the left, or to the right of the
	   marker depending on how much room is available on the map,
	   or depending on whether the area is already occupied by
	   another marker or label.  If no room or clear space is
	   available on the map to place the marker and its associated
	   text, then the marker and text are not written to the map. */
 
	int	a, b, c, byte;
	char	ok2print, occupied;
	double	x, y, lat, lon, textx=0.0, texty=0.0, xmin, xmax,
		ymin, ymax, p1, p3, p6, p8, p12, p16, p24, label_length;

	xmin=min_north;
	xmax=max_north;
	ymin=min_west;
	ymax=max_west;
	lat=location.lat;
	lon=location.lon;

	if (lat<xmax && lat>xmin && (LonDiff(lon,ymax)<0.0) && (LonDiff(lon,ymin)>0.0))
	{
		p1=1.0/1200.0;
		p3=3.0/1200.0;
		p6=6.0/1200.0;
		p8=8.0/1200.0;
		p12=12.0/1200.0;
		p16=16.0/1200.0;
		p24=24.0/1200.0;
		ok2print=0;
		occupied=0;

		/* Is Marker Position Clear Of Text Or Other Markers? */

		for (x=lat-p3; (x<=xmax && x>=xmin && x<=lat+p3); x+=p1)
			for (y=lon-p3; (LonDiff(y,ymax)<=0.0) && (LonDiff(y,ymin)>=0.0) && (LonDiff(y,lon+p3)<=0.0); y+=p1)
				occupied|=(GetMask(x,y)&2);

		if (occupied==0)
		{
			/* Determine Where Text Can Be Positioned */

			/* label_length=length in pixels.
			   Each character is 8 pixels wide. */

			label_length=p1*(double)(strlen(location.name)<<3);

			if ((LonDiff(lon+label_length,ymax)<=0.0) && (LonDiff(lon-label_length,ymin)>=0.0))
			{
				/* Default: Centered Text */

				texty=lon+label_length/2.0;

				if ((lat-p8)>=p16)
				{
					/* Position Text Below The Marker */

					textx=lat-p8;

					x=textx;
					y=texty;
	
					/* Is This Position Clear Of
					   Text Or Other Markers? */

					for (a=0, occupied=0; a<16; a++)
					{
						for (b=0; b<(int)strlen(location.name); b++)
							for (c=0; c<8; c++, y-=p1)
								occupied|=(GetMask(x,y)&2);
						x-=p1;
						y=texty;
					}

					x=textx;
					y=texty;

					if (occupied==0)
						ok2print=1;
				}

				else
				{
					/* Position Text Above The Marker */

					textx=lat+p24;

					x=textx;
					y=texty;
	
					/* Is This Position Clear Of
					   Text Or Other Markers? */

					for (a=0, occupied=0; a<16; a++)
					{
						for (b=0; b<(int)strlen(location.name); b++)
							for (c=0; c<8; c++, y-=p1)
								occupied|=(GetMask(x,y)&2);
						x-=p1;
						y=texty;
					}

					x=textx;
					y=texty;

					if (occupied==0)
						ok2print=1;
				}
			}

			if (ok2print==0)
			{
				if (LonDiff(lon-label_length,ymin)>=0.0)
				{
					/* Position Text To The
					   Right Of The Marker */

					textx=lat+p6;
					texty=lon-p12;

					x=textx;
					y=texty;
	
					/* Is This Position Clear Of
					   Text Or Other Markers? */

					for (a=0, occupied=0; a<16; a++)
					{
						for (b=0; b<(int)strlen(location.name); b++)
							for (c=0; c<8; c++, y-=p1)
								occupied|=(GetMask(x,y)&2);
						x-=p1;
						y=texty;
					}

					x=textx;
					y=texty;

					if (occupied==0)
						ok2print=1;
				}

				else
				{
					/* Position Text To The
					   Left Of The Marker */

					textx=lat+p6;
					texty=lon+p8+(label_length);

					x=textx;
					y=texty;
	
					/* Is This Position Clear Of
					   Text Or Other Markers? */

					for (a=0, occupied=0; a<16; a++)
					{
						for (b=0; b<(int)strlen(location.name); b++)
							for (c=0; c<8; c++, y-=p1)
								occupied|=(GetMask(x,y)&2);
						x-=p1;
						y=texty;
					}

					x=textx;
					y=texty;

					if (occupied==0)
						ok2print=1;
				}
			}

			/* textx and texty contain the latitude and longitude
			   coordinates that describe the placement of the text
			   on the map. */
	
			if (ok2print)
			{
				/* Draw Text */

				x=textx;
				y=texty;

				for (a=0; a<16 && ok2print; a++)
				{
					for (b=0; b<(int)strlen(location.name); b++)
					{
						byte=fontdata[16*(location.name[b])+a];

						for (c=128; c>0; c=c>>1, y-=p1)
							if (byte&c)
								OrMask(x,y,2);
					}

					x-=p1;
					y=texty;
				}
	
				/* Draw Square Marker Centered
				   On Location Specified */
	
				for (x=lat-p3; (x<=xmax && x>=xmin && x<=lat+p3); x+=p1)
					for (y=lon-p3; (LonDiff(y,ymax)<=0.0) && (LonDiff(y,ymin)>=0.0) && (LonDiff(y,lon+p3)<=0.0); y+=p1)
						OrMask(x,y,2);
			}
		}
	}
}

double ReadBearing(char *input)
{
	/* This function takes numeric input in the form of a character
	   string, and returns an equivalent bearing in degrees as a
	   decimal number (double).  The input may either be expressed
	   in decimal format (40.139722) or degree, minute, second
	   format (40 08 23).  This function also safely handles
	   extra spaces found either leading, trailing, or
	   embedded within the numbers expressed in the
	   input string.  Decimal seconds are permitted. */
 
	double	seconds, bearing=0.0;
	char	string[20];
	int	a, b, length, degrees, minutes;

	/* Copy "input" to "string", and ignore any extra
	   spaces that might be present in the process. */

	string[0]=0;
	length=strlen(input);

	for (a=0, b=0; a<length && a<18; a++)
	{
		if ((input[a]!=32 && input[a]!='\n') || (input[a]==32 && input[a+1]!=32 && input[a+1]!='\n' && b!=0))
		{
			string[b]=input[a];
			b++;
		}	 
	}

	string[b]=0;

	/* Count number of spaces in the clean string. */

	length=strlen(string);

	for (a=0, b=0; a<length; a++)
		if (string[a]==32)
			b++;

	if (b==0)  /* Decimal Format (40.139722) */
		sscanf(string,"%lf",&bearing);

	if (b==2)  /* Degree, Minute, Second Format (40 08 23) */
	{
		sscanf(string,"%d %d %lf",&degrees, &minutes, &seconds);

		bearing=(double)abs(degrees)+((double)abs(minutes)/60)+(fabs(seconds)/3600);

		if ((degrees<0) || (minutes<0) || (seconds<0.0))
			bearing=-bearing;
	}

	/* Anything else returns a 0.0 */

	if (bearing>360.0 || bearing<-90.0)
		bearing=0.0;

	return bearing;
}

struct site LoadQTH(char *filename)
{
	/* This function reads SPLAT! .qth (site location) files.
	   The latitude and longitude may be expressed either in
	   decimal degrees, or in degree, minute, second format.
	   Antenna height is assumed to be expressed in feet above
	   ground level (AGL), unless followed by the letter 'M',
	   or 'm', or by the word "meters" or "Meters", in which
	   case meters is assumed, and is handled accordingly. */

	int	x;
	char	string[50], qthfile[255];
	struct	site tempsite;
	FILE	*fd=NULL;

	for (x=0; filename[x]!='.' && filename[x]!=0 && x<250; x++)
		qthfile[x]=filename[x];

	qthfile[x]='.';
	qthfile[x+1]='q';
	qthfile[x+2]='t';
	qthfile[x+3]='h';
	qthfile[x+4]=0;

	tempsite.lat=91.0;
	tempsite.lon=361.0;
	tempsite.alt=0.0;
	tempsite.name[0]=0;

	fd=fopen(qthfile,"r");

	if (fd!=NULL)
	{
		/* Site Name */
		fgets(string,49,fd);

		/* Strip <CR> and/or <LF> from end of site name */

		for (x=0; string[x]!=13 && string[x]!=10 && string[x]!=0; tempsite.name[x]=string[x], x++);

		tempsite.name[x]=0;

		/* Site Latitude */
		fgets(string,49,fd);
		tempsite.lat=ReadBearing(string);

		/* Site Longitude */
		fgets(string,49,fd);
		tempsite.lon=ReadBearing(string);

		/* Antenna Height */
		fgets(string,49,fd);
		fclose(fd);

		/* Remove <CR> and/or <LF> from antenna height string */
		for (x=0; string[x]!=13 && string[x]!=10 && string[x]!=0; x++);

		string[x]=0;

		/* Antenna height may either be in feet or meters.
		   If the letter 'M' or 'm' is discovered in
		   the string, then this is an indication that
		   the value given is expressed in meters, and
		   must be converted to feet before exiting. */

		for (x=0; string[x]!='M' && string[x]!='m' && string[x]!=0 && x<48; x++);
		if (string[x]=='M' || string[x]=='m')
		{
			string[x]=0;
			sscanf(string,"%f",&tempsite.alt);
			tempsite.alt*=3.28084;
		}

		else
		{
			string[x]=0;
			sscanf(string,"%f",&tempsite.alt);
		}
	}

	return tempsite;
}

void LoadPAT(char *filename)
{
	/* This function reads and processes antenna pattern (.az
	   and .el) files that correspond in name to previously
	   loaded SPLAT! .lrp files.  */

	int	a, b, w, x, y, z, last_index, next_index, span;
	char	string[255], azfile[255], elfile[255], *pointer=NULL;
	float	az, xx, elevation, amplitude, rotation, valid1, valid2,
		delta, azimuth[361], azimuth_pattern[361], el_pattern[10001],
		elevation_pattern[361][1001], slant_angle[361], tilt,
		mechanical_tilt, tilt_azimuth, tilt_increment, sum;
	FILE	*fd=NULL;
	unsigned char read_count[10001];

	for (x=0; filename[x]!='.' && filename[x]!=0 && x<250; x++)
	{
		azfile[x]=filename[x];
		elfile[x]=filename[x];
	}

	azfile[x]='.';
	azfile[x+1]='a';
	azfile[x+2]='z';
	azfile[x+3]=0;

	elfile[x]='.';
	elfile[x+1]='e';
	elfile[x+2]='l';
	elfile[x+3]=0;

	rotation=0.0;

	/* Load .az antenna pattern file */

	fd=fopen(azfile,"r");

	if (fd!=NULL)
	{
		/* Clear azimuth pattern array */

		for (x=0; x<=360; x++)
		{
			azimuth[x]=0.0;
			read_count[x]=0;
		}


		/* Read azimuth pattern rotation
		   in degrees measured clockwise
		   from true North. */

		fgets(string,254,fd);
		pointer=strchr(string,';');

		if (pointer!=NULL)
			*pointer=0;

		sscanf(string,"%f",&rotation);


		/* Read azimuth (degrees) and corresponding
		   normalized field radiation pattern amplitude
		   (0.0 to 1.0) until EOF is reached. */

		fgets(string,254,fd);
		pointer=strchr(string,';');

		if (pointer!=NULL)
			*pointer=0;

		sscanf(string,"%f %f",&az, &amplitude);

		do
		{
			x=(int)rintf(az);

			if (x>=0 && x<=360 && fd!=NULL)
			{
				azimuth[x]+=amplitude;
				read_count[x]++;
			}

			fgets(string,254,fd);
			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			sscanf(string,"%f %f",&az, &amplitude);

		} while (feof(fd)==0);

		fclose(fd);


		/* Handle 0=360 degree ambiguity */

		if ((read_count[0]==0) && (read_count[360]!=0))
		{
			read_count[0]=read_count[360];
			azimuth[0]=azimuth[360];
		}

		if ((read_count[0]!=0) && (read_count[360]==0))
		{
			read_count[360]=read_count[0];
			azimuth[360]=azimuth[0];
		}

		/* Average pattern values in case more than
		    one was read for each degree of azimuth. */

		for (x=0; x<=360; x++)
		{
			if (read_count[x]>1)
				azimuth[x]/=(float)read_count[x];
		}

		/* Interpolate missing azimuths
		   to completely fill the array */

		last_index=-1;
		next_index=-1;

		for (x=0; x<=360; x++)
		{
			if (read_count[x]!=0)
			{
				if (last_index==-1)
					last_index=x;
				else
					next_index=x;
			}

			if (last_index!=-1 && next_index!=-1)
			{
				valid1=azimuth[last_index];
				valid2=azimuth[next_index];

				span=next_index-last_index;
				delta=(valid2-valid1)/(float)span;

				for (y=last_index+1; y<next_index; y++)
					azimuth[y]=azimuth[y-1]+delta;

				last_index=y;
				next_index=-1;
			}
		}

		/* Perform azimuth pattern rotation
		   and load azimuth_pattern[361] with
		   azimuth pattern data in its final form. */

		for (x=0; x<360; x++)
		{
			y=x+(int)rintf(rotation);

			if (y>=360)
				y-=360;

			azimuth_pattern[y]=azimuth[x];
		}

		azimuth_pattern[360]=azimuth_pattern[0];

		got_azimuth_pattern=255;
	}

	/* Read and process .el file */

	fd=fopen(elfile,"r");

	if (fd!=NULL)
	{
		for (x=0; x<=10000; x++)
		{
			el_pattern[x]=0.0;
			read_count[x]=0;
		}

		/* Read mechanical tilt (degrees) and
		   tilt azimuth in degrees measured
		   clockwise from true North. */  

		fgets(string,254,fd);
		pointer=strchr(string,';');

		if (pointer!=NULL)
			*pointer=0;

		sscanf(string,"%f %f",&mechanical_tilt, &tilt_azimuth);

		/* Read elevation (degrees) and corresponding
		   normalized field radiation pattern amplitude
		   (0.0 to 1.0) until EOF is reached. */

		fgets(string,254,fd);
		pointer=strchr(string,';');

		if (pointer!=NULL)
			*pointer=0;

		sscanf(string,"%f %f", &elevation, &amplitude);

		while (feof(fd)==0)
		{
			/* Read in normalized radiated field values
			   for every 0.01 degrees of elevation between
			   -10.0 and +90.0 degrees */

			x=(int)rintf(100.0*(elevation+10.0));

			if (x>=0 && x<=10000)
			{
				el_pattern[x]+=amplitude;
				read_count[x]++;
			}

			fgets(string,254,fd);
			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			sscanf(string,"%f %f", &elevation, &amplitude);
		}

		fclose(fd);

		/* Average the field values in case more than
		   one was read for each 0.01 degrees of elevation. */

		for (x=0; x<=10000; x++)
		{
			if (read_count[x]>1)
				el_pattern[x]/=(float)read_count[x];
		}

		/* Interpolate between missing elevations (if
		   any) to completely fill the array and provide
		   radiated field values for every 0.01 degrees of
		   elevation. */

		last_index=-1;
		next_index=-1;

		for (x=0; x<=10000; x++)
		{
			if (read_count[x]!=0)
			{
				if (last_index==-1)
					last_index=x;
				else
					next_index=x;
			}

			if (last_index!=-1 && next_index!=-1)
			{
				valid1=el_pattern[last_index];
				valid2=el_pattern[next_index];

				span=next_index-last_index;
				delta=(valid2-valid1)/(float)span;

				for (y=last_index+1; y<next_index; y++)
					el_pattern[y]=el_pattern[y-1]+delta;

				last_index=y;
				next_index=-1;
			}
		}

		/* Fill slant_angle[] array with offset angles based
		   on the antenna's mechanical beam tilt (if any)
		   and tilt direction (azimuth). */

		if (mechanical_tilt==0.0)
		{
			for (x=0; x<=360; x++)
				slant_angle[x]=0.0;
		}

		else
		{
			tilt_increment=mechanical_tilt/90.0;

			for (x=0; x<=360; x++)
			{
				xx=(float)x;
				y=(int)rintf(tilt_azimuth+xx);

				while (y>=360)
					y-=360;

				while (y<0)
					y+=360;

				if (x<=180)
					slant_angle[y]=-(tilt_increment*(90.0-xx));

				if (x>180)
					slant_angle[y]=-(tilt_increment*(xx-270.0));
			}
		}

		slant_angle[360]=slant_angle[0];   /* 360 degree wrap-around */

		for (w=0; w<=360; w++)
		{
			tilt=slant_angle[w];

			/** Convert tilt angle to
			    an array index offset **/

			y=(int)rintf(100.0*tilt);

			/* Copy shifted el_pattern[10001] field
			   values into elevation_pattern[361][1001]
			   at the corresponding azimuth, downsampling
			   (averaging) along the way in chunks of 10. */

			for (x=y, z=0; z<=1000; x+=10, z++)
			{
				for (sum=0.0, a=0; a<10; a++)
				{
					b=a+x;

					if (b>=0 && b<=10000)
						sum+=el_pattern[b];
					if (b<0)
						sum+=el_pattern[0];
					if (b>10000)
						sum+=el_pattern[10000];
				}

				elevation_pattern[w][z]=sum/10.0;
			}
		}

		got_elevation_pattern=255;
	}

	for (x=0; x<=360; x++)
	{
		for (y=0; y<=1000; y++)
		{
			if (got_elevation_pattern)
				elevation=elevation_pattern[x][y];
			else
				elevation=1.0;

			if (got_azimuth_pattern)
				az=azimuth_pattern[x];
			else
				az=1.0;

			LR.antenna_pattern[x][y]=az*elevation;
		}
	}
}

int LoadSDF_SDF(char *name)
{
	/* This function reads uncompressed SPLAT Data Files (.sdf)
	   containing digital elevation model data into memory.
	   Elevation data, maximum and minimum elevations, and
	   quadrangle limits are stored in the first available
	   dem[] structure. */

	int	x, y, data, indx, minlat, minlon, maxlat, maxlon;
	char	found, free_slot=0, line[20], sdf_file[255],
		path_plus_name[255];
	FILE	*fd;

	for (x=0; name[x]!='.' && name[x]!=0 && x<250; x++)
		sdf_file[x]=name[x];

	sdf_file[x]=0;

	/* Parse filename for minimum latitude and longitude values */

	sscanf(sdf_file,"%d:%d:%d:%d",&minlat,&maxlat,&minlon,&maxlon);

	sdf_file[x]='.';
	sdf_file[x+1]='s';
	sdf_file[x+2]='d';
	sdf_file[x+3]='f';
	sdf_file[x+4]=0;

	/* Is it already in memory? */

	for (indx=0, found=0; indx<MAXSLOTS && found==0; indx++)
	{
		if (minlat==dem[indx].min_north && minlon==dem[indx].min_west && maxlat==dem[indx].max_north && maxlon==dem[indx].max_west)
			found=1;
	}

	/* Is room available to load it? */

	if (found==0)
	{	
		for (indx=0, free_slot=0; indx<MAXSLOTS && free_slot==0; indx++)
			if (dem[indx].max_north==-90)
				free_slot=1;
	}

	indx--;

	if (free_slot && found==0 && indx>=0 && indx<MAXSLOTS)
	{
		/* Search for SDF file in current working directory first */

		strncpy(path_plus_name,sdf_file,255);

		fd=fopen(path_plus_name,"rb");

		if (fd==NULL)
		{
			/* Next, try loading SDF file from path specified
			   in $HOME/.splat_path file or by -d argument */

			strncpy(path_plus_name,sdf_path,255);
			strncat(path_plus_name,sdf_file,255);

			fd=fopen(path_plus_name,"rb");
		}

		if (fd!=NULL)
		{
			fprintf(stdout,"Loading \"%s\" into slot %d...",path_plus_name,indx+1);
			fflush(stdout);

			fgets(line,19,fd);
			sscanf(line,"%d",&dem[indx].max_west);

			fgets(line,19,fd);
			sscanf(line,"%d",&dem[indx].min_north);

			fgets(line,19,fd);
			sscanf(line,"%d",&dem[indx].min_west);

			fgets(line,19,fd);
			sscanf(line,"%d",&dem[indx].max_north);

			for (x=0; x<1200; x++)
				for (y=0; y<1200; y++)
				{
					fgets(line,19,fd);
					sscanf(line,"%d",&data);

					dem[indx].data[x][y]=data;

					if (data>dem[indx].max_el)
						dem[indx].max_el=data;

					if (data<dem[indx].min_el)
						dem[indx].min_el=data;
				}

			fclose(fd);

			if (dem[indx].min_el<min_elevation)
				min_elevation=dem[indx].min_el;

			if (dem[indx].max_el>max_elevation)
				max_elevation=dem[indx].max_el;

			if (max_north==-90)
				max_north=dem[indx].max_north;

			else if (dem[indx].max_north>max_north)
				max_north=dem[indx].max_north;

			if (min_north==90)
				min_north=dem[indx].min_north;

			else if (dem[indx].min_north<min_north)
				min_north=dem[indx].min_north;

			if (max_west==-1)
				max_west=dem[indx].max_west;

			else
			{
				if (abs(dem[indx].max_west-max_west)<180)
				{
 					if (dem[indx].max_west>max_west)
						max_west=dem[indx].max_west;
				}

				else
				{
 					if (dem[indx].max_west<max_west)
						max_west=dem[indx].max_west;
				}
			}

			if (min_west==360)
				min_west=dem[indx].min_west;

			else
			{
				if (abs(dem[indx].min_west-min_west)<180)
				{
 					if (dem[indx].min_west<min_west)
						min_west=dem[indx].min_west;
				}

				else
				{
 					if (dem[indx].min_west>min_west)
						min_west=dem[indx].min_west;
				}
			}

			fprintf(stdout," Done!\n");
			fflush(stdout);
			return 1;
		}

		else
			return -1;
	}

	else
		return 0;
}

char *BZfgets(BZFILE *bzfd, unsigned length)
{
	/* This function returns at most one less than 'length' number
	   of characters from a bz2 compressed file whose file descriptor
	   is pointed to by *bzfd.  In operation, a buffer is filled with
	   uncompressed data (size = BZBUFFER), which is then parsed
	   and doled out as NULL terminated character strings every time
	   this function is invoked.  A NULL string indicates an EOF
	   or error condition. */

	static int x, y, nBuf;
	static char buffer[BZBUFFER+1], output[BZBUFFER+1];
	char done=0;

	if (opened!=1 && bzerror==BZ_OK)
	{
		/* First time through.  Initialize everything! */

		x=0;
		y=0;
		nBuf=0;
		opened=1;
		output[0]=0;
	}

	do
	{
		if (x==nBuf && bzerror!=BZ_STREAM_END && bzerror==BZ_OK && opened)
		{
			/* Uncompress data into a static buffer */

			nBuf=BZ2_bzRead(&bzerror, bzfd, buffer, BZBUFFER);
			buffer[nBuf]=0;
			x=0;
		}

		/* Build a string from buffer contents */

		output[y]=buffer[x];

		if (output[y]=='\n' || output[y]==0 || y==(int)length-1)
		{
			output[y+1]=0;
			done=1;
			y=0;
		}

		else
			y++;
		x++;

	} while (done==0);

	if (output[0]==0)
		opened=0;

	return (output);
}

int LoadSDF_BZ(char *name)
{
	/* This function reads .bz2 compressed SPLAT Data Files containing
	   digital elevation model data into memory.  Elevation data,
	   maximum and minimum elevations, and quadrangle limits are
	   stored in the first available dem[] structure. */

	int	x, y, data, indx, minlat, minlon, maxlat, maxlon;
	char	found, free_slot=0, sdf_file[255], path_plus_name[255];
	FILE	*fd;
	BZFILE	*bzfd;

	for (x=0; name[x]!='.' && name[x]!=0 && x<247; x++)
		sdf_file[x]=name[x];

	sdf_file[x]=0;

	/* Parse sdf_file name for minimum latitude and longitude values */

	sscanf(sdf_file,"%d:%d:%d:%d",&minlat,&maxlat,&minlon,&maxlon);

	sdf_file[x]='.';
	sdf_file[x+1]='s';
	sdf_file[x+2]='d';
	sdf_file[x+3]='f';
	sdf_file[x+4]='.';
	sdf_file[x+5]='b';
	sdf_file[x+6]='z';
	sdf_file[x+7]='2';
	sdf_file[x+8]=0;

	/* Is it already in memory? */

	for (indx=0, found=0; indx<MAXSLOTS && found==0; indx++)
	{
		if (minlat==dem[indx].min_north && minlon==dem[indx].min_west && maxlat==dem[indx].max_north && maxlon==dem[indx].max_west)
			found=1;
	}

	/* Is room available to load it? */

	if (found==0)
	{	
		for (indx=0, free_slot=0; indx<MAXSLOTS && free_slot==0; indx++)
			if (dem[indx].max_north==-90)
				free_slot=1;
	}

	indx--;

	if (free_slot && found==0 && indx>=0 && indx<MAXSLOTS)
	{
		/* Search for SDF file in current working directory first */

		strncpy(path_plus_name,sdf_file,255);

		fd=fopen(path_plus_name,"rb");
		bzfd=BZ2_bzReadOpen(&bzerror,fd,0,0,NULL,0);

		if (fd==NULL || bzerror!=BZ_OK)
		{
			/* Next, try loading SDF file from path specified
			   in $HOME/.splat_path file or by -d argument */

			strncpy(path_plus_name,sdf_path,255);
			strncat(path_plus_name,sdf_file,255);

			fd=fopen(path_plus_name,"rb");
			bzfd=BZ2_bzReadOpen(&bzerror,fd,0,0,NULL,0);
		}

		if (fd!=NULL && bzerror==BZ_OK)
		{
			fprintf(stdout,"Loading \"%s\" into slot %d...",path_plus_name,indx+1);
			fflush(stdout);

			sscanf(BZfgets(bzfd,255),"%d",&dem[indx].max_west);
			sscanf(BZfgets(bzfd,255),"%d",&dem[indx].min_north);
			sscanf(BZfgets(bzfd,255),"%d",&dem[indx].min_west);
			sscanf(BZfgets(bzfd,255),"%d",&dem[indx].max_north);
	
			for (x=0; x<1200; x++)
				for (y=0; y<1200; y++)
				{
					sscanf(BZfgets(bzfd,20),"%d",&data);

					dem[indx].data[x][y]=data;

					if (data>dem[indx].max_el)
						dem[indx].max_el=data;

					if (data<dem[indx].min_el)
						dem[indx].min_el=data;
				}

			fclose(fd);

			BZ2_bzReadClose(&bzerror,bzfd);

			if (dem[indx].min_el<min_elevation)
				min_elevation=dem[indx].min_el;
	
			if (dem[indx].max_el>max_elevation)
				max_elevation=dem[indx].max_el;

			if (max_north==-90)
				max_north=dem[indx].max_north;

			else if (dem[indx].max_north>max_north)
				max_north=dem[indx].max_north;

			if (min_north==90)
				min_north=dem[indx].min_north;

			else if (dem[indx].min_north<min_north)
				min_north=dem[indx].min_north;

			if (max_west==-1)
				max_west=dem[indx].max_west;

			else
			{
				if (abs(dem[indx].max_west-max_west)<180)
				{
 					if (dem[indx].max_west>max_west)
						max_west=dem[indx].max_west;
				}

				else
				{
 					if (dem[indx].max_west<max_west)
						max_west=dem[indx].max_west;
				}
			}

			if (min_west==360)
				min_west=dem[indx].min_west;

			else
			{
				if (abs(dem[indx].min_west-min_west)<180)
				{
 					if (dem[indx].min_west<min_west)
						min_west=dem[indx].min_west;
				}

				else
				{
 					if (dem[indx].min_west>min_west)
						min_west=dem[indx].min_west;
				}
			}

			fprintf(stdout," Done!\n");
			fflush(stdout);
			return 1;
		}

		else
			return -1;
	}

	else
		return 0;
}

char LoadSDF(char *name)
{
	/* This function loads the requested SDF file from the filesystem.
	   It first tries to invoke the LoadSDF_SDF() function to load an
	   uncompressed SDF file (since uncompressed files load slightly
	   faster).  If that attempt fails, then it tries to load a
	   compressed SDF file by invoking the LoadSDF_BZ() function.
	   If that fails, then we can assume that no elevation data
	   exists for the region requested, and that the region
	   requested must be entirely over water. */

	int	x, y, indx, minlat, minlon, maxlat, maxlon;
	char	found, free_slot=0;
	int	return_value=-1;

	/* Try to load an uncompressed SDF first. */

	return_value=LoadSDF_SDF(name);

	/* If that fails, try loading a compressed SDF. */

	if (return_value==0 || return_value==-1)
		return_value=LoadSDF_BZ(name);

	/* If neither format can be found, then assume the area is water. */

	if (return_value==0 || return_value==-1)
	{
		/* Parse SDF name for minimum latitude and longitude values */

		sscanf(name,"%d:%d:%d:%d",&minlat,&maxlat,&minlon,&maxlon);

		/* Is it already in memory? */

		for (indx=0, found=0; indx<MAXSLOTS && found==0; indx++)
		{
			if (minlat==dem[indx].min_north && minlon==dem[indx].min_west && maxlat==dem[indx].max_north && maxlon==dem[indx].max_west)
				found=1;
		}

		/* Is room available to load it? */

		if (found==0)
		{	
			for (indx=0, free_slot=0; indx<MAXSLOTS && free_slot==0; indx++)
				if (dem[indx].max_north==-90)
					free_slot=1;
		}

		indx--;

		if (free_slot && found==0 && indx>=0 && indx<MAXSLOTS)
		{
			fprintf(stdout,"Region  \"%s\" assumed as sea-level into slot %d...",name,indx+1);
			fflush(stdout);

			dem[indx].max_west=maxlon;
			dem[indx].min_north=minlat;
			dem[indx].min_west=minlon;
			dem[indx].max_north=maxlat;

			/* Fill DEM with sea-level topography */

			for (x=0; x<1200; x++)
				for (y=0; y<1200; y++)
				{
		    			dem[indx].data[x][y]=0;

					if (dem[indx].min_el>0)
						dem[indx].min_el=0;
				}

			if (dem[indx].min_el<min_elevation)
				min_elevation=dem[indx].min_el;

			if (dem[indx].max_el>max_elevation)
				max_elevation=dem[indx].max_el;

			if (max_north==-90)
				max_north=dem[indx].max_north;

			else if (dem[indx].max_north>max_north)
				max_north=dem[indx].max_north;

			if (min_north==90)
				min_north=dem[indx].min_north;

			else if (dem[indx].min_north<min_north)
				min_north=dem[indx].min_north;

			if (max_west==-1)
				max_west=dem[indx].max_west;

			else
			{
				if (abs(dem[indx].max_west-max_west)<180)
				{
 					if (dem[indx].max_west>max_west)
						max_west=dem[indx].max_west;
				}

				else
				{
 					if (dem[indx].max_west<max_west)
						max_west=dem[indx].max_west;
				}
			}

			if (min_west==360)
				min_west=dem[indx].min_west;

			else
			{
				if (abs(dem[indx].min_west-min_west)<180)
				{
 					if (dem[indx].min_west<min_west)
						min_west=dem[indx].min_west;
				}

				else
				{
 					if (dem[indx].min_west>min_west)
						min_west=dem[indx].min_west;
				}
			}

			fprintf(stdout," Done!\n");
			fflush(stdout);

			return_value=1;
		}
	}

	return return_value;
}

void LoadCities(char *filename)
{
	/* This function reads SPLAT! city/site files, and plots
	   the locations and names of the cities and site locations
	   read on topographic maps generated by SPLAT! */

	int	x, y, z;
	char	input[80], str[3][80];
	struct	site city_site;
	FILE	*fd=NULL;

	fd=fopen(filename,"r");

	if (fd!=NULL)
	{
		fgets(input,78,fd);

		fprintf(stdout,"Reading \"%s\"... ",filename);
		fflush(stdout);

		while (fd!=NULL && feof(fd)==0)
		{
			/* Parse line for name, latitude, and longitude */

			for (x=0, y=0, z=0; x<78 && input[x]!=0 && z<3; x++)
			{
				if (input[x]!=',' && y<78)
				{
					str[z][y]=input[x];
					y++;
				}

				else
				{
					str[z][y]=0;
					z++;
					y=0;
				}
			}

			strncpy(city_site.name,str[0],49);
			city_site.lat=ReadBearing(str[1]);
			city_site.lon=ReadBearing(str[2]);
			city_site.alt=0.0;

			PlaceMarker(city_site);

			fgets(input,78,fd);
		}

		fclose(fd);
		fprintf(stdout,"Done!\n");
		fflush(stdout);
	}

	else
		fprintf(stderr,"*** ERROR: \"%s\": not found!\n",filename);
}

void LoadUDT(char *filename)
{
	/* This function reads a file containing User-Defined Terrain
	   features for their addition to the digital elevation model
	   data used by SPLAT!.  Elevations in the UDT file are evaluated
	   and then copied into a temporary file under /tmp.  Then the
	   contents of the temp file are scanned, and if found to be unique,
	   are added to the ground elevations described by the digital
	   elevation data already loaded into memory. */

	int	i, x, y, z, fd=0;
	char	input[80], str[3][80], tempname[15], *pointer=NULL;
	double	latitude, longitude, height, templat, templon,
		tempheight, one_pixel;
	FILE	*fd1=NULL, *fd2=NULL;

	strcpy(tempname,"/tmp/XXXXXX\0");
	one_pixel=1.0/1200.0;

	fd1=fopen(filename,"r");

	if (fd1!=NULL)
	{
		fd=mkstemp(tempname);
		fd2=fopen(tempname,"w");

		fgets(input,78,fd1);

		pointer=strchr(input,';');

		if (pointer!=NULL)
			*pointer=0;

		fprintf(stdout,"Reading \"%s\"... ",filename);
		fflush(stdout);

		while (feof(fd1)==0)
		{
			/* Parse line for latitude, longitude, height */

			for (x=0, y=0, z=0; x<78 && input[x]!=0 && z<3; x++)
			{
				if (input[x]!=',' && y<78)
				{
					str[z][y]=input[x];
					y++;
				}

				else
				{
					str[z][y]=0;
					z++;
					y=0;
				}
			}

			latitude=ReadBearing(str[0]);
			longitude=ReadBearing(str[1]);

			/* Remove <CR> and/or <LF> from antenna height string */

			for (i=0; str[2][i]!=13 && str[2][i]!=10 && str[2][i]!=0; i++);

			str[2][i]=0;

			/* The terrain feature may be expressed in either
			   feet or meters.  If the letter 'M' or 'm' is
			   discovered in the string, then this is an
			   indication that the value given is expressed
			   in meters.  Otherwise the height is interpreted
			   as being expressed in feet.  */

			for (i=0; str[2][i]!='M' && str[2][i]!='m' && str[2][i]!=0 && i<48; i++);

			if (str[2][i]=='M' || str[2][i]=='m')
			{
				str[2][i]=0;
				height=rint(atof(str[2]));
			}

			else
			{
				str[2][i]=0;
				height=rint(3.28084*atof(str[2]));
			}

			if (height>0.0)
				fprintf(fd2,"%f, %f, %f\n",latitude, longitude, height);

			fgets(input,78,fd1);

			pointer=strchr(input,';');

			if (pointer!=NULL)
				*pointer=0;
		}

		fclose(fd1);
		fclose(fd2);
		close(fd);

		fprintf(stdout,"Done!\n");
		fflush(stdout);

		fd1=fopen(tempname,"r");
		fd2=fopen(tempname,"r");

		fscanf(fd1,"%lf, %lf, %lf", &latitude, &longitude, &height);

		for (y=0; feof(fd1)==0; y++)
		{
			rewind(fd2);

			fscanf(fd2,"%lf, %lf, %lf", &templat, &templon, &tempheight);

			for (x=0, z=0; feof(fd2)==0; x++)
			{
				if (x>y)
					if (fabs(latitude-templat)<=one_pixel && fabs(longitude-templon)<=one_pixel)
						z=1;

				fscanf(fd2,"%lf, %lf, %lf", &templat, &templon, &tempheight);
			}

			if (z==0)
				AddElevation(latitude, longitude, height);

			fscanf(fd1,"%lf, %lf, %lf", &latitude, &longitude, &height);
		}

		fclose(fd1);
		fclose(fd2);
		unlink(tempname);
	}

	else
		fprintf(stderr,"*** ERROR: \"%s\": not found!\n",filename);
}

void LoadBoundaries(char *filename)
{
	/* This function reads Cartographic Boundary Files available from
	   the U.S. Census Bureau, and plots the data contained in those
	   files on the PPM Map generated by SPLAT!.  Such files contain
	   the coordinates that describe the boundaries of cities,
	   counties, and states. */

	int	x;
	double	lat0, lon0, lat1, lon1;
	char	string[80];
	struct	site source, destination;
	FILE	*fd=NULL;

	fd=fopen(filename,"r");

	if (fd!=NULL)
	{
		fgets(string,78,fd);

		fprintf(stdout,"Reading \"%s\"... ",filename);
		fflush(stdout);

		do
		{
			fgets(string,78,fd);
			sscanf(string,"%lf %lf", &lon0, &lat0);
			fgets(string,78,fd);

			do
			{
				sscanf(string,"%lf %lf", &lon1, &lat1);

				lon0=fabs(lon0);
				lon1=fabs(lon1);

				source.lat=lat0;
				source.lon=lon0;
				destination.lat=lat1;
				destination.lon=lon1;

				ReadPath(source,destination);

				for (x=0; x<path.length; x++)
					OrMask(path.lat[x],path.lon[x],4);

				lat0=lat1;
				lon0=lon1;

				fgets(string,78,fd);

			} while (strncmp(string,"END",3)!=0 && feof(fd)==0);

			fgets(string,78,fd);

		} while (strncmp(string,"END",3)!=0 && feof(fd)==0);

		fclose(fd);

		fprintf(stdout,"Done!\n");
		fflush(stdout);
	}

	else
		fprintf(stderr,"*** ERROR: \"%s\": not found!\n",filename);
}

void ReadLRParm(char *txsite_filename)
{
	/* This function reads Longley-Rice parameter data for the
	   transmitter site.  The file name is the same as the txsite,
	   except the filename extension is .lrp.  If the needed file
	   is not found, then the file "splat.lrp" is read from the
	   current working directory.  Failure to load this file will
	   result in the default parameters hard coded into this
	   function to be used and written to "splat.lrp". */

	double	din;
	char	filename[255], string[80], *pointer=NULL;
	int	iin, ok=0, x;
	FILE	*fd=NULL, *outfile=NULL;

	/* Default parameters in case things go bad */

	LR.eps_dielect=15.0;
	LR.sgm_conductivity=0.005;
	LR.eno_ns_surfref=301.0;
	LR.frq_mhz=300.0;
	LR.radio_climate=5;
	LR.pol=0;
	LR.conf=0.50;
	LR.rel=0.50;

	/* Modify txsite filename to one with a .lrp extension. */

	strncpy(filename,txsite_filename,255);

	for (x=0; filename[x]!='.' && filename[x]!=0 && filename[x]!='\n' && x<249; x++);

	filename[x]='.';
	filename[x+1]='l';
	filename[x+2]='r';
	filename[x+3]='p';
	filename[x+4]=0;

	fd=fopen(filename,"r");

	if (fd==NULL)
	{
		/* Load default "splat.lrp" file */

		strncpy(filename,"splat.lrp\0",10);
		fd=fopen(filename,"r");
	}

	if (fd!=NULL)
	{
		fgets(string,80,fd);

		pointer=strchr(string,';');

		if (pointer!=NULL)
			*pointer=0;

		ok=sscanf(string,"%lf", &din);

		if (ok)
		{
			LR.eps_dielect=din;

			fgets(string,80,fd);

			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			ok=sscanf(string,"%lf", &din);
		}

		if (ok)
		{
			LR.sgm_conductivity=din;

			fgets(string,80,fd);

			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			ok=sscanf(string,"%lf", &din);
		}

		if (ok)
		{
			LR.eno_ns_surfref=din;

			fgets(string,80,fd);

			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			ok=sscanf(string,"%lf", &din);
		}

		if (ok)
		{
			LR.frq_mhz=din;

			fgets(string,80,fd);

			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			ok=sscanf(string,"%d", &iin);
		}

		if (ok)
		{
			LR.radio_climate=iin;

			fgets(string,80,fd);

			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			ok=sscanf(string,"%d", &iin);
		}

		if (ok)
		{
			LR.pol=iin;

			fgets(string,80,fd);

			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			ok=sscanf(string,"%lf", &din);
		}

		if (ok)
		{
			LR.conf=din;

			fgets(string,80,fd);

			pointer=strchr(string,';');

			if (pointer!=NULL)
				*pointer=0;

			ok=sscanf(string,"%lf", &din);
		}

		fclose(fd);

		if (ok)
		{
			LR.rel=din;
			LoadPAT(filename);
		}
	} 

	if (fd==NULL)
	{
		/* Create a "splat.lrp" file since one
		   could not be successfully loaded. */

		outfile=fopen("splat.lrp","w");

		fprintf(outfile,"%.3f\t; Earth Dielectric Constant (Relative permittivity)\n",LR.eps_dielect);
		fprintf(outfile,"%.3f\t; Earth Conductivity (Siemens per meter)\n", LR.sgm_conductivity);
		fprintf(outfile,"%.3f\t; Atmospheric Bending Constant (N-Units)\n",LR.eno_ns_surfref);
		fprintf(outfile,"%.3f\t; Frequency in MHz (20 MHz to 20 GHz)\n", LR.frq_mhz);
		fprintf(outfile,"%d\t; Radio Climate\n",LR.radio_climate);
		fprintf(outfile,"%d\t; Polarization (0 = Horizontal, 1 = Vertical)\n", LR.pol);
		fprintf(outfile,"%.2f\t; Fraction of situations\n",LR.conf);
		fprintf(outfile, "%.2f\t; Fraction of time\n",LR.rel);
		fprintf(outfile,"\nPlease consult SPLAT! documentation for the meaning and use of this data.\n");

		fclose(outfile);

		fprintf(stderr,"\n%c*** There were problems reading your \"%s\" file! ***\nA \"splat.lrp\" file was written to your directory with default data.\n",7,filename);
	}

	if (fd==NULL || ok==0)
		fprintf(stderr,"Longley-Rice default parameters have been assumed for this analysis.\n");
}

void PlotPath(struct site source, struct site destination, char mask_value)
{
	/* This function analyzes the path between the source and
	   destination locations.  It determines which points along
	   the path have line-of-sight visibility to the source.
	   Points along with path having line-of-sight visibility
	   to the source at an AGL altitude equal to that of the
	   destination location are stored by setting bit 1 in the
	   mask[][] array, which are displayed in green when PPM
	   maps are later generated by SPLAT!. */

	char block;
	int x, y;
	register double cos_xmtr_angle, cos_test_angle, test_alt;
	double distance, rx_alt, tx_alt;

	ReadPath(source,destination);

	for (y=0; y<path.length; y++)
	{
		/* Test this point only if it hasn't been already
		   tested and found to be free of obstructions. */

		if ((GetMask(path.lat[y],path.lon[y])&mask_value)==0)
		{
			distance=5280.0*path.distance[y];
			tx_alt=earthradius+source.alt+path.elevation[0];
			rx_alt=earthradius+destination.alt+path.elevation[y];

			/* Calculate the cosine of the elevation of the
			   transmitter as seen at the temp rx point. */

			cos_xmtr_angle=((rx_alt*rx_alt)+(distance*distance)-(tx_alt*tx_alt))/(2.0*rx_alt*distance);

			for (x=y, block=0; x>=0 && block==0; x--)
			{
				distance=5280.0*(path.distance[y]-path.distance[x]);
				test_alt=earthradius+path.elevation[x];

				cos_test_angle=((rx_alt*rx_alt)+(distance*distance)-(test_alt*test_alt))/(2.0*rx_alt*distance);

				/* Compare these two angles to determine if
				   an obstruction exists.  Since we're comparing
				   the cosines of these angles rather than
				   the angles themselves, the following "if"
				   statement is reversed from what it would
				   be if the actual angles were compared. */

				if (cos_xmtr_angle>cos_test_angle)
					block=1;
			}

			if (block==0)
				OrMask(path.lat[y],path.lon[y],mask_value);
		}
	}
}

void PlotLRPath(struct site source, struct site destination, FILE *fd)
{
	/* This function plots the RF path loss between source and
	   destination points based on the Longley-Rice propagation
	   model, taking into account antenna pattern data, if available. */

	char	block=0, strmode[100];
	int	x, y, errnum;
	double	loss, azimuth, pattern=0.0, 
		source_alt, dest_alt, source_alt2, dest_alt2,
		cos_xmtr_angle, cos_test_angle=0.0, test_alt,
		elevation, distance=0.0,
		four_thirds_earth;
	struct	site temp;

	ReadPath(source,destination);

	four_thirds_earth=EARTHRADIUS*(4.0/3.0);

	/* Copy elevations along path into the elev_l[] array. */

	for (x=0; x<path.length; x++)
		elev_l[x+2]=path.elevation[x]*METERS_PER_FOOT;

	/* Since the only energy the Longley-Rice model considers
	   reaching the destination is based on what is scattered
	   or deflected from the first obstruction along the path,
	   we first need to find the location and elevation angle
	   of that first obstruction (if it exists).  This is done
	   using a 4/3rds Earth radius to match the model used by
	   Longley-Rice.  This information is required for properly
	   integrating the antenna's elevation pattern into the
	   calculation for overall path loss.  (Using path.length-1
	   below avoids a Longley-Rice model error from occuring at
	   the destination point.) */

	for (y=2; (y<(path.length-1) && path.distance[y]<=max_range); y++)
	{
		/* Process this point only if it
		   has not already been processed. */

		if (GetMask(path.lat[y],path.lon[y])==0)
		{
			distance=5280.0*path.distance[y];
			source_alt=four_thirds_earth+source.alt+path.elevation[0];
			dest_alt=four_thirds_earth+destination.alt+path.elevation[y];
			dest_alt2=dest_alt*dest_alt;
			source_alt2=source_alt*source_alt;

			/* Calculate the cosine of the elevation of
			   the receiver as seen by the transmitter. */

			cos_xmtr_angle=((source_alt2)+(distance*distance)-(dest_alt2))/(2.0*source_alt*distance);

			if (got_elevation_pattern || fd!=NULL)
			{
				/* If no antenna elevation pattern is available, and
				   no output file is designated, the following code
				   that determines the elevation angle to the first
				   obstruction along the path is bypassed. */

				for (x=2, block=0; x<y && block==0; x++)
				{
					distance=5280.0*(path.distance[y]-path.distance[x]);
					test_alt=four_thirds_earth+path.elevation[x];

					/* Calculate the cosine of the elevation
					   angle of the terrain (test point)
					   as seen by the transmitter. */

					cos_test_angle=((source_alt2)+(distance*distance)-(test_alt*test_alt))/(2.0*source_alt*distance);

					/* Compare these two angles to determine if
					   an obstruction exists.  Since we're comparing
					   the cosines of these angles rather than
					   the angles themselves, the sense of the
					   following "if" statement is reversed from
					   what it would be if the angles themselves
					   were compared. */

					if (cos_xmtr_angle>cos_test_angle)
						block=1;
				}

				/* At this point, we have the elevation angle
				   to the first obstruction (if it exists). */
			}

			/* Determine attenuation for each point along the
			   path using Longley-Rice's point_to_point mode
			   starting at y=2 (number_of_points = 1), the
			   shortest distance terrain can play a role in
			   path loss. */
 
			elev_l[0]=y-1;  /* (number of points - 1) */

			/* Distance between elevation samples */
			elev_l[1]=METERS_PER_MILE*(path.distance[y]-path.distance[y-1]);

			point_to_point(elev_l,source.alt*METERS_PER_FOOT, 
	   		destination.alt*METERS_PER_FOOT, LR.eps_dielect,
			LR.sgm_conductivity, LR.eno_ns_surfref, LR.frq_mhz,
			LR.radio_climate, LR.pol, LR.conf, LR.rel, loss,
			strmode, errnum);

			if (block)
				elevation=((acos(cos_test_angle))/deg2rad)-90.0;

			else
				elevation=((acos(cos_xmtr_angle))/deg2rad)-90.0;

			temp.lat=path.lat[y];
			temp.lon=path.lon[y];

			azimuth=(Azimuth(source,temp));

			if (fd!=NULL)
			{
				/* Write path loss data to output file */

				fprintf(fd,"%.7f, %.7f, %.3f, %.3f, %.2f\n",path.lat[y], path.lon[y], azimuth, elevation, loss);
			}

			/* Integrate the antenna's radiation
			   pattern into the overall path loss. */

			x=(int)rint(10.0*(10.0-elevation));

			if (x>=0 && x<=1000)
			{
				azimuth=rint(azimuth);

				pattern=(double)LR.antenna_pattern[(int)azimuth][x];

				if (pattern!=0.0)
				{
					pattern=20.0*log10(pattern);
					loss-=pattern;
				}
			}

			if (loss>225.0)
				loss=225.0;

			if (loss<75.0)
				loss=75.0;

			loss-=75.0;
			loss/=10.0;
			loss+=1.0;
		
			OrMask(path.lat[y],path.lon[y],((unsigned char)(loss))<<3);
		}

		else if (GetMask(path.lat[y],path.lon[y])==0 && path.distance[y]>max_range)
			OrMask(path.lat[y],path.lon[y],1);
	}
}

void PlotCoverage(struct site source, double altitude)
{
	/* This function performs a 360 degree sweep around the
	   transmitter site (source location), and plots the
	   line-of-sight coverage of the transmitter on the SPLAT!
	   generated topographic map based on a receiver located
	   at the specified altitude (in feet AGL).  Results
	   are stored in memory, and written out in the form
	   of a topographic map when the WritePPM() function
	   is later invoked. */

	float lat, lon, one_pixel;
	static unsigned char mask_value;
	int z, count;
	struct site edge;
	unsigned char symbol[4], x;

	/* Initialize mask_value */

	if (mask_value!=8 && mask_value!=16 && mask_value!=32)
		mask_value=1;

	one_pixel=1.0/1200.0;

	symbol[0]='.';
	symbol[1]='o';
	symbol[2]='O';
	symbol[3]='o';

	count=0;	

	fprintf(stdout,"\nComputing line-of-sight coverage of %s with an RX antenna\nat %.2f %s AGL:\n\n 0%c to  25%c ",source.name,metric?altitude*METERS_PER_FOOT:altitude,metric?"meters":"feet",37,37);
	fflush(stdout);

	/* 18.75=1200 pixels/degree divided by 64 loops
	   per progress indicator symbol (.oOo) printed. */

	z=(int)(18.75*ReduceAngle(max_west-min_west));

	for (lon=min_west, x=0; (LonDiff(lon,max_west)<=0.0); lon+=one_pixel)
	{
		if (lon>=360.0)
			lon-=360.0;

		edge.lat=max_north;
		edge.lon=lon;
		edge.alt=altitude;

		PlotPath(source,edge,mask_value);
		count++;

		if (count==z) 
		{
			fprintf(stdout,"%c",symbol[x]);
			fflush(stdout);
			count=0;

			if (x==3)
				x=0;
			else
				x++;
		}
	}

	count=0;
	fprintf(stdout,"\n25%c to  50%c ",37,37);
	fflush(stdout);
	
	z=(int)(18.75*(max_north-min_north));

	for (lat=max_north, x=0; lat>=min_north; lat-=one_pixel)
	{
		edge.lat=lat;
		edge.lon=min_west;
		edge.alt=altitude;

		PlotPath(source,edge,mask_value);
		count++;

		if (count==z) 
		{
			fprintf(stdout,"%c",symbol[x]);
			fflush(stdout);
			count=0;

			if (x==3)
				x=0;
			else
				x++;
		}
	}

	count=0;
	fprintf(stdout,"\n50%c to  75%c ",37,37);
	fflush(stdout);

	z=(int)(18.75*ReduceAngle(max_west-min_west));

	for (lon=min_west, x=0; (LonDiff(lon,max_west)<=0.0); lon+=one_pixel)
	{
		if (lon>=360.0)
			lon-=360.0;

		edge.lat=min_north;
		edge.lon=lon;
		edge.alt=altitude;

		PlotPath(source,edge,mask_value);
		count++;

		if (count==z)
		{
			fprintf(stdout,"%c",symbol[x]);
			fflush(stdout);
			count=0;

			if (x==3)
				x=0;
			else
				x++;
		}
	}

	count=0;
	fprintf(stdout,"\n75%c to 100%c ",37,37);
	fflush(stdout);
	
	z=(int)(18.75*(max_north-min_north));

	for (lat=min_north, x=0; lat<=max_north; lat+=one_pixel)
	{
		edge.lat=lat;
		edge.lon=max_west;
		edge.alt=altitude;

		PlotPath(source,edge,mask_value);
		count++;

		if (count==z)
		{
			fprintf(stdout,"%c",symbol[x]);
			fflush(stdout);
			count=0;

			if (x==3)
				x=0;
			else
				x++;
		}
	}

	fprintf(stdout,"\nDone!\n");
	fflush(stdout);

	/* Assign next mask value */

	switch (mask_value)
	{
		case 1:
			mask_value=8;
			break;

		case 8:
			mask_value=16;
			break;

		case 16:
			mask_value=32;
	}
}

void PlotLRMap(struct site source, double altitude, char *plo_filename)
{
	/* This function performs a 360 degree sweep around the
	   transmitter site (source location), and plots the
	   Longley-Rice attenuation on the SPLAT! generated
	   topographic map based on a receiver located at
	   the specified altitude (in feet AGL).  Results
	   are stored in memory, and written out in the form
	   of a topographic map when the WritePPMLR() function
	   is later invoked. */

	int z, count;
	struct site edge;
	float lat, lon, one_pixel;
	unsigned char symbol[4], x;
	FILE *fd=NULL;

	one_pixel=1.0/1200.0;

	symbol[0]='.';
	symbol[1]='o';
	symbol[2]='O';
	symbol[3]='o';

	count=0;

	fprintf(stdout,"\nComputing Longley-Rice coverage of %s ", source.name);

	fprintf(stdout,"out to a radius\nof %.2f %s with an RX antenna at %.2f %s AGL:\n\n 0%c to  25%c ",metric?max_range*KM_PER_MILE:max_range,metric?"kilometers":"miles",metric?altitude*METERS_PER_FOOT:altitude,metric?"meters":"feet",37,37);
	fflush(stdout);

	if (plo_filename[0]!=0)
		fd=fopen(plo_filename,"wb");

	if (fd!=NULL)
	{
		/* Write header information to output file */

		fprintf(fd,"%d, %d\t; max_west, min_west\n%d, %d\t; max_north, min_north\n",max_west, min_west, max_north, min_north);
	}

	/* 18.75=1200 pixels/degree divided by 64 loops
	   per progress indicator symbol (.oOo) printed. */

	z=(int)(18.75*ReduceAngle(max_west-min_west));

	for (lon=min_west, x=0; (LonDiff(lon,max_west)<=0.0); lon+=one_pixel)
	{
		if (lon>=360.0)
			lon-=360.0;

		edge.lat=max_north;
		edge.lon=lon;
		edge.alt=altitude;

		PlotLRPath(source,edge,fd);
		count++;

		if (count==z) 
		{
			fprintf(stdout,"%c",symbol[x]);
			fflush(stdout);
			count=0;

			if (x==3)
				x=0;
			else
				x++;
		}
	}

	count=0;
	fprintf(stdout,"\n25%c to  50%c ",37,37);
	fflush(stdout);
	
	z=(int)(18.75*(max_north-min_north));

	for (lat=max_north, x=0; lat>=min_north; lat-=one_pixel)
	{
		edge.lat=lat;
		edge.lon=min_west;
		edge.alt=altitude;

		PlotLRPath(source,edge,fd);
		count++;

		if (count==z) 
		{
			fprintf(stdout,"%c",symbol[x]);
			fflush(stdout);
			count=0;

			if (x==3)
				x=0;
			else
				x++;
		}
	}

	count=0;
	fprintf(stdout,"\n50%c to  75%c ",37,37);
	fflush(stdout);

	z=(int)(18.75*ReduceAngle(max_west-min_west));

	for (lon=min_west, x=0; (LonDiff(lon,max_west)<=0.0); lon+=one_pixel)
	{
		if (lon>=360.0)
			lon-=360.0;

		edge.lat=min_north;
		edge.lon=lon;
		edge.alt=altitude;

		PlotLRPath(source,edge,fd);
		count++;

		if (count==z)
		{
			fprintf(stdout,"%c",symbol[x]);
			fflush(stdout);
			count=0;

			if (x==3)
				x=0;
			else
				x++;
		}
	}

	count=0;
	fprintf(stdout,"\n75%c to 100%c ",37,37);
	fflush(stdout);
	
	z=(int)(18.75*(max_north-min_north));

	for (lat=min_north, x=0; lat<=max_north; lat+=one_pixel)
	{
		edge.lat=lat;
		edge.lon=max_west;
		edge.alt=altitude;

		PlotLRPath(source,edge,fd);
		count++;

		if (count==z)
		{
			fprintf(stdout,"%c",symbol[x]);
			fflush(stdout);
			count=0;

			if (x==3)
				x=0;
			else
				x++;
		}
	}

	if (fd!=NULL)
		fclose(fd);

	fprintf(stdout,"\nDone!\n");
	fflush(stdout);
}

void WritePPM(char *filename, unsigned char geo)
{
	/* This function generates a topographic map in Portable Pix Map
	   (PPM) format based on logarithmically scaled topology data,
	   as well as the content of flags held in the mask[][] array.
	   The image created is rotated counter-clockwise 90 degrees
	   from its representation in dem[][] so that north points
	   up and east points right in the image generated. */

	char mapfile[255], geofile[255];
	unsigned char found, mask;
	unsigned width, height, output;
	int indx, x, y, x0=0, y0=0, minlat, minlon;
	float lat, lon, one_pixel, conversion, one_over_gamma;
	FILE *fd;

	one_pixel=1.0/1200.0;
	one_over_gamma=1.0/GAMMA;
	conversion=255.0/pow((double)(max_elevation-min_elevation),one_over_gamma);

	width=(unsigned)(1200*ReduceAngle(max_west-min_west));
	height=(unsigned)(1200*ReduceAngle(max_north-min_north));

	if (filename[0]==0)
		strncpy(mapfile, "map.ppm\0",8);
	else
	{
		for (x=0; filename[x]!='.' && filename[x]!=0 && x<250; x++)
		{
			mapfile[x]=filename[x];
			geofile[x]=filename[x];
		}

		mapfile[x]='.';
		geofile[x]='.';
		mapfile[x+1]='p';
		geofile[x+1]='g';
		mapfile[x+2]='p';
		geofile[x+2]='e';
		mapfile[x+3]='m';
		geofile[x+3]='o';
		mapfile[x+4]=0;
		geofile[x+4]=0;
	}

	if (geo)
	{
		fd=fopen(geofile,"wb");

		fprintf(fd,"FILENAME\t%s\n",mapfile);
		fprintf(fd,"#\t\tX\tY\tLong\t\tLat\n");
		fprintf(fd,"TIEPOINT\t0\t0\t%d.000\t\t%d.000\n",(max_west<180?-max_west:360-max_west),max_north);
		fprintf(fd,"TIEPOINT\t%u\t%u\t%d.000\t\t%d.000\n",width-1,height-1,(min_west<180?-min_west:360-min_west),min_north);
		fprintf(fd,"IMAGESIZE\t%u\t%u\n",width,height);
		fprintf(fd,"#\n# Auto Generated by SPLAT! v%s\n#\n",splat_version);

		fclose(fd);
	}

	fd=fopen(mapfile,"wb");

	fprintf(fd,"P6\n%u %u\n255\n",width,height);

	fprintf(stdout,"\nWriting \"%s\" (%ux%u pixmap image)... ",mapfile,width,height);
	fflush(stdout);

	for (y=0, lat=((double)max_north)-one_pixel; y<(int)height; y++, lat-=one_pixel)
	{
		minlat=(int)floor(lat);

		for (x=0, lon=((double)max_west)-one_pixel; x<(int)width; x++, lon-=one_pixel)
		{
			if (lon<0.0)
				lon+=360.0;

			minlon=(int)floor(lon);

			for (indx=0, found=0; indx<MAXSLOTS && found==0;)
				if (minlat==dem[indx].min_north && minlon==dem[indx].min_west)
					found=1;
				else
					indx++;

			if (found)
			{
				x0=(int)(1199.0*(lat-floor(lat)));
				y0=(int)(1199.0*(lon-floor(lon)));

				mask=dem[indx].mask[x0][y0];

				if (mask&2)
					/* Text Labels: Red */
					fprintf(fd,"%c%c%c",255,0,0);

				else if (mask&4)
					/* County Boundaries: Light Cyan */
					fprintf(fd,"%c%c%c",128,128,255);

				else switch (mask&57)
				{
					case 1:
					/* TX1: Green */
					fprintf(fd,"%c%c%c",0,255,0);
					break;

					case 8:
					/* TX2: Cyan */
					fprintf(fd,"%c%c%c",0,255,255);
					break;

					case 9:
					/* TX1 + TX2: Yellow */
					fprintf(fd,"%c%c%c",255,255,0);
					break;

					case 16:
					/* TX3: Medium Violet */
					fprintf(fd,"%c%c%c",147,112,219);
					break;

					case 17:
					/* TX1 + TX3: Pink */
					fprintf(fd,"%c%c%c",255,192,203);
					break;

					case 24:
					/* TX2 + TX3: Orange */
					fprintf(fd,"%c%c%c",255,165,0);
					break;

				 	case 25:
					/* TX1 + TX2 + TX3: Dark Green */
					fprintf(fd,"%c%c%c",0,100,0);
					break;

					case 32:
					/* TX4: Sienna 1 */
					fprintf(fd,"%c%c%c",255,130,71);
					break;

					case 33:
					/* TX1 + TX4: Green Yellow */
					fprintf(fd,"%c%c%c",173,255,47);
					break;

					case 40:
					/* TX2 + TX4: Dark Sea Green 1 */
					fprintf(fd,"%c%c%c",193,255,193);
					break;

					case 41:
					/* TX1 + TX2 + TX4: Blanched Almond */
					fprintf(fd,"%c%c%c",255,235,205);
					break;

					case 48:
					/* TX3 + TX4: Dark Turquoise */
					fprintf(fd,"%c%c%c",0,206,209);
					break;

					case 49:
					/* TX1 + TX3 + TX4: Medium Spring Green */
					fprintf(fd,"%c%c%c",0,250,154);
					break;

					case 56:
					/* TX2 + TX3 + TX4: Tan */
					fprintf(fd,"%c%c%c",210,180,140);
					break;

					case 57:
					/* TX1 + TX2 + TX3 + TX4: Gold2 */
					fprintf(fd,"%c%c%c",238,201,0);
					break;

					default:
					/* Water: Medium Blue */
					if (dem[indx].data[x0][y0]==0)
						fprintf(fd,"%c%c%c",0,0,170);
					else
					{
						/* Elevation: Greyscale */
						output=(unsigned)(0.5+pow((double)(dem[indx].data[x0][y0]-min_elevation),one_over_gamma)*conversion);
						fprintf(fd,"%c%c%c",output,output,output);
					}
				}
			}

			else
			{
				/* We should never get here, but if */
				/* we do, display the region as black */

				fprintf(fd,"%c%c%c",0,0,0);
			}
		}
	}

	fclose(fd);
	fprintf(stdout,"Done!\n");
	fflush(stdout);
}

void WritePPMLR(char *filename, unsigned char geo)
{
	/* This function generates a topographic map in Portable Pix Map
	   (PPM) format based on the content of flags held in the mask[][] 
	   array (only).  The image created is rotated counter-clockwise
	   90 degrees from its representation in dem[][] so that north
	   points up and east points right in the image generated. */

	char mapfile[255], geofile[255];
	unsigned width, height, output;
	unsigned char found, mask, cityorcounty;
	int indx, x, y, t, t2, x0, y0, minlat, minlon, loss;
	float lat, lon, one_pixel, conversion, one_over_gamma;
	FILE *fd;

	one_pixel=1.0/1200.0;
	one_over_gamma=1.0/GAMMA;
	conversion=255.0/pow((double)(max_elevation-min_elevation),one_over_gamma);

	width=(unsigned)(1200*ReduceAngle(max_west-min_west));
	height=(unsigned)(1200*ReduceAngle(max_north-min_north));

	if (filename[0]==0)
		strncpy(mapfile, "map.ppm\0",8);
	else
	{
		for (x=0; filename[x]!='.' && filename[x]!=0 && x<250; x++)
		{
			mapfile[x]=filename[x];
			geofile[x]=filename[x];
		}

		mapfile[x]='.';
		geofile[x]='.';
		mapfile[x+1]='p';
		geofile[x+1]='g';
		mapfile[x+2]='p';
		geofile[x+2]='e';
		mapfile[x+3]='m';
		geofile[x+3]='o';
		mapfile[x+4]=0;
		geofile[x+4]=0;
	}

	if (geo)
	{
		fd=fopen(geofile,"wb");

		fprintf(fd,"FILENAME\t%s\n",mapfile);
		fprintf(fd,"#\t\tX\tY\tLong\t\tLat\n");
		fprintf(fd,"TIEPOINT\t0\t0\t%d.000\t\t%d.000\n",(max_west<180?-max_west:360-max_west),max_north);
		fprintf(fd,"TIEPOINT\t%u\t%u\t%d.000\t\t%.3f\n",width-1,height+29,(min_west<180?-min_west:360-min_west),(double)(min_north-0.025));
		fprintf(fd,"IMAGESIZE\t%u\t%u\n",width,height+30);
		fprintf(fd,"#\n# Auto Generated by SPLAT! v%s\n#\n",splat_version);

		fclose(fd);
	}

	fd=fopen(mapfile,"wb");

	fprintf(fd,"P6\n%u %u\n255\n",width,height+30);

	fprintf(stdout,"\nWriting \"%s\" (%ux%u pixmap image)... ",mapfile,width,height+30);
	fflush(stdout);

	for (y=0, lat=((double)max_north)-one_pixel; y<(int)height; y++, lat-=one_pixel)
	{
		minlat=(int)floor(lat);

		for (x=0, lon=((double)max_west)-one_pixel; x<(int)width; x++, lon-=one_pixel)
		{
			if (lon<0.0)
				lon+=360.0;

			minlon=(int)floor(lon);

			for (indx=0, found=0; indx<MAXSLOTS && found==0;)
				if (minlat==dem[indx].min_north && minlon==dem[indx].min_west)
					found=1;
				else
					indx++;
			if (found)
			{
				x0=(int)(1199.0*(lat-floor(lat)));
				y0=(int)(1199.0*(lon-floor(lon)));

				mask=dem[indx].mask[x0][y0];
				loss=70+(10*(int)((mask&248)>>3));
				cityorcounty=0;

	 			if (mask&2)
				{
					/* Text Labels - Black or Red */

					if ((mask&120) && (loss<=90))
						fprintf(fd,"%c%c%c",0,0,0);
					else
						fprintf(fd,"%c%c%c",255,0,0);

					cityorcounty=1;
				}

				else if (mask&4)
				{
					/* County Boundaries: Black */

					fprintf(fd,"%c%c%c",0,0,0);

					cityorcounty=1;
				}

				if (cityorcounty==0)
				{
					if (loss>maxdB)

					{ /* Display land or sea elevation */

						if (dem[indx].data[x0][y0]==0)
							fprintf(fd,"%c%c%c",0,0,170);
						else
						{
							/* Elevation: Greyscale */
							output=(unsigned)(0.5+pow((double)(dem[indx].data[x0][y0]-min_elevation),one_over_gamma)*conversion);
							fprintf(fd,"%c%c%c",output,output,output);
						}
					}

					else switch (loss)
					{
						/* Plot signal loss in color */

						case 80:
						fprintf(fd,"%c%c%c",255,0,0);
						break;

						case 90:
						fprintf(fd,"%c%c%c",255,128,0);
						break;

						case 100:
						fprintf(fd,"%c%c%c",255,165,0);
						break;

						case 110:
						fprintf(fd,"%c%c%c",255,206,0);
						break;

						case 120:
						fprintf(fd,"%c%c%c",255,255,0);
						break;

						case 130:
						fprintf(fd,"%c%c%c",184,255,0);
						break;

					 	case 140:
						fprintf(fd,"%c%c%c",0,255,0);
						break;

						case 150:
						fprintf(fd,"%c%c%c",0,208,0);
						break;

						case 160:
						fprintf(fd,"%c%c%c",0,196,196);
						break;

						case 170:
						fprintf(fd,"%c%c%c",0,148,255);
						break;

						case 180:
						fprintf(fd,"%c%c%c",80,80,255);
						break;

						case 190:
						fprintf(fd,"%c%c%c",0,38,255);
						break;

						case 200:
						fprintf(fd,"%c%c%c",142,63,255);
						break;

						case 210:
						fprintf(fd,"%c%c%c",196,54,255);
						break;

						case 220:
						fprintf(fd,"%c%c%c",255,0,255);
						break;

						case 230:
						fprintf(fd,"%c%c%c",255,194,204);
						break;

						default:

						if (dem[indx].data[x0][y0]==0)
							fprintf(fd,"%c%c%c",0,0,170);
						else
						{
							/* Elevation: Greyscale */
							output=(unsigned)(0.5+pow((double)(dem[indx].data[x0][y0]-min_elevation),one_over_gamma)*conversion);
							fprintf(fd,"%c%c%c",output,output,output);
						}
					}
				}
			}

			else
			{
				/* We should never get here, but if */
				/* we do, display the region as black */

				fprintf(fd,"%c%c%c",0,0,0);
			}
		}
	}

	/* Display legend along bottom of image */

	x0=width/16;

	for (y0=0; y0<30; y0++)
	{
		for (indx=0; indx<16; indx++)
		{
			for (x=0; x<x0; x++)
			{
		       		t=indx;  
		       		t2=indx+8;

		       		if (y0>=10 && y0<=18)
				{  
					if (t2>9)
					{
				  		if (x>=11 && x<=17)     
				      			if (smallfont[t2/10][y0-10][x-11])
								t=255; 
			    		}

					if (x>=19 && x<=25)     
						if (smallfont[t2%10][y0-10][x-19])
							t=255;
 
					if (x>=27 && x<=33)
						if (smallfont[0][y0-10][x-27])
							t=255; 
		       		}

				switch (t)
				{
					case 0:
					fprintf(fd,"%c%c%c",255,0,0);
					break;

					case 1:
					fprintf(fd,"%c%c%c",255,128,0);
					break;

					case 2:
					fprintf(fd,"%c%c%c",255,165,0);
					break;

					case 3:
					fprintf(fd,"%c%c%c",255,206,0);
					break;

					case 4:
					fprintf(fd,"%c%c%c",255,255,0);
					break;

					case 5:
					fprintf(fd,"%c%c%c",184,255,0);
					break;

					case 6:
					fprintf(fd,"%c%c%c",0,255,0);
					break;

				 	case 7:
					fprintf(fd,"%c%c%c",0,208,0);
					break;

					case 8:
					fprintf(fd,"%c%c%c",0,196,196);
					break;

					case 9:
					fprintf(fd,"%c%c%c",0,148,255);
					break;

					case 10:
					fprintf(fd,"%c%c%c",80,80,255);
					break;

					case 11:
					fprintf(fd,"%c%c%c",0,38,255);
					break;

					case 12:
					fprintf(fd,"%c%c%c",142,63,255);
					break;

					case 13:
					fprintf(fd,"%c%c%c",196,54,255);
					break;

					case 14:
					fprintf(fd,"%c%c%c",255,0,255);
					break;

					case 255:
					/* Black */
					fprintf(fd,"%c%c%c",0,0,0);
					break;

					default:
					fprintf(fd,"%c%c%c",255,194,204);
				}
			} 
		}
	}

	fclose(fd);
	fprintf(stdout,"Done!\n");
	fflush(stdout);
}

void GraphTerrain(struct site source, struct site destination, char *name)
{
	/* This function invokes gnuplot to generate an appropriate
	   output file indicating the terrain profile between the source
	   and destination locations.  "filename" is the name assigned
	   to the output file generated by gnuplot.  The filename extension
	   is used to set gnuplot's terminal setting and output file type.
	   If no extension is found, .png is assumed.  */

	int	x, y, z;
	char	filename[255], term[30], ext[15];
	FILE	*fd=NULL;

	ReadPath(destination,source);

	fd=fopen("profile.gp","wb");

	for (x=0; x<path.length; x++)
	{
		if (metric)
			fprintf(fd,"%f\t%f\n",KM_PER_MILE*path.distance[x],METERS_PER_FOOT*path.elevation[x]);

		else
			fprintf(fd,"%f\t%f\n",path.distance[x],path.elevation[x]);
	}

	fclose(fd);

	if (name[0]==0)
	{
		/* Default filename and output file type */

		strncpy(filename,"profile\0",8);
		strncpy(term,"png\0",4);
		strncpy(ext,"png\0",4);
	}

	else
	{
		/* Grab extension and terminal type from "name" */

		for (x=0; name[x]!='.' && name[x]!=0 && x<254; x++)
			filename[x]=name[x];

		if (name[x]=='.')
		{
			for (y=0, z=x, x++; name[x]!=0 && x<254 && y<14; x++, y++)
			{
				term[y]=tolower(name[x]);
				ext[y]=term[y];
			}

			ext[y]=0;
			term[y]=0;
			filename[z]=0;
		}

		else
		{	/* No extension -- Default is png */

			filename[x]=0;
			strncpy(term,"png\0",4);
			strncpy(ext,"png\0",4);
		}
	}

	/* Either .ps or .postscript may be used
	   as an extension for postscript output. */

	if (strncmp(term,"postscript",10)==0)
		strncpy(ext,"ps\0",3);

	else if (strncmp(ext,"ps",2)==0)
		strncpy(term,"postscript enhanced color\0",26);

	fprintf(stdout,"Writing \"%s.%s\"...",filename,ext);
	fflush(stdout);

	fd=fopen("splat.gp","w");
	fprintf(fd,"set grid\n");
	fprintf(fd,"set autoscale\n");
	fprintf(fd,"set encoding iso_8859_1\n");
	fprintf(fd,"set term %s\n",term);
	fprintf(fd,"set title \"SPLAT! Terrain Profile Between %s and %s (%.2f%c Azimuth)\"\n",destination.name, source.name, Azimuth(destination,source),176);

	if (metric)
	{
		fprintf(fd,"set xlabel \"Distance Between %s and %s (%.2f kilometers)\"\n",destination.name,source.name,KM_PER_MILE*Distance(source,destination));
		fprintf(fd,"set ylabel \"Ground Elevation Above Sea Level (meters)\"\n");


	}

	else
	{
		fprintf(fd,"set xlabel \"Distance Between %s and %s (%.2f miles)\"\n",destination.name,source.name,Distance(source,destination));
		fprintf(fd,"set ylabel \"Ground Elevation Above Sea Level (feet)\"\n");
	}

	fprintf(fd,"set output \"%s.%s\"\n",filename,ext);
	fprintf(fd,"plot \"profile.gp\" title \"\" with lines\n");
	fclose(fd);
			
	x=system("gnuplot splat.gp");

	if (x!=-1)
	{
		unlink("splat.gp");
		unlink("profile.gp");
		fprintf(stdout," Done!\n");
		fflush(stdout);
	}

	else
		fprintf(stderr,"\n*** ERROR: Error occurred invoking gnuplot!\n");
}

void GraphElevation(struct site source, struct site destination, char *name)
{
	/* This function invokes gnuplot to generate an appropriate
	   output file indicating the terrain profile between the source
	   and destination locations.  "filename" is the name assigned
	   to the output file generated by gnuplot.  The filename extension
	   is used to set gnuplot's terminal setting and output file type.
	   If no extension is found, .png is assumed. */

	int	x, y, z;
	char	filename[255], term[30], ext[15];
	double	angle, refangle, maxangle=-90.0;
	struct	site remote;
	FILE	*fd=NULL, *fd2=NULL;

	ReadPath(destination,source);  /* destination=RX, source=TX */
	refangle=ElevationAngle(destination,source);

	fd=fopen("profile.gp","wb");
	fd2=fopen("reference.gp","wb");

	for (x=1; x<path.length-1; x++)
	{
		remote.lat=path.lat[x];
		remote.lon=path.lon[x];
		remote.alt=0.0;
		angle=ElevationAngle(destination,remote);

		if (metric)
		{
			fprintf(fd,"%f\t%f\n",KM_PER_MILE*path.distance[x],angle);
			fprintf(fd2,"%f\t%f\n",KM_PER_MILE*path.distance[x],refangle);
		}

		else
		{
			fprintf(fd,"%f\t%f\n",path.distance[x],angle);
			fprintf(fd2,"%f\t%f\n",path.distance[x],refangle);
		}

		if (angle>maxangle)
			maxangle=angle;
	}

	if (metric)
	{
		fprintf(fd,"%f\t%f\n",KM_PER_MILE*path.distance[path.length-1],refangle);
		fprintf(fd2,"%f\t%f\n",KM_PER_MILE*path.distance[path.length-1],refangle);
	}

	else
	{
		fprintf(fd,"%f\t%f\n",path.distance[path.length-1],refangle);
		fprintf(fd2,"%f\t%f\n",path.distance[path.length-1],refangle);
	}

	fclose(fd);
	fclose(fd2);

	if (name[0]==0)
	{
		/* Default filename and output file type */

		strncpy(filename,"profile\0",8);
		strncpy(term,"png\0",4);
		strncpy(ext,"png\0",4);
	}

	else
	{
		/* Grab extension and terminal type from "name" */

		for (x=0; name[x]!='.' && name[x]!=0 && x<254; x++)
			filename[x]=name[x];

		if (name[x]=='.')
		{
			for (y=0, z=x, x++; name[x]!=0 && x<254 && y<14; x++, y++)
			{
				term[y]=tolower(name[x]);
				ext[y]=term[y];
			}

			ext[y]=0;
			term[y]=0;
			filename[z]=0;
		}

		else
		{	/* No extension -- Default is png */

			filename[x]=0;
			strncpy(term,"png\0",4);
			strncpy(ext,"png\0",4);
		}
	}

	/* Either .ps or .postscript may be used
	   as an extension for postscript output. */

	if (strncmp(term,"postscript",10)==0)
		strncpy(ext,"ps\0",3);

	else if (strncmp(ext,"ps",2)==0)
		strncpy(term,"postscript enhanced color\0",26);

	fprintf(stdout,"Writing \"%s.%s\"...",filename,ext);
	fflush(stdout);

	fd=fopen("splat.gp","w");

	fprintf(fd,"set grid\n");
	fprintf(fd,"set yrange [%2.3f to %2.3f]\n", (-fabs(refangle)-0.25), maxangle+0.25);
	fprintf(fd,"set encoding iso_8859_1\n");
	fprintf(fd,"set term %s\n",term);
	fprintf(fd,"set title \"SPLAT! Elevation Profile Between %s and %s (%.2f%c azimuth)\"\n",destination.name,source.name,Azimuth(destination,source),176);

	if (metric)
		fprintf(fd,"set xlabel \"Distance Between %s and %s (%.2f kilometers)\"\n",destination.name,source.name,KM_PER_MILE*Distance(source,destination));
	else
		fprintf(fd,"set xlabel \"Distance Between %s and %s (%.2f miles)\"\n",destination.name,source.name,Distance(source,destination));


	fprintf(fd,"set ylabel \"Elevation Angle Along LOS Path Between %s and %s (degrees)\"\n",destination.name,source.name);
	fprintf(fd,"set output \"%s.%s\"\n",filename,ext);
	fprintf(fd,"plot \"profile.gp\" title \"Real Earth Profile\" with lines, \"reference.gp\" title \"Line of Sight Path (%.2f%c elevation)\" with lines\n",refangle,176);

	fclose(fd);
			
	x=system("gnuplot splat.gp");

	if (x!=-1)
	{
		unlink("splat.gp");
		unlink("profile.gp");
		unlink("reference.gp"); 

		fprintf(stdout," Done!\n");
		fflush(stdout);
	}

	else
		fprintf(stderr,"\n*** ERROR: Error occurred invoking gnuplot!\n");
}

void GraphHeight(struct site source, struct site destination, char *name, double f, unsigned char n)
{
	/* This function invokes gnuplot to generate an appropriate
	   output file indicating the terrain profile between the source
	   and destination locations referenced to the line-of-sight path
	   between the receive and transmit sites.  "filename" is the name
	   assigned to the output file generated by gnuplot.  The filename
	   extension is used to set gnuplot's terminal setting and output
	   file type.  If no extension is found, .png is assumed. */

	int	x, y, z;
	char	filename[255], term[30], ext[15];
	double	a, b, c, height=0.0, refangle, cangle, maxheight=-100000.0,
		minheight=100000.0, lambda=0.0, f_zone=0.0, fpt6_zone=0.0,
		nm=0.0, nb=0.0, ed=0.0, es=0.0, r=0.0, d=0.0, d1=0.0,
		terrain, azimuth, distance, dheight=0.0, minterrain=100000.0,
		minearth=100000.0, miny, maxy, min2y, max2y;
	struct	site remote;
	FILE	*fd=NULL, *fd2=NULL, *fd3=NULL, *fd4=NULL, *fd5=NULL;

	ReadPath(destination,source);  /* destination=RX, source=TX */
	azimuth=Azimuth(destination,source);
	distance=Distance(destination,source);
	refangle=ElevationAngle(destination,source);
	b=GetElevation(destination)+destination.alt+earthradius;

	/* Wavelength and path distance (great circle) in feet. */

	if (f)
	{
		lambda=9.8425e8/(f*1e6);
		d=5280.0*path.distance[path.length-1];
	}

	if (n)
	{
		ed=GetElevation(destination);
		es=GetElevation(source);
		nb=-destination.alt-ed;
		nm=(-source.alt-es-nb)/(path.distance[path.length-1]);
	}

	fd=fopen("profile.gp","wb");
	fd2=fopen("reference.gp","wb");
	fd5=fopen("curvature.gp", "wb");

	if (f)
	{
		fd3=fopen("fresnel.gp", "wb");
		fd4=fopen("fresnel_pt_6.gp", "wb");
	}

	for (x=0; x<path.length; x++)
	{
		remote.lat=path.lat[x];
		remote.lon=path.lon[x];
		remote.alt=0.0;

		terrain=GetElevation(remote);

		if (x==0)
			terrain+=destination.alt;  /* RX antenna spike */

		a=terrain+earthradius;
 		cangle=5280.0*Distance(destination,remote)/earthradius;
		c=b*sin(refangle*deg2rad+HALFPI)/sin(HALFPI-refangle*deg2rad-cangle);

		height=a-c;

		/* Per Fink and Christiansen, Electronics
		 * Engineers' Handbook, 1989:
		 *
		 *   H = sqrt(lamba * d1 * (d - d1)/d)
		 *
		 * where H is the distance from the LOS
		 * path to the first Fresnel zone boundary.
		 */

		if (f)
		{
			d1=5280.0*path.distance[x];
			f_zone=-1*sqrt(lambda*d1*(d-d1)/d);
			fpt6_zone=0.6*f_zone;
		}

		if (n)
		{
			r=-(nm*path.distance[x])-nb;
			height+=r;

			if (f>0) 
			{
				f_zone+=r;
				fpt6_zone+=r;
			}
		}

		else
			r=0.0;

		if (metric)
		{
			fprintf(fd,"%f\t%f\n",KM_PER_MILE*path.distance[x],METERS_PER_FOOT*height);
			fprintf(fd2,"%f\t%f\n",KM_PER_MILE*path.distance[x],METERS_PER_FOOT*r);
			fprintf(fd5,"%f\t%f\n",KM_PER_MILE*path.distance[x],METERS_PER_FOOT*(height-terrain));
		}

		else
		{
			fprintf(fd,"%f\t%f\n",path.distance[x],height);
			fprintf(fd2,"%f\t%f\n",path.distance[x],r);
			fprintf(fd5,"%f\t%f\n",path.distance[x],height-terrain);
		}

		if (f)
		{
			if (metric)
			{
				fprintf(fd3,"%f\t%f\n",KM_PER_MILE*path.distance[x],METERS_PER_FOOT*f_zone);
				fprintf(fd4,"%f\t%f\n",KM_PER_MILE*path.distance[x],METERS_PER_FOOT*fpt6_zone);
			}

			else
			{
				fprintf(fd3,"%f\t%f\n",path.distance[x],f_zone);
				fprintf(fd4,"%f\t%f\n",path.distance[x],fpt6_zone);
			}

			if (f_zone<minheight)
				minheight=f_zone;
		}

		if (height>maxheight)
			maxheight=height;

		if (height<minheight)
			minheight=height;

		if (r>maxheight)
			maxheight=r;

		if (terrain<minterrain)
			minterrain=terrain;

		if ((height-terrain)<minearth)
			minearth=height-terrain;
	}

	if (n)
		r=-(nm*path.distance[path.length-1])-nb;
	else
		r=0.0;

	if (metric)
	{
		fprintf(fd,"%f\t%f\n",KM_PER_MILE*path.distance[path.length-1],METERS_PER_FOOT*r);
		fprintf(fd2,"%f\t%f\n",KM_PER_MILE*path.distance[path.length-1],METERS_PER_FOOT*r);
	}

	else
	{
		fprintf(fd,"%f\t%f\n",path.distance[path.length-1],r);
		fprintf(fd2,"%f\t%f\n",path.distance[path.length-1],r);
	}

	if (f)
	{
		if (metric)
		{
			fprintf(fd3,"%f\t%f\n",path.distance[path.length-1],r);
			fprintf(fd4,"%f\t%f\n",path.distance[path.length-1],r);
		}

		else
		{
			fprintf(fd3,"%f\t%f\n",KM_PER_MILE*path.distance[path.length-1],METERS_PER_FOOT*r);
			fprintf(fd4,"%f\t%f\n",KM_PER_MILE*path.distance[path.length-1],METERS_PER_FOOT*r);
		}
	}
	
	if (r>maxheight)
		maxheight=r;

	if (r<minheight)
		minheight=r;

	fclose(fd);
	fclose(fd2);
	fclose(fd5);

	if (f)
	{
		fclose(fd3);
		fclose(fd4);
	}

	if (name[0]==0)
	{
		/* Default filename and output file type */

		strncpy(filename,"height\0",8);
		strncpy(term,"png\0",4);
		strncpy(ext,"png\0",4);
	}

	else
	{
		/* Grab extension and terminal type from "name" */

		for (x=0; name[x]!='.' && name[x]!=0 && x<254; x++)
			filename[x]=name[x];

		if (name[x]=='.')
		{
			for (y=0, z=x, x++; name[x]!=0 && x<254 && y<14; x++, y++)
			{
				term[y]=tolower(name[x]);
				ext[y]=term[y];
			}

			ext[y]=0;
			term[y]=0;
			filename[z]=0;
		}

		else
		{	/* No extension -- Default is png */

			filename[x]=0;
			strncpy(term,"png\0",4);
			strncpy(ext,"png\0",4);
		}
	}

	/* Either .ps or .postscript may be used
	   as an extension for postscript output. */

	if (strncmp(term,"postscript",10)==0)
		strncpy(ext,"ps\0",3);

	else if (strncmp(ext,"ps",2)==0)
		strncpy(term,"postscript enhanced color\0",26);

	fprintf(stdout,"Writing \"%s.%s\"...",filename,ext);
	fflush(stdout);

	fd=fopen("splat.gp","w");

	dheight=maxheight-minheight;
	miny=minheight-0.15*dheight;
	maxy=maxheight+0.05*dheight;

	if (maxy<20.0)
		maxy=20.0;

	dheight=maxheight-minheight;
	min2y=miny-minterrain+0.05*dheight;

	if (minearth<min2y)
	{
		miny-=min2y-minearth+0.05*dheight;
		min2y=minearth-0.05*dheight;
	}

	max2y=min2y+maxy-miny;
 
	fprintf(fd,"set grid\n");
	fprintf(fd,"set yrange [%2.3f to %2.3f]\n", metric?miny*METERS_PER_FOOT:miny, metric?maxy*METERS_PER_FOOT:maxy);
	fprintf(fd,"set y2range [%2.3f to %2.3f]\n", metric?min2y*METERS_PER_FOOT:min2y, metric?max2y*METERS_PER_FOOT:max2y);
	fprintf(fd,"set xrange [-0.5 to %2.3f]\n",metric?KM_PER_MILE*rint(distance+0.5):rint(distance+0.5));
	fprintf(fd,"set encoding iso_8859_1\n");
	fprintf(fd,"set term %s\n",term);

	if (f)
		fprintf(fd,"set title \"SPLAT! Path Profile Between %s and %s (%.2f%c azimuth)\\nWith First Fresnel Zone\"\n",destination.name, source.name, azimuth,176);

	else
		fprintf(fd,"set title \"SPLAT! Height Profile Between %s and %s (%.2f%c azimuth)\"\n",destination.name, source.name, azimuth,176);

	if (metric)
		fprintf(fd,"set xlabel \"Distance Between %s and %s (%.2f kilometers)\"\n",destination.name,source.name,KM_PER_MILE*Distance(source,destination));
	else
		fprintf(fd,"set xlabel \"Distance Between %s and %s (%.2f miles)\"\n",destination.name,source.name,Distance(source,destination));

	if (n)
	{
		if (metric)
			fprintf(fd,"set ylabel \"Normalized Height Referenced To LOS Path Between\\n%s and %s (meters)\"\n",destination.name,source.name);

		else
			fprintf(fd,"set ylabel \"Normalized Height Referenced To LOS Path Between\\n%s and %s (feet)\"\n",destination.name,source.name);

	}

	else
	{
		if (metric)
			fprintf(fd,"set ylabel \"Height Referenced To LOS Path Between %s and %s (meters)\"\n",destination.name,source.name);

		else
			fprintf(fd,"set ylabel \"Height Referenced To LOS Path Between %s and %s (feet)\"\n",destination.name,source.name);
	}

	fprintf(fd,"set output \"%s.%s\"\n",filename,ext);

	if (f)
		fprintf(fd,"plot \"profile.gp\" title \"Point-to-Point Profile\" with lines, \"reference.gp\" title \"Line of Sight Path\" with lines, \"curvature.gp\" axes x1y2 title \"Earth's Curvature Contour\" with lines, \"fresnel.gp\" axes x1y1 title \"First Fresnel Zone (%.3f MHz)\" with lines, \"fresnel_pt_6.gp\" title \"60%% of First Fresnel Zone\" with lines\n",f);

	else
		fprintf(fd,"plot \"profile.gp\" title \"Point-to-Point Profile\" with lines, \"reference.gp\" title \"Line Of Sight Path\" with lines, \"curvature.gp\" axes x1y2 title \"Earth's Curvature Contour\" with lines\n");

	fclose(fd);

	x=system("gnuplot splat.gp");

	if (x!=-1)
	{
		unlink("splat.gp");
		unlink("profile.gp");
		unlink("reference.gp");
		unlink("curvature.gp");

		if (f)
		{
			unlink("fresnel.gp");
			unlink("fresnel_pt_6.gp");
		}

		fprintf(stdout," Done!\n");
		fflush(stdout);
	}

	else
		fprintf(stderr,"\n*** ERROR: Error occurred invoking gnuplot!\n");
}

void GraphLongley(struct site source, struct site destination, char *name)
{
	/* This function invokes gnuplot to generate an appropriate
	   output file indicating the Longley-Rice model loss between
	   the source and destination locations.  "filename" is the
	   name assigned to the output file generated by gnuplot.
	   The filename extension is used to set gnuplot's terminal
	   setting and output file type.  If no extension is found,
	   .png is assumed. */

	int	x, y, z, errnum, errflag=0;
	char	filename[255], term[30], ext[15], strmode[100],
		report_name[80], block=0;
	double	maxloss=-100000.0, minloss=100000.0, loss, haavt,
		angle1, angle2, azimuth, pattern=1.0, patterndB=0.0,
		total_loss=0.0, cos_xmtr_angle, cos_test_angle=0.0,
		source_alt, test_alt, dest_alt, source_alt2, dest_alt2,
		distance, elevation, four_thirds_earth;
	FILE	*fd=NULL, *fd2=NULL;

	sprintf(report_name,"%s-to-%s.lro",source.name,destination.name);

	four_thirds_earth=EARTHRADIUS*(4.0/3.0);

	for (x=0; report_name[x]!=0; x++)
		if (report_name[x]==32 || report_name[x]==17 || report_name[x]==92 || report_name[x]==42 || report_name[x]==47)
			report_name[x]='_';	

	fd2=fopen(report_name,"w");

	fprintf(fd2,"\n\t--==[ SPLAT! v%s Longley-Rice Model Path Loss Report ]==--\n\n",splat_version);
	fprintf(fd2,"Analysis of RF path conditions between %s and %s:\n",source.name, destination.name);
	fprintf(fd2,"\n-------------------------------------------------------------------------\n\n");
	fprintf(fd2,"Transmitter site: %s\n",source.name);

	if (source.lat>=0.0)
	{
		fprintf(fd2,"Site location: %.4f North / %.4f West",source.lat, source.lon);
		fprintf(fd2, " (%s N / ", dec2dms(source.lat));
	}

	else
	{

		fprintf(fd2,"Site location: %.4f South / %.4f West",-source.lat, source.lon);
		fprintf(fd2, " (%s S / ", dec2dms(source.lat));
	}
	
	fprintf(fd2, "%s W)\n", dec2dms(source.lon));

	if (metric)
	{
		fprintf(fd2,"Ground elevation: %.2f meters AMSL\n",METERS_PER_FOOT*GetElevation(source));
		fprintf(fd2,"Antenna height: %.2f meters AGL / %.2f meters AMSL\n",METERS_PER_FOOT*source.alt,METERS_PER_FOOT*(source.alt+GetElevation(source)));
	}

	else
	{
		fprintf(fd2,"Ground elevation: %.2f feet AMSL\n",GetElevation(source));
		fprintf(fd2,"Antenna height: %.2f feet AGL / %.2f feet AMSL\n",source.alt, source.alt+GetElevation(source));
	}

	haavt=haat(source);

	if (haavt>-4999.0)
	{
		if (metric)
			fprintf(fd2,"Antenna height above average terrain: %.2f meters\n",METERS_PER_FOOT*haavt);
		else
			fprintf(fd2,"Antenna height above average terrain: %.2f feet\n",haavt);
	}

	azimuth=Azimuth(source,destination);
	angle1=ElevationAngle(source,destination);
	angle2=ElevationAngle2(source,destination,earthradius);

	if (got_azimuth_pattern || got_elevation_pattern)
	{
		x=(int)rint(10.0*(10.0-angle2));

		if (x>=0 && x<=1000)
			pattern=(double)LR.antenna_pattern[(int)rint(azimuth)][x];

		patterndB=20.0*log10(pattern);

		fprintf(fd2,"Antenna pattern between %s and %s: %.3f (%.2f dB)\n",source.name, destination.name, pattern, patterndB);

	}

	if (metric)
		fprintf(fd2,"Distance to %s: %.2f kilometers\n",destination.name,METERS_PER_FOOT*Distance(source,destination));

	else
		fprintf(fd2,"Distance to %s: %.2f miles.\n",destination.name,Distance(source,destination));

	fprintf(fd2,"Azimuth to %s: %.2f degrees\n",destination.name,azimuth);

	if (angle1>=0.0)
		fprintf(fd2,"Elevation angle to %s: %+.4f degrees\n",destination.name,angle1);

	else
		fprintf(fd2,"Depression angle to %s: %+.4f degrees\n",destination.name,angle1);

	if (angle1!=angle2)
	{
		if (angle2<0.0)
			fprintf(fd2,"Depression");
		else
			fprintf(fd2,"Elevation");

		fprintf(fd2," angle to the first obstruction: %+.4f degrees\n",angle2);
	}

	fprintf(fd2,"\n-------------------------------------------------------------------------\n\n");

	/* Receiver */

	fprintf(fd2,"Receiver site: %s\n",destination.name);

	if (destination.lat>=0.0)
	{
		fprintf(fd2,"Site location: %.4f North / %.4f West",destination.lat, destination.lon);
		fprintf(fd2, " (%s N / ", dec2dms(destination.lat));
	}

	else
	{
		fprintf(fd2,"Site location: %.4f South / %.4f West",-destination.lat, destination.lon);
		fprintf(fd2, " (%s S / ", dec2dms(destination.lat));
	}

	fprintf(fd2, "%s W)\n", dec2dms(destination.lon));

	if (metric)
	{
		fprintf(fd2,"Ground elevation: %.2f meters AMSL\n",METERS_PER_FOOT*GetElevation(destination));
		fprintf(fd2,"Antenna height: %.2f meters AGL / %.2f meters AMSL\n",METERS_PER_FOOT*destination.alt, METERS_PER_FOOT*(destination.alt+GetElevation(destination)));
	}

	else
	{
		fprintf(fd2,"Ground elevation: %.2f feet AMSL\n",GetElevation(destination));
		fprintf(fd2,"Antenna height: %.2f feet AGL / %.2f feet AMSL\n",destination.alt, destination.alt+GetElevation(destination));
	}

	haavt=haat(destination);

	if (haavt>-4999.0)
	{
		if (metric)
			fprintf(fd2,"Antenna height above average terrain: %.2f meters\n",METERS_PER_FOOT*haavt);
		else
			fprintf(fd2,"Antenna height above average terrain: %.2f feet\n",haavt);
	}

	if (metric)
		fprintf(fd2,"Distance to %s: %.2f kilometers\n",source.name,KM_PER_MILE*Distance(source,destination));

	else
		fprintf(fd2,"Distance to %s: %.2f miles\n",source.name,Distance(source,destination));

	azimuth=Azimuth(destination,source);

	angle1=ElevationAngle(destination,source);
	angle2=ElevationAngle2(destination,source,earthradius);

	fprintf(fd2,"Azimuth to %s: %.2f degrees.\n",source.name,azimuth);

	if (angle1>=0.0)
		fprintf(fd2,"Elevation angle to %s: %+.4f degrees\n",source.name,angle1);

	else
		fprintf(fd2,"Depression angle to %s: %+.4f degrees\n",source.name,angle1);

	if (angle1!=angle2)
	{
		if (angle2<0.0)
			fprintf(fd2,"Depression");
		else
			fprintf(fd2,"Elevation");

		fprintf(fd2," angle to the first obstruction: %+.4f degrees\n",angle2);
	}

	fprintf(fd2,"\n-------------------------------------------------------------------------\n\n");

	fprintf(fd2,"Longley-Rice path calculation parameters used in this analysis:\n\n");
	fprintf(fd2,"Earth's Dielectric Constant: %.3lf\n",LR.eps_dielect);
	fprintf(fd2,"Earth's Conductivity: %.3lf\n",LR.sgm_conductivity);
	fprintf(fd2,"Atmospheric Bending Constant (N): %.3lf\n",LR.eno_ns_surfref);
	fprintf(fd2,"Frequency: %.3lf (MHz)\n",LR.frq_mhz);
	fprintf(fd2,"Radio Climate: %d (",LR.radio_climate);

	switch (LR.radio_climate)
	{
		case 1:
		fprintf(fd2,"Equatorial");
		break;

		case 2:
		fprintf(fd2,"Continental Subtropical");
		break;

		case 3:
		fprintf(fd2,"Maritime Subtropical");
		break;

		case 4:
		fprintf(fd2,"Desert");
		break;

		case 5:
		fprintf(fd2,"Continental Temperate");
		break;

		case 6:
		fprintf(fd2,"Martitime Temperate, Over Land");
		break;

		case 7:
		fprintf(fd2,"Maritime Temperate, Over Sea");
		break;

		default:
		fprintf(fd2,"Unknown");
	}

	fprintf(fd2,")\nPolarization: %d (",LR.pol);

	if (LR.pol==0)
		fprintf(fd2,"Horizontal");

	if (LR.pol==1)
		fprintf(fd2,"Vertical");

	fprintf(fd2,")\nFraction of Situations: %.1lf%c\n",LR.conf*100.0,37);
	fprintf(fd2,"Fraction of Time: %.1lf%c\n",LR.rel*100.0,37);
	fprintf(fd2,"\n-------------------------------------------------------------------------\n\n");

	fprintf(fd2,"Analysis Results:\n\n");

	ReadPath(source, destination);  /* source=TX, destination=RX */

	/* Copy elevations along path into the elev_l[] array. */

	for (x=0; x<path.length; x++)
		elev_l[x+2]=path.elevation[x]*METERS_PER_FOOT;  

	if (metric)
		fprintf(fd2,"Distance (km)");
	else
		fprintf(fd2,"Distance (mi)");

	fprintf(fd2,"\tLoss (dB)\tErrnum\tComment\n\n");

	fd=fopen("profile.gp","w");

	azimuth=rint(Azimuth(source,destination));

	for (y=2; y<(path.length-1); y++)  /* path.length-1 avoids LR error */
	{
		distance=5280.0*path.distance[y];
		source_alt=four_thirds_earth+source.alt+path.elevation[0];
		dest_alt=four_thirds_earth+destination.alt+path.elevation[y];
		dest_alt2=dest_alt*dest_alt;
		source_alt2=source_alt*source_alt;

		/* Calculate the cosine of the elevation of
		   the receiver as seen by the transmitter. */

		cos_xmtr_angle=((source_alt2)+(distance*distance)-(dest_alt2))/(2.0*source_alt*distance);

		if (got_elevation_pattern)
		{
			/* If an antenna elevation pattern is available, the
			   following code determines the elevation angle to
			   the first obstruction along the path. */

			for (x=2, block=0; x<y && block==0; x++)
			{
				distance=5280.0*(path.distance[y]-path.distance[x]);
				test_alt=four_thirds_earth+path.elevation[x];

				/* Calculate the cosine of the elevation
				   angle of the terrain (test point)
				   as seen by the transmitter. */

				cos_test_angle=((source_alt2)+(distance*distance)-(test_alt*test_alt))/(2.0*source_alt*distance);

				/* Compare these two angles to determine if
				   an obstruction exists.  Since we're comparing
				   the cosines of these angles rather than
				   the angles themselves, the sense of the
				   following "if" statement is reversed from
				   what it would be if the angles themselves
				   were compared. */

				if (cos_xmtr_angle>cos_test_angle)
					block=1;
			}

			/* At this point, we have the elevation angle
			   to the first obstruction (if it exists). */
		}

		/* Determine path loss for each point along the
		   path using Longley-Rice's point_to_point mode
	  	   starting at x=2 (number_of_points = 1), the
	  	   shortest distance terrain can play a role in
	  	   path loss. */

		elev_l[0]=y-1;	/* (number of points - 1) */

		/* Distance between elevation samples */
		elev_l[1]=METERS_PER_MILE*(path.distance[y]-path.distance[y-1]);

		point_to_point(elev_l, source.alt*METERS_PER_FOOT, 
		destination.alt*METERS_PER_FOOT, LR.eps_dielect,
		LR.sgm_conductivity, LR.eno_ns_surfref, LR.frq_mhz,
		LR.radio_climate, LR.pol, LR.conf, LR.rel, loss,
		strmode, errnum);

		if (block)
			elevation=((acos(cos_test_angle))/deg2rad)-90.0;
		else
			elevation=((acos(cos_xmtr_angle))/deg2rad)-90.0;

		/* Integrate the antenna's radiation
		   pattern into the overall path loss. */

		x=(int)rint(10.0*(10.0-elevation));

		if (x>=0 && x<=1000)
		{
			pattern=(double)LR.antenna_pattern[(int)azimuth][x];

			if (pattern!=0.0)
				patterndB=20.0*log10(pattern);
		}

		else
			patterndB=0.0;

		total_loss=loss-patterndB;

		if (metric)
		{
			fprintf(fd,"%f\t%f\n",KM_PER_MILE*(path.distance[path.length-1]-path.distance[y]),total_loss);
			fprintf(fd2,"%7.2f\t\t%7.2f\t\t  %d\t%s\n",KM_PER_MILE*path.distance[y],total_loss, errnum, strmode);
		}

		else
		{
			fprintf(fd,"%f\t%f\n",path.distance[path.length-1]-path.distance[y],total_loss);
			fprintf(fd2,"%7.2f\t\t%7.2f\t\t  %d\t%s\n",path.distance[y],total_loss, errnum, strmode);
		}

		errflag|=errnum;

		if (total_loss>maxloss)
			maxloss=total_loss;

		if (total_loss<minloss)
			minloss=total_loss;
	}

	fclose(fd);

	fprintf(fd2,"\nLongley-Rice path loss between %s and %s is %.2f dB.\n",source.name, destination.name, loss);

	if (patterndB!=0.0)
		fprintf(fd2,"Total path loss including TX antenna pattern is %.2f dB.\n",total_loss);

	if (errflag)
	{
		fprintf(fd2,"\nNotes on \"errnum\":\n\n");
		fprintf(fd2,"  0: No error.  :-)\n");
		fprintf(fd2,"  1: Warning!  Some parameters are nearly out of range.\n");
		fprintf(fd2,"     Results should be used with caution.\n");
		fprintf(fd2,"  2: Note: Default parameters have been substituted for impossible ones.\n");
		fprintf(fd2,"  3: Warning!  A combination of parameters is out of range.\n");
		fprintf(fd2,"     Results are probably invalid.\n");
		fprintf(fd2,"  Other: Warning!  Some parameters are out of range.\n");
		fprintf(fd2,"	 Results are probably invalid.\n\nEnd of Report\n");
	}

	fprintf(stdout,"Longley-Rice path loss between %s and %s is %.2f dB.\n",source.name, destination.name, loss);

	if (patterndB!=0.0)
		fprintf(stdout,"Total path loss including TX antenna pattern is %.2f dB.\n",total_loss);

	fprintf(stdout,"Path Loss Report written to: \"%s\"\n",report_name);
	fflush(stdout);

	fclose(fd2);

	if (name[0]==0)
	{
		/* Default filename and output file type */

		strncpy(filename,"loss\0",5);
		strncpy(term,"png\0",4);
		strncpy(ext,"png\0",4);
	}

	else
	{
		/* Grab extension and terminal type from "name" */

		for (x=0; name[x]!='.' && name[x]!=0 && x<254; x++)
			filename[x]=name[x];

		if (name[x]=='.')
		{
			for (y=0, z=x, x++; name[x]!=0 && x<254 && y<14; x++, y++)
			{
				term[y]=tolower(name[x]);
				ext[y]=term[y];
			}

			ext[y]=0;
			term[y]=0;
			filename[z]=0;
		}

		else
		{	/* No extension -- Default is png */

			filename[x]=0;
			strncpy(term,"png\0",4);
			strncpy(ext,"png\0",4);
		}
	}

	/* Either .ps or .postscript may be used
	   as an extension for postscript output. */

	if (strncmp(term,"postscript",10)==0)
		strncpy(ext,"ps\0",3);

	else if (strncmp(ext,"ps",2)==0)
		strncpy(term,"postscript enhanced color\0",26);

	fprintf(stdout,"Writing \"%s.%s\"...",filename,ext);
	fflush(stdout);

	fd=fopen("splat.gp","w");

	fprintf(fd,"set grid\n");
	fprintf(fd,"set yrange [%2.3f to %2.3f]\n", minloss, maxloss);
	fprintf(fd,"set encoding iso_8859_1\n");
	fprintf(fd,"set term %s\n",term);
	fprintf(fd,"set title \"SPLAT! Loss Profile Along Path Between %s and %s (%.2f%c azimuth)\"\n",destination.name, source.name, Azimuth(destination,source),176);

	if (metric)
		fprintf(fd,"set xlabel \"Distance Between %s and %s (%.2f kilometers)\"\n",destination.name,source.name,KM_PER_MILE*Distance(destination,source));
	else
		fprintf(fd,"set xlabel \"Distance Between %s and %s (%.2f miles)\"\n",destination.name,source.name,Distance(destination,source));

	if (got_azimuth_pattern || got_elevation_pattern)
		fprintf(fd,"set ylabel \"Total Path Loss (including TX antenna pattern) (dB)");
	else
		fprintf(fd,"set ylabel \"Longley-Rice Path Loss (dB)");

	fprintf(fd,"\"\nset output \"%s.%s\"\n",filename,ext);
	fprintf(fd,"plot \"profile.gp\" title \"Path Loss\" with lines\n");

	fclose(fd);
			
	x=system("gnuplot splat.gp");

	if (x!=-1)
	{
		unlink("splat.gp");
		unlink("profile.gp");
		unlink("reference.gp"); 

		fprintf(stdout," Done!\n");
		fflush(stdout);
	}

	else
		fprintf(stderr,"\n*** ERROR: Error occurred invoking gnuplot!\n");
}

void ObstructionReport(struct site xmtr, struct site rcvr, char report, double f)
{
	int	x;
	struct	site site_x;
	double	h_r, h_t, h_x, h_r_orig, cos_tx_angle, cos_test_angle,
		cos_tx_angle_f1, cos_tx_angle_fpt6, haavt, d_tx, d_x,
		h_r_f1, h_r_fpt6, h_f, h_los, lambda=0.0, azimuth,
		pattern, patterndB, distance, angle1, angle2;
	char	report_name[80], string[255], string_fpt6[255],
		string_f1[255];
	FILE	*fd;

	sprintf(report_name,"%s-to-%s.txt",xmtr.name,rcvr.name);

	for (x=0; report_name[x]!=0; x++)
		if (report_name[x]==32 || report_name[x]==17 || report_name[x]==92 || report_name[x]==42 || report_name[x]==47)
			report_name[x]='_';	

	fd=fopen(report_name,"w");

	fprintf(fd,"\n\t\t--==[ SPLAT! v%s Obstruction Report ]==--\n\n",splat_version);
	fprintf(fd,"Analysis of great circle path between %s and %s:\n",xmtr.name, rcvr.name);
	fprintf(fd,"\n-------------------------------------------------------------------------\n\n");
	fprintf(fd,"Transmitter site: %s\n",xmtr.name);


	if (xmtr.lat>=0.0)
	{
		fprintf(fd,"Site location: %.4f North / %.4f West",xmtr.lat, xmtr.lon);
		fprintf(fd, " (%s N / ", dec2dms(xmtr.lat));
	}

	else
	{
		fprintf(fd,"Site location: %.4f South / %.4f West",-xmtr.lat, xmtr.lon);
		fprintf(fd, " (%s S / ", dec2dms(xmtr.lat));
	}

	fprintf(fd, "%s W)\n", dec2dms(xmtr.lon));

	if (metric)
	{
		fprintf(fd,"Ground elevation: %.2f meters AMSL\n",METERS_PER_FOOT*GetElevation(xmtr));
		fprintf(fd,"Antenna height: %.2f meters AGL / %.2f meters AMSL\n",METERS_PER_FOOT*xmtr.alt, METERS_PER_FOOT*(xmtr.alt+GetElevation(xmtr)));
	}

	else
	{
		fprintf(fd,"Ground elevation: %.2f feet AMSL\n",GetElevation(xmtr));
		fprintf(fd,"Antenna height: %.2f feet AGL / %.2f feet AMSL\n",xmtr.alt, xmtr.alt+GetElevation(xmtr));
	}

	haavt=haat(xmtr);

	if (haavt>-4999.0)
	{
		if (metric)
			fprintf(fd,"Antenna height above average terrain: %.2f meters\n",METERS_PER_FOOT*haavt);
		else
			fprintf(fd,"Antenna height above average terrain: %.2f feet\n",haavt);
	}

	pattern=1.0;
	patterndB=0.0;
	distance=Distance(xmtr,rcvr);
	azimuth=Azimuth(xmtr,rcvr);
	angle1=ElevationAngle(xmtr,rcvr);
	angle2=ElevationAngle2(xmtr,rcvr,earthradius);

	if (got_azimuth_pattern || got_elevation_pattern)
	{
		x=(int)rint(10.0*(10.0-angle2));

		if (x>=0 && x<=1000)
			pattern=(double)LR.antenna_pattern[(int)rint(azimuth)][x];

		if (pattern!=1.0)
		{
			fprintf(fd,"Antenna pattern toward %s: %.3f",rcvr.name,pattern);
			patterndB=20.0*log10(pattern);
			fprintf(fd," (%.2f dB)\n",patterndB);
		}
	}

	if (metric)
		fprintf(fd,"Distance to %s: %.2f kilometers\n",rcvr.name,KM_PER_MILE*distance);

	else
		fprintf(fd,"Distance to %s: %.2f miles\n",rcvr.name,distance);

	fprintf(fd,"Azimuth to %s: %.2f degrees\n",rcvr.name,azimuth);

	if (angle1>=0.0)
		fprintf(fd,"Elevation angle to %s: %+.4f degrees\n",rcvr.name,angle1);

	else
		fprintf(fd,"Depression angle to %s: %+.4f degrees\n",rcvr.name,angle1);

	if (angle1!=angle2)
	{
		if (angle2<0.0)
			fprintf(fd,"Depression");
		else
			fprintf(fd,"Elevation");

		fprintf(fd," angle to the first obstruction: %+.4f degrees\n",angle2);
	}

	fprintf(fd,"\n-------------------------------------------------------------------------\n\n");

	/* Receiver */

	fprintf(fd,"Receiver site: %s\n",rcvr.name);

	if (rcvr.lat>=0.0)
	{
		fprintf(fd,"Site location: %.4f North / %.4f West",rcvr.lat, rcvr.lon);
		fprintf(fd, " (%s N / ", dec2dms(rcvr.lat));
	}

	else
	{
		fprintf(fd,"Site location: %.4f South / %.4f West",-rcvr.lat, rcvr.lon);
		fprintf(fd, " (%s S / ", dec2dms(rcvr.lat));
	}

	fprintf(fd, "%s W)\n", dec2dms(rcvr.lon));

	if (metric)
	{
		fprintf(fd,"Ground elevation: %.2f meters AMSL\n",METERS_PER_FOOT*GetElevation(rcvr));
		fprintf(fd,"Antenna height: %.2f meters AGL / %.2f meters AMSL\n",METERS_PER_FOOT*rcvr.alt, METERS_PER_FOOT*(rcvr.alt+GetElevation(rcvr)));
	}

	else
	{
		fprintf(fd,"Ground elevation: %.2f feet AMSL\n",GetElevation(rcvr));
		fprintf(fd,"Antenna height: %.2f feet AGL / %.2f feet AMSL\n",rcvr.alt, rcvr.alt+GetElevation(rcvr));
	}

	haavt=haat(rcvr);

	if (haavt>-4999.0)
	{
		if (metric)
			fprintf(fd,"Antenna height above average terrain: %.2f meters\n",METERS_PER_FOOT*haavt);
		else
			fprintf(fd,"Antenna height above average terrain: %.2f feet\n",haavt);
	}

	azimuth=Azimuth(rcvr,xmtr);
	angle1=ElevationAngle(rcvr,xmtr);
	angle2=ElevationAngle2(rcvr,xmtr,earthradius);

	if (metric)
		fprintf(fd,"Distance to %s: %.2f kilometers\n",xmtr.name,KM_PER_MILE*distance);
	else
		fprintf(fd,"Distance to %s: %.2f miles\n",xmtr.name,distance);

	fprintf(fd,"Azimuth to %s: %.2f degrees\n",xmtr.name,azimuth);

	if (angle1>=0.0)
		fprintf(fd,"Elevation to %s: %+.4f degrees\n",xmtr.name,angle1);

	else
		fprintf(fd,"Depression angle to %s: %+.4f degrees\n",xmtr.name,angle1);

	if (angle1!=angle2)
	{
		if (angle2<0.0)
			fprintf(fd,"Depression");
		else
			fprintf(fd,"Elevation");

		fprintf(fd," angle to the first obstruction: %+.4f degrees\n",angle2);

	}

	fprintf(fd,"\n-------------------------------------------------------------------------\n\n");

	if (report=='y')
	{
		/* Generate profile of the terrain.  Create the path
		   from transmitter to receiver because that's the
		   way the original los() function did it, and going
		   the other way can yield slightly different results. */

		ReadPath(xmtr,rcvr);
		h_r=GetElevation(rcvr)+rcvr.alt+earthradius;
		h_r_f1=h_r;
		h_r_fpt6=h_r;
		h_r_orig=h_r;
		h_t=GetElevation(xmtr)+xmtr.alt+earthradius;
		d_tx=5280.0*Distance(rcvr,xmtr);
		cos_tx_angle=((h_r*h_r)+(d_tx*d_tx)-(h_t*h_t))/(2.0*h_r*d_tx);
		cos_tx_angle_f1=cos_tx_angle;
		cos_tx_angle_fpt6=cos_tx_angle;

		if (f)
			lambda=9.8425e8/(f*1e6);

		/* At each point along the path calculate the cosine
		   of a sort of "inverse elevation angle" at the receiver.
		   From the antenna, 0 deg. looks at the ground, and 90 deg.
		   is parallel to the ground.

		   Start at the receiver.  If this is the lowest antenna,
		   then terrain obstructions will be nearest to it.  (Plus,
		   that's the way the original los() did it.)

		   Calculate cosines only.  That's sufficient to compare
		   angles and it saves the extra computational burden of
		   acos().  However, note the inverted comparison: if
		   acos(A) > acos(B), then B > A. */
		
		for (x=path.length-1; x>0; x--)
		{
			site_x.lat=path.lat[x];
			site_x.lon=path.lon[x];
			site_x.alt=0.0;

			h_x=GetElevation(site_x)+earthradius;
			d_x=5280.0*Distance(rcvr,site_x);

			/* Deal with the LOS path first. */

			cos_test_angle=((h_r*h_r)+(d_x*d_x)-(h_x*h_x))/(2.0*h_r*d_x);

			if (cos_tx_angle>cos_test_angle)
			{
				if (h_r==h_r_orig)
					fprintf(fd,"SPLAT! detected obstructions at:\n\n");

				if (site_x.lat>=0.0)
				{
					if (metric)
						fprintf(fd,"\t%.4f N, %.4f W, %5.2f kilometers, %6.2f meters AMSL\n",site_x.lat, site_x.lon, KM_PER_MILE*(d_x/5280.0), METERS_PER_FOOT*(h_x-earthradius));
					else
						fprintf(fd,"\t%.4f N, %.4f W, %5.2f miles, %6.2f feet AMSL\n",site_x.lat, site_x.lon, d_x/5280.0, h_x-earthradius);
				}

				else
				{
					if (metric)
						fprintf(fd,"\t%.4f S, %.4f W, %5.2f kilometers, %6.2f meters AMSL\n",-site_x.lat, site_x.lon, KM_PER_MILE*(d_x/5280.0), METERS_PER_FOOT*(h_x-earthradius));
					else

						fprintf(fd,"\t%.4f S, %.4f W, %5.2f miles, %6.2f feet AMSL\n",-site_x.lat, site_x.lon, d_x/5280.0, h_x-earthradius);
				}
			}

			while (cos_tx_angle>cos_test_angle)
			{
				h_r+=1;
				cos_test_angle=((h_r*h_r)+(d_x*d_x)-(h_x*h_x))/(2.0*h_r*d_x);
				cos_tx_angle=((h_r*h_r)+(d_tx*d_tx)-(h_t*h_t))/(2.0*h_r*d_tx);
			}

			if (f)
			{
				/* Now clear the first Fresnel zone, but don't
				   clutter the obstruction report. */

				cos_tx_angle_f1=((h_r_f1*h_r_f1)+(d_tx*d_tx)-(h_t*h_t))/(2.0*h_r_f1*d_tx);
				h_los=sqrt(h_r_f1*h_r_f1+d_x*d_x-2*h_r_f1*d_x*cos_tx_angle_f1);
				h_f=h_los-sqrt(lambda*d_x*(d_tx-d_x)/d_tx);

				while (h_f<h_x)
				{
					h_r_f1+=1;
					cos_tx_angle_f1=((h_r_f1*h_r_f1)+(d_tx*d_tx)-(h_t*h_t))/(2.0*h_r_f1*d_tx);
					h_los=sqrt(h_r_f1*h_r_f1+d_x*d_x-2*h_r_f1*d_x*cos_tx_angle_f1);
					h_f=h_los-sqrt(lambda*d_x*(d_tx-d_x)/d_tx);
				}

				/* And clear the 60% F1 zone. */

				cos_tx_angle_fpt6=((h_r_fpt6*h_r_fpt6)+(d_tx*d_tx)-(h_t*h_t))/(2.0*h_r_fpt6*d_tx);
				h_los=sqrt(h_r_fpt6*h_r_fpt6+d_x*d_x-2*h_r_fpt6*d_x*cos_tx_angle_fpt6);
				h_f=h_los-0.6*sqrt(lambda*d_x*(d_tx-d_x)/d_tx);

				while (h_f<h_x)
				{
					h_r_fpt6+=1;
					cos_tx_angle_fpt6=((h_r_fpt6*h_r_fpt6)+(d_tx*d_tx)-(h_t*h_t))/(2.0*h_r_fpt6*d_tx);
					h_los=sqrt(h_r_fpt6*h_r_fpt6+d_x*d_x-2*h_r_fpt6*d_x*cos_tx_angle_fpt6);
					h_f=h_los-0.6*sqrt(lambda*d_x*(d_tx-d_x)/d_tx);
				}
			}
		}
		
		if (h_r>h_r_orig)
		{
			if (metric)
				sprintf(string,"\nAntenna at %s must be raised to at least %.2f meters AGL\nto clear all obstructions detected by SPLAT!\n",rcvr.name, METERS_PER_FOOT*(h_r-GetElevation(rcvr)-earthradius));
			else
				sprintf(string,"\nAntenna at %s must be raised to at least %.2f feet AGL\nto clear all obstructions detected by SPLAT!\n",rcvr.name, h_r-GetElevation(rcvr)-earthradius);
		}

		else
			sprintf(string,"\nNo obstructions to LOS path due to terrain were detected by SPLAT!\n");

		if (f)
		{
			if (h_r_fpt6>h_r_orig)
			{
				if (metric)
					sprintf(string_fpt6,"\nAntenna at %s must be raised to at least %.2f meters AGL\nto clear 60%c of the first Fresnel zone.\n",rcvr.name, METERS_PER_FOOT*(h_r_fpt6-GetElevation(rcvr)-earthradius),37);

				else
					sprintf(string_fpt6,"\nAntenna at %s must be raised to at least %.2f feet AGL\nto clear 60%c of the first Fresnel zone.\n",rcvr.name, h_r_fpt6-GetElevation(rcvr)-earthradius,37);
			}

			else
				sprintf(string_fpt6,"\n60%c of the first Fresnel zone is clear.\n",37);
	
			if (h_r_f1>h_r_orig)
			{
				if (metric)
					sprintf(string_f1,"\nAntenna at %s must be raised to at least %.2f meters AGL\nto clear the first Fresnel zone.\n",rcvr.name, METERS_PER_FOOT*(h_r_f1-GetElevation(rcvr)-earthradius));

				else			
					sprintf(string_f1,"\nAntenna at %s must be raised to at least %.2f feet AGL\nto clear the first Fresnel zone.\n",rcvr.name, h_r_f1-GetElevation(rcvr)-earthradius);

			}

			else
    			    sprintf(string_f1,"\nThe first Fresnel zone is clear.\n\n");
		}
	}
	
	fprintf(fd,"%s",string);

	if (f)
	{
		fprintf(fd,"%s",string_f1);
		fprintf(fd,"%s",string_fpt6);
	}

	fclose(fd);

	/* Display report summary on terminal */

	/* Line-of-sight status */

	fprintf(stdout,"%s",string);

	if (f)
	{
		/* Fresnel zone status */

		fprintf(stdout,"%s",string_f1);
		fprintf(stdout,"%s",string_fpt6);
	}

	fprintf(stdout, "\nObstruction report written to: \"%s\"\n",report_name);

	fflush(stdout);
}

void SiteReport(struct site xmtr)
{
	char	report_name[80];
	double	terrain;
	int	x, azi;
	FILE	*fd;

	sprintf(report_name,"%s-site_report.txt",xmtr.name);

	for (x=0; report_name[x]!=0; x++)
		if (report_name[x]==32 || report_name[x]==17 || report_name[x]==92 || report_name[x]==42 || report_name[x]==47)
			report_name[x]='_';	

	fd=fopen(report_name,"w");

	fprintf(fd,"\n\t--==[ SPLAT! v%s Site Analysis Report For: %s ]==--\n\n",splat_version,xmtr.name);

	fprintf(fd,"---------------------------------------------------------------------------\n\n");

	if (xmtr.lat>=0.0)
	{
		fprintf(fd,"Site location: %.4f North / %.4f West",xmtr.lat, xmtr.lon);
		fprintf(fd, " (%s N / ",dec2dms(xmtr.lat));
	}

	else
	{
		fprintf(fd,"Site location: %.4f South / %.4f West",-xmtr.lat, xmtr.lon);
		fprintf(fd, " (%s S / ",dec2dms(xmtr.lat));
	}

	fprintf(fd, "%s W)\n",dec2dms(xmtr.lon));

	if (metric)
	{
		fprintf(fd,"Ground elevation: %.2f meters AMSL\n",METERS_PER_FOOT*GetElevation(xmtr));
		fprintf(fd,"Antenna height: %.2f meters AGL / %.2f meters AMSL\n",METERS_PER_FOOT*xmtr.alt, METERS_PER_FOOT*(xmtr.alt+GetElevation(xmtr)));
	}

	else
	{
		fprintf(fd,"Ground elevation: %.2f feet AMSL\n",GetElevation(xmtr));
		fprintf(fd,"Antenna height: %.2f feet AGL / %.2f feet AMSL\n",xmtr.alt, xmtr.alt+GetElevation(xmtr));
	}

	terrain=haat(xmtr);

	if (terrain>-4999.0)
	{
		if (metric)
			fprintf(fd,"Antenna height above average terrain: %.2f meters\n\n",METERS_PER_FOOT*terrain);
		else
			fprintf(fd,"Antenna height above average terrain: %.2f feet\n\n",terrain);

		/* Display the average terrain between 2 and 10 miles
		   from the transmitter site at azimuths of 0, 45, 90,
		   135, 180, 225, 270, and 315 degrees. */

		for (azi=0; azi<=315; azi+=45)
		{
			fprintf(fd,"Average terrain at %3d degrees azimuth: ",azi);
			terrain=AverageTerrain(xmtr,(double)azi,2.0,10.0);

			if (terrain>-4999.0)
			{
				if (metric)
					fprintf(fd,"%.2f meters AMSL\n",METERS_PER_FOOT*terrain);
				else
					fprintf(fd,"%.2f feet AMSL\n",terrain);
			}

			else
				fprintf(fd,"No terrain\n");
		}
	}

	fprintf(fd,"\n---------------------------------------------------------------------------\n\n");
	fclose(fd);
	fprintf(stdout,"\nSite analysis report written to: \"%s\"\n",report_name);
}

void LoadTopoData(int max_lon, int min_lon, int max_lat, int min_lat)
{
	/* This function loads the SDF files required
	   to cover the limits of the region specified. */ 

	int	x, y, width, ymin, ymax;

	width=ReduceAngle(max_lon-min_lon);

	if ((max_lon-min_lon)<=180.0)
	{
		for (y=0; y<=width; y++)
			for (x=min_lat; x<=max_lat; x++)
			{
				ymin=(int)(min_lon+(double)y);

				while (ymin<0)
					ymin+=360;

				while (ymin>=360)
					ymin-=360;

				ymax=ymin+1;

				while (ymax<0)
					ymax+=360;

				while (ymax>=360)
					ymax-=360;

				sprintf(string,"%d:%d:%d:%d",x, x+1, ymin, ymax);
				LoadSDF(string);
			}
	}

	else
	{
		for (y=0; y<=width; y++)
			for (x=min_lat; x<=max_lat; x++)
			{
				ymin=max_lon+y;

				while (ymin<0)
					ymin+=360;

				while (ymin>=360)
					ymin-=360;
					
				ymax=ymin+1;

				while (ymax<0)
					ymax+=360;

				while (ymax>=360)
					ymax-=360;

				sprintf(string,"%d:%d:%d:%d",x, x+1, ymin, ymax);
				LoadSDF(string);
			}
	}
}

int LoadPLI(char *filename)
{
	int	error=0, max_west, min_west, max_north, min_north;
	char	string[80], *pointer=NULL;
	double	latitude=0.0, longitude=0.0, azimuth=0.0, elevation=0.0,
		loss=0.0;
	FILE	*fd;

	fd=fopen(filename,"r");

	if (fd!=NULL)
	{
		fgets(string,78,fd);
		pointer=strchr(string,';');

		if (pointer!=NULL)
			*pointer=0;

		sscanf(string,"%d, %d",&max_west, &min_west);

		fgets(string,78,fd);
		pointer=strchr(string,';');

		if (pointer!=NULL)
			*pointer=0;

		sscanf(string,"%d, %d",&max_north, &min_north);

		fgets(string,78,fd);
		pointer=strchr(string,';');

		if (pointer!=NULL)
			*pointer=0;

		LoadTopoData(max_west-1, min_west, max_north-1, min_north);

		fprintf(stdout,"\nReading \"%s\"... ",filename);
		fflush(stdout);

		fscanf(fd,"%lf, %lf, %lf, %lf, %lf",&latitude, &longitude, &azimuth, &elevation, &loss);

		while (feof(fd)==0)
		{
			if (loss>225.0)
				loss=225.0;

			if (loss<75.0)
				loss=75.0;

			loss-=75.0;
			loss/=10.0;
			loss+=1.0;

			if (loss<=(double)maxdB)
				OrMask(latitude,longitude,((unsigned char)(loss))<<3);

			fscanf(fd,"%lf, %lf, %lf, %lf, %lf",&latitude, &longitude, &azimuth, &elevation, &loss);
		}

		fclose(fd);

		fprintf(stdout," Done!\n");
		fflush(stdout);
	}

	else
		error=1;

	return error;
}

void WriteKML(struct site source, struct site destination)
{
	int	x, y;
	char	block, report_name[80];
	double	distance, rx_alt, tx_alt, cos_xmtr_angle,
		azimuth, cos_test_angle, test_alt;
	FILE	*fd=NULL;

	ReadPath(source,destination);

	sprintf(report_name,"%s-to-%s.kml",source.name,destination.name);

	for (x=0; report_name[x]!=0; x++)
		if (report_name[x]==32 || report_name[x]==17 || report_name[x]==92 || report_name[x]==42 || report_name[x]==47)
			report_name[x]='_';	

	fd=fopen(report_name,"w");

	fprintf(fd,"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
	fprintf(fd,"<kml xmlns=\"http://earth.google.com/kml/2.0\">\n");
	fprintf(fd,"<!-- Generated by SPLAT! Version %s -->\n",splat_version);
	fprintf(fd,"<Folder>\n");
	fprintf(fd,"<name>SPLAT! Path</name>\n");
	fprintf(fd,"<open>1</open>\n");
	fprintf(fd,"<description>Path Between %s and %s</description>\n",source.name,destination.name);

	fprintf(fd,"<Placemark>\n");
	fprintf(fd,"    <name>%s</name>\n",source.name);
	fprintf(fd,"    <description>\n");
	fprintf(fd,"       Transmit Site\n");

	if (source.lat>=0.0)
		fprintf(fd,"       <BR>%s North</BR>\n",dec2dms(source.lat));
	else
		fprintf(fd,"       <BR>%s South</BR>\n",dec2dms(source.lat));

	fprintf(fd,"       <BR>%s West</BR>\n",dec2dms(source.lon));

	azimuth=Azimuth(source,destination);
	distance=Distance(source,destination);

	if (metric)
		fprintf(fd,"       <BR>%.2f km",distance*KM_PER_MILE);
	else
		fprintf(fd,"       <BR>%.2f miles",distance);

	fprintf(fd," to %s</BR>\n       <BR>toward an azimuth of %.2f%c</BR>\n",destination.name,azimuth,176);

	fprintf(fd,"    </description>\n");
	fprintf(fd,"    <visibility>1</visibility>\n");
	fprintf(fd,"    <Style>\n");
	fprintf(fd,"      <IconStyle>\n");
	fprintf(fd,"        <Icon>\n");
	fprintf(fd,"          <href>root://icons/palette-5.png</href>\n");
	fprintf(fd,"          <x>224</x>\n");
	fprintf(fd,"          <y>224</y>\n");
	fprintf(fd,"          <w>32</w>\n");
	fprintf(fd,"          <h>32</h>\n");
	fprintf(fd,"        </Icon>\n");
	fprintf(fd,"      </IconStyle>\n");
	fprintf(fd,"    </Style>\n");
	fprintf(fd,"    <Point>\n");
	fprintf(fd,"      <extrude>1</extrude>\n");
	fprintf(fd,"      <altitudeMode>relativeToGround</altitudeMode>\n");
	fprintf(fd,"      <coordinates>%f,%f,30</coordinates>\n",(source.lon<180.0?-source.lon:360.0-source.lon),source.lat);
	fprintf(fd,"    </Point>\n");
	fprintf(fd,"</Placemark>\n");

	fprintf(fd,"<Placemark>\n");
	fprintf(fd,"    <name>%s</name>\n",destination.name);
	fprintf(fd,"    <description>\n");
	fprintf(fd,"       Receive Site\n");

	if (destination.lat>=0.0)
		fprintf(fd,"       <BR>%s North</BR>\n",dec2dms(destination.lat));
	else
		fprintf(fd,"       <BR>%s South</BR>\n",dec2dms(destination.lat));

	fprintf(fd,"       <BR>%s West</BR>\n",dec2dms(destination.lon));


	if (metric)
		fprintf(fd,"       <BR>%.2f km",distance*KM_PER_MILE);
	else
		fprintf(fd,"       <BR>%.2f miles",distance);

	fprintf(fd," to %s</BR>\n       <BR>toward an azimuth of %.2f%c</BR>\n",source.name,Azimuth(destination,source),176);

	fprintf(fd,"    </description>\n");
	fprintf(fd,"    <visibility>1</visibility>\n");
	fprintf(fd,"    <Style>\n");
	fprintf(fd,"      <IconStyle>\n");
	fprintf(fd,"        <Icon>\n");
	fprintf(fd,"          <href>root://icons/palette-5.png</href>\n");
	fprintf(fd,"          <x>224</x>\n");
	fprintf(fd,"          <y>224</y>\n");
	fprintf(fd,"          <w>32</w>\n");
	fprintf(fd,"          <h>32</h>\n");
	fprintf(fd,"        </Icon>\n");
	fprintf(fd,"      </IconStyle>\n");
	fprintf(fd,"    </Style>\n");
	fprintf(fd,"    <Point>\n");
	fprintf(fd,"      <extrude>1</extrude>\n");
	fprintf(fd,"      <altitudeMode>relativeToGround</altitudeMode>\n");
	fprintf(fd,"      <coordinates>%f,%f,30</coordinates>\n",(destination.lon<180.0?-destination.lon:360.0-destination.lon),destination.lat);
	fprintf(fd,"    </Point>\n");
	fprintf(fd,"</Placemark>\n");

	fprintf(fd,"<Placemark>\n");
	fprintf(fd,"<name>Point-to-Point Path</name>\n");
	fprintf(fd,"  <visibility>1</visibility>\n");
	fprintf(fd,"  <open>0</open>\n");
	fprintf(fd,"  <Style>\n");
	fprintf(fd,"    <LineStyle>\n");
	fprintf(fd,"      <color>7fffffff</color>\n");
	fprintf(fd,"    </LineStyle>\n");
	fprintf(fd,"    <PolyStyle>\n");
	fprintf(fd,"       <color>7fffffff</color>\n");
	fprintf(fd,"    </PolyStyle>\n");
	fprintf(fd,"  </Style>\n");
	fprintf(fd,"  <LineString>\n");
	fprintf(fd,"    <extrude>1</extrude>\n");
	fprintf(fd,"    <tessellate>1</tessellate>\n");
	fprintf(fd,"    <altitudeMode>relativeToGround</altitudeMode>\n");
	fprintf(fd,"    <coordinates>\n");

	for (x=0; x<path.length; x++)
		fprintf(fd,"      %f,%f,5\n",(path.lon[x]<180.0?-path.lon[x]:360.0-path.lon[x]),path.lat[x]);

	fprintf(fd,"    </coordinates>\n");
	fprintf(fd,"   </LineString>\n");
	fprintf(fd,"</Placemark>\n");

	fprintf(fd,"<Placemark>\n");
	fprintf(fd,"<name>Line-of-Sight Path</name>\n");
	fprintf(fd,"  <visibility>1</visibility>\n");
	fprintf(fd,"  <open>0</open>\n");
	fprintf(fd,"  <Style>\n");
	fprintf(fd,"    <LineStyle>\n");
	fprintf(fd,"      <color>ff00ff00</color>\n");
	fprintf(fd,"    </LineStyle>\n");
	fprintf(fd,"    <PolyStyle>\n");
	fprintf(fd,"       <color>7f00ff00</color>\n");
	fprintf(fd,"    </PolyStyle>\n");
	fprintf(fd,"  </Style>\n");
	fprintf(fd,"  <LineString>\n");
	fprintf(fd,"    <extrude>1</extrude>\n");
	fprintf(fd,"    <tessellate>1</tessellate>\n");
	fprintf(fd,"    <altitudeMode>relativeToGround</altitudeMode>\n");
	fprintf(fd,"    <coordinates>\n");

	/* Walk across the "path", indentifying obstructions along the way */

	for (y=0; y<path.length; y++)
	{
		distance=5280.0*path.distance[y];
		tx_alt=earthradius+source.alt+path.elevation[0];
		rx_alt=earthradius+destination.alt+path.elevation[y];

		/* Calculate the cosine of the elevation of the
		   transmitter as seen at the temp rx point. */

		cos_xmtr_angle=((rx_alt*rx_alt)+(distance*distance)-(tx_alt*tx_alt))/(2.0*rx_alt*distance);

		for (x=y, block=0; x>=0 && block==0; x--)
		{
			distance=5280.0*(path.distance[y]-path.distance[x]);
			test_alt=earthradius+path.elevation[x];

			cos_test_angle=((rx_alt*rx_alt)+(distance*distance)-(test_alt*test_alt))/(2.0*rx_alt*distance);

			/* Compare these two angles to determine if
			   an obstruction exists.  Since we're comparing
			   the cosines of these angles rather than
			   the angles themselves, the following "if"
			   statement is reversed from what it would
			   be if the actual angles were compared. */

			if (cos_xmtr_angle>cos_test_angle)
				block=1;
		}

		if (block)
			fprintf(fd,"      %f,%f,-30\n",(path.lon[y]<180.0?-path.lon[y]:360.0-path.lon[y]),path.lat[y]);
		else
			fprintf(fd,"      %f,%f,5\n",(path.lon[y]<180.0?-path.lon[y]:360.0-path.lon[y]),path.lat[y]);
	}

	fprintf(fd,"    </coordinates>\n");
	fprintf(fd,"  </LineString>\n");
	fprintf(fd,"</Placemark>\n");

	fprintf(fd,"    <LookAt>\n");
	fprintf(fd,"      <longitude>%f</longitude>\n",(source.lon<180.0?-source.lon:360.0-source.lon));
	fprintf(fd,"      <latitude>%f</latitude>\n",source.lat);
	fprintf(fd,"      <range>300.0</range>\n");
	fprintf(fd,"      <tilt>45.0</tilt>\n");
	fprintf(fd,"      <heading>%f</heading>\n",azimuth);
	fprintf(fd,"    </LookAt>\n");

	fprintf(fd,"</Folder>\n");
	fprintf(fd,"</kml>\n");

	fclose(fd);

	fprintf(stdout, "KML file written to: \"%s\"\n",report_name);

	fflush(stdout);
}

int main(char argc, char *argv[])
{
	int		x, y, z=0, min_lat, min_lon, max_lat, max_lon,
			rxlat, rxlon, txlat, txlon, west_min, west_max,
			north_min, north_max;

	unsigned char	coverage=0, LRmap=0, ext[20], terrain_plot=0,
			elevation_plot=0, height_plot=0, 
			longley_plot=0, cities=0, bfs=0, txsites=0,
			count, report='y', norm=0, topomap=0, geo=0,
			kml=0;
 
	char		mapfile[255], header[80], city_file[5][255], 
			elevation_file[255], height_file[255], 
			longley_file[255], terrain_file[255],
			string[255], rxfile[255], *env=NULL,
			txfile[255], map=0, boundary_file[5][255],
			udt_file[255], rxsite=0, plo_filename[255],
			pli_filename[255], nf=0;

	double		altitude=0.0, altitudeLR=0.0, tx_range=0.0,
			rx_range=0.0, deg_range=0.0, deg_limit,
			deg_range_lon, er_mult, freq=0.0;

	struct		site tx_site[4], rx_site;

	FILE		*fd;


	if (argc==1)
	{
		fprintf(stdout,"\n\t\t --==[ SPLAT! v%s Available Options... ]==--\n\n",splat_version);
		fprintf(stdout,"       -t txsite(s).qth (max of 4)\n");
		fprintf(stdout,"       -r rxsite.qth\n");
		fprintf(stdout,"       -c plot coverage of TX(s) with an RX antenna at X feet/meters AGL\n");
		fprintf(stdout,"       -L plot path loss map of TX based on an RX at X feet/meters AGL\n");
		fprintf(stdout,"       -s filename(s) of city/site file(s) to import (5 max)\n");
		fprintf(stdout,"       -b filename(s) of cartographic boundary file(s) to import (max of 5)\n");
		fprintf(stdout,"       -p filename of terrain profile graph to plot\n");
		fprintf(stdout,"       -e filename of terrain elevation graph to plot\n");
		fprintf(stdout,"       -h filename of terrain height graph to plot\n");
		fprintf(stdout,"       -H filename of normalized terrain height graph to plot\n");
		fprintf(stdout,"       -l filename of Longley-Rice graph to plot\n");
		fprintf(stdout,"       -o filename of topographic map to generate (.ppm)\n");
		fprintf(stdout,"       -u filename of user-defined terrain file to import\n");
		fprintf(stdout,"       -d sdf file directory path (overrides path in ~/.splat_path file)\n");
		fprintf(stdout,"       -n no analysis, brief report\n");
		fprintf(stdout,"       -N no analysis, no report\n");
		fprintf(stdout,"       -m earth radius multiplier\n");
		fprintf(stdout,"       -f frequency for Fresnel zone calculation (MHz)\n");
		fprintf(stdout,"       -R modify default range for -c or -L (miles/kilometers)\n");
		fprintf(stdout,"      -db maximum loss contour to display on path loss maps (80-230 dB)\n");
		fprintf(stdout,"      -nf do not plot Fresnel zones in height plots\n");
		fprintf(stdout,"     -plo filename of path-loss output file\n");
		fprintf(stdout,"     -pli filename of path-loss input file\n");
		fprintf(stdout,"     -udt filename of user defined terrain input file\n");
		fprintf(stdout,"     -geo generate an Xastir .geo georeference file (with .ppm output)\n");
		fprintf(stdout,"     -kml generate a Google Earth .kml file (for point-to-point links)\n");
		fprintf(stdout,"  -metric employ metric rather than imperial units for all user I/O\n\n");

		fprintf(stdout,"If that flew by too fast, consider piping the output through 'less':\n");
		fprintf(stdout,"\n\tsplat | less\n\n");
		fprintf(stdout,"Type 'man splat', or see the documentation for more details.\n\n");
		fflush(stdout);
		return 1;
	}

	y=argc-1;

	metric=0;
	rxfile[0]=0;
	txfile[0]=0;
	string[0]=0;
	mapfile[0]=0;
	elevation_file[0]=0;
	terrain_file[0]=0;
	sdf_path[0]=0;
	udt_file[0]=0;
	path.length=0;
	LR.frq_mhz=0.0;
	rx_site.lat=91.0;
	rx_site.lon=361.0;
	plo_filename[0]=0;
	pli_filename[0]=0;
	earthradius=EARTHRADIUS;

	sprintf(header,"\n\t\t--==[ Welcome To SPLAT! v%s ]==--\n\n", splat_version);

	for (x=0; x<4; x++)
	{
		tx_site[x].lat=91.0;
		tx_site[x].lon=361.0;
	}

	for (x=0; x<MAXSLOTS; x++)
	{
		dem[x].min_el=32768;
		dem[x].max_el=-32768;
		dem[x].min_north=90;
		dem[x].max_north=-90;
		dem[x].min_west=360;
		dem[x].max_west=-1;
	}

	/* Scan for command line arguments */

	for (x=1; x<=y; x++)
	{
		if (strcmp(argv[x],"-R")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				sscanf(argv[z],"%lf",&max_range);

				if (max_range<0.0)
					max_range=0.0;

				if (max_range>1000.0)
					max_range=1000.0;
			}			 
		}

		if (strcmp(argv[x],"-m")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				sscanf(argv[z],"%lf",&er_mult);

				if (er_mult<0.1)
					er_mult=1.0;

				if (er_mult>1.0e6)
					er_mult=1.0e6;

				earthradius*=er_mult;
			}			 
		}

		if (strcmp(argv[x],"-o")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
				strncpy(mapfile,argv[z],253);
			map=1;
		}

		if (strcmp(argv[x],"-u")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
				strncpy(udt_file,argv[z],253);
		}

		if (strcmp(argv[x],"-c")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				sscanf(argv[z],"%lf",&altitude);
				coverage=1;
			}
		}

		if (strcmp(argv[x],"-db")==0 || strcmp(argv[x],"-dB")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				sscanf(argv[z],"%d",&maxdB);

				maxdB=abs(maxdB);

				if (maxdB<80)
					maxdB=80;

				if (maxdB>230)
					maxdB=230;
			}			 
		}

		if (strcmp(argv[x],"-p")==0)
		{ 
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				strncpy(terrain_file,argv[z],253);
				terrain_plot=1;
			}
		}

		if (strcmp(argv[x],"-e")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				strncpy(elevation_file,argv[z],253);
				elevation_plot=1;
			}
		}

		if (strcmp(argv[x],"-h")==0 || strcmp(argv[x],"-H")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				strncpy(height_file,argv[z],253);
				height_plot=1;
			}

			if (strcmp(argv[x],"-H")==0)
				norm=1;
			else
				norm=0;
		}

		if (strcmp(argv[x],"-n")==0)
		{
			report='n';
			map=1;
		}

		if (strcmp(argv[x],"-N")==0)
		{
			report='N';
			map=1;
		}

		if (strcmp(argv[x],"-metric")==0)
			metric=1;

		if (strcmp(argv[x],"-geo")==0)
			geo=1;

		if (strcmp(argv[x],"-kml")==0)
			kml=1;

		if (strcmp(argv[x],"-nf")==0)
			nf=1;

		if (strcmp(argv[x],"-d")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
				strncpy(sdf_path,argv[z],253);
		}

		if (strcmp(argv[x],"-t")==0)
		{
			/* Read Transmitter Location */

			z=x+1;

			while (z<=y && argv[z][0] && argv[z][0]!='-' && txsites<4)
			{
				strncpy(txfile,argv[z],253);
				tx_site[txsites]=LoadQTH(txfile);
				txsites++;
				z++;
			}

			z--;
		}

		if (strcmp(argv[x],"-L")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				sscanf(argv[z],"%lf",&altitudeLR);

				if (coverage)
					fprintf(stdout,"c and L are exclusive options, ignoring L.\n");
				else
				{
					LRmap=1;
					ReadLRParm(txfile);
				} 
			}
		}

		if (strcmp(argv[x],"-l")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				strncpy(longley_file,argv[z],253);
				longley_plot=1;
				/* Doing this twice is harmless */
				ReadLRParm(txfile);
			}
		}

		if (strcmp(argv[x],"-r")==0)
		{
			/* Read Receiver Location */

			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				strncpy(rxfile,argv[z],253);
				rx_site=LoadQTH(rxfile);
				rxsite=1;
			}
		}

		if (strcmp(argv[x],"-s")==0)
		{
			/* Read city file(s) */

			z=x+1;

			while (z<=y && argv[z][0] && argv[z][0]!='-' && cities<5)
			{
				strncpy(city_file[cities],argv[z],253);
				cities++;
				z++;
			}

			z--;
		}

		if (strcmp(argv[x],"-b")==0)
		{
			/* Read Boundary File(s) */

			z=x+1;

			while (z<=y && argv[z][0] && argv[z][0]!='-' && bfs<5)
			{
				strncpy(boundary_file[bfs],argv[z],253);
				bfs++;
				z++;
			}

			z--;
		}
		
		if (strcmp(argv[x],"-f")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
			{
				sscanf(argv[z],"%lf",&freq);

				if (freq<20)
					freq=20;

				if (freq>20e3)
					freq=20e3;
			}			 
		}

		if (strcmp(argv[x],"-plo")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
				strncpy(plo_filename,argv[z],253);
		}

		if (strcmp(argv[x],"-pli")==0)
		{
			z=x+1;

			if (z<=y && argv[z][0] && argv[z][0]!='-')
				strncpy(pli_filename,argv[z],253);
		}
	}

	/* Perform some error checking on the arguments
	   and switches parsed from the command-line.
	   If an error is encountered, print a message
	   and exit gracefully. */

	if (txsites==0)
	{
		fprintf(stderr,"\n%c*** ERROR: No transmitter site(s) specified!\n\n",7);
		exit (-1);
	}

	for (x=0, y=0; x<txsites; x++)
	{
		if (tx_site[x].lat==91.0 && tx_site[x].lon==361.0)
		{
			fprintf(stderr,"\n*** ERROR: Transmitter site #%d not found!",x+1);
			y++;
		}
	}

	if (y)
	{
		fprintf(stderr,"%c\n\n",7);
		exit (-1);
	}

	if ((coverage+LRmap+pli_filename[0])==0 && rx_site.lat==91.0 && rx_site.lon==361.0)
	{
		if (max_range!=0.0 && txsites!=0)
		{
			/* Plot topographic map of radius "max_range" */

			map=0;
			topomap=1;
			report='N';
		}

		else
		{
			fprintf(stderr,"\n%c*** ERROR: No receiver site found or specified!\n\n",7);
			exit (-1);
		}
	}

	/* No major errors were detected.  Whew!  :-) */

	/* Adjust input parameters if -metric option is used */

	if (metric)
	{
		altitudeLR/=METERS_PER_FOOT;	/* meters --> feet */
		max_range/=KM_PER_MILE;		/* kilometers --> miles */
		altitude/=METERS_PER_FOOT;	/* meters --> feet */
	}

	/* If no SDF path was specified on the command line (-d), check
	   for a path specified in the $HOME/.splat_path file.  If the
	   file is not found, then sdf_path[] remains NULL, and the
	   current working directory is assumed to contain the SDF
	   files. */

	if (sdf_path[0]==0)
	{
		env=getenv("HOME");
		sprintf(string,"%s/.splat_path",env);
		fd=fopen(string,"r");

		if (fd!=NULL)
		{
			fgets(string,253,fd);

			/* Remove <CR> and/or <LF> from string */

			for (x=0; string[x]!=13 && string[x]!=10 && string[x]!=0 && x<253; x++);
			string[x]=0;

			strncpy(sdf_path,string,253);

			fclose(fd);
		}
	}

	/* Ensure a trailing '/' is present in sdf_path */

	if (sdf_path[0])
	{
		x=strlen(sdf_path);

		if (sdf_path[x-1]!='/' && x!=0)
		{
			sdf_path[x]='/';
			sdf_path[x+1]=0;
		}
	}

	fprintf(stdout,"%s",header);
	fflush(stdout);

	if (pli_filename[0])
	{
		y=LoadPLI(pli_filename);

		for (x=0; x<txsites; x++)
			PlaceMarker(tx_site[x]);

		if (rxsite)
			PlaceMarker(rx_site);

		if (bfs)
		{
			for (x=0; x<bfs; x++)
				LoadBoundaries(boundary_file[x]);
		}

		if (cities)
		{
			for (x=0; x<cities; x++)
				LoadCities(city_file[x]);
		}

		WritePPMLR(mapfile,geo);

		exit(0);
	}

	x=0;
	y=0;

	min_lat=90;
	max_lat=-90;

	min_lon=(int)floor(tx_site[0].lon);
	max_lon=(int)floor(tx_site[0].lon);

	for (y=0, z=0; z<txsites; z++)
	{
		txlat=(int)floor(tx_site[z].lat);
		txlon=(int)floor(tx_site[z].lon);

		if (txlat<min_lat)
			min_lat=txlat;

		if (txlat>max_lat)
			max_lat=txlat;

		if (LonDiff(txlon,min_lon)<0.0)
			min_lon=txlon;

		if (LonDiff(txlon,max_lon)>0.0)
			max_lon=txlon;
	}

	if (rxsite)
	{
		rxlat=(int)floor(rx_site.lat);
		rxlon=(int)floor(rx_site.lon);

		if (rxlat<min_lat)
			min_lat=rxlat;

		if (rxlat>max_lat)
			max_lat=rxlat;

		if (LonDiff(rxlon,min_lon)<0.0)
			min_lon=rxlon;

		if (LonDiff(rxlon,max_lon)>0.0)
			max_lon=rxlon;
	}

	/* Load the required SDF files */ 

	LoadTopoData(max_lon, min_lon, max_lat, min_lat);

	if (coverage | LRmap | topomap)
	{
		if (LRmap)
			txsites=1;

		for (z=0; z<txsites; z++)
		{
			/* "Ball park" estimates used to load any additional
			   SDF files required to conduct this analysis. */

			tx_range=sqrt(1.5*(tx_site[z].alt+GetElevation(tx_site[z])));

			if (LRmap)
				rx_range=sqrt(1.5*altitudeLR);
			else
				rx_range=sqrt(1.5*altitude);

			/* deg_range determines the maximum
			   amount of topo data we read */

			deg_range=(tx_range+rx_range)/69.0;

			/* max_range sets the maximum size of the
			   analysis.  A small, non-zero amount can
			   be used to shrink the size of the analysis
			   and limit the amount of topo data read by
			   SPLAT!  A very large number will only increase
			   the width of the analysis, not the size of
			   the map. */

			if (max_range==0.0)
				max_range=tx_range+rx_range;

			if (max_range<(tx_range+rx_range))
				deg_range=max_range/69.0;

			/* Prevent the demand for a really wide coverage
			   from allocating more slots than are available
			   in memory. */

			switch (MAXSLOTS)
			{
				case 2: deg_limit=0.25;
					break;

				case 4: deg_limit=0.5;
					break;

				case 9: deg_limit=1.0;
					break;

				case 16: deg_limit=2.0;
					break;

				case 25: deg_limit=3.0;
			}

			if (tx_site[z].lat<70.0)
				deg_range_lon=deg_range/cos(deg2rad*tx_site[z].lat);
			else
				deg_range_lon=deg_range/cos(deg2rad*70.0);

			/* Correct for squares in degrees not being square in miles */  

			if (deg_range>deg_limit)
				deg_range=deg_limit;

			if (deg_range_lon>deg_limit)
				deg_range_lon=deg_limit;


			north_min=(int)floor(tx_site[z].lat-deg_range);
			north_max=(int)floor(tx_site[z].lat+deg_range);

			west_min=(int)floor(tx_site[z].lon-deg_range_lon);

			while (west_min<0)
				west_min+=360;

			while (west_min>=360)
				west_min-=360;

			west_max=(int)floor(tx_site[z].lon+deg_range_lon);

			while (west_max<0)
				west_max+=360;

			while (west_max>=360)
				west_max-=360;

			if (north_min<min_lat)
				min_lat=north_min;

			if (north_max>max_lat)
				max_lat=north_max;

			if (LonDiff(west_min,min_lon)<0.0)
				min_lon=west_min;

			if (LonDiff(west_max,max_lon)>0.0)
				max_lon=west_max;
		}

		/* Load any additional SDF files, if required */ 

		LoadTopoData(max_lon, min_lon, max_lat, min_lat);
	}

	if (udt_file[0])
		LoadUDT(udt_file);

	if (mapfile[0] && topomap==0)
		map=1;

	if (freq==0.0 && nf==0)
		freq=LR.frq_mhz;
	else
		freq=0.0;

	if (coverage | LRmap)
	{
		for (x=0; x<txsites; x++)
		{
			if (coverage)
				PlotCoverage(tx_site[x],altitude);

			if (LRmap)
				PlotLRMap(tx_site[x],altitudeLR,plo_filename);

			PlaceMarker(tx_site[x]);

			if (report!='N')
				SiteReport(tx_site[x]);
		}

		map=1;
	}

 	if (coverage==0 && LRmap==0)       
	{
		PlaceMarker(rx_site);

		for (x=0; x<txsites; x++)
		{
			PlaceMarker(tx_site[x]);

			if (report=='y')
			{
				switch (x)
				{
					case 0:
						PlotPath(tx_site[x],rx_site,1);
						break;

					case 1:
						PlotPath(tx_site[x],rx_site,8);
						break;

					case 2:
						PlotPath(tx_site[x],rx_site,16);
						break;

					case 3:
						PlotPath(tx_site[x],rx_site,32);
				}
			}

			if (report!='N')
				ObstructionReport(tx_site[x],rx_site,report,freq);

			if (kml)
				WriteKML(tx_site[x],rx_site);
		}
	}

	if (map | topomap)
	{
		if (bfs)
		{
			for (x=0; x<bfs; x++)
				LoadBoundaries(boundary_file[x]);
		}

		if (cities)
		{
			for (x=0; x<cities; x++)
				LoadCities(city_file[x]);
		}

		if (LRmap)
			WritePPMLR(mapfile,geo);
		else
			WritePPM(mapfile,geo);
	}

	if (terrain_plot)
	{
		if (txsites>1)
		{
			for (x=0; terrain_file[x]!='.' && terrain_file[x]!=0 && x<80; x++);

			if (terrain_file[x]=='.')  /* extension */
			{
				ext[0]='.';
				for (y=1, z=x, x++; terrain_file[x]!=0 && x<253 && y<14; x++, y++)
					ext[y]=terrain_file[x];

				ext[y]=0;
				terrain_file[z]=0;
			}

			else
			{
				ext[0]=0;  /* No extension */
				terrain_file[x]=0;
			}

			for (count=0; count<txsites; count++)
			{
				sprintf(string,"%s-%c%s%c",terrain_file,'1'+count,ext,0);
				GraphTerrain(tx_site[count],rx_site,string);
			}
		}

		else
			GraphTerrain(tx_site[0],rx_site,terrain_file);
	}

	if (elevation_plot)
	{
		if (txsites>1)
		{
			for (x=0; elevation_file[x]!='.' && elevation_file[x]!=0 && x<80; x++);

			if (elevation_file[x]=='.')  /* extension */
			{
				ext[0]='.';
				for (y=1, z=x, x++; elevation_file[x]!=0 && x<253 && y<14; x++, y++)
					ext[y]=elevation_file[x];

				ext[y]=0;
				elevation_file[z]=0;
			}

			else
			{
				ext[0]=0;  /* No extension */
				elevation_file[x]=0;
			}

			for (count=0; count<txsites; count++)
			{
				sprintf(string,"%s-%c%s%c",elevation_file,'1'+count,ext,0);
				GraphElevation(tx_site[count],rx_site,string);
			}
		}

		else
			GraphElevation(tx_site[0],rx_site,elevation_file);
	}

	if (height_plot)
	{
		if (txsites>1)
		{
			for (x=0; height_file[x]!='.' && height_file[x]!=0 && x<80; x++);

			if (height_file[x]=='.')  /* extension */
			{
				ext[0]='.';
				for (y=1, z=x, x++; height_file[x]!=0 && x<253 && y<14; x++, y++)
					ext[y]=height_file[x];

				ext[y]=0;
				height_file[z]=0;
			}

			else
			{
				ext[0]=0;  /* No extension */
				height_file[x]=0;
			}

			for (count=0; count<txsites; count++)
			{
				sprintf(string,"%s-%c%s%c",height_file,'1'+count,ext,0);
				GraphHeight(tx_site[count],rx_site,string,freq,norm);
			}
		}

		else
			GraphHeight(tx_site[0],rx_site,height_file,freq,norm);
	}
	
	if (longley_plot)
	{
		if (txsites>1)
		{
			for (x=0; longley_file[x]!='.' && longley_file[x]!=0 && x<80; x++);

			if (longley_file[x]=='.')  /* extension */
			{
				ext[0]='.';
				for (y=1, z=x, x++; longley_file[x]!=0 && x<253 && y<14; x++, y++)
					ext[y]=longley_file[x];

				ext[y]=0;
				longley_file[z]=0;
			}

			else
			{
				ext[0]=0;  /* No extension */
				longley_file[x]=0;
			}

			for (count=0; count<txsites; count++)
			{
				sprintf(string,"%s-%c%s%c",longley_file,'1'+count,ext,0);
				GraphLongley(tx_site[count],rx_site,string);
			}
		}

		else
			GraphLongley(tx_site[0],rx_site,longley_file);
	}

	return 0;
}

