#!/bin/sh
#
#	srecord - manipulate eprom load files
#	Copyright (C) 2001, 2003 Peter Miller;
#	All rights reserved.
#
#	This program is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with this program; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
#
# MANIFEST: Test the DEC Binary functionality
#
here=`pwd`
if test $? -ne 0 ; then exit 2; fi
work=${TMP_DIR-/tmp}/$$

pass()
{
	cd $here
	rm -rf $work
	echo PASSED
	exit 0
}

fail()
{
	cd $here
	rm -rf $work
	echo 'FAILED test of the DEC Binary functionality'
	exit 1
}

no_result()
{
	cd $here
	rm -rf $work
	echo 'NO RESULT for test of the DEC Binary functionality'
	exit 2
}

trap "no_result" 1 2 3 15

bin=$here/${1-.}/bin
mkdir $work
if test $? -ne 0; then no_result; fi
cd $work
if test $? -ne 0; then no_result; fi

# -------------------------------------------------------------------------
#
# Test reading the format.
#
# (This is a tad verbose, but it's a "real" DEC Binary file,
# found in the wild on the Internet.)
#
cat > test.in.srec << 'fubar'
S00600004844521B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S9030000FC
fubar
if test $? -ne 0; then no_result; fi

$bin/srec_cat test.in.srec -o test.in -bin
if test $? -ne 0; then no_result; fi

cat > test.ok << 'fubar'
S00600004844521B
S11F00005F00DC000002E0000202E0000402E0000602E0000A02E0000802E0003B
S12300408A13C000000000000000000000000000000000000000000000000000000000003F
S123006000000000000000000000000000000000000000000000000000000000000000007C
S123008000000000000000000000000000000000000000000000CE000000000000009E00F0
S11500A000000000000000000000000000000000B0009A
S12300D0DE1380004A1480003202E0000500C0152A18C115FC1F01E0810C810CD01500F7C5
S11300F0100AC10AFB06C015E034C115FE3F01E039
S5030007F5
S9030000FC
fubar
if test $? -ne 0; then no_result; fi

$bin/srec_cat test.in -decbin -crop 0 0x100 -o test.out -header HDR
if test $? -ne 0; then fail; fi

diff test.ok test.out
if test $? -ne 0; then fail; fi

# --------------------------------------------------------------------------
#
# Test writing the format.
#
$bin/srec_cat test.in -decbin -o test.out -decbin -multi > LOG 2>&1
if test $? -ne 0; then cat LOG; fail; fi

$bin/srec_cmp test.in -decbin test.out -decbin
if test $? -ne 0; then fail; fi

#
# The things tested here, worked.
# No other guarantees are made.
#
pass
