/* Copyright (C) 2000-2005  Thomas Bopp, Thorsten Hampel, Robert Hinn
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */

inherit "/kernel/module";
import httplib;

#include <macros.h>
#include <database.h>
#include <attributes.h>
#include <classes.h>

#define WIKITRAIL_SIZE 6

class WikiContext {
    object filepath;
    object      obj;
    array footnotes = ({ });
    int footnote_cnt = 0;

    void create(object fp, object o) {
	filepath = fp;
	obj = o;
    }
    object resolve_path(object obj, string path) {
	return filepath->resolve_path(obj, path);
    }
    string object_to_filename(object obj) {
	return filepath->object_to_filename(obj);
    }
    object path_to_object(string path) {
	return filepath->path_to_object(path);
    }
    string object_to_path(object obj) {
	return filepath->object_to_path(obj);
    }
    int add_footnote(string desc) {
	footnotes += ({ desc });
	footnote_cnt++;
	return footnote_cnt;
    }
}

string html_show_annotations(array(object) annotations)
{
    array(object) next_annotations;
    string html = "<UL>\n";
    
    foreach ( annotations, object annotation ) {
	if ( !objectp(annotation) ) continue;
	    
	next_annotations = annotation->get_annotations();
	html += "<LI><strong>"+annotation->query_attribute(OBJ_NAME)+"</strong> ( id: " + annotation->get_object_id()+") by "+
	    "<i><a href=\"JavaScript:open_userdetails('" + 
	    annotation->get_creator()->get_object_id() +"');\">"+
	    annotation->get_creator()->get_identifier() + "</a>"+
	    "</i> on "+
	    ctime(annotation->query_attribute(OBJ_CREATION_TIME))+
	    "<BR/>\n";
	switch ( annotation->query_attribute(DOC_MIME_TYPE) ) {
	case "text/plain":
	case "text/html":
	    html += annotation->get_content();
	    break;
	case "image/jpeg":
	case "image/gif":
	    html += "<image src=\"/scripts/get.pike?object="+annotation->get_object_id()+"\"/>\n";
	    break;
	default:
	    html += "<a href=\"/scripts/get.pike?object="+annotation->get_object_id()+"\">"+annotation->get_identifier()+"</a>\n";
	    break;
	}
	
	html += "</LI>\n";
	    annotation->get_content()+"</LI>\n";
	html += html_show_annotations(next_annotations);
    }
    html += "</UL>\n";
    return html;
}

string pikeWiki(object obj, object fp, string pcode)
{
  string result;

  sscanf(pcode, "@@PIKE%s@@", pcode);
  string code = "#include <macros.h>\n#include <attributes.h>\n#include <classes.h>\n#include <database.h>\nmixed exec(object env, object fp) {\n" + pcode + "\n}";
  
  object e = master()->ErrorContainer();
  master()->set_inhibit_compile_errors(e);
  mixed err = catch { 
    program prg = compile_string(code); 
    object o = new(prg);
    result = o->exec(obj, fp);
  };
  if ( err != 0 ) {
    return "<!-- error calling function:"+sprintf("%O\n%O\n",err, e->get())+"-->\n";
  }
  return result;
}


string embedWiki(object obj, object fp, string embed)
{
  string link, prefix, args;
  mapping      vars = ([ ]);
  
  if ( sscanf(embed, "{%s}", link) == 0 )
    return "<!-- unable to embed " + embed + " -->\n";

  while ( sscanf(link, "%s:%s", prefix, link) >= 1 ) {
    object namespace = get_wiki_room(prefix, obj);
    if ( !objectp(namespace) )
      return "<!-- Wiki Prefix not found: "+prefix+"-->\n";
    if ( objectp(namespace) )
      obj = namespace;
  }
  if ( sscanf(link, "%s$%s", link, args) == 2 ) {
    array params = args / ",";
    foreach(params, string param) {
      string key, val;
      if ( sscanf(param, "%s=%s", key, val) == 2 )
	vars[key] = val;
    }
  }

  object o;
  int  oid;

  if ( sscanf(link, "#%d", oid) ) {
    o = find_object(oid);
  }
  else {
    o = fp->resolve_path(obj, link);
  }
  if ( !objectp(o) )
    return "<!-- object " + embed + " not found !-->\n";
  
  if ( o->get_object_class() & CLASS_SCRIPT )
    return o->execute( vars );


  string mtype = o->query_attribute(DOC_MIME_TYPE);

  if ( mtype == "image/svg+xml" ) {
    int h, w;
    h = o->query_attribute(OBJ_HEIGHT);
    w = o->query_attribute(OBJ_WIDTH);
    return sprintf("<embed height='%d' width='%d' type='%s' "+
		   " source='/scripts/get.pike?object='%d' />\n",
		   h, w, mtype, o->get_object_id());
  }
  if ( search(mtype, "image") >= 0 ) {
    return "<img src='"+replace_uml(fp->object_to_filename(o))+"'/>";
  }

  if ( mtype == "text/wiki" ) 
    return "<div class='embedded'>"+wiki_to_html_plain(o, fp, ([ ])) + "</div>";

  if ( search(mtype, "text") == -1 && search(mtype, "source") == -1 )
    return "<a href='"+replace_uml(fp->object_to_filename(o))+"'>"+
      o->get_identifier()+"</a>\n";


  string embed_class = replace(mtype, "/", "_");

  string maint, subt;
  sscanf(mtype, "%s/%s", maint, subt);

  return "<div class='embedded'><div class='"+maint+"'><div class='"+subt+"'>"+
    o->get_content()+"</div></div></div>\n";
}

string annotationWiki(object obj, object fp, string ann) 
{
  string     text;
  object        o;
  int          id;
  
  if ( sscanf(ann, "[%s[#%d]]", text, id) == 2 ) {
    o   = find_object(id);
    if ( objectp(o) ) {
      ann = o->get_content();
      replace(ann, "\"", "'");
    }
  }
  else {
    id = time();
    sscanf(ann, "[%s[%s]]", text, ann);
  }
  
  return sprintf("<div class='annotate'><div class='annotated'>%s</div><div class='annotation'>%s</div></div>", text, ann);
}

string tagWiki(object obj, object fp, string tagStr) 
{
    return replace(tagStr, ({ "<", ">" }), ({ "&lt;", "&gt;" }));
}

string linkInternalWiki(object obj, object fp, string link) 
{
  string desc, prefix, space, image;
  space = "";

  if ( strlen(link) > 1 && link[0] == ' ' ) {
    link = link[1..];
    space = " ";
  }
  link = String.trim_whites(link); // remove other whitespaces
  
  
  if ( sscanf(link, "[[%s|%s]]", link, desc) == 0 ) {
      sscanf(link, "[[%s]]", link);
      desc = link;
  }

  string footnote;
  if ( sscanf(link, "#%s", footnote) ) {
      int id = fp->add_footnote(desc);
      return "<a href=\"#"+id + "\" class=\"footnote\">"+id+"</a>\n";
  }


  object namespace, lnk; // namespace and link object

  while ( sscanf(link, "%s:%s", prefix, link) >= 1 ) {
    if ( prefix == "wikipedia" )
      return "<a href='http://de.wikipedia.org/wiki/"+link+"'>"+desc+"</a>";
    
    namespace = get_wiki_room(prefix, obj);

    if ( objectp(namespace) )
      obj = namespace;
    else {
      object factory = get_factory(CLASS_CONTAINER);
      object cont = factory->execute( ([ "name": prefix]) );
      cont->move(obj);
      obj = cont;
    }
  }
  if ( objectp(namespace) ) {
    if ( namespace->get_object_class() & CLASS_EXIT )
      lnk = namespace->get_exit();
    else if ( namespace->get_object_class() & CLASS_LINK )
      lnk = namespace->get_object();
    else
      lnk = namespace;
    // treat containers as several links
    if ( !stringp(link) || link == "" ) {
      int id = lnk->get_object_id();
      string html = "<div class='container'>\n"; 
      html += "<div class='title'>"+ 
	desc + "&nbsp;<img src='/images/unfold.gif' onClick='contClick("+id+
	", event);'/></div>\n";
      html += "<div class='inv' id='"+id+"'><div class='topbar'>"+
	"<img src='/images/closetop.gif' onClick='closeClick("+id+
	", event);'/></div>\n";
      
      array inv;
      if ( catch(inv = lnk->get_inventory_by_class(CLASS_DOCUMENT)) )
	return "<!-- unreadable container in " + link + "--></div></div>\n";
      foreach(inv, object o) {
	if ( !objectp(o) ) continue;
	switch(o->query_attribute(DOC_MIME_TYPE)) {
	case "text/wiki": 
	  string n;
	  sscanf(o->get_identifier(), "%s.%*s", n);
	  html += "<div class='wikilink'>\n";
	  html += "<a href='"+fp->object_to_filename(o)+"'>"+ n + "</a>\n";
	  html += "<div class='description'>"+o->query_attribute(OBJ_DESC)+
	    "</div>\n";
	  html += "</div>\n";
	  break;
	default:
	  html += "<div class='contlink'>\n";
	  html += "<a href='"+fp->object_to_filename(o)+"'>"+ 
	    o->get_identifier() + "</a>\n";
	  html += "<div class='description'>"+o->query_attribute(OBJ_DESC)+
	    "</div>\n";
	  html += "</div>\n";
	  break;
	}
      }
      html += "</div></div>\n";
      return html;
    }
  }
  

  if ( sscanf(desc, "{%s}", image) == 1 ) 
    desc = embedWiki(obj, fp, desc);
  
  if ( !stringp(link) || link == "" )
    lnk = obj;
  else
    lnk = fp->resolve_path(obj, link);

  mapping linkMap;
  catch(linkMap = fp->obj->query_attribute("OBJ_WIKILINKS"));
  if ( !mappingp(linkMap) )
      linkMap = ([ ]);

  if ( !objectp(lnk) ) {
      // try to find the link object if any
      if ( objectp(linkMap[link]) ) {
	  lnk = linkMap[link];
	  link = fp->object_to_filename(lnk);
      }
  }
  else {
      linkMap[link] = lnk;
  }

  if ( !objectp(lnk) || lnk->get_object_class() & CLASS_CONTAINER ) {
      link += ".wiki";  
      lnk = fp->resolve_path(obj, link);
      if ( !objectp(lnk) ) {
	  if ( objectp(linkMap[link]) ) {
	      lnk = linkMap[link];
	      link = fp->object_to_filename(lnk);
	  }
      }
      else
	  linkMap[link]= lnk;
  }
  
  catch(fp->obj->set_attribute("OBJ_WIKILINKS", linkMap));
  

  if ( !objectp(lnk) ) {
    string wiki_edit = "/scripts/wikiedit.pike";
    
    link = replace_uml(fp->object_to_filename(obj) + "/"+link);
    object wedit = fp->path_to_object("/scripts/wikiedit.pike");
    if ( objectp(wedit) )
      wiki_edit = replace_uml(fp->object_to_filename(wedit));
    return sprintf("%s<a class=\"%s\" href=\"%s\">%s</a>",
		   space, "notexistant",
		   wiki_edit+"?path="+ link+"&mode=create", desc);
  }
  catch(lnk->add_reference(fp->obj));
  if ( objectp(namespace) )
    link = fp->object_to_filename(lnk);
  link = replace_uml(link);
  return sprintf("%s<a class=\"%s\" href=\"%s\">%s</a>%s",
		 space, "internal", link, desc, (!objectp(lnk)?"?":""));
}

string hyperlinkWiki(object obj, object fp, string link) 
{
  string dest, desc;
  if ( sscanf(link, "[%s|%s]", dest, desc) != 2 && 
       sscanf(link, "[%s %s]", dest, desc) != 2 ) 
  {
    sscanf(link, "[%s]", link);
    dest = link;
    desc = link;
  }
  if ( sscanf(desc, "{%*s}") )
    desc = embedWiki(obj, fp, desc);

  return "<a class=\"external\" href=\""+dest+"\">"+desc+"</a>";
}

string barelinkWiki(object obj, object fp, string link) 
{
  return "<a class=\"external\" href=\""+link+"\">"+link+"</a>";
}

string imageWiki(object obj, object fp, string link) 
{
  string img, alt;
  if ( sscanf(link, "[[image:%s|%s]]", img, alt) != 2 )
    sscanf(link, "[[image:%s]]", img);
  
  return "<img alt=\""+alt+"\" src=\""+img+"\">";
}

static string add_footnotes(object wikiContext) 
{
  string html = "";
  
  if ( wikiContext->footnote_cnt > 0 ) {
    html += "<hr/>";
      for ( int i = 0; i < wikiContext->footnote_cnt; i++ ) {
	html += 
	  sprintf(
		  "<div class=\"footnote\">"+
		  "<a name=\"%d\" class=\"footnote\">%d</a>"+
		  "%s</div>", 
		  (i+1), (i+1), wikiContext->footnotes[i]);
      }
  }
  return html;
}

static array add_wiki_trail(object user, object doc)
{
  array wikiTrail = user->query_attribute(USER_WIKI_TRAIL);
  if ( !arrayp(wikiTrail) )
      wikiTrail = ({ });
  wikiTrail -= ({ doc });
  
  wikiTrail += ({ doc });
  
  if ( sizeof(wikiTrail) > WIKITRAIL_SIZE )
      wikiTrail = wikiTrail[..WIKITRAIL_SIZE-1];
  
  user->set_attribute(USER_WIKI_TRAIL, wikiTrail);
  return wikiTrail;
}

string wiki_to_html_plain(object doc, void|object fp, void|mapping vars)
{
  if ( doc->query_attribute(DOC_MIME_TYPE) != "text/wiki" )
    return "<!-- Source Document is not a wiki file !-->\n";

  if ( !objectp(fp) )
    fp = _FILEPATH;

  add_wiki_trail(this_user(), doc);

  object WParser = wiki.Parser(this_object());
  object wikiroom = doc->get_environment();

  wikiroom = get_room_environment(wikiroom);
  string res;
  string content = doc->get_content();
  object context = WikiContext(fp, doc);
  if ( stringp(content) ) {
    string enc = doc->query_attribute(DOC_ENCODING);
    if ( enc == "iso-8859-1" || !xml.utf8_check(content) )
      content = string_to_utf8(content);
     
     res = WParser->wiki_to_html(wikiroom, context, content);
     res += add_footnotes(context);
  }
  res = "<div class='html'>"+res+"</div>";
  return res;
}

string wiki_to_html(object doc, void|object fp, void|mapping vars)
{
  object env = doc->get_environment();
  string envname = "";
  if ( objectp(env) )
    envname = env->get_identifier();
  // get the room if we are currently in a container
  object wikiroom = get_room_environment(env);
  string wikipath = "";
  if ( objectp(wikiroom) )
    wikipath = fp->object_to_filename(wikiroom);
  string title = doc->get_identifier();
  if ( objectp(env) && wikiroom != env )
    title = env->get_identifier() + ": " + title;

  string html = "<html><head>"+
    "<meta content=\"text/html; charset=utf-8\" http-equiv=\"Content-Type\">"+
    "<link rel=\"alternate\" type=\"application/rss+xml\" title=\""+doc->get_identifier()+"\" href=\"/scripts/rss.pike?feed="+doc->get_object_id()+"\"/>\n"+
    "<LINK REL=\"stylesheet\" HREF=\"/gui_css/default/wiki.css\"/>"+
    "<script src='/gui_js/wiki.js' type='text/javascript'>\n"+
    "<script src='/gui_js/main.js' type='text/javascript'>\n"+
    "<title>WIKI: "+title+"</title>"+
    "</head><body>";
  
  if ( !objectp(fp) )
    fp = _FILEPATH;

  object WParser = wiki.Parser(this_object());
  string path = replace_uml(fp->object_to_path(doc));
  object cuser = doc->query_attribute(DOC_USER_MODIFIED);
  object user = this_user();
  string content = doc->get_content();
  object wikiContext = WikiContext(fp, doc);

  array wikiTrail = add_wiki_trail(user, doc);

  html += 
    "<div class='engine_top'>"+
    "<img class='logo' src='/images/RoomWiki.gif' border='0'/>";
  html += //edit part
    "<div class='edit'>"+
    "<a href='/scripts/wikiedit.pike?path="+
    replace_uml(fp->object_to_filename(doc))+"'>EditText</a>"+
    " (last changed " + ctime(doc->query_attribute(DOC_LAST_MODIFIED)) +
      (objectp(cuser) ? " by <strong><a href='#' onClick=\"open_userdetails('"+
       cuser->get_object_id() + "');\">" +
       cuser->get_identifier() + "</a>": "") + "</strong>),"+
    "</div>";

  if ( this_user() != USER("guest") ) {
      html += "<div class='wiki_trail'>";
      foreach(wikiTrail, object trail) {
	  if ( !objectp(trail) )
	      continue;
	  html += "&nbsp; &gt; <a class='wiki_trail' href='"+ 
	      fp->object_to_filename(trail) + "'>" + trail->get_identifier() + 
	      "</a>";
      }
      html += "</div>";
  }

  html += "</div>";

  string result = "";
  if ( stringp(content) ) {
    string enc = doc->query_attribute(DOC_ENCODING);
    if ( (stringp(enc) && enc == "iso-8859-1") || !xml.utf8_check(content) )
      content = string_to_utf8(content);
    
    result = WParser->wiki_to_html(wikiroom, wikiContext, content);
  }
  
  html += "<div class='wiki_content'>\n"+result+"</div>\n";
  
  html += add_footnotes(wikiContext);
  html += "<div class='engine_bottom'>\n"+
    "<div class='room'>"+
    "<strong>" + (objectp(wikiroom) ? wikiroom->get_identifier() + "(<a href='"+wikipath+"'>"+wikipath+"</a>)": "none")+"</strong>&nbsp;"+
    "<a href='"+wikipath+"?type=wiki'>&Uuml;bersicht</a> &nbsp; &nbsp; <a href='?type=wiki'>Details</a>\n"+
    "</div><div class='search'>"+
    "<form action='/scripts/browser.pike' enctype='multipart/form-data' method='POST'>"+
    "<input type='hidden' name='_action' value='search' />"+
    "<input type='hidden' name='advsearch_mimetype' value='text/wiki' />"+
    "<input type='hidden' name='advsearch_objtype' value='document' />"+
    "Suche: <input type='text' name='keywords' />&nbsp;<input type='submit' value='Wiki finden'/>"+ 
    "</form></div>";

  html += "</div>";

  html += "<div class='annotations'>";
  html += html_show_annotations(doc->get_annotations());
  html += "</div></body></html>";
  return html;
}

object get_wiki_room(string wiki, void|object env)
{
  if ( objectp(env) ) {
    object room = env;
    object cont = room->get_object_byname(wiki);
    if ( objectp(cont) ) {
      if ( cont->get_object_class() & CLASS_EXIT )
	return cont->get_exit();
      return cont;
    }
  }
  return 0;
}

object get_room_environment(object obj)
{
  if ( !objectp(obj) )
    return 0;

  if ( !(obj->get_object_class() & CLASS_ROOM) )
    return obj->get_environment();
  else
    return obj;
}

string make_diff(object newv, object oldv)
{
  if ( !objectp(newv) )
    error("Param 1: No new version object found !");
  if ( !objectp(oldv) )
    error("Param 2: No old version object found !");

  int i = 0; // compare element #
  string diff = sprintf("Diff of %s (Versions %d and %d):\n\n",
			newv->get_identifier(), 
			newv->query_attribute(DOC_VERSION),
			oldv->query_attribute(DOC_VERSION));
  string first, second;
  first = newv->get_content();
  second = oldv->get_content();

}


string get_identifier() { return "wiki"; }

