/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: ldap.pike,v 1.8 2006/05/05 19:51:27 exodusd Exp $
 */

constant cvs_version="$Id: ldap.pike,v 1.8 2006/05/05 19:51:27 exodusd Exp $";

inherit "/kernel/module";

#include <configure.h>
#include <macros.h>
#include <config.h>
#include <attributes.h>
#include <classes.h>
#include <events.h>
#include <database.h>

//#define LDAP_DEBUG 1

#ifdef LDAP_DEBUG
#define LDAP_LOG(s, args...) werror("ldap: "+s+"\n", args)
#else
#define LDAP_LOG(s, args...)
#endif

#define DEPENDENCIES cache

//! This module is a ldap client inside sTeam. It reads configuration
//! parameters of sTeam to contact a ldap server.
//! All the user management can be done with ldap this way. Some
//! special scenario might require to modify the modules code to 
//! make it work.
//!
//! The configuration variables used are:
//! server - the server name to connect to (if none is specified, then ldap will not be used)
//! cacheTime - how long (in seconds) shall ldap entries be cached? (to reduce server requests)
//! reconnectTime - how long (in seconds) between reconnection attempts in case the LDAP server cannot be connected (if not set, or set to 0, then no reconnection attempts will be made)
//! user   - ldap user for logging in
//! password - the password for the user
//! base_dc - ldap base dc, consult ldap documentation
//! userdn - the dn path where new users are stored
//! groupdn - the dn path where new groups are stored
//! objectName - the ldap object name to be used for the search
//! userAttr - the attribute containing the user's login name
//! passwordAttr - the attribute containing the password
//! emailAttr - the attribute containing the user's email address
//! iconAttr - the attribute containing the user's icon
//! fullnameAttr - the attribute containing the user's surname
//! nameAttr - the attribute containing the user's first name
//! userClass - an attribute value that identifies users
//! userId - an attribute that contains a value that can be used to match users to groups
//! groupAttr - the attribute containing the group's name
//! groupClass - an attribute value that identifies groups
//! groupId - an attribute that contains a value that can be used to match users to groups
//! memberAttr - an attribute that contains a value that can be used to match users to groups
//! descriptionAttr - the attribute that contains a user or group description
//! notfound - defines what should be done if a user or group could not be found in LDAP:
//!            "create" : create and insert a new user in LDAP (at userdn)
//!            "ignore" : do nothing
//! sync - "true"/"false" : sync user or group data
//! authorize - "ldap" : authorize users through LDAP
//! bindUser - "true" : when authorizing users, try ldap bind first (you will need this if the ldap lookup doesn't return a password for the user)
//! requiredAttr : required attributes for creating new users in LDAP
//! adminAccount - "root" : sTeam account that will receive administrative mails about ldap
//! charset - "utf-8" : charset used in ldap server

static object      oLDAP;
static string sServerURL;
static mapping    config;

static object charset_decoder;
static object charset_encoder;

static object user_cache;
static object group_cache;
static object authorize_cache;
static int cache_time = 0;

static object admin_account = 0;
static array ldap_conflicts = ({ });

static bool reconnecting = false;


private bool bind_root ()
{
  if ( stringp(config["user"]) && sizeof(config["user"])>0 ) {
    if ( !objectp( oLDAP ) )
      return false;
    return oLDAP->bind("cn="+config["user"] + 
		       (stringp(config->root_dn) && strlen(config->root_dn)>0?","+config->root_dn:""),
		       config["password"]);
  }
  else return true;  // if no user is set, then we don't need to bind: just return true
}

private static void connect_ldap()
{
  if ( !stringp(config["server"]) )
    return;

  mixed err = catch {
    oLDAP = Protocols.LDAP.client(config["server"]);
    if ( objectp(oLDAP) && reconnecting )
    reconnecting = false;
  };
  if ( ! objectp(oLDAP) ) {
    mixed reconnect_time = config["reconnectTime"];
    if ( !intp(reconnect_time) ) reconnect_time = 0;
    string msg = "Failed to connect to LDAP server " + config["server"];
    if ( reconnect_time > 0 )
      msg += sprintf( ", trying to reconnect in %d seconds", reconnect_time );
    FATAL( msg );
    if ( reconnect_time < 1 ) return;
    // try reconnect:
    reconnecting = true;
    call_out( connect_ldap, reconnect_time );
    return;
  }
  err = catch {
    if ( ! bind_root() )
      throw( "Could not bind to server " + config["server"] );
    oLDAP->set_scope(2);
    oLDAP->set_basedn(config["base_dc"]);
  };
  if ( err != 0 ) {
    string error_msg = "";
    if ( oLDAP->error_number() > 0 )
      error_msg = "\n" + oLDAP->error_string();
    FATAL( "Failed to bind ldap" + error_msg );
  }
  MESSAGE( "Connected to LDAP: %s", config["server"] );
}

static void init_module()
{
    config = Config.read_config_file( _Server.get_config_dir()+"/modules/ldap.cfg", "ldap" );
    if ( !mappingp(config) ) {
	MESSAGE("LDAP Service not started - missing configuration !");
	return; // ldap not started !
    }
    if ( !stringp(config["server"]) ) return;  // ldap deactivated

    LDAP_LOG("configuration is %O", config);
    if ( stringp(config["charset"]) )
      charset_decoder = Locale.Charset.decoder( config["charset"] );
    else charset_decoder = 0;
    charset_encoder = Locale.Charset.encoder( "utf-8" );

    if ( intp(config->cacheTime) )
	cache_time = config->cacheTime;
    else if ( !stringp(config->cacheTime) || (sscanf(config->cacheTime, "%d", cache_time) < 1) )
	cache_time = 0;
    object cache_module = get_module("cache");
    if ( objectp(cache_module) ) {
      user_cache = get_module("cache")->create_cache( "ldap:users", cache_time );
      group_cache = get_module("cache")->create_cache("ldap:groups", cache_time );
      authorize_cache = get_module("cache")->create_cache("ldap:auth", cache_time );
    }

    if ( !config->objectName )
      steam_error("objectName configuration missing !");

    connect_ldap();
/*
    // if our main dc does not exist - create it
    oLDAP->add(config->base_dc, 
	       ([ 
		 "objectclass": ({ "dcObject", "organization" }),
		 "o": "sTeam Authorization Directory",
		 "dc": "steam"
	       ]));
*/
}

void load_module()
{
  add_global_event(EVENT_USER_CHANGE_PW, sync_password, PHASE_NOTIFY);
  add_global_event(EVENT_USER_NEW_TICKET, sync_ticket, PHASE_NOTIFY);
}

private static bool notify_admin ( string msg ) {
  if ( zero_type( config["adminAccount"] ) ) return false;
  object admin = USER( config["adminAccount"] );
  if ( !objectp(admin) ) admin = GROUP( config["adminAccount"] );
  if ( !objectp(admin) ) return false;
  string msg_text = "The following LDAP situation occured on the server "
    + _Server->get_server_name() + " at " + ctime(time()) + " :\n" + msg;
  admin->mail( msg_text, "LDAP on " + _Server->get_server_name(), 0, "text/plain" );
  return true;
}

static mixed map_results(object results)
{
  array result = ({ });
  
  for ( int i = 1; i <= results->num_entries(); i++ ) {
    mapping            data = ([ ]);
    mapping res = results->fetch(i);
    
    foreach(indices(res), string attr) {
      if ( arrayp(res[attr]) ) {
	if ( sizeof(res[attr]) == 1 )
	  data[attr] = res[attr][0];
	else
	  data[attr] = res[attr];
      }
    }
    if ( results->num_entries() == 1 )
      return data;
    result += ({ data });
  }
  return result;
}

mapping search_user ( string search_str, void|string user, void|string pass )
{
    mapping udata = ([ ]);
    object results;
    
    object ldap;
    if(Config.bool_value(config->bindUser) && user && pass)
        ldap = bind_user(user, pass);
    if(!objectp(ldap))
        ldap = oLDAP;
    if(!objectp(ldap))
        return UNDEFINED;

    LDAP_LOG("looking up user in LDAP: %s\n", search_str);
    if ( config->userdn )
    {
      ldap->set_basedn(config->userdn+","+config->base_dc);
      if ( ldap->error_number() )
        werror( "LDAP: %s", oLDAP->error_string() );
    }

    if ( catch(results = 
	       ldap->search(search_str)) )
    {
      // something went wrong. if we were using the global connection
      // try rebuilding it and retry:
      if(!Config.bool_value(config->bindUser)) {
        connect_ldap(); // try to rebuild connection;
	if ( catch(results = ldap->search(search_str)) )
	  return UNDEFINED;  // still doesn't work, so return
      }
    }

    if ( !objectp(results) ) {
        werror( "LDAP: invalid results: %O\n", results );
	return UNDEFINED;
    }
      
    if ( ldap->error_number() )
      werror("LDAP: Error while searching user: %s\n", ldap->error_string());
    
    if ( results->num_entries() == 0 ) {
      LDAP_LOG("user not found in LDAP directory: %s", search_str);
      return UNDEFINED;
    }
    udata = map_results(results);

    /*
    if ( stringp(udata[config->passwordAttr]) )
      sscanf(udata[config->passwordAttr], 
	     "{crypt}%s", udata[config->passwordAttr]);
    */
    return udata;
}


mixed fix_charset ( string|mapping|array v )
{
  if ( !objectp(charset_encoder) || !objectp(charset_decoder) ) return v;
  if ( stringp(v) ) {
    if ( xml.utf8_check(v) ) return v;  // already utf-8
    string tmp = charset_decoder->feed(v)->drain();
    tmp = charset_encoder->feed(tmp)->drain();
    //LDAP_LOG( "charset conversion: from \"%s\" to \"%s\".", v, tmp );
    return tmp;
  }
  else if ( arrayp(v) ) {
    array tmp = ({ });
    foreach ( v, mixed i )
      tmp += ({ fix_charset(i) });
    return tmp;
  }
  else if ( mappingp(v) ) {
    mapping tmp = ([ ]);
    foreach ( indices(v), mixed i )
      tmp += ([ fix_charset(i) : fix_charset(v[i]) ]);
    return tmp;
  }
}


mapping fetch_user ( string identifier, void|string pass )
{
  if ( !objectp(oLDAP) ) return UNDEFINED;
  if ( !stringp(identifier) )
    return UNDEFINED;

  if ( !objectp(user_cache) )
    user_cache = get_module("cache")->create_cache( "ldap:users", cache_time );

  LDAP_LOG("fetch_user(%s)", identifier);
  return user_cache->get( identifier, lambda(){ return fix_charset( search_user( "("+config->userAttr+"="+identifier+")", identifier, pass ) ); } );
}

mapping search_group ( string search_str )
{
  mapping gdata = ([ ]);
  object results;

  if ( !objectp(oLDAP) )
    return UNDEFINED;
  
  if ( !stringp(config->groupAttr) || sizeof(config->groupAttr)<1 )
    return UNDEFINED;

  if ( config->groupdn )
    oLDAP->set_basedn(config->groupdn+"," + config->base_dc);

  if ( catch(results = oLDAP->search(search_str)) ) {
    connect_ldap(); // try to rebuild connection;
    return UNDEFINED;
  }
  
  if ( oLDAP->error_number() )
    FATAL("Error: %s", oLDAP->error_string());
  
  if ( results->num_entries() == 0 ) {
    LDAP_LOG("LDAP: Group not found in LDAP directory: %s", search_str);
    return UNDEFINED;
  }
  gdata = map_results(results);
  return gdata;
}

mapping fetch_group ( string identifier )
{
  if ( !objectp(oLDAP) ) return UNDEFINED;
  if ( !stringp(identifier) )
    return UNDEFINED;

  if ( !stringp(config->groupAttr) || sizeof(config->groupAttr)<1 )
    return UNDEFINED;

  if ( !objectp(group_cache) )
    group_cache = get_module("cache")->create_cache("ldap:groups", cache_time );

  return group_cache->get( identifier, lambda(){ return fix_charset( search_group( "("+config->groupAttr+"="+identifier+")" ) ); } );
}

mapping fetch ( string dn, string pattern )
{
    if ( !objectp(oLDAP) )
	return UNDEFINED;

  // caller must be module...

  mapping   data;
  object results;

  if ( !_Server->is_module(CALLER) )
    steam_error("Access for non-module denied !");

  if ( stringp(dn) && sizeof(dn)>0 )
    oLDAP->set_basedn(dn+"," + config->base_dc);
  else
    oLDAP->set_basedn(config->base_dc);
  
  if ( catch(results = oLDAP->search(pattern)) )
    {
      connect_ldap(); // try to rebuild connection;
      return UNDEFINED;
    }
  
  if ( oLDAP->error_number() )
    FATAL("Error: %s", oLDAP->error_string());
  
  if ( results->num_entries() == 0 ) {
    return UNDEFINED;
  }
  data = map_results(results);
  return data;
}

static bool check_password(string pass, string user_pw)
{
  if ( !stringp(pass) || !stringp(user_pw) )
    return false;

  LDAP_LOG("check_password()");
  if ( user_pw[0..4] == "{SHA}" )
    return user_pw[5..] == MIME.encode_base64( sha_hash(pass) );
  if ( user_pw[0..6] == "{crypt}" )
    return crypt(pass, user_pw[7..]);
  return verify_crypt_md5(pass, user_pw);
}

object bind_user(string user, string pass)
{
    LDAP_LOG("bind_user(%s/%s)", (string)config["server"], user);
    if(!stringp(config["server"]) || config["server"] == "")
      return UNDEFINED;
    object ldap = Protocols.LDAP.client(config["server"]);
    if(objectp(ldap))
    {
        string userdn="";
        if(config->userdn)
          userdn=config->userdn+",";
        if(ldap->bind(sprintf("%s=%s,%s%s", config->userAttr, user, userdn, 
                                            config->base_dc), pass))
        {
          LDAP_LOG("bind successfull");
          ldap->set_scope(2);
          return ldap;
	}
    }
    else
    {
      LDAP_LOG("connect failed.");
      return UNDEFINED;
    }
}

bool authorize_ldap ( object user, string pass )
{
  if ( config->authorize != "ldap" )
    return false;
  
  if ( !stringp(config["server"]) )  // ldap disabled
    return false;

  if ( !objectp(user) )
    steam_error("User object expected for authorization !");

  if ( !stringp(pass) || sizeof(pass)<1 )
    return false;

  string uname = user->get_user_name();

  // don't authorize restricted users:
  if ( _Persistence->user_restricted( uname ) ) return false;

  if ( !objectp(authorize_cache) )
    authorize_cache = get_module("cache")->create_cache("ldap:auth", cache_time );

  string cached = authorize_cache->get( uname );
  if (stringp(cached)) {
      if ( cached == "*" ) {  // "*" means: authorized via ldap bind
	LDAP_LOG("user %s LDAP cache authorized (cached bind)", uname);
	return true;
      }
      if ( check_password( pass, cached ) ) 
      {
        LDAP_LOG("user %s LDAP cache authorized", uname);
        return true;
      }
      else 
      {
        LDAP_LOG("user %s found in LDAP cache - password failed: %O", uname, cached);
        return false;
      }
  }
  
  // try authorizing via bind:
  if ( Config.bool_value(config->bindUser) && stringp(uname) && stringp(pass) ) {
    if ( bind_user(uname, pass) != 0 ) {
      authorize_cache->put( uname, "*" );  // "*" means: authorized via ldap bind
      return true;
    }
  }

  // fetch user data and authorize via password:
  mapping udata = fetch_user(uname, pass);

  if ( mappingp(udata) ) {
    string dn = udata["dn"];
    if ( !stringp(dn) ) dn = "";

    // check for conflicts (different user in sTeam than in LDAP):
    if (config->checkConflicts && !stringp(user->query_attribute("ldap:dn")) ) {
      if ( search(ldap_conflicts,uname)<0 ) {
	ldap_conflicts += ({ uname });
        if ( notify_admin(
            "Dear LDAP administrator at "+_Server->get_server_name()
	    +",\n\nthere has been a conflict between LDAP and sTeam:\n"
	    +"User \""+uname+"\" already exists in sTeam, but now "
	    +"there is also an LDAP user with the same name/id.\nYou "
	    +"will need to remove/rename one of them or, if they are "
	    +"the same user, you can overwrite the sTeam data from LDAP "
	    +"by adding a \"dn\" attribute to the sTeam user." ) );
	else
	  werror( "LDAP: user conflict: %s in sTeam vs. %s in LDAP\n", uname, dn );
	return false;
      }
    }
    else if ( search(ldap_conflicts,uname) >= 0 )
      ldap_conflicts -= ({ uname });

    if ( check_password( pass, udata[config->passwordAttr] ) ) {
      // need to synchronize passwords from ldap if ldap is down ?!
      // this is only done when the ldap password is received
      if ( udata[config->passwordAttr] != user->get_user_password()) {
	//catch(LDAP_LOG("sync PW: %s:%s", udata[config->passwordAttr],
	//      user->get_user_password()));
	user->set_user_password(udata[config->passwordAttr], 1);
      }
      authorize_cache->put( uname, udata[config->passwordAttr] );

      LDAP_LOG("user %s LDAP authorized !", uname);
      return true;
    }
    else {
      LDAP_LOG("user %s found in LDAP directory - password failed!", uname);
      return false;
    }
  }
  LDAP_LOG("user " + uname + " was not found in LDAP directory.");
  // if notfound configuration is set to create, then we should create
  // a user.
  if ( config->notfound == "create" )
    add_user(uname, pass, user);
  return false;
}

object sync_user(string name)
{
  // don't sync restricted users:
  if ( _Persistence->user_restricted( name ) ) return UNDEFINED;

  mapping udata = fetch_user(name);
  if ( !mappingp(udata) ) return UNDEFINED;

  LDAP_LOG("sync of ldap user \"%s\": %O", name, udata);
  object user = get_module("users")->get_value(name);
  if ( objectp(user) ) {
    // update user date from LDAP
    if ( ! user->set_attributes( ([
             "pw" : udata[config->passwordAttr],
	     "email" : udata[config->emailAttr],
	     "fullname" : udata[config->fullnameAttr],
	     "firstname" : udata[config->nameAttr],
	     "OBJ_DESC" : udata[config->descriptionAttr],
	   ]) ) )
      werror( "LDAP: Could not sync user attributes with ldap for \"%s\".\n", name );
  } else {
    // create new user to match LDAP user
    object factory = get_factory(CLASS_USER);
    user = factory->execute( ([
	     "name" : name,
	     "pw" : udata[config->passwordAttr],
	     "email" : udata[config->emailAttr],
	     "fullname" : udata[config->fullnameAttr],
	     "firstname" : udata[config->nameAttr],
	     "OBJ_DESC" : udata[config->descriptionAttr],
	   ]) );
    user->set_user_password(udata[config->passwordAttr], 1);
    user->activate_user(factory->get_activation());
  }
  // sync group membership:
  if ( objectp( user ) ) {
    string primaryGroupId = udata[config->groupId];
    if ( stringp( primaryGroupId ) ) {
      mapping group = search_group("("+config->groupId+"="+primaryGroupId+")");
    }
  }

  return user;
}

object sync_group(string name)
{
  // don't syncronize restricted groups:
  if ( _Persistence->group_restricted( name ) ) return null;

  mapping gdata = fetch_group(name);
  if ( !mappingp(gdata) ) return UNDEFINED;
  LDAP_LOG("sync of ldap group: %O", gdata);
  //object group = get_module("groups")->lookup(name);
  object group = get_module("groups")->get_value(name);
  if ( objectp(group) ) {
//TODO: check and update group memberships
    // update group date from LDAP
    group->set_attributes( ([
	     "OBJ_DESC" : gdata[config->descriptionAttr],
	   ]) );
  } else {
    // create new group to match LDAP user
    object factory = get_factory(CLASS_GROUP);
    group = factory->execute( ([
	      "name": name,
	      "OBJ_DESC" : gdata[config->descriptionAttr],
	    ]) );
  }
  return group;
}

static void sync_password(int event, object user, object caller)
{
  if ( !mappingp(config) || !stringp(config["server"]) 
       || config["server"] == "" || !Config.bool_value(config->sync) )
    return;
  string oldpw = user->get_old_password();
  string crypted = user->get_user_password();
  string name = user->get_user_name();
  // don't sync password for restricted users:
  if ( _Persistence->group_restricted( name ) ) return;
  LDAP_LOG("password sync for " + user->get_user_name());

  object ldap;
  string dn;

  if(Config.bool_value(config->bindUser) && oldpw && (ldap=bind_user(name, oldpw)))
  {
    if(config->userdn)
      dn = sprintf("%s=%s,%s,%s", config->userAttr, name, config->userdn, config->base_dc);
    else
      dn = sprintf("%s=%s,%s", config->userAttr, name, config->base_dc);
  }
  else
  {
    ldap = oLDAP;
      dn = config->base_dc + " , " + config->userAttr + "=" + name;
  }

  mixed err;
  if(crypted[..2]=="$1$")
    crypted="{crypt}"+crypted;
  err=catch(ldap->modify(dn, ([ config->passwordAttr: ({ 2,crypted }),])));
  authorize_cache->remove( name );
  user_cache->remove( name );
  LDAP_LOG("sync_password(): %s - %s - %O\n", crypted, dn, ldap->error_string());
}

static void sync_ticket(int event, object user, object caller, string ticket)
{
  mixed err;
  if ( !mappingp(config) || !Config.bool_value(config->sync) ) return;
  string name = user->get_user_name();
  string dn = config->base_dc + " , " + config->userAttr + "=" + name;
  err=catch(oLDAP->modify(dn, ([ "userCertificate": ({ 2,ticket }),])));
}

bool is_user(string user)
{
    object results = oLDAP->search("("+config["userAttr"]+"="+user+")");
    return (results->num_entries() > 0);
}

static bool add_user(string name, string password, object user)
{
  if ( !objectp(oLDAP) ) return false;

  // don't add restricted users:
  if ( _Persistence->group_restricted( name ) ) return false;
  
  string fullname = user->get_name();
  string firstname = user->query_attribute(USER_FIRSTNAME);
  string email = user->query_attribute(USER_EMAIL);
  
  mapping attributes = ([
    config["userAttr"]: ({ name }),
    config["fullnameAttr"]: ({ fullname }),
    "objectClass": ({ config["objectName"] }),
    config["passwordAttr"]: ({ make_crypt_md5(password) }),
  ]);
  if ( stringp(firstname) && strlen(firstname) > 0 )
    config["nameAttr"] = ({ firstname });
  if ( stringp(email) && strlen(email) > 0 )
    config["emailAttr"] = ({ email });

  array(string) requiredAttributes =  config["requiredAttr"];
  
  if ( arrayp(requiredAttributes) && sizeof(requiredAttributes) > 0 ) {
    foreach(requiredAttributes, string attr) {
      if ( zero_type(attributes[attr]) )
	attributes[attr] = ({ "-" });
    }
  }
  
  oLDAP->add(config["userAttr"]+"="+name+","+config["base_dc"], attributes);
  int err = oLDAP->error_number();
  if ( err != 0 )
    FATAL("Failed to add user , error number is " + oLDAP->error_string());
  return oLDAP->error_number() == 0;
}

string get_identifier() { return "ldap"; }
