/*
 *
 * s y s t e m . c				-- System relative primitives
 *
 * Copyright  1994-2003 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 *
 * Permission to use, copy, modify, distribute,and license this
 * software and its documentation for any purpose is hereby granted,
 * provided that existing copyright notices are retained in all
 * copies and that this notice is included verbatim in any
 * distributions.  No written agreement, license, or royalty fee is
 * required for any of the authorized uses.
 * This software is provided ``AS IS'' without express or implied
 * warranty.
 *
 *           Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date: 29-Mar-1994 10:57
 * Last file update:  5-Jun-2003 16:28 (eg)
 */

#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <time.h> 
#include "stklos.h"

#ifndef MAXBUFF
#  define MAXBUFF 1024
#endif

/******************************************************************************
 *
 * Utilities
 *
 ******************************************************************************/
static void error_bad_path(SCM path)
{
  STk_error("~S is a bad pathname", path);
}

static void error_bad_string(SCM path)
{
  STk_error("~S is a bad string", path);
}

static void error_bad_int_or_out_of_bounds(SCM val)
{
  STk_error("bad integer ~S (or out of range)", val);
}

static void error_cannot_copy(SCM f1, SCM f2)
{
  STk_error("cannot copy file ~S to ~S", f1, f2);
}


static SCM my_access(SCM path, int mode)
{
  if (!STRINGP(path)) error_bad_path(path);
  return MAKE_BOOLEAN(access(STRING_CHARS(path), mode) == 0);
}


int STk_dirp(const char *path)
{
  struct stat buf;
  
  if (stat(path, &buf) >= 0) 
    return S_ISDIR(buf.st_mode);
  return FALSE;
}



#ifdef FIXME
//EG: void STk_whence(char *exec, char *path)
//EG: {
//EG:   char *p, *q, dir[MAX_PATH_LENGTH];
//EG:   struct stat buf;
//EG:  
//EG:   if (ISABSOLUTE(exec)) {
//EG:     strncpy(path, exec, MAX_PATH_LENGTH);
//EG:     return;
//EG:   }
//EG:   
//EG:   /* the executable path may be specified by relative path from the cwd. */
//EG:   /* Patch suggested by Shiro Kawai <shiro@squareusa.com> */
//EG:   if (strchr(exec, DIRSEP) != NULL) {
//EG:     getcwd(dir, MAX_PATH_LENGTH);
//EG:     sprintf(dir + strlen(dir), "%c%s", DIRSEP, exec);
//EG:     absolute(dir, path);
//EG:     return;
//EG:   }
//EG: 
//EG: #ifdef FREEBSD 
//EG:   /* I don't understand why this is needed */
//EG:   if (access(path, X_OK) == 0) {
//EG:     stat(path, &buf);
//EG:     if (!S_ISDIR(buf.st_mode)) return;
//EG:   }  
//EG: #endif
//EG: 
//EG:   p = getenv("PATH");
//EG:   if (p == NULL) {
//EG:     p = "/bin:/usr/bin";
//EG:   }
//EG: 
//EG:   while (*p) {
//EG:     /* Copy the stuck of path in dir */
//EG:     for (q = dir; *p && *p != PATHSEP; p++, q++) *q = *p;
//EG:     *q = '\000';
//EG: 
//EG:     if (!*dir) { 
//EG:       /* patch suggested by Erik Ostrom <eostrom@vesuvius.ccs.neu.edu> */
//EG:       getcwd(path, MAX_PATH_LENGTH);
//EG:       sprintf(path + strlen(path), "%c%s", DIRSEP, exec);
//EG:     }
//EG:     else
//EG:       sprintf(path, "%s%c%s", dir, DIRSEP, exec);
//EG: 
//EG:     sprintf(path, "%s%c%s", dir, DIRSEP, exec);
//EG:     if (access(path, X_OK) == 0) {
//EG:       stat(path, &buf);
//EG:       if (!S_ISDIR(buf.st_mode)) return;
//EG:     }
//EG: 	 
//EG:     /* Try next path */
//EG:     if (*p) p++;
//EG:   }
//EG:   /* Not found. Set path to "" */
//EG:   path[0] = '\0';
//EG: }
#endif

/******************************************************************************
 *
 * Primitives
 *
 ******************************************************************************/

/*
<doc EXT expand-file-name
 * (expand-file-name path)
 * 
 * @cindex tilde expansion
 * |Expand-file-name| expands the filename given in |path| to
 * an absolute path. 
 * @lisp
 *   ;; Current directory is ~eg/STklos (i.e. /users/eg/STklos)
 *   (expand-file-name "..")            => "/users/eg"
 *   (expand-file-name "~eg/../eg/bin") => "/users/eg/bin"
 *   (expand-file-name "~/STklos)"      => "/users/eg/STk"
 * @end lisp 
doc>
*/

DEFINE_PRIMITIVE("expand-file-name", expand_fn, subr1, (SCM s))
{
  ENTER_PRIMITIVE(expand_fn);
  if (!STRINGP(s)) error_bad_string(s);
  return STk_Cstring2string(STk_expand_file_name(STRING_CHARS(s)));
}


/*
<doc EXT canonical-file-name
 * (canonical-file-name path)
 *
 * Expands all symbolic links in |path| and returns its canonicalized
 * absolute path name. The resulting path does not have symbolic links. 
 * If |path| doesn't designate a valid path name, |canonical-file-name| 
 * returns |f|.
doc>
*/
DEFINE_PRIMITIVE("canonical-file-name", canonical_path, subr1, (SCM str))
{
  ENTER_PRIMITIVE(canonical_path);

  if (!STRINGP(str)) error_bad_string(str);
  return STk_resolve_link(STRING_CHARS(str), 0);
}

/*
<doc EXT getcwd
 * (getcwd)
 *
 * Returns a string containing the current working directory.
doc>
*/
DEFINE_PRIMITIVE("getcwd", getcwd, subr0, (void))
{
  char buf[MAX_PATH_LENGTH], *s;
  SCM z;

  ENTER_PRIMITIVE(getcwd);

  s = getcwd(buf, MAX_PATH_LENGTH);
  if (!s) STk_error("cannot determine current directory");
  z = STk_Cstring2string(buf);

  return z;
}


/*
<doc EXT chdir
 * (chdir dir)
 *
 * Changes the current directory to the directory given in string |dir|.
doc>
*/
DEFINE_PRIMITIVE("chdir", chdir, subr1, (SCM s))
{
  ENTER_PRIMITIVE(chdir);

  if (!STRINGP(s)) error_bad_path(s);
  
  if (chdir(STk_expand_file_name(STRING_CHARS(s))))
    STk_error("cannot change directory to ~S", s);
 
  return STk_void;
}


/*
<doc EXT getpid
 * (getpid)
 *
 * Returns the system process number of the current program (i.e. the
 * Unix @i{pid}) as an integer.
doc>
*/
DEFINE_PRIMITIVE("getpid", getpid, subr0, (void))
{
  return (MAKE_INT((int) getpid()));
}


/*
<doc EXT system
 * (system string)
 *
 * Sends the given |string| to the system shell |/bin/sh|. The result of
 * |system| is the integer status code the shell returns.
doc>
*/
DEFINE_PRIMITIVE("system", system, subr1, (SCM com))
{
  ENTER_PRIMITIVE(system);
  
  if (!STRINGP(com)) error_bad_string(com);
  return MAKE_INT(system(STRING_CHARS(com)));
}

/*
<doc EXT file-is-directory? file-is-regular? file-is-writable? file-is-readable? file-is-executable? file-exists?
 * (file-is-directory?  string)
 * (file-is-regular?    string)
 * (file-is-readable?   string)
x * (file-is-writable?   string)
 * (file-is-executable? string)
 * (file-exists?        string)
 *
 * Returns |#t| if the predicate is true for the path name given in
 * |string|; returns |#f| otherwise (or if |string| denotes a file
 * which does not exist).
doc>
 */
DEFINE_PRIMITIVE("file-is-directory?", file_is_directoryp, subr1, (SCM f))
{
  struct stat info;

  ENTER_PRIMITIVE(file_is_directoryp);

  if (!STRINGP(f)) error_bad_path(f);
  if (stat(STRING_CHARS(f), &info) != 0) return STk_false;

  return MAKE_BOOLEAN((S_ISDIR(info.st_mode)));
}


DEFINE_PRIMITIVE("file-is-regular?", file_is_regularp, subr1, (SCM f))
{
  struct stat info;

  ENTER_PRIMITIVE(file_is_regularp);

  if (!STRINGP(f)) error_bad_path(f);
  if (stat(STRING_CHARS(f), &info) != 0) return STk_false;

  return MAKE_BOOLEAN((S_ISREG(info.st_mode)));
}


DEFINE_PRIMITIVE("file-is-readable?", file_is_readablep, subr1, (SCM f))
{
  ENTER_PRIMITIVE(file_is_readablep);
  return my_access(f, R_OK);
}


DEFINE_PRIMITIVE("file-is-writable?", file_is_writablep, subr1, (SCM f))
{
  ENTER_PRIMITIVE(file_is_writablep);
  return my_access(f, W_OK);
}


DEFINE_PRIMITIVE("file-is-executable?", file_is_executablep, subr1, (SCM f))
{
  ENTER_PRIMITIVE(file_is_executablep);
  return my_access(f, X_OK);
}


DEFINE_PRIMITIVE("file-exists?", file_existsp, subr1, (SCM f))
{
  ENTER_PRIMITIVE(file_existsp)
  return my_access(f, F_OK);
}

/*
<doc EXT glob
 * (glob pattern ...)
 *
 * |Glob| performs file name ``globbing'' in a fashion similar to the 
 * csh shell. |Glob| returns a list of the filenames that match at least
 * one of |pattern| arguments.  The |pattern| arguments may contain
 * the following special characters:
 * @itemize @bullet
 * @item |?| Matches any single character.
 * @item |*| Matches any sequence of zero or more characters.
 * @item |[chars]| Matches any single character in |chars|. 
 * If chars contains a sequence of the form |a-b| then any character 
 * between |a| and |b| (inclusive) will match.
 * @item  |\\| Matches the character |x|.
 * @item |@{a,b,...@}| Matches any of the strings |a|, |b|, etc.
 * @end itemize
 * 
 * As with csh, a '.' at the beginning of a file's name or just after 
 * a '/ must be matched explicitly or with a |@{@}| construct.  
 * In addition, all '/' characters must be matched explicitly.
 * 
 * If the first character in a pattern is '~' then it refers to
 * the home directory of the user whose name follows the '~'.
 * If the '~' is followed immediately by `/' then the value of
 * the environment variable HOME is used.
 *
 * |Glob| differs from csh globbing in two ways.  First, it does not
 * sort its result list (use the |sort| procedure if you want the list
 * sorted).
 * Second, |glob| only returns the names of files that actually exist; 
 * in csh no check for existence is made unless a pattern contains a 
 * |?|, |*|, or |@{@}| construct.
doc>
*/
DEFINE_PRIMITIVE("glob", glob, vsubr, (int argc, SCM *argv))
{
  ENTER_PRIMITIVE(glob);
  return STk_do_glob(argc, argv);
}


/*
<doc EXT remove-file
 * (remove-file string)
 *
 * Removes the file whose path name is given in |string|.
 * The result of |remove-file| is @emph{void}.
doc>
*/
DEFINE_PRIMITIVE("remove-file", remove_file, subr1, (SCM filename))
{
  ENTER_PRIMITIVE(remove_file);
  
  if (!STRINGP(filename)) error_bad_string(filename);
  if (remove(STRING_CHARS(filename)) != 0)
    STk_error("cannot remove ~S", filename);
  return STk_void;
}
 

/*
<doc EXT rename-file
 * (rename-file string1 string2)
 *
 * Renames the file whose path-name is |string1| to a file whose path-name is
 * |string2|. The result of |rename-file| is @emph{void}.
doc>
*/
DEFINE_PRIMITIVE("rename-file", rename_file, subr2, (SCM filename1, SCM filename2))
{
  ENTER_PRIMITIVE(rename_file);
  
  if (!STRINGP(filename1)) error_bad_string(filename1);
  if (!STRINGP(filename2)) error_bad_string(filename2);
  if (rename(STRING_CHARS(filename1), STRING_CHARS(filename2)) != 0)
    STk_error("cannot rename file ~S in ~S", filename1, filename2);
  return STk_void;
}
 

/*
<doc EXT copy-file
 * (copy-file string1 string2)
 *
 * Copies the file whose path-name is |string1| to a file whose path-name is
 * |string2|. If the file |string2| already exists, its content prior 
 * the call to |copy-file| is lost. The result of |copy-file| is @emph{void}.
doc>
*/
DEFINE_PRIMITIVE("copy-file", copy_file, subr2, (SCM filename1, SCM filename2))
{
  char buff[1024];
  int f1, f2, n;

  /* Should I use sendfile on Linux here? */
  ENTER_PRIMITIVE(copy_file);
  
  if (!STRINGP(filename1)) error_bad_string(filename1);
  if (!STRINGP(filename2)) error_bad_string(filename2);
  
  f1 = open(STRING_CHARS(filename1), O_RDONLY);
  f2 = open(STRING_CHARS(filename2), O_WRONLY|O_CREAT|O_TRUNC, 0666);
  
  if ((f1==-1) || (f2==-1)) {
    error_cannot_copy(filename1, filename2);
  }

  while ((n = read(f1, buff, MAXBUFF)) > 0) {
    if ((n < 0) || (write(f2, buff, n) < n)) {
      close(f1); close(f2); 
      error_cannot_copy(filename1, filename2);
    }
  }
  
  close(f1); close(f2);
  return STk_void;
}


/*
<doc EXT temporary-file-name
 * (temporary-file-name)
 *
 * Generates a unique temporary file name. The value returned by
 * |temporary-file-name| is the newly generated name of |#f|
 * if a unique name cannot be generated.
doc>
*/
DEFINE_PRIMITIVE("temporary-file-name", tmp_file, subr0, (void))
{
#ifdef WIN32
  char buff[MAX_PATH_LENGTH], *s;
  
  s = tmpnam(buff);
  return s ? STk_Cstring2string(s) : STk_false;
#else
  static int cpt=0;
  char buff[MAX_PATH_LENGTH];

  for ( ; ; ) {
    sprintf(buff, "/tmp/stklos%05x", cpt++);
    if (cpt > 100000)		/* arbitrary limit to avoid infinite search */
      return STk_false; 
    if (access(buff, F_OK) == -1) break;
  }

  return STk_Cstring2string(buff);
#endif
}


/*
<doc EXT exit
 * (exit) 
 * (exit ret-code)
 *
 * Exits the program with the specified integer return code. If |ret-code|
 * is omitted, the program terminates with a return code of 0.
doc>
*/
DEFINE_PRIMITIVE("exit", quit, subr01, (SCM retcode))
{
  long ret = 0;
  
  ENTER_PRIMITIVE(quit);

  if (retcode) {
    ret = STk_integer_value(retcode);
    if (ret == LONG_MIN) STk_error("bad return code ~S", retcode);
  }
#ifdef FIXME
//EG:  /* Execute all the terminal thunks of pending dynamic-wind */
//EG:  STk_unwind_all();
//EG:
//EG:  /* call user finalization code */
//EG:  STk_user_cleanup();
//EG:
//EG:#if defined(WIN32) && defined(USE_SOCKET)
//EG:  /* Unregister the interpreter from Winsock */
//EG:  WSACleanup();  
//EG:#endif
#endif
  exit(ret);
  return STk_void; /* never reached */
}


/*
<doc EXT machine-type
 * (machine-type)
 *
 * Returns a string identifying the kind of machine which is running the
 * program. The result string is of the form 
 * |[os-name]-[os-version]-[processor-type]|.
doc>
*/
DEFINE_PRIMITIVE("machine-type", machine_type, subr0, (void))
{
  return STk_Cstring2string(BUILD_MACHINE);
}


#ifdef FIXME
//EG: PRIMITIVE STk_random(SCM n)
//EG: {
//EG:   if (NEXACTP(n) || STk_negativep(n) == STk_true || STk_zerop(n) == STk_true)
//EG:     Err("random: bad number", n);
//EG:   return STk_modulo(STk_makeinteger(rand()), n);
//EG: }
//EG: 
//EG: PRIMITIVE STk_set_random_seed(SCM n)
//EG: {
//EG:   if (NEXACTP(n)) Err("set-random-seed!: bad number", n);
//EG:   srand((unsigned int) STk_integer_value_no_overflow(n));
//EG:   return STk_unsepecified;
//EG: }
//EG: 
//EG: #ifndef HZ
//EG: #define HZ 60.0
//EG: #endif
//EG: 
//EG: #ifdef CLOCKS_PER_SEC
//EG: #  define TIC CLOCKS_PER_SEC
//EG: #else 
//EG: #  define TIC HZ
//EG: #endif
//EG: 
//EG: PRIMITIVE STk_get_internal_info(void)
//EG: {
//EG:   SCM z = STk_makevect(7, STk_nil);
//EG:   long allocated, used, calls;
//EG: 
//EG:   /* The result is a vector which contains
//EG:    *	0 The total cpu used in ms
//EG:    *	1 The number of cells currently in use.
//EG:    *    2 Total number of allocated cells
//EG:    *	3 The number of cells used since the last call to get-internal-info
//EG:    *	4 Number of gc calls
//EG:    *    5 Total time used in the gc
//EG:    *	6 A boolean indicating if Tk is initialized
//EG:    */
//EG: 
//EG:   STk_gc_count_cells(&allocated, &used, &calls);
//EG: 
//EG:   VECT(z)[0] = STk_makenumber(STk_my_time());
//EG:   VECT(z)[1] = STk_makeinteger(used);
//EG:   VECT(z)[2] = STk_makeinteger(allocated);
//EG:   VECT(z)[3] = STk_makenumber((double) STk_alloc_cells);
//EG:   VECT(z)[4] = STk_makeinteger(calls);
//EG:   VECT(z)[5] = STk_makenumber((double) STk_total_gc_time);
//EG: #ifdef USE_TK
//EG:   VECT(z)[6] = Tk_initialized ? STk_true: STk_false;
//EG: #else
//EG:   VECT(z)[6] = STk_false;
//EG: #endif
//EG:   
//EG:   STk_alloc_cells = 0;
//EG:   return z;
//EG: }
#endif


/*
<doc EXT clock 
 * (clock)
 *
 * Returns an approximation of processor time, in milliseconds, used so far by the
 * program.
doc>
 */
DEFINE_PRIMITIVE("clock", clock, subr0, (void))
{
  return STk_double2real((double) clock() / CLOCKS_PER_SEC * 1000.0);
}

/*
<doc EXT current-time 
 * (current-time)
 *
 * Returns the time since the Epoch (that is 00:00:00 UTC, January 1, 1970), 
 * measured in seconds.
doc>
 */
DEFINE_PRIMITIVE("current-time", current_time, subr0, (void))
{
  return STk_ulong2integer(time(NULL));
}

/*
<doc EXT full-current-time
 * (full-current-time)
 * 
 * Returns the time of the day as a pair where 
 * @itemize -
 * @item the  first element is the time since the Epoch 
 * (that is 00:00:00 UTC, January 1, 1970),  measured in seconds. 
 * 
 * @item the second element is the number of microseconds in the given 
 * second.
 * @end itemize
doc>
*/
DEFINE_PRIMITIVE("full-current-time", full_current_time, subr0, (void))
{
  struct timeval now;
  gettimeofday(&now, NULL); 

  return STk_cons(STk_long2integer(now.tv_sec),
		  STk_long2integer(now.tv_usec));
}
		  
/*
<doc EXT seconds->date
 * (seconds->date sec)
 * 
 * Returns a keyword list for the date given by |sec| (a date based on the 
 * Epoch). The keyed values returned are
 * @itemize -
 * @item second : 0 to 59 (but can be up to 61 to allow for leap seconds)
 * @item minute : 0 to 59
 * @item hour : 0 to 23
 * @item day : 1 to 31
 * @item month : 1 to 12
 * @item year : e.g., 2002
 * @item week-day : 0 (Sunday) to 6 (Saturday)
 * @item year-day : 0 to 365 (365 in leap years)
 * @item dst? : #t (daylight savings time) or #f
 * @item time-zone-offset : the difference between Coordinated Universal Time
 * (UTC) and local standard time in seconds.
 * @end itemize
 * @noindent
 * Example:
 * @lisp
 * (seconds->date (current-time))   => (:second 49 :minute 32 :hour 23
 *                                      :day 2 :month 4 :year 2002 
 *                                      :week-day 2 :year-day 91 
 *                                      :dst #t :time-zone-offset -3600)
 * @end lisp
doc>
*/
DEFINE_PRIMITIVE("%seconds->date", seconds2date, subr1, (SCM seconds))
{
  int overflow; 
  SCM v, *vect;
  struct tm *t;
  time_t tt;

  ENTER_PRIMITIVE(seconds2date);
  tt = (time_t) STk_integer2int32(seconds, &overflow);
  
  if (overflow) error_bad_int_or_out_of_bounds(seconds);
  
  t        = localtime(&tt);

  v 	   = STk_makevect(20, STk_false);
  vect     = VECTOR_DATA(v);
  
  vect[0]  = STk_makekey("second");
  vect[1]  = MAKE_INT(t->tm_sec);
  vect[2]  = STk_makekey("minute");
  vect[3]  = MAKE_INT(t->tm_min);
  vect[4]  = STk_makekey("hour");
  vect[5]  = MAKE_INT(t->tm_hour);
  vect[6]  = STk_makekey("day");
  vect[7]  = MAKE_INT(t->tm_mday);
  vect[8]  = STk_makekey("month");
  vect[9]  = MAKE_INT(t->tm_mon + 1);
  vect[10] = STk_makekey("year");
  vect[11] = MAKE_INT(1900 + t->tm_year);
  vect[12] = STk_makekey("week-day");
  vect[13] = MAKE_INT(t->tm_wday);
  vect[14] = STk_makekey("year-day");
  vect[15] = MAKE_INT(t->tm_yday);
  vect[16] = STk_makekey("dst");
  vect[17] = MAKE_BOOLEAN(t->tm_isdst > 0);
  vect[18] = STk_makekey("time-zone-offset");
  vect[19] = STk_long2integer(timezone);

  return STk_vector2list(v);
}

DEFINE_PRIMITIVE("%date->seconds", date2seconds, subr1, (SCM v))
{
  struct tm t;
  SCM *vect;
  time_t n;

  ENTER_PRIMITIVE(date2seconds);
  
  if (!(VECTORP(v) && VECTOR_SIZE(v) == 7)) STk_error("bad date vector ~S", v);
  vect = VECTOR_DATA(v);

  t.tm_sec   = STk_integer_value(vect[0]);
  t.tm_min   = STk_integer_value(vect[1]);
  t.tm_hour  = STk_integer_value(vect[2]);
  t.tm_mday  = STk_integer_value(vect[3]);
  t.tm_mon   = STk_integer_value(vect[4]) - 1;
  t.tm_year  = STk_integer_value(vect[5]) - 1900;
  t.tm_isdst = (vect[6] != STk_false);

  n = mktime(&t);
  if (n == (time_t)(-1)) STk_error("cannot convert date to seconds (~S)", v);
  
  return STk_ulong2integer((long) n);
}


DEFINE_PRIMITIVE("%seconds->string", date2string, subr2, (SCM fmt, SCM seconds))
{
  char buffer[1024];
  struct tm *p;
  time_t tt;  
  int len, overflow;

  ENTER_PRIMITIVE(date2string);
  
  tt = (time_t) STk_integer2int32(seconds, &overflow);
  
  if (!STRINGP(fmt)) error_bad_string(fmt);
  if (overflow)      error_bad_int_or_out_of_bounds(seconds);
  
  p   = localtime(&tt);
  len = strftime(buffer, 1023, STRING_CHARS(fmt), p);

  if (len > 0)
    return STk_Cstring2string(buffer);
  else 
    STk_error("buffer too short!");
  
  return STk_void; /* never reached */
}


/*
<doc EXT running-os
 * (running-os)
 * 
 * Returns the name of the underlying Operating System which is running 
 * the program. 
 * The value returned by |runnin-os| is a symbol. For now, this procedure 
 * returns either |unix| or |windows|.
doc>
*/
DEFINE_PRIMITIVE("running-os", running_os, subr0, (void))
{
#ifdef WIN32
  return STk_intern("windows");
#else
  return STk_intern("unix");
#endif
}



/*
<doc EXT getenv
 * (getenv str)
 * (getenv)
 *
 * Looks for the environment variable named |str| and returns its
 * value as a string, if it exists. Otherwise, |getenv| returns |#f|.
 * If |getenv| is called without parameter, it returns the list of 
 * all the environment variables accessible from the program as an 
 * A-list.
 * @lisp
 * (getenv "SHELL")   
 *      => "/bin/zsh"
 * (getenv)
 *      => (("TERM" . "xterm") ("PATH" . "/bin:/usr/bin") ...)
 * @end lisp
doc>
 */
static SCM build_posix_environment(char **env)
{
  SCM l;

  for (l=STk_nil; *env; env++) {
    char *s, *p;

    s = *env; p = strchr(s, '=');
    if (p)
      l = STk_cons(STk_cons(STk_makestring(p-s, s), STk_Cstring2string(p+1)),l);
  }
  return l;
}


DEFINE_PRIMITIVE("getenv", getenv, subr01, (SCM str))
{
  char *tmp;
  
  ENTER_PRIMITIVE(getenv);
  if (str) {		/* One parameter: find the value of the given variable */
    if (!STRINGP(str)) error_bad_string(str);
    
    tmp = getenv(STRING_CHARS(str));
    return tmp ? STk_Cstring2string(tmp) : STk_false;
  } else {		/* No parameter: give the complete environment */
    extern char **environ;
    return build_posix_environment(environ);
  }
}

/*
<doc EXT setenv!
 * (setenv! var value)
 *
 * Sets the environment variable |var| to |value|. |Var| and
 * |value| must be strings. The result of |setenv!| is @emph{void}.
doc>
 */
DEFINE_PRIMITIVE("setenv!", setenv, subr2, (SCM var, SCM value))
{
  char *s;
  ENTER_PRIMITIVE(setenv);
  if (!STRINGP(var)) 		      STk_error("variable ~S is not a string", var);
  if (strchr(STRING_CHARS(var), '=')) STk_error("variable ~S contains a '='", var);
  if (!STRINGP(value)) 		      STk_error("value ~S is not a string", value);

  s = STk_must_malloc(strlen(STRING_CHARS(var))   + 
		      strlen(STRING_CHARS(value)) + 2); /* 2 because of '=' & \0 */
  sprintf(s, "%s=%s", STRING_CHARS(var), STRING_CHARS(value));
  putenv(s);
  return STk_void;
}


/*
 * Undocumented primitives
 *
 */

DEFINE_PRIMITIVE("%library-prefix", library_prefix, subr0, (void))
{
  return STk_Cstring2string(PREFIXDIR);
}



int STk_init_system(void)
{
  ADD_PRIMITIVE(clock);
  ADD_PRIMITIVE(current_time);
  ADD_PRIMITIVE(full_current_time);
  ADD_PRIMITIVE(seconds2date);
  ADD_PRIMITIVE(date2seconds);
  ADD_PRIMITIVE(date2string);
  ADD_PRIMITIVE(running_os);
  ADD_PRIMITIVE(getenv);
  ADD_PRIMITIVE(setenv);
  ADD_PRIMITIVE(library_prefix);

  ADD_PRIMITIVE(getcwd);
  ADD_PRIMITIVE(chdir);
  ADD_PRIMITIVE(getpid);
  ADD_PRIMITIVE(system);
    
  ADD_PRIMITIVE(file_is_directoryp);
  ADD_PRIMITIVE(file_is_regularp);
  ADD_PRIMITIVE(file_is_readablep);
  ADD_PRIMITIVE(file_is_writablep);
  ADD_PRIMITIVE(file_is_executablep);
  ADD_PRIMITIVE(file_existsp);
  ADD_PRIMITIVE(glob);
  ADD_PRIMITIVE(expand_fn);
  ADD_PRIMITIVE(canonical_path);

  ADD_PRIMITIVE(remove_file);
  ADD_PRIMITIVE(rename_file);
  ADD_PRIMITIVE(copy_file);
  ADD_PRIMITIVE(tmp_file);
  ADD_PRIMITIVE(quit);
  ADD_PRIMITIVE(machine_type);

  return TRUE;
}
