use lang:bs;
use core:lang;
use core:asm; // Note: also contains a Block class!

/**
 * An expression representing creating a template syntax tree.
 */
class TemplateExpr extends ExprBlock {
	init(Block block, Expr env, SExpr create) {
		init(create.pos, block);
		var atoms = create.allChildren(named{PatternAtom});
		var arrays = findArrays(create);

		// PatternBlock exprs(<env>, atoms.count, actuals.count);
		Actuals params;
		params.add(env);
		params.add(NumLiteral(pos, atoms.count.long));
		params.add(NumLiteral(pos, arrays.count.long));

		Var exprVars(this, Value(named{PatternBlock}), SStr("block"), params);
		add(exprVars);
		LocalVarAccess exprs(pos, exprVars.var);

		for (id, node in atoms) {
			if (node as PatternAtom) {
				addExpr(block, exprs, node.pattern, "push");
				node.pattern = PatternExprSrc(id);
			} else {
				throw SyntaxError(pos, "Internal error!");
			}
		}

		for (node in arrays) {
			addExpr(block, exprs, node, "pushArray");
		}

		// Call transform on the stored syntax tree.
		ReferSExpr src(create);
		add(namedExpr(this, SStr("transform"), exprs, Actuals(src)));

		// Return 'exprs'.
		add(exprs);
	}

	void code(CodeGen state, CodeResult r) {
		super:code(state, r);
	}

private:
	Array<SPatternExpr> findArrays(SExpr create) {
		SPatternExpr[] result;
		for (x in create.allChildren(named{PatternActual})) {
			if (x as PatternActual) {
				result << x.pattern;
				x.pattern = PatternExprSrc(result.count - 1);
			}
		}
		for (x in create.allChildren(named{PatternArray})) {
			if (x as PatternArray) {
				result << x.pattern;
				x.pattern = PatternExprSrc(result.count - 1);
			}
		}
		for (x in create.allChildren(named{PatternBlockItem})) {
			if (x as PatternBlockItem) {
				result << x.pattern;
				x.pattern = PatternExprSrc(result.count - 1);
			}
		}
		result;
	}

	void addExpr(Block parent, LocalVarAccess exprs, SPatternExpr node, Str member) {
		if (var = node as PatternVar) {
			addExpr(member, exprs, namedExpr(parent, var.name.transform, Actuals()));
		} else if (expr = node as PatternExpr) {
			addExpr(member, exprs, expr.expr.transform(parent));
		} else {
			throw SyntaxError(pos, "Unexpected subtype of SPatternExpr found. Can not continue.");
		}
	}

	void addExpr(Str member, LocalVarAccess exprs, Expr insert) {
		add(namedExpr(this, SStr(member), exprs, Actuals(insert)));
	}
}


/**
 * Block containing expressions to be inserted into a pattern.
 *
 * Attempts to appear invisible to the user. Variables added here are actually added to the parent scope.
 */
class PatternBlock extends Block {
	// Expressions to be inserted later.
	Array<Expr> exprs;

	// Actuals to be inserted later.
	Array<Array<Expr>> actuals;

	// Parent block.
	private Block parent;

	// Contained expr.
	private Expr data;

	init(Block parent, Nat atomCount, Nat actualsCount) {
		init(parent.pos, parent) {
			parent = parent;
			// Create a dummy object for now.
			data(parent.pos);
		}

		exprs.reserve(atomCount);
		actuals.reserve(actualsCount);
	}

	// Add an expression.
	void push(Expr expr) {
		exprs.push(expr);
	}

	// Add a set of actual parameters.
	void pushArray(Array<Expr> expr) {
		actuals.push(expr);
	}

	// Transform an SExpr and contain the new value.
	void transform(SExpr src) {
		data = src.transform(this);
	}

	// Do not add variables here. Add them to the parent scope instead!
	void add(LocalVar v) {
		parent.add(v);
	}

	// Result of this block.
	ExprResult result() {
		data.result();
	}

	// Generate code for this block.
	void blockCode(CodeGen state, CodeResult to) {
		data.code(state, to);
	}

	// To string.
	void toS(StrBuf to) {
		to << data;
	}
}


/**
 * Subclass to the SPatternExpr used to tell the transformation where to fetch the proper
 * expressions from.
 */
class PatternExprSrc extends SPatternExpr {
	// The index we're concerned of.
	private Nat id;

	// Create.
	init(Nat id) {
		init() {
			id = id;
		}
	}

	// Get an expression.
	Expr expr(Block block) {
		if (r = findPattern(block)) {
			r.exprs[id];
		} else {
			throw SyntaxError(block.pos, "Can not use \${} syntax outside of pattern blocks.");

			// Can be removed when we have exceptions.
			Expr(block.pos);
		}
	}

	// Get an array.
	Array<Expr> array(Block block) {
		if (r = findPattern(block)) {
			r.actuals[id];
		} else {
			throw SyntaxError(block.pos, "Can not use @{} syntax outside of pattern blocks.");

			// Can be removed when we have exceptions.
			Array<Expr>();
		}
	}

	// Dummy transformation function.
	void transform() {}

	// To string.
	void toS(StrBuf to) {
		to << "Captured expression " << id;
	}

	private Maybe<PatternBlock> findPattern(Block src) {
		Block now = src;
		while (true) {
			if (now as PatternBlock)
				return now;

			if (l = now.lookup.parent as BlockLookup) {
				now = l.block;
			} else {
				return null;
			}
		}

		null;
	}
}


// Insert a previously generated Expr here.
Expr insertExpr(SrcPos pos, Block block, SPatternExpr pattern) {
	if (pattern as PatternExprSrc) {
		pattern.expr(block);
	} else {
		throw SyntaxError(pos, "Can not use \${} syntax outside of pattern blocks.");
		// When we have exceptions, we can ignore this.
		Expr(pos);
	}
}

// Insert a previously generated Array<Expr> here.
void insertActuals(SrcPos pos, Actuals actuals, Block block, SPatternExpr pattern) {
	if (pattern as PatternExprSrc) {
		for (x in pattern.array(block)) {
			actuals.add(x);
		}
	} else {
		throw SyntaxError(pos, "Can not use @{} syntax outside of pattern blocks.");
	}
}

// Insert a previously generated Array<Expr> here.
void insertExprs(SrcPos pos, Array<Expr> into, Block block, SPatternExpr pattern) {
	if (pattern as PatternExprSrc) {
		for (x in pattern.array(block)) {
			into.push(x);
		}
	} else {
		throw SyntaxError(pos, "Can not use @{} syntax outside of pattern blocks.");
	}
}

void insertExprs(SrcPos pos, ExprBlock block, SPatternExpr pattern) {
	if (pattern as PatternExprSrc) {
		for (x in pattern.array(block)) {
			block.add(x);
		}
	} else {
		throw SyntaxError(pos, "Can not use @{} syntax outside of pattern blocks.");
	}
}


/**
 * Refer to a SExpr object.
 */
class ReferSExpr extends Expr {
	// The expression referred to.
	private SExpr expr;

	init(SExpr expr) {
		init(expr.pos) { expr = expr; }
	}

	ExprResult result() {
		ExprResult(Value(named{lang:bs:SExpr}));
	}

	void code(CodeGen state, CodeResult result) {
		if (result.needed) {
			var v = result.location(state);
			state.l << mov(v, objPtr(expr));
			result.created(state);
		}
	}

	Str toS() {
		"<captured syntax tree>";
	}
}
