/*
 * $Id: st-transfer.c,v 1.73 2004/03/26 18:04:57 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <curl/curl.h>
#include <gtk/gtk.h>
#include <string.h>
#include "gettext.h"
#include "sg-util.h"
#include "sgtk-auth-dialog.h"
#include "st-dialog-api.h"
#include "st-settings.h"
#include "st-transfer.h"
#include "st-thread.h"
#include "st-main.h"

#define AGENT_STRING			"streamtuner"

/* maintain compatibility with old libcurl versions */

#ifndef CURLOPT_WRITEDATA
#define CURLOPT_WRITEDATA		CURLOPT_FILE
#endif

#ifndef CURLOPT_HEADERDATA
#define CURLOPT_HEADERDATA		CURLOPT_WRITEHEADER
#endif

/*** type definitions ********************************************************/

/*
 * libcurl already provides this prototype as curl_write_callback, but
 * we are not gonna rely on it as it's undocumented
 */
typedef size_t (CurlCallback) (const char *buffer,
			       size_t size,
			       size_t nitems,
			       gpointer data);

typedef struct
{
  STThread			*thread;
  double			downloaded;

  const char			*url;
  unsigned int			flags;

  STTransferSession		*session;
  char				error[CURL_ERROR_SIZE];

  CurlCallback			*header_cb;
  gpointer			header_data;

  CurlCallback			*body_cb;
  gpointer			body_data;

  curl_proxytype		proxy_type;
  char				*proxy_server;
  int				proxy_port;
  char				*proxy_userpwd;
} STTransfer;

struct _STTransferSession
{
  CURL				*handle;
};

typedef struct
{
  STTransfer			*transfer;

  GString			*line;
  STTransferLineCallback	*line_cb;
  gpointer			line_cb_data;
} STTransferLineData;

/*** function declarations ***************************************************/

static gboolean st_transfer_perform (STTransfer *transfer, GError **err);

static gboolean st_transfer_set_proxy_settings (STTransfer *transfer);
static void st_transfer_free_proxy_settings (STTransfer *transfer);

static char *st_transfer_password_prompt (const char *username);

static gboolean st_transfer_progress_cb (gpointer data,
					 double dltotal,
					 double dlnow,
					 double ultotal,
					 double ulnow);

static gboolean st_transfer_session_get_internal (STTransferSession *session,
						  const char *url,
						  unsigned int flags,
						  char **headers,
						  char **body,
						  GError **err);
static gboolean st_transfer_session_get_by_line_internal (STTransferSession *session,
							  const char *url,
							  unsigned int flags,
							  STTransferLineCallback *header_cb,
							  gpointer header_data,
							  STTransferLineCallback *body_cb,
							  gpointer body_data,
							  GError **err);

static size_t st_transfer_session_get_cb (const char *buffer,
					  size_t size,
					  size_t nitems,
					  gpointer data);
static size_t st_transfer_session_get_binary_cb (const char *buffer,
						 size_t size,
						 size_t nitems,
						 gpointer data);
static size_t st_transfer_session_get_by_line_cb (const char *buffer,
						  size_t size,
						  size_t nitems,
						  gpointer data);

static void st_transfer_zero_and_free (char *buffer);

/*** API implementation ******************************************************/

STTransferSession *
st_transfer_session_new (void)
{
  STTransferSession *session;

  session = g_new(STTransferSession, 1);
  session->handle = curl_easy_init();

  return session;
}

void
st_transfer_session_free (STTransferSession *session)
{
  g_return_if_fail(session != NULL);

  curl_easy_cleanup(session->handle);
  g_free(session);
}

/*
 * Deprecated.
 */
char *
st_transfer_get_full (const char *url, GError **err)
{
  STTransferSession *session;
  gboolean status;
  char *body;

  g_return_val_if_fail(url != NULL, FALSE);

  session = st_transfer_session_new();
  status = st_transfer_session_get_internal(session, url, 0, NULL, &body, err);
  st_transfer_session_free(session);

  return status ? body : NULL;
}

/*
 * Fetches URL.
 *
 * If HEADERS is not null, write a newly-allocated string containing
 * the HTTP headers at its address.
 *
 * A newly-allocated string containing the body will be written at the
 * address pointed to by BODY.
 *
 * Returns TRUE in case of success, FALSE in case of abort or error
 * (if there was an error, ERR will be set).
 */
gboolean
st_transfer_session_get (STTransferSession *session,
			 const char *url,
			 unsigned int flags,
			 char **headers,
			 char **body,
			 GError **err)
{
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);

  status = st_transfer_session_get_internal(session, url, flags, headers, body, err);

  return status;
}

/*
 * Fetches URL as binary data.
 *
 * If HEADERS is not null, write a newly-allocated string containing
 * the HTTP headers at its address.
 *
 * A newly-allocated buffer containing the body will be written at the
 * address pointed to by BODY.
 *
 * The length of the body will be written at the address pointed to by
 * BODY_LEN.
 *
 * Returns TRUE in case of success, FALSE in case of abort or error
 * (if there was an error, ERR will be set).
 */
gboolean
st_transfer_session_get_binary (STTransferSession *session,
				const char *url,
				unsigned int flags,
				char **headers,
				guint8 **body,
				unsigned int *body_len,
				GError **err)
{
  STTransfer transfer;
  GString *header_string;
  GByteArray *body_array;
  gboolean status;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  
  transfer.url = url;
  transfer.flags = flags;
  transfer.session = session;

  if (headers)
    {
      header_string = g_string_new(NULL);
      transfer.header_cb = st_transfer_session_get_cb;
      transfer.header_data = header_string;
    }
  else
    {
      transfer.header_cb = NULL;
      transfer.header_data = NULL;
    }

  if (body)
    {
      body_array = g_byte_array_new();
      transfer.body_cb = st_transfer_session_get_binary_cb;
      transfer.body_data = body_array;
    }
  else
    {
      transfer.body_cb = NULL;
      transfer.body_data = NULL;
    }
  
  status = st_transfer_perform(&transfer, err);
  if (status)
    {
      if (headers)
	*headers = header_string->str;
      if (body)
	{
	  *body = body_array->data;
	  *body_len = body_array->len;
	}
    }

  if (headers)
    g_string_free(header_string, ! status);
  if (body)
    g_byte_array_free(body_array, ! status);

  return status;
}

/*
 * Deprecated
 */
char *
st_transfer_get_full_with_session (STTransferSession *session,
				   const char *url,
				   GError **err)
{
  gboolean status;
  char *body;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);

  status = st_transfer_session_get_internal(session, url, 0, NULL, &body, err);
  
  return status ? body : NULL;
}

static gboolean
st_transfer_session_get_internal (STTransferSession *session,
				  const char *url,
				  unsigned int flags,
				  char **headers,
				  char **body,
				  GError **err)
{
  STTransfer transfer;
  GString *header_string;
  GString *body_string;
  gboolean status;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  
  transfer.url = url;
  transfer.flags = flags;
  transfer.session = session;

  if (headers)
    {
      header_string = g_string_new(NULL);
      transfer.header_cb = st_transfer_session_get_cb;
      transfer.header_data = header_string;
    }
  else
    {
      transfer.header_cb = NULL;
      transfer.header_data = NULL;
    }

  if (body)
    {
      body_string = g_string_new(NULL);
      transfer.body_cb = st_transfer_session_get_cb;
      transfer.body_data = body_string;
    }
  else
    {
      transfer.body_cb = NULL;
      transfer.body_data = NULL;
    }
  
  status = st_transfer_perform(&transfer, err);
  if (status)
    {
      if (headers)
	*headers = header_string->str;
      if (body)
	*body = body_string->str;
    }

  if (headers)
    g_string_free(header_string, ! status);
  if (body)
    g_string_free(body_string, ! status);

  return status;
}

/*
 * Deprecated.
 */
gboolean
st_transfer_get_lines (const char *url,
		       STTransferLineCallback *cb,
		       gpointer data,
		       GError **err)
{
  STTransferSession *session;
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(cb != NULL, FALSE);

  session = st_transfer_session_new();
  status = st_transfer_session_get_by_line_internal(session, url, 0, NULL, NULL, cb, data, err);
  st_transfer_session_free(session);

  return status;
}

gboolean
st_transfer_session_get_by_line (STTransferSession *session,
				 const char *url,
				 unsigned int flags,
				 STTransferLineCallback *header_cb,
				 gpointer header_data,
				 STTransferLineCallback *body_cb,
				 gpointer body_data,
				 GError **err)
{
  gboolean status;

  g_return_val_if_fail(url != NULL, FALSE);

  status = st_transfer_session_get_by_line_internal(session, url, flags, header_cb, header_data, body_cb, body_data, err);

  return status;
}

/*
 * Deprecated.
 */
gboolean
st_transfer_get_lines_with_session (STTransferSession *session,
				    const char *url,
				    STTransferLineCallback *cb,
				    gpointer data,
				    GError **err)
{
  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  g_return_val_if_fail(cb != NULL, FALSE);

  return st_transfer_session_get_by_line_internal(session, url, 0, NULL, NULL, cb, data, err);
}

static gboolean
st_transfer_session_get_by_line_internal (STTransferSession *session,
					  const char *url,
					  unsigned int flags,
					  STTransferLineCallback *header_cb,
					  gpointer header_data,
					  STTransferLineCallback *body_cb,
					  gpointer body_data,
					  GError **err)
{
  STTransfer transfer;
  STTransferLineData header_line_data;
  STTransferLineData body_line_data;
  gboolean status;

  g_return_val_if_fail(session != NULL, FALSE);
  g_return_val_if_fail(url != NULL, FALSE);
  
  transfer.url = url;
  transfer.flags = flags;
  transfer.session = session;

  if (header_cb)
    {
      header_line_data.transfer = &transfer;
      header_line_data.line = g_string_new(NULL);
      header_line_data.line_cb = header_cb;
      header_line_data.line_cb_data = header_data;

      transfer.header_cb = st_transfer_session_get_by_line_cb;
      transfer.header_data = &header_line_data;
    }
  else
    {
      transfer.header_cb = NULL;
      transfer.header_data = NULL;
    }
   
  if (body_cb)
    {
      body_line_data.transfer = &transfer;
      body_line_data.line = g_string_new(NULL);
      body_line_data.line_cb = body_cb;
      body_line_data.line_cb_data = body_data;
      
      transfer.body_cb = st_transfer_session_get_by_line_cb;
      transfer.body_data = &body_line_data;
    }
  else
    {
      transfer.body_cb = NULL;
      transfer.body_data = NULL;
    }

  status = st_transfer_perform(&transfer, err);
  
  if (header_cb)
    g_string_free(header_line_data.line, TRUE);
  if (body_cb)
    g_string_free(body_line_data.line, TRUE);
  
  return status;
}

static gboolean
st_transfer_perform (STTransfer *transfer, GError **err)
{
  int status;

  g_return_val_if_fail(transfer != NULL, FALSE);
  g_return_val_if_fail(transfer->url != NULL, FALSE);
  g_return_val_if_fail(transfer->session != NULL, FALSE);

  transfer->thread = st_thread_get();
  g_return_val_if_fail(transfer->thread != NULL, FALSE);

  if (st_thread_is_aborted(transfer->thread))
    return FALSE;

  transfer->downloaded = 0;

  GDK_THREADS_ENTER();
  st_thread_set_progress(transfer->thread, 0);
  gdk_flush();
  GDK_THREADS_LEAVE();

  if (! st_transfer_set_proxy_settings(transfer))
    {
      GDK_THREADS_ENTER();
      st_thread_abort(transfer->thread);
      gdk_flush();
      GDK_THREADS_LEAVE();
      
      return FALSE;		/* aborted */
    }

  GDK_THREADS_ENTER();
  st_thread_printf(transfer->thread, _("Connecting to %s..."), transfer->url);
  gdk_flush();
  GDK_THREADS_LEAVE();
  
  curl_easy_setopt(transfer->session->handle, CURLOPT_USERAGENT, AGENT_STRING);
  curl_easy_setopt(transfer->session->handle, CURLOPT_URL, transfer->url);
  curl_easy_setopt(transfer->session->handle, CURLOPT_HEADERFUNCTION, transfer->header_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_HEADERDATA, transfer->header_data);
  curl_easy_setopt(transfer->session->handle, CURLOPT_WRITEFUNCTION, transfer->body_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_WRITEDATA, transfer->body_data);
  curl_easy_setopt(transfer->session->handle, CURLOPT_NOPROGRESS, FALSE);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROGRESSFUNCTION, st_transfer_progress_cb);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROGRESSDATA, transfer);
  curl_easy_setopt(transfer->session->handle, CURLOPT_ERRORBUFFER, transfer->error);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXYTYPE, transfer->proxy_type);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXY, transfer->proxy_server);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXYPORT, transfer->proxy_port);
  curl_easy_setopt(transfer->session->handle, CURLOPT_PROXYUSERPWD, transfer->proxy_userpwd);
  curl_easy_setopt(transfer->session->handle, CURLOPT_COOKIEFILE, "");
  curl_easy_setopt(transfer->session->handle, CURLOPT_FOLLOWLOCATION, TRUE);
  curl_easy_setopt(transfer->session->handle, CURLOPT_NOBODY, transfer->body_cb == NULL);
  curl_easy_setopt(transfer->session->handle, CURLOPT_ENCODING, "");
  curl_easy_setopt(transfer->session->handle, CURLOPT_NOSIGNAL, TRUE);
  
  status = curl_easy_perform(transfer->session->handle);
  
  st_transfer_free_proxy_settings(transfer);

  if (status != CURLE_OK
      && status != CURLE_ABORTED_BY_CALLBACK
      && ! st_thread_is_aborted(transfer->thread))
    g_set_error(err, 0, 0, "%s", transfer->error); /* failure which isn't abort */

  return status == CURLE_OK;
}

static gboolean
st_transfer_set_proxy_settings (STTransfer *transfer)
{
  gboolean status = TRUE;

  g_return_val_if_fail(transfer != NULL, FALSE);

  transfer->proxy_type = 0;
  transfer->proxy_server = NULL;
  transfer->proxy_port = 0;
  transfer->proxy_userpwd = NULL;

  G_LOCK(st_settings);

  /*
   * We have to set transfer->proxy_userpwd, because even if the proxy
   * is disabled in streamtuner, libcurl might still use http_proxy,
   * ftp_proxy, etc...
   */
  if (st_settings.proxy_auth_enabled && st_settings.proxy_auth_name)
    {
      char *password;

      if (st_settings.proxy_auth_password)
	password = g_strdup(st_settings.proxy_auth_password);
      else
	{
	  GDK_THREADS_ENTER();
	  st_thread_printf(transfer->thread, _("Waiting for proxy password..."));
	  password = st_transfer_password_prompt(st_settings.proxy_auth_name);
	  gdk_flush();
	  GDK_THREADS_LEAVE();
	}

      if (password)
	{
	  transfer->proxy_userpwd = g_strconcat(st_settings.proxy_auth_name, ":", password, NULL);
	  st_transfer_zero_and_free(password);
	}
      else			/* cancelled by user */
	status = FALSE;
    }
  else
    transfer->proxy_userpwd = g_strdup(""); /* libcurl crashes in handleSock5Proxy() if this is NULL */

  if (status && st_settings.proxy_enabled && st_settings.proxy_server)
    { 
      transfer->proxy_type = st_settings.proxy_type;
      transfer->proxy_server = g_strdup(st_settings.proxy_server);
      transfer->proxy_port = st_settings.proxy_port;
    }

  G_UNLOCK(st_settings);

  return status;
}

static void
st_transfer_free_proxy_settings (STTransfer *transfer)
{
  g_return_if_fail(transfer != NULL);
  
  g_free(transfer->proxy_server);
  st_transfer_zero_and_free(transfer->proxy_userpwd);
}

char *
st_transfer_escape (const char *url)
{
  return curl_escape(url, 0);
}

static gboolean
st_transfer_progress_cb (gpointer data,
			 double dltotal,
			 double dlnow,
			 double ultotal,
			 double ulnow)
{
  STTransfer *transfer = data;
  gboolean aborted = FALSE;

  aborted = st_thread_is_aborted(transfer->thread);
  if (! aborted)
    {
      GDK_THREADS_ENTER();

      if (dltotal == 0)
	st_thread_printf(transfer->thread,
			 ngettext("Receiving (%u byte so far)...",
				  "Receiving (%u bytes so far)...",
				  (unsigned int) dlnow),
			 (unsigned int) dlnow);
      else
	st_thread_printf(transfer->thread,
			 ngettext("Receiving (%u byte out of %u)...",
				  "Receiving (%u bytes out of %u)...",
				  (unsigned int) dlnow),
			 (unsigned int) dlnow,
			 (unsigned int) dltotal);
	  
      if (dlnow != transfer->downloaded)
	{
	  st_thread_set_progress(transfer->thread, dltotal == 0 ? -1 : dlnow / dltotal);
	  transfer->downloaded = dlnow;
	}

      gdk_flush();
      GDK_THREADS_LEAVE();
    }

  return aborted;
}

static char *
st_transfer_password_prompt (const char *username)
{
  GtkWidget *dialog;
  char *password = NULL;

  g_return_val_if_fail(username != NULL, NULL);

  dialog = sgtk_auth_dialog_new(st_main_get_transient(),
				_("Please enter your proxy password."));

  sgtk_auth_dialog_set_name(SGTK_AUTH_DIALOG(dialog), username);
  sgtk_auth_dialog_set_name_sensitive(SGTK_AUTH_DIALOG(dialog), FALSE);

  if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
    password = g_strdup(sgtk_auth_dialog_get_password(SGTK_AUTH_DIALOG(dialog)));

  gtk_widget_destroy(dialog);

  return password;
}

static size_t
st_transfer_session_get_cb (const char *buffer,
			    size_t size,
			    size_t nitems,
			    gpointer data)
{
  GString *string = data;
  size_t len;

  len = size * nitems;
  g_string_append_len(string, buffer, len);
  
  return len;
}

static size_t
st_transfer_session_get_binary_cb (const char *buffer,
				   size_t size,
				   size_t nitems,
				   gpointer data)
{
  GByteArray *array = data;
  size_t len;

  len = size * nitems;
  g_byte_array_append(array, buffer, len);
  
  return len;
}

static size_t
st_transfer_session_get_by_line_cb (const char *buffer,
				    size_t size,
				    size_t nitems,
				    gpointer data)
{
  STTransferLineData *line_data = data;
  size_t len;
  int start = 0;
  int i = 0;

  len = size * nitems;

  /* handle lf (UNIX), crlf (DOS) and cr (Mac) */
  while (i < len)
    if (buffer[i] == '\n' || buffer[i] == '\r')
      {
	int newline_len;

	newline_len = buffer[i] == '\r' && i < len && buffer[i + 1] == '\n'
	  ? 2			/* crlf */
	  : 1;			/* cr or lf */

	g_string_append_len(line_data->line, &buffer[start], i - start);
	if (line_data->transfer->flags & ST_TRANSFER_PASS_NEWLINE)
	  g_string_append_len(line_data->line, &buffer[i], newline_len);

	line_data->line_cb(line_data->line->str, line_data->line_cb_data);
	g_string_truncate(line_data->line, 0);

	start = i += newline_len;
      }
    else
      i++;
  
  if (start < len)
    g_string_append_len(line_data->line, &buffer[start], len - start);
  
  return len;
}

static void
st_transfer_zero_and_free (char *buffer)
{
  if (buffer)
    {
      int i;

      for (i = 0; buffer[i]; i++)
	buffer[i] = 0;
      
      g_free(buffer);
    }
}

/*** private implementation **************************************************/

void
st_transfer_init (void)
{
  if (curl_global_init(CURL_GLOBAL_ALL) != CURLE_OK)
    {
      st_error_dialog(_("A fatal error has occurred."),
		      _("Unable to initialize the curl library."));
      exit(1);
    }
}
