/*
 * Copyright (C) 2008 Martin Willi
 * Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

/**
 * @defgroup xcbc xcbc
 * @{ @ingroup xcbc_p
 */

#ifndef XCBC_H_
#define XCBC_H_

typedef struct xcbc_t xcbc_t;

#include <crypto/hashers/hasher.h>

/**
 * Message authentication using CBC crypter.
 *
 * This class implements the message authenticaion algorithm
 * described in RFC3566.
 */
struct xcbc_t {
	
	/**
	 * Generate message authentication code.
	 * 
	 * If buffer is NULL, no result is given back. A next call will
	 * append the data to already supplied data. If buffer is not NULL, 
	 * the mac of all apended data is calculated, returned and the
	 * state of the xcbc_t is reseted.
	 * 
	 * @param data		chunk of data to authenticate
	 * @param buffer	pointer where the generated bytes will be written
	 */
	void (*get_mac) (xcbc_t *this, chunk_t data, u_int8_t *buffer);
	
	/**
	 * Get the block size of this xcbc_t object.
	 * 
	 * @return			block size in bytes
	 */
	size_t (*get_block_size) (xcbc_t *this);
	
	/**
	 * Set the key for this xcbc_t object.
	 * 
	 * @param key		key to set
	 */
	void (*set_key) (xcbc_t *this, chunk_t key);
	
	/**
	 * Destroys a xcbc_t object.
	 */
	void (*destroy) (xcbc_t *this);
};

/**
 * Creates a new xcbc_t object.
 * 
 * @param algo			underlying crypto algorithm
 * @param key_size		key size to use, if required for algorithm
 * @return				xcbc_t object, NULL if not supported
 */
xcbc_t *xcbc_create(encryption_algorithm_t algo, size_t key_size);

#endif /** XCBC_H_ @}*/
