/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "LogDialog.h"
#include "LogEntryLvi.h"
#include "RevisionWidget.h"
#include "RepositoryModel.h"
#include "PostCmdResult.h"
#include "Project.h"
#include "ExternProviderImpl.h"
#include "ScModel.h"
#include "DiffDialog.h"
#include "TextEdit.h"
#include "TextWindow.h"
#include "dialogs/DiffDialogCmd.h"
#include "dialogs/MergeDialogCmd.h"
#include "settings/FontSettings.h"
#include "commands/ScCmd.h"
#include "commands/LogParam.h"
#include "commands/DiffParam.h"
#include "commands/MergeParam.h"
#include "commands/CatParam.h"
#include "commands/PropSetRevParam.h"
#include "events/LogEvent.h"
#include "events/ScParamEvent.h"
#include "events/EventSupport.h"
#include "sublib/TargetId.h"
#include "sublib/MessageBox.h"
#include "sublib/SplitLayout.h"
#include "sublib/Gui.h"
#include "util/Mutex.h"
#include "util/Guard.h"
#include "util/Id.h"
#include "svn/ClientTypes.h"
#include "svn/LogBaton.h"
#include "svn/LogEntry.h"
#include "svn/Error.h"

// qt
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qcheckbox.h>
#include <qlistview.h>
#include <qtextedit.h>
#include <qaction.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qspinbox.h>
#include <qmenubar.h>
#include <qlineedit.h>
#include <qlabel.h>



///////////////////////////////////////////////////////////////////////////////

class LogDialogResultVisitor :
  public ParamVisitor<LogParam>,
  public ParamVisitor<DiffParam>,
  public ParamVisitor<MergeParam>,
  public ParamVisitor<CatParam>,
  public ParamVisitor<PropSetRevParam>
{
public:
  LogDialogResultVisitor(LogDialog* w, BaseModel* model)
    : _w(w), _m(model)
  {
  }

  void visit( LogParam* p )
  {
    _w->finished();
  }

  void visit( DiffParam* p )
  {
    if( p->getPatch() )
    {
      QString title = _q("subcommander:patch ");
      title += p->getPathOrUrl1();

      if(p->getPatchFile().isEmpty())
      {
        return;
      }

      TextWindow* tw = new TextWindow( title, _m->getModel()->getFontSettings(), _w );
      tw->loadText(QString::fromUtf8(p->getPatchFile()));
      tw->show();
    }
  }

  void visit( MergeParam* p )
  {
  }

  void visit( CatParam* p )
  {
    _w->showCat( QString::fromUtf8(p->getPathOrUrl()), QString::fromUtf8(p->getValue()) );
  }

  void visit( PropSetRevParam* p )
  {
  }

private:
  LogDialog* _w;
  BaseModel* _m;
};

//
///////////////////////////////////////////////////////////////////////////////
//

class LogBaton : public svn::LogBaton
{
public:
  LogBaton( LogDialog* dlg ) :_dlg(dlg)
  {
  }

  svn::Error* receiveMessage( const svn::LogEntryPtr log )
  {
    postEvent( _dlg, new LogEvent(log) );
    return 0;
  }

private:
  LogDialog* _dlg;
};

//
///////////////////////////////////////////////////////////////////////////////
//

class MergeRevisions
{
public:
  MergeRevisions( const LogDialog::LogEntryLvis& entries, bool undo )
    : _undo(undo)
  {
    if( entries.size() == 1 )
    {
      _revStop  = entries.front()->getLogEntry()->getRevnumber();
      _revStart = getPrevRevision( entries.front() );
    }
    else
    {
      _revStop  = entries.front()->getLogEntry()->getRevnumber();
      _revStart = entries.back()->getLogEntry()->getRevnumber();
    }
  }

  svn::Revnumber getPrevRevision( const LogEntryLvi* entry )
  {
    // try to get the previous revision number on this path/file.
    // This may be not just rev-1 because of commits on other paths/files.
    // if we don't have a previous log entry, fallback to prev-1.

    LogEntryLvi* prevEntry = (LogEntryLvi*)entry->nextSibling();

    svn::Revnumber prevRev;
    if( prevEntry )
    {
      prevRev = prevEntry->getLogEntry()->getRevnumber();
    }
    else
    {
      prevRev = entry->getLogEntry()->getRevnumber()-1;
    }

    return prevRev;
  }

  svn::Revnumber getStartRevision()
  {
    if( _undo )
    {
      return _revStop;
    }
    else
    {
      return _revStart;
    }
  }

  svn::Revnumber getStopRevision()
  {
    if( _undo )
    {
      return _revStart;
    }
    else
    {
      return _revStop;
    }
  }

private:
  bool           _undo;
  svn::Revnumber _revStart;
  svn::Revnumber _revStop;
};

//
///////////////////////////////////////////////////////////////////////////////

Qt::WFlags LogDialogFlags = Qt::WStyle_Customize | Qt::WType_TopLevel
  | Qt::WStyle_MinMax  | Qt::WStyle_NormalBorder | Qt::WStyle_Title
  | Qt::WStyle_SysMenu | Qt::WDestructiveClose;

LogDialog::LogDialog( BaseModel* model, const Project* prj, const sc::String&
  src, bool folder, QWidget *parent )
: super( parent, 0, LogDialogFlags ), TargetId(this), _model(model), _prj(prj),
  _source(src), _folder(folder), _findIdx(0), _findLastLvi(0)
{
  setCaption( QString(_q("subcommander:log (%1)")).arg((const char*)_source) );

  QVBoxLayout *vbl = new QVBoxLayout(this,5,8);
  vbl->setSpacing(10);
  {
    QGroupBox* gb = new QGroupBox(1,Qt::Vertical,this);
    gb->setTitle( _q("log options: ") );
    //gb->setInsideSpacing(10);
    gb->setInsideMargin(0);
    gb->setFlat(true);
    vbl->addWidget(gb);
    {
      QHBox* hb = new QHBox(gb);
      hb->setSpacing(5);
      {
        QVBox* vb = new QVBox(hb);
        {
          _strictHistory = new QCheckBox(_q("&stop on copy"),vb);
          _strictHistory->setChecked(true);
          _changedPaths = new QCheckBox(_q("&changed paths"),vb);
          _changedPaths->setChecked(true);
        }

        _limitGroup = new QGroupBox(hb);
        _limitGroup->setFlat(true);
        _limitGroup->setTitle(_q("limit"));
        _limitGroup->setCheckable(true);
        _limitGroup->setColumnLayout( 1, Qt::Vertical );
        {
          _limit = new QSpinBox(_limitGroup);
          _limit->setMinValue(1);
          _limit->setMaxValue(1000000);
          _limit->setValue(50);
        }

        _rwStart = new RevisionWidget(false,"SDN","HBCP",0,hb);
        _rwStart->setTitle( _q("start revision") );
        _rwStop = new RevisionWidget(false,"NDS","HBCP",0,hb);
        _rwStop->setTitle( _q("stop revision") );
      }
    }

    QWidget*      hw     = new QWidget(this);
    HSplitLayout* hsplit = new HSplitLayout(hw);
    vbl->addWidget(hw);
    {
      _revs = new QListView(hw);
      _revs->addColumn( _q("revision") );
      _revs->addColumn( _q("date") );
      _revs->addColumn( _q("author") );
      _revs->addColumn( "" );
      _revs->setColumnAlignment( 0, Qt::AlignRight );
      _revs->setColumnAlignment( 1, Qt::AlignRight );
      _revs->setColumnAlignment( 2, Qt::AlignRight );
      _revs->setResizeMode(QListView::LastColumn);
      _revs->setSortOrder(Qt::Descending);
      _revs->setAllColumnsShowFocus(true);
      _revs->showSortIndicator();
      _revs->setItemMargin( 1 );
      _revs->setSelectionMode( QListView::Extended );
      hsplit->addWidgetOne(_revs,false,2);

      connect( _revs, SIGNAL(selectionChanged()), this, SLOT(updateSelection()) );
      connect( _revs, SIGNAL(currentChanged(QListViewItem*)), this, SLOT(clicked(QListViewItem*)) );
      connect( _revs, SIGNAL(contextMenuRequested(QListViewItem*,const QPoint&,int)),
        SLOT(contextMenuRequest(QListViewItem*,const QPoint&,int)) );


      QWidget*      vw     = new QWidget(hw);
      VSplitLayout* vsplit = new VSplitLayout(vw,SplitLayout::pTwo);
      hsplit->addWidgetTwo(vw,false,3);
      {
        QWidget*     ivw = new QWidget(vw);
        QVBoxLayout* vb  = new QVBoxLayout(ivw);
        vb->setSpacing(5);
        vsplit->addWidgetOne(ivw,false,3);
        {
          // todo uh...
          QFont font = _model->getModel()->getFontSettings()->getEditorFont();

          _log = new TextEdit(ivw);
          _log->setWordWrap( QTextEdit::NoWrap );
          _log->setTextFormat( QTextEdit::PlainText );
          _log->setFont(font);
          _log->setEnabled(false);
          vb->addWidget(_log);

          connect( _log, SIGNAL(textChanged()), SLOT(logChanged()) );

          QHBoxLayout* ih = new QHBoxLayout;
          vb->addLayout(ih);
          ih->addStretch(1);

          _set = new QPushButton(ivw);
          _set->setText( _q("&Set") );
          _set->setEnabled(false);
          ih->addWidget(_set);

          connect( _set, SIGNAL(clicked()), SLOT(set()) );
        }

        _paths = new QListView(vw);
        _paths->setAllColumnsShowFocus(true);
        _paths->setItemMargin( 2 );
        _paths->addColumn( _q("action") );
        _paths->addColumn( _q("path") );
        _paths->addColumn( _q("copy from path") );
        _paths->addColumn( _q("copy from rev") );
        vsplit->addWidgetTwo(_paths,false,2);
      }
    }

    QHBoxLayout* hu = new QHBoxLayout;
    vbl->addLayout(hu);
    {
      QLabel* fl = new QLabel( _q("&find"), this );
      hu->addWidget(fl);

      _find = new QLineEdit(this);
      hu->addWidget(_find);
      fl->setBuddy(_find);

      _findNext = new QPushButton( _q("&next"), this );
      _findPrev = new QPushButton( _q("&prev"), this );
      hu->addWidget(_findNext);
      hu->addWidget(_findPrev);
      _findNext->setDisabled(true);
      _findPrev->setDisabled(true);

      connect( _find, SIGNAL(textChanged(const QString&)), SLOT(find(const QString&)) );
      connect( _findNext, SIGNAL(clicked()), SLOT(findNext()) );
      connect( _findPrev, SIGNAL(clicked()), SLOT(findPrev()) );

      // eats extra space, so the buttons keep their size
      hu->addStretch(1); 

      _run = new QPushButton(this);
      _run->setText( _q("&Run") );
      _run->setFocus();
      hu->addWidget(_run);
      _done = new QPushButton(this);
      _done->setText( _q("&Done") );
      hu->addWidget(_done);

      hu->addSpacing(getSizeGripSpacing());
      
      connect( _run,  SIGNAL(clicked()), SLOT(run()) );
      connect( _done, SIGNAL(clicked()), SLOT(close()) );
    }
  }

  // diff
  _diffAction = new QAction( _q("&diff.."), _q("Ctrl+Shift+D"), this );
  _diffAction->setStatusTip( _q("show diff between 2 selected revisions") );
  _diffAction->setEnabled( true );
  connect( _diffAction, SIGNAL(activated()), SLOT(diff()) );

  _diffcAction = new QAction( _q("diff &commit"), _q("Ctrl+Shift+C"), this );
  _diffcAction->setStatusTip( _q("show diff between the selected revision and the previous revision") );
  _diffcAction->setEnabled( true );
  connect( _diffcAction, SIGNAL(activated()), SLOT(diffc()) );

  _diffwcAction = new QAction( _q("diff &wc"), _q("Ctrl+Shift+W"), this );
  _diffwcAction->setStatusTip( _q("show diff between the selected revision and the working copy") );
  _diffwcAction->setEnabled( true );
  connect( _diffwcAction, SIGNAL(activated()), SLOT(diffwc()) );

  // merge
  _mergeAction = new QAction( _q("&merge.."), _q("Ctrl+Shift+M"), this );
  _mergeAction->setStatusTip( _q("merge revision(s)") );
  _mergeAction->setEnabled( false );
  connect( _mergeAction, SIGNAL(activated()), SLOT(merge()) );

  // undo
  _undoAction = new QAction( _q("&undo.."), _q("Ctrl+Shift+U"), this );
  _undoAction->setStatusTip( _q("undo commit by reverse merging") );
  _undoAction->setEnabled( false );
  connect( _undoAction, SIGNAL(activated()), SLOT(undo()) );

  // cat
  _catAction = new QAction( _q("ca&t"), _q("Ctrl+Shift+T"), this );
  _catAction->setStatusTip( _q("view file") );
  _catAction->setEnabled( false );
  connect( _catAction, SIGNAL(activated()), SLOT(cat()) );

  QMenuBar* mb = new QMenuBar
  (
#ifndef _MACOSX
   this
#endif // _MACOSX
  );
  {
    _menu = new QPopupMenu( mb );
    mb->insertItem( _q("&Actions"), _menu );

    _mergeAction->addTo(_menu);
    _undoAction->addTo(_menu);

    _diffcAction->addTo(_menu);
    _diffwcAction->addTo(_menu);
    _diffAction->addTo(_menu);
    _catAction->addTo(_menu);
  }

  resize( QSize( 700, 600 ) );
}

LogDialog::~LogDialog()
{
}

void LogDialog::updateSelection()
{
  LogEntryLvis entries;
  getSelectedRevisions( entries );

  _diffAction->setEnabled( (entries.size() == 1 || entries.size() == 2) ? true : false );
  _diffcAction->setEnabled( (entries.size() == 1) ? true : false );
  _diffwcAction->setEnabled( (entries.size() == 1) ? true : false );
  _mergeAction->setEnabled( (entries.size() > 0) ? true : false );
  _undoAction->setEnabled( (entries.size() > 0) ? true : false );
  _catAction->setEnabled( (entries.size() == 1) ? true : false );
}

void LogDialog::customEvent( QCustomEvent* ce )
{
  switch( ce->type() )
  {
  case ScLogEvent:
    {
      LogEvent* le = (LogEvent*)ce;
      LogEntryLvi* lvi = new LogEntryLvi( _revs, le->getLog() );
      if( _revs->childCount() == 1 )
      {
        _revs->setFocus();
        _revs->setSelected(lvi,true);
        clicked(lvi);

        _findNext->setEnabled(true);
        _findPrev->setEnabled(true);
      }
      break;
    }
  case ScParameterEvent:
    {
      LogDialogResultVisitor v(this, _model);
      ScParamEvent* pe = dynamic_cast<ScParamEvent*>(ce);
      pe->getParam()->accept(&v);
      break;
    }
  default:
    {
      printf( "LogDialog: unknown custom event type %d!\n", ce->type() );
    }
  }
}

void LogDialog::showCat( const QString& pathOrUrl, const QString& text )
{
  // todo uh...
  QFont font = _model->getModel()->getFontSettings()->getEditorFont();

  QTextEdit* e = new QTextEdit();
  e->setCaption( QString(_q("subcommander:log (%1)")).arg(pathOrUrl) );
  e->setText( text );
  e->setWordWrap( QTextEdit::NoWrap );
  e->setFont( font );
  e->resize( QSize( 700, 600 ) );
  e->show();
}

void LogDialog::finished()
{
  _run->setEnabled(true);
  _done->setEnabled(true);
}

void LogDialog::run()
{
  _run->setEnabled(false);
  _done->setEnabled(false);
  _log->setEnabled(false);
  _set->setEnabled(false);
  _findNext->setEnabled(false);
  _findPrev->setEnabled(false);
  _revs->setFocus();

  _revs->clear();
  _log->clear();
  _paths->clear();

  svn::Paths paths;
  paths.push_back(_source);

  LogParam* param = new LogParam( paths, _rwStart->getRevision(),
    _rwStop->getRevision(), _limitGroup->isChecked() ? _limit->value() : 0,
    _changedPaths->isChecked(), _strictHistory->isChecked(), new LogBaton(this) );
  PostCmdResult* pcres = new PostCmdResult(this);

  _model->log( param, pcres );
}

void LogDialog::clicked( QListViewItem* item )
{
  if( ! item )
  {
    return;
  }

  LogEntryLvi*           lvi = (LogEntryLvi*)item;
  const svn::LogEntryPtr e   = lvi->getLogEntry();

  _log->setText( QString::fromUtf8(e->getMessage()) );
  _log->setEnabled(true);
  _set->setEnabled(false);

  _paths->clear();
  const svn::LogEntry::ChangedPaths& paths = e->getPaths();
  for( svn::LogEntry::ChangedPaths::const_iterator it = paths.begin(); it != paths.end(); it++ )
  {
    const svn::LogEntry::Path& path = *it;

    QChar   action      = path._action;
    QString copyfromrev = (path._copyFromRev >= 0) ? QString().sprintf( "%10ld",
      (unsigned long)path._copyFromRev) : "";

    new QListViewItem( _paths, QString(action), QString::fromUtf8(path._path),
      QString::fromUtf8(path._copyFromPath), copyfromrev );
  }
}

void setSelection( QTextEdit* log, const QString& text, int pos )
{
  int line   = 0;
  int length = 0;
  for( /*line*/; line < log->paragraphs(); line++ )
  {
    int len = log->paragraphLength(line);
    if( length + len < pos )
    {
      length += len + 1;
    }
    else
    {
      break;
    }
  }

  log->setSelection( line, pos - length, line, pos - length + text.length() );
  log->ensureCursorVisible();
}

void LogDialog::find( const QString& text, bool forward, int findIdx )
{
  LogEntryLvi* lvi = (LogEntryLvi*)_revs->currentItem();

  if( ! lvi )
  {
    return;
  }

  if( lvi != _findLastLvi || text.isEmpty() )
  {
    _findLastLvi = lvi;
    _findIdx     = forward ? 0 : -1;
    findIdx      = _findIdx;
  }

  QString source = QString::fromUtf8(lvi->getLogEntry()->getMessage());

  int pos;
  if( forward )
  {
    pos = source.find( text, findIdx, false );
  }
  else
  {
    pos = source.findRev( text, findIdx, false );
  }

  if( pos != -1 )
  {
    _findIdx = pos;
    setSelection( _log, text, pos );
  }
  else
  {
    int pos = -1;
    while( lvi && pos == -1 )
    {
      if( forward )
      {
        lvi = (LogEntryLvi*)lvi->itemBelow();
      }
      else
      {
        lvi = (LogEntryLvi*)lvi->itemAbove();
      }

      if( ! lvi )
      {
        return;
      }

      QString source = QString::fromUtf8(lvi->getLogEntry()->getMessage());

      if( forward )
      {
        pos = source.find( text, 0, false );
      }
      else
      {
        pos = source.findRev( text, -1, false );
      }

      if( pos != -1 )
      {
        _revs->clearSelection();
        _revs->setCurrentItem(lvi);
        _revs->setSelected( lvi, true );
        clicked( lvi );
        _revs->ensureItemVisible(lvi);

        _findIdx     = pos;
        _findLastLvi = lvi;
        setSelection( _log, text, pos );
      }
    }
  }
}

void LogDialog::find( const QString& text )
{
  find( text, true, _findIdx );
}

void LogDialog::findNext()
{
  find( _find->text(), true, _findIdx+1 );
}

void LogDialog::findPrev()
{
  find( _find->text(), false,
    // if the first char was the last match (_findIdx = 0) we will find
    // the same match again instead of moving to a previous item.
    (_findIdx == 0) ? -1 * (_log->text().length() + 1) : _findIdx-1
    );
}

void LogDialog::getSelectedRevisions( LogEntryLvis& entries )
{
  QListViewItemIterator it( _revs, QListViewItemIterator::Selected );
  while( it.current() )
  {
    entries.push_back( (LogEntryLvi*)it.current() );
    ++it;
  }
}

void LogDialog::logChanged()
{
  LogEntryLvis entries;
  getSelectedRevisions( entries );

  if( entries.size() != 1 )
  {
    return;
  }

  _set->setEnabled(true);
}

void LogDialog::contextMenuRequest( QListViewItem* item, const QPoint& p ,int col )
{
  _menu->exec(p);
}

void LogDialog::set()
{
  QString msg = "<qt>";
  msg += " <nobr><center>";
  msg += _q("Overwrite the log message?");
  msg += "</center></nobr>";
  msg += " <nobr><center>";
  msg += _q("Its current value is lost and can NOT be restored!");
  msg += "</center></nobr>";
  msg += "</qt>";

  int answer = msgInformation( _q("subcommander:set log message"), msg,
    _q("&Ok"), _q("&Cancel") );

  if( answer != QMessageBox::Ok )
  {
    return;
  }

  _set->setEnabled(false);

  LogEntryLvis entries;
  getSelectedRevisions( entries );

  entries[0]->getLogEntry()->setMessage( sc::String(_log->text().utf8()) );

  PropSetRevParam* param = new PropSetRevParam(
    sc::String("svn:log"), sc::String(_log->text().utf8()), _source,
    new svn::RevisionNumber( entries[0]->getLogEntry()->getRevnumber() ),
    false );

   PostCmdResult* pcres = new PostCmdResult(this);

  _model->propsetrev( param, pcres );
}

void LogDialog::diff()
{
  DiffDialogCmd* diff = new DiffDialogCmd( 0, _model, _prj, false, false ); 

  LogEntryLvis entries;
  getSelectedRevisions( entries );
  MergeRevisions mRevs( entries, false );
  QString url1 = QString::fromUtf8(_source);
  svn::RevisionNumber rev1( mRevs.getStartRevision() );
  svn::RevisionNumber rev2( mRevs.getStopRevision() );

  diff->run( url1, url1, &rev1, &rev2, _folder );
}

void LogDialog::diffc()
{
  LogEntryLvis entries;
  getSelectedRevisions( entries );

  MergeRevisions mRevs( entries, false );

  DiffParam* param = new DiffParam( 
    _source, new svn::RevisionNumber( mRevs.getStartRevision() ), 
    _source, new svn::RevisionNumber( mRevs.getStopRevision() ),
    0/*no peg*/, 
    _folder ? _model->isCmdRecursive() : false,
    true, false, false,
    _folder ? true : false );

  PostCmdResult* pcres = new PostCmdResult(this);

  _model->diff( param, pcres );
}

void LogDialog::diffwc()
{
  LogEntryLvis entries;
  getSelectedRevisions( entries );

  DiffParam* param = new DiffParam( 
    _source, new svn::RevisionNumber( entries[0]->getLogEntry()->getRevnumber() ), 
    _source, new svn::Revision(svn::Revision_Working),
    0/*no peg*/, 
    _folder ? _model->isCmdRecursive() : false,
    true, false, false,
    _folder ? true : false );

  PostCmdResult* pcres = new PostCmdResult(this);

  _model->diff( param, pcres );
}

void LogDialog::merge( bool undo )
{
  MergeDialogCmd* merge = new MergeDialogCmd( 0, getTid(), _model, _prj, false, false ); 

  LogEntryLvis entries;
  getSelectedRevisions( entries );
  MergeRevisions mRevs( entries, undo );
  QString url = QString::fromUtf8(_source);
  QString wc  = QString::fromUtf8(_prj->getCurWorkingCopyPath());
  svn::RevisionNumber rev1( mRevs.getStartRevision() );
  svn::RevisionNumber rev2( mRevs.getStopRevision() );

  merge->run( url, url, &rev1, &rev2, wc );
}

void LogDialog::merge()
{
  merge(false);
}

void LogDialog::undo()
{
  merge(true);
}

void LogDialog::cat()
{
  LogEntryLvis entries;
  getSelectedRevisions( entries );

  PostCmdResult* pcres = new PostCmdResult(this);
  CatParam* param      = new CatParam(_source,
    new svn::RevisionNumber( entries[0]->getLogEntry()->getRevnumber() ) );
      
  _model->cat( param, pcres );
}
