/* ====================================================================
 * Copyright (c) 2007-2008  Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "Splitter.h"
#include "util/String.h"

// qt
#include <QtCore/QEvent>
#include <QtGui/QSplitterHandle>
#include <QtGui/QMoveEvent>
#include <QtGui/QShowEvent>

/** Initial @a position of SplitterHandle. */
const int UnsetPos = -1;

/**
 * Custom splitter handle for @a Splitter widget. Hides or shows the @a First or
 * @a Last widget when double clicking on the handle.
 */
class SplitterHandle : public QSplitterHandle
{
  typedef QSplitterHandle super;

public:
  SplitterHandle( QSplitter* parent, Qt::Orientation orientation, Splitter::Hide hide )
    : super(orientation,parent), _hide(hide), _pos(UnsetPos), _visible(true),
    _delayedShowHide(false)
  {
  }
  
  void showEvent( QShowEvent*e )
  {
    super::showEvent(e);

    if( _delayedShowHide )
      showHide(_delayedShow);
  }

  void moveEvent( QMoveEvent* e )
  {
    super::moveEvent(e);

    if( !isVisible() )
      return;

    int cpos = getCurrentPos();
    _visible = cpos != getHidePos();

    if( cpos != getHidePos() )
      setPos(cpos);
  }

  void mouseDoubleClickEvent( QMouseEvent* e )
  {
    if( _visible )
    {
      moveSplitter(getHidePos());
    }
    else
    {
      moveSplitter( _pos != UnsetPos ? _pos : getParentSize()/2 );
    }
  }

  /** Show or hide @c Splitter::Hide */
  void showHide( bool show )
  {
    if( !isVisible() )
    {
      _delayedShowHide = true;
      _delayedShow     = show;
      return;
    }
    else
    {
      _delayedShowHide = false;
      _delayedShow     = show;
    }
    
    if( (show && !_visible) || (!show && _visible) )
      mouseDoubleClickEvent(NULL);
  }

  /**
   * Get the hide position respecting the @a orientation and @c Splitter::Hide.
   */
  int getHidePos() const
  {
    if( _hide == Splitter::First )
    {
      return 0; 
    }
    else // Splitter::Last
    {
      return getParentSize() - getSize();
    }
  }

  /**
   * Get the size of the handle respecting the @a orientation. Ie. it returns
   * @c width() for @a horizontal and @c height() for @a vertical layout.
   */
  int getSize() const
  {
    if( orientation() == Qt::Vertical )
    {
      return height();
    }
    else // Qt::Horizontal
    {
      return width();
    }
  }

  /**
   * Get the handle position respecting the @a orientation. Ie. it returns
   * @c x() for @a horizontal and @c y() for @a vertical layout.
   */ 
  int getCurrentPos() const
  {
    if( orientation() == Qt::Vertical )
    {
      return y();
    }
    else // Qt::Horizontal
    {
      return x();
    }
  }

  /** Set the click position. */
  void setPos( int pos )
  {
    _pos = pos;
    moveSplitter(_pos);
  }

  /** Get the click position. */
  int getPos() const
  {
    return _pos;
  }

  /**
   * Get the size of the handles parent respecting the @a orientation. Ie.
   * it returns @c width() for @a horizontal and @c height() for @a vertical
   * layout.
   */
  int getParentSize() const
  {
    if( orientation() == Qt::Vertical )
    {
      return ((QWidget*)parent())->height();
    }
    else // Qt::Horizontal
    {
      return ((QWidget*)parent())->width();
    }
  }

private:
  Splitter::Hide _hide;

  int            _pos;             ///< visible click position
  bool           _visible;         ///< splitter content is visible

  bool           _delayedShowHide; ///< showHide called when not shown?
  bool           _delayedShow;
};


///////////////////////////////////////////////////////////////////////////////

Splitter::Splitter( QWidget* parent, Hide hide ) : super(parent), _hide(hide),
_handle(NULL)
{
}

Splitter::Splitter( QWidget* parent, Qt::Orientation orientation, Hide hide )
: super(orientation,parent), _hide(hide), _handle(NULL)
{
}

QSplitterHandle* Splitter::createHandle()
{
  if( count() == 1 )
  {
    _handle = new SplitterHandle( this, orientation(), _hide );
    return _handle;
  }
  else
  {
    return new QSplitterHandle( orientation(), this );
  }
}

void Splitter::showHide( bool b )
{
  _handle->showHide(b);
}

void Splitter::enableHide( bool b )
{
  _handle->setEnabled(b);
}

void Splitter::setHandlePos( int pos )
{
  _handle->setPos(pos);
}

int Splitter::getHandlePos() const
{
  return _handle->getPos();
}
