/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "DialogFind.h"
#include "ISubtitleEditor.h"
#include "utility.h"
#include "Config.h"

DialogFindAndReplace::DialogFindAndReplace(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::Dialog(cobject)
{
	refGlade->get_widget("entry-find", m_entryFind);
	refGlade->get_widget("entry-replace", m_entryReplace);
	refGlade->get_widget("check-ignore-case", m_checkIgnoreCase);
	refGlade->get_widget("check-used-regular-expression", m_checkUsedRegularExpression);

	refGlade->get_widget("label-replace", m_labelReplace);
	refGlade->get_widget("button-replace", m_buttonReplace);
	refGlade->get_widget("button-replace-all", m_buttonReplaceAll);
	
	Config &cfg = Config::getInstance();

	Glib::ustring text;
	bool state;

	if(cfg.get_value_string("dialog-find-and-replace", "find", text))
		m_entryFind->set_text(text);
	if(cfg.get_value_string("dialog-find-and-replace", "replace", text))
		m_entryReplace->set_text(text);
	if(cfg.get_value_bool("dialog-find-and-replace", "ignore-case", state))
		m_checkIgnoreCase->set_active(state);
	if(cfg.get_value_bool("dialog-find-and-replace", "used-regular-expression", state))
		m_checkUsedRegularExpression->set_active(state);

	show_replace(false);
}

DialogFindAndReplace::~DialogFindAndReplace()
{
	Config &cfg = Config::getInstance();

	cfg.set_value_string("dialog-find-and-replace", "find", m_entryFind->get_text());
	cfg.set_value_string("dialog-find-and-replace", "replace", m_entryReplace->get_text());
	cfg.set_value_bool("dialog-find-and-replace", "ignore-case", m_checkIgnoreCase->get_active());
	cfg.set_value_bool("dialog-find-and-replace", "used-regular-expression", m_checkUsedRegularExpression->get_active());
}

void DialogFindAndReplace::execute()
{
	Document *doc = SE::getInstance()->getDocument();
	g_return_if_fail(doc);
	
	show();


	{
		SubtitleColumnRecorder m_column;

		Gtk::TreeIter iter = doc->get_subtitle_model()->getFirst();

		while(int response = run())
		{
			if(response == CLOSE || response == Gtk::RESPONSE_CLOSE)
				break;
			else if(response == FIND)
			{
				if(!iter)
					iter = doc->get_subtitle_model()->getFirst();
				else
				{
					Glib::ustring exp = m_entryFind->get_text();

					bool found = false;

					// pour ne pas rester bloquer sur la selection
					Gtk::TreeIter selected = SE::getInstance()->getSubtitleView()->getSelected();
					if(selected)
						if(iter == selected)
							++iter;

					Glib::ustring::size_type start, len;

					// recherche dans le model
					for(; iter; ++iter)
					{
						Glib::ustring text = (*iter)[m_column.text];
						if(found = find(text, exp, start, len))
							break;
					}

					if(found && iter)
					{
						SE::getInstance()->setStatusbar("");
						SE::getInstance()->getSubtitleView()->select_and_set_cursor(iter);
					}
					else
					{
						iter = doc->get_subtitle_model()->getFirst();

						if(m_checkUsedRegularExpression->get_active())
							SE::getInstance()->setStatusbar(_("The regular expression \"%s\" was not found."), exp.c_str());
						else
							SE::getInstance()->setStatusbar(_("The text \"%s\" was not found."), exp.c_str());
					}
				}
			}
			else if(response == REPLACE)
			{
				if(iter)
				{
					Glib::ustring exp = m_entryFind->get_text();
					Glib::ustring replace = m_entryReplace->get_text();

					Glib::ustring text = (*iter)[m_column.text];

					if(find_and_replace(text, exp, replace))
						(*iter)[m_column.text] = text;
				}
			}
			else if(response == REPLACE_ALL)
			{
				iter = SE::getInstance()->getDocument()->get_subtitle_model()->getFirst();
				
				if(iter)
				{
					Glib::ustring exp = m_entryFind->get_text();
					Glib::ustring replace = m_entryReplace->get_text();

					Glib::ustring text = (*iter)[m_column.text];
					
					// recherche dans le model
					for(; iter; ++iter)
					{
						Glib::ustring text = (*iter)[m_column.text];
						if(find_and_replace(text, exp, replace))
							(*iter)[m_column.text] = text;
					}
				}
			}
			else
				break;
		}
	}

	hide();
}

/*
 *	text : la ligne
 *	exp : la recherche
 */
bool DialogFindAndReplace::find(const Glib::ustring &text, const Glib::ustring &exp, 
																	Glib::ustring::size_type &start, Glib::ustring::size_type &len)
{
	if(m_checkUsedRegularExpression->get_active())
	{
		int flag = REG_EXTENDED | REG_NEWLINE;
		// sensible a la case ?
		if(m_checkIgnoreCase->get_active())
			flag |= REG_ICASE;

		// création de l'expression
		RegEx ex(exp, flag);

		return ex.exec(text, start, len);
	}
	else
	{
		if(m_checkIgnoreCase->get_active())
		{
			Glib::ustring exp_lc = exp.lowercase();
			Glib::ustring text_lc = text.lowercase();

			Glib::ustring::size_type res = text_lc.find(exp_lc);
			if(res != Glib::ustring::npos)
			{
				start = res;
				len = exp.size();
				return true;
			}
		}
		else
		{
			Glib::ustring::size_type res = text.find(exp);
			if(res != Glib::ustring::npos)
			{
				start = res;
				len = exp.size();
				return true;
			}
		}
	}

	return false;
}

/*
 *
 */
bool DialogFindAndReplace::find_and_replace(Glib::ustring &text, const Glib::ustring &exp, const Glib::ustring &replace)
{
	bool found = false;
	Glib::ustring::size_type pos=0;
	Glib::ustring::size_type start, len;
	
	// TODO: tant que dans la boucle on recherche (si plusieurs fois dans la même ligne)
	while(find(text.substr(pos, text.size() - pos), exp, start, len))
	{
		if(start != Glib::ustring::npos && len != Glib::ustring::npos)
		{
			text.replace(start+pos, len, replace);
			found = true;
			pos = start + len;
		}
	}
	return found;
}

void DialogFindAndReplace::findNext()
{
	Document *doc = SE::getInstance()->getDocument();
	g_return_if_fail(doc);

	Glib::ustring exp = m_entryFind->get_text();
	
	if(exp.empty())
		return;

	SubtitleColumnRecorder m_column;

	Gtk::TreeIter iter = doc->get_subtitle_model()->getFirst();

	if(!iter)
		iter = doc->get_subtitle_model()->getFirst();
	else
	{
		bool found = false;

		// pour ne pas rester bloquer sur la selection
		Gtk::TreeIter selected = SE::getInstance()->getSubtitleView()->getSelected();
		if(selected)
			if(iter == selected)
				++iter;

		Glib::ustring::size_type start, len;

		// recherche dans le model
		for(; iter; ++iter)
		{
			Glib::ustring text = (*iter)[m_column.text];
			if(found = find(text, exp, start, len))
				break;
		}

		if(found && iter)
		{
			SE::getInstance()->setStatusbar("");
			SE::getInstance()->getSubtitleView()->select_and_set_cursor(iter);
		}
		else
		{
			if(m_checkUsedRegularExpression->get_active())
				SE::getInstance()->setStatusbar(_("The regular expression \"%s\" was not found."), exp.c_str());
			else
				SE::getInstance()->setStatusbar(_("The text \"%s\" was not found."), exp.c_str());
			
			iter = doc->get_subtitle_model()->getFirst();
		}
	}
}

void DialogFindAndReplace::show_replace(bool state)
{
	if(state)
	{
		//set_title(_("Find"));
		m_labelReplace->show();
		m_entryReplace->show();
		m_buttonReplace->show();
		m_buttonReplaceAll->show();
	}
	else
	{
		//set_title(_("Find And Replace"));
		m_labelReplace->hide();
		m_entryReplace->hide();
		m_buttonReplace->hide();
		m_buttonReplaceAll->hide();
	}
}

