/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "SubtitleSubRip.h"
#include <iostream>
#include <fstream>
#include <glibmm/ustring.h>
#include "utility.h"

/*
 *
 */
Glib::ustring SubtitleSubRip::get_name()
{
	return "SubRip";
}

/*
 *
 */
Glib::ustring SubtitleSubRip::get_extension()
{
	return "srt";
}

/*
 *
 */
bool SubtitleSubRip::check(const std::string &line)
{
	static RegEx ex(
				"[0-9][0-9]:[0-9][0-9]:[0-9][0-9],[0-9]{3}"
				" --> "
				"[0-9][0-9]:[0-9][0-9]:[0-9][0-9],[0-9]{3}");

	return ex.exec(line);
}

/*
 *
 */
SubtitleSubRip::SubtitleSubRip(Document* doc)
:m_model(doc->m_subtitleModel)
{
}

/*
 *
 */
SubtitleSubRip::~SubtitleSubRip()
{
}


/*
 *
 */
bool SubtitleSubRip::open(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	SubtitleFormat::open(filename,encoding);
	
	std::ifstream file(filename.c_str());
	if(!file)
	{
		throw SubtitleException("SubtitleSubRip", _("I can't open this file."));
	}

	SubtitleColumnRecorder column;

	std::string line;
	while(!file.eof() && std::getline(file, line))
	{
		unsigned int num = 0;

		// lecture du numero
		if(sscanf(line.c_str(), "%d", &num) > 0)
		{
			// lecture du temps
			if(std::getline(file, line))
			{
				int start[4], end[4];

				if(sscanf(line.c_str(), "%d:%d:%d,%d --> %d:%d:%d,%d",
							&start[0], &start[1], &start[2], &start[3], 
							&end[0], &end[1], &end[2], &end[3]) == 8)
				{
					Glib::ustring text;
					int count = 0; // permet de compter le nombre de ligne

					while(std::getline(file, line))
					{
						line = check_end_char(charset_to_utf8(line));

						if(line.size()==0)
							break;
						else
						{
							if(count > 0) // s'il y a deja une ligne alors on utlise le | pour le retour a la ligne
								text += "|";

							text += line;
							++count;
						}
					}

					Gtk::TreeIter iter = m_model->append();

					(*iter)[column.text]	= text;
					(*iter)[column.start] = SubtitleTime(start[0], start[1], start[2], start[3]).str();
					(*iter)[column.end]		= SubtitleTime(end[0], end[1], end[2], end[3]).str();
				}
			}
		}
	}

	file.close();

	return true;
}

/*
 *
 */
bool SubtitleSubRip::save(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	SubtitleFormat::save(filename,encoding);

	std::ofstream file(filename.c_str());
	if(!file)
	{
		throw SubtitleException("SubtitleSubRip", _("I can't open this file."));
	}

	SubtitleColumnRecorder column;
	Glib::ustring text;

	Gtk::TreeNodeChildren rows = m_model->children();
	for(Gtk::TreeIter it = rows.begin(); it; ++it)
	{
		file << (*it)[column.num] << std::endl;
		file << convert_time_for_save((*it)[column.start]) << " --> " << convert_time_for_save((*it)[column.end]) << std::endl;

		text = (*it)[column.text];

		Glib::ustring::size_type pos=0;

		while((pos = text.find('|', pos)) != Glib::ustring::npos)
		{
			text.replace(pos, 1, "\n");
		}

		file << utf8_to_charset(text) << std::endl;
		file << std::endl;
	}
	
	file.close();
	return true;
}


Glib::ustring SubtitleSubRip::convert_time_for_save(const Glib::ustring &time)
{
	Glib::ustring str = time;
	Glib::ustring::size_type f = str.find('.');
	str.replace(f, 1, ",");
	return str;
}

