/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include <iostream>
#include "SubtitleView.h"
#include "utility.h"
#include <gtkmm/cellrenderercombo.h>
#include <gdk/gdkkeysyms.h>

#include "Config.h"
#include "ISubtitleEditor.h"


SubtitleView::SubtitleView(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::TreeView(cobject)
{
	m_refDocument = SE::getInstance()->getDocument();
		
	set_model(m_refDocument->m_subtitleModel);

	m_subtitleModel = m_refDocument->m_subtitleModel;
	m_styleModel = m_refDocument->m_styleModel;
	
	createColumnNum();
	createColumnLayer();
	createColumnStart();
	createColumnEnd();
	createColumnStyle();
	createColumnName();
	
	createColumnMarginL();
	createColumnMarginR();
	createColumnMarginV();
	
	createColumnEffect();

	createColumnText();
	createColumnTranslation();

	set_rules_hint(true);
	set_enable_search(false);

	set_view_type("ssa");

	// config
	loadCfg();
	saveCfg();

	get_selection()->signal_changed().connect(
			sigc::mem_fun(*this, &SubtitleView::on_selection_changed));
}

void SubtitleView::loadCfg()
{
	bool state = false;
	
	Config &cfg = Config::getInstance();

#define SET(name)	if(cfg.get_value_bool("setup-view", "show-column-"name, state)) set_column_visible(name, state);

	SET("number");
	SET("layer");
	SET("start");
	SET("end");
	SET("style");
	SET("name");
	SET("margin-l");
	SET("margin-r");
	SET("margin-v");
	SET("effect");
	SET("text");
	SET("translation");
	
#undef SET
}

void SubtitleView::saveCfg()
{
	Config &cfg = Config::getInstance();

#define SET(name)	cfg.set_value_bool("setup-view", "show-column-"name, get_column_visible(name));

	SET("number");
	SET("layer");
	SET("start");
	SET("end");
	SET("style");
	SET("name");
	SET("margin-l");
	SET("margin-r");
	SET("margin-v");
	SET("effect");
	SET("text");
	SET("translation");
	
#undef SET
}

/*
 *	create columns
 */
void SubtitleView::createColumnNum()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("num")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.num);
	
	append_column(*column);
	//append_column_numeric_editable("num", m_column.num, "%d");
	renderer->property_editable() = false;

	m_columns["number"] = column;
}

/*
 *
 */
void SubtitleView::createColumnLayer()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("layer")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.layer);

	renderer->property_editable() = true;
	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_layer));

	append_column(*column);

	m_columns["layer"] = column;
}

/*
 *
 */
void SubtitleView::createColumnStart()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("start")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.start);
	
	renderer->property_editable() = true;
	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_start));

	append_column(*column);

	m_columns["start"] = column;
}

/*
 *
 */
void SubtitleView::createColumnEnd()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("end")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.end);
	
	renderer->property_editable() = true;
	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_end));

	append_column(*column);

	m_columns["end"] = column;
}

/*
 *
 */
void SubtitleView::createColumnStyle()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererCombo* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("style")));
	renderer = manage(new Gtk::CellRendererCombo);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.style);
	
	renderer->property_model() =	m_styleModel;
	renderer->property_text_column() = 0;
	renderer->property_editable() = true;
	renderer->property_has_entry() = false;
	
	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_style));

	append_column(*column);

	m_columns["style"] = column;
}

/*
 *
 */
void SubtitleView::createColumnName()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("name")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.name);
	
	renderer->property_editable() = true;

	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_name));
	
	append_column(*column);

	m_columns["name"] = column;
}

/*
 *
 */
void SubtitleView::createColumnText()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("text")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.text);
	
	append_column(*column);
	
	renderer->property_editable() = true;
	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_text));

	//renderer->signal_editing_started().connect(
	//		sigc::mem_fun(*this, &SubtitleView::editing_start));

	column->set_resizable(true);
//	column->set_expand(true);

	m_columns["text"] = column;
}

/*
 *
 */
void SubtitleView::createColumnTranslation()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("translation")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.translation);
	
	append_column(*column);
	
	renderer->property_editable() = true;
	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_translation));

	column->set_resizable(true);
//	column->set_expand(true);

	m_columns["translation"] = column;
}

/*
 *
 */
void SubtitleView::createColumnEffect()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("effect")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.effect);
	
	append_column(*column);
	
	renderer->property_editable() = true;
	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_effect));

	column->set_resizable(true);
//	column->set_expand(true);

	m_columns["effect"] = column;
}

/*
 *
 */
void SubtitleView::createColumnMarginR()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("R")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.marginR);
	
	renderer->property_editable() = true;

	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_margin_r));

	append_column(*column);

	m_columns["margin-r"] = column;
}

/*
 *
 */
void SubtitleView::createColumnMarginL()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("L")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.marginL);
	
	renderer->property_editable() = true;

	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_margin_l));

	append_column(*column);

	m_columns["margin-l"] = column;
}

/*
 *
 */
void SubtitleView::createColumnMarginV()
{
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* renderer = NULL;
	
	column = manage(new Gtk::TreeViewColumn(_("V")));
	renderer = manage(new Gtk::CellRendererText);
	
	column->pack_start(*renderer, false);
	column->add_attribute(renderer->property_text(), m_column.marginV);
	
	renderer->property_editable() = true;

	renderer->signal_edited().connect(
		sigc::mem_fun(*this, &SubtitleView::on_edited_margin_v));

	append_column(*column);

	m_columns["margin-v"] = column;
}

/*
 *	retourne l'item select ou NULL
 */
Gtk::TreeIter SubtitleView::getSelected()
{
	return get_selection()->get_selected();
}

/*
 *
 */
void SubtitleView::on_selection_changed()
{
	Signal::getInstance().subtitle_view_selection_changed(getSelected());
}

/*
 * 
 */
void SubtitleView::on_edited_layer( const Glib::ustring &path, const Glib::ustring &value)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		int val = 0;
		if(from_string(value, val))
			(*iter)[m_column.layer] = value;
	}
}


/*
 *	callback utiliser pour modifier le temps directement depuis la list (treeview)
 */
void SubtitleView::on_edited_start( const Glib::ustring &path, 
																		const Glib::ustring &newtext)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		if(SubtitleTime::validate(newtext))
		{
			SubtitleTime time(newtext);

			// ici on convertir en MSecs pour ensuite le reconvertir en temps normal (h,m,s,ms)
			// afin de pouvoir recevoir des chiffres "bizarre" ex: "00:05:99,000" (99 secs est faux)
			// il y aura donc une conversion ce qui va donner : "00:06:39,000"
			(*iter)[m_column.start] = time.str();
			
			Signal::getInstance().subtitle_time_changed(iter);
		}
		else
		{
			std::cerr << "on_edited_start::invalide> " << newtext << std::endl;
		}
	}
}

/*
 *	callback utiliser pour modifier le temps directement depuis la list (treeview)
 */
void SubtitleView::on_edited_end( const Glib::ustring &path, 
																	const Glib::ustring &newtext)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		if(SubtitleTime::validate(newtext))
		{
			SubtitleTime time(newtext);

			// ici on convertir en MSecs pour ensuite le reconvertir en temps normal (h,m,s,ms)
			// afin de pouvoir recevoir des chiffres "bizarre" ex: "00:05:99,000" (99 secs et faux)
			// il y aura donc une conversion ce qui va donner : "00:06:39,000"
			(*iter)[m_column.end] = time.str();

			Signal::getInstance().subtitle_time_changed(iter);
		}
		else
		{
			std::cerr << "on_edited_end::invalide> " << newtext << std::endl;
		}
	}
}

/*
 *	callback utiliser pour modifier le texte
 */
void SubtitleView::on_edited_text( const Glib::ustring &path, 
																		const Glib::ustring &newtext)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		(*iter)[m_column.text] = newtext;
	}
}

/*
 *	callback utiliser pour modifier le texte
 */
void SubtitleView::on_edited_translation( const Glib::ustring &path, 
																		const Glib::ustring &newtext)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		(*iter)[m_column.translation] = newtext;
	}
}

/*
 */
void SubtitleView::on_edited_effect( const Glib::ustring &path, 
																		const Glib::ustring &newtext)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		(*iter)[m_column.effect] = newtext;
	}
}


/*
 *	callback utiliser pour modifier le style a partir d'un menu
 */
void SubtitleView::on_edited_style( const Glib::ustring &path, 
																		const Glib::ustring &newstyle)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		(*iter)[m_column.style] = newstyle;
	}
}

/*
 *	callback utiliser pour modifier le nom
 */
void SubtitleView::on_edited_name( const Glib::ustring &path, 
																		const Glib::ustring &newname)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		(*iter)[m_column.name] = newname;
	}
}

/*
 * 
 */
void SubtitleView::on_edited_margin_l( const Glib::ustring &path, const Glib::ustring &value)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		int val = 0;
		if(from_string(value, val))
			(*iter)[m_column.marginL] = value;
	}
}

/*
 * 
 */
void SubtitleView::on_edited_margin_r( const Glib::ustring &path, const Glib::ustring &value)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		int val = 0;
		if(from_string(value, val))
			(*iter)[m_column.marginR] = value;
	}
}

/*
 * 
 */
void SubtitleView::on_edited_margin_v( const Glib::ustring &path, const Glib::ustring &value)
{
	Gtk::TreeIter iter = m_subtitleModel->get_iter(path);
	if(iter)
	{
		int val = 0;
		if(from_string(value, val))
			(*iter)[m_column.marginV] = value;
	}
}

/*
 *
 */
void SubtitleView::select_and_set_cursor(const Gtk::TreeIter &iter)
{
	get_selection()->select(iter);
	set_cursor(m_subtitleModel->get_path(iter), *m_columns["text"], false);
}


/*
 *	passe d'une vue simple pour le format SRT 
 *	a la vue total (SSA)
 *	afficher certaine colonne ou non
 *	"srt" ou "ssa"
 */
void SubtitleView::set_view_type(const Glib::ustring &type)
{
	if(type == "srt")
	{
		m_columns["number"]->set_visible(true);
		m_columns["layer"]->set_visible(false);
		m_columns["start"]->set_visible(true);
		m_columns["end"]->set_visible(true);
		m_columns["style"]->set_visible(false);
		m_columns["name"]->set_visible(false);
		m_columns["margin-l"]->set_visible(false);
		m_columns["margin-r"]->set_visible(false);
		m_columns["margin-v"]->set_visible(false);
		m_columns["text"]->set_visible(true);
//	m_columns["effect"]->set_visible(false);
		m_columns["translation"]->set_visible(false);
	}
	else if(type == "ssa")
	{
		m_columns["number"]->set_visible(true);
		m_columns["layer"]->set_visible(true);
		m_columns["start"]->set_visible(true);
		m_columns["end"]->set_visible(true);
		m_columns["style"]->set_visible(true);
		m_columns["name"]->set_visible(true);
		m_columns["margin-l"]->set_visible(false);
		m_columns["margin-r"]->set_visible(false);
		m_columns["margin-v"]->set_visible(false);
		m_columns["text"]->set_visible(true);
//	m_columns["effect"]->set_visible(false);
		m_columns["translation"]->set_visible(false);
	}
	else
		std::cerr << "SubtitleView::set_view_type type no found :" << type << std::endl;
}

/*
 *
 */
void SubtitleView::set_column_visible(const Glib::ustring& name, bool state)
{
	std::map<Glib::ustring, Gtk::TreeViewColumn*>::iterator it = m_columns.find(name);
	if(it == m_columns.end())
	{
		std::cerr << "SubtitleView::set_column_visible : " << name << " not found" << std::endl;
	}
	else
	{
		it->second->set_visible(state);
		Config &cfg = Config::getInstance();
		cfg.set_value_bool("setup-view", "show-column-"+name, state);
	}
}

/*
 *
 */
bool SubtitleView::get_column_visible(const Glib::ustring &name)
{
	std::map<Glib::ustring, Gtk::TreeViewColumn*>::iterator it = m_columns.find(name);
	if(it == m_columns.end())
	{
		std::cerr << "SubtitleView::set_column_visible : " << name << " not found" << std::endl;
		return false;
	}
	
	return it->second->get_visible();
}

/*
 *
 */
/*
bool SubtitleView::on_key_press_event(GdkEventKey *event)
{
	bool res = Gtk::TreeView::on_key_press_event(event);
	return res;
}

#include <gtkmm/entry.h>

Gtk::Entry *__entry = NULL;

void SubtitleView::editing_start(Gtk::CellEditable* cell, const Glib::ustring &path)
{
	std::cout << "editing_start:" << path << std::endl;

	Gtk::Entry* entry = static_cast<Gtk::Entry*>(cell);
	if(entry != NULL)
	{
		__entry = entry;

		entry->signal_key_press_event().connect(
				sigc::mem_fun(*this, &SubtitleView::entry_key_press_event));

		entry->grab_focus();
	}
}

bool SubtitleView::entry_key_press_event(GdkEventKey *ev)
{
	std::cout << "entry_key_press_event" << std::endl;

	guint state = ev->state;
	guint keyval = ev->keyval;

	if(state & GDK_CONTROL_MASK && keyval == GDK_Return)
	{
		Glib::ustring enter("\\n");
		int position = __entry->get_position();

		std::cout << "entrer" << std::endl;
		__entry->insert_text(enter, enter.size(), position);

		__entry->set_position(position + enter.size());
		return false;
	}
	return true;
}
*/

