/*
	SuperCollider real time audio synthesis system
    Copyright (c) 2002 James McCartney. All rights reserved.
	http://www.audiosynth.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "SC_PlugIn.h"

static InterfaceTable *ft;

struct Ramp : public Unit
{
	double m_level, m_slope;
	int m_counter;
};

struct Lag : public Unit
{
	float m_lag, m_b1, m_y1;
};

struct Lag2 : public Unit
{
	float m_lag, m_b1, m_y1a, m_y1b;
};

struct Lag3 : public Unit
{
	float m_lag, m_b1, m_y1a, m_y1b, m_y1c;
};

struct OnePole : public Unit
{
	float m_b1, m_y1;
};

struct OneZero : public Unit
{
	float m_b1, m_x1;
};

struct Integrator : public Unit
{
	float m_b1, m_y1;
};

struct Decay : public Unit
{
	float m_decayTime, m_y1, m_b1;
};

struct Decay2 : public Unit
{
	float m_attackTime, m_y1a, m_b1a;
	float m_decayTime, m_y1b, m_b1b;
};

struct LeakDC : public Unit
{
	float m_b1, m_x1, m_y1;
};

struct TwoPole : public Unit
{
	float m_y1, m_y2, m_b1, m_b2, m_freq, m_reson;
};

struct APF : public Unit
{
	float m_y1, m_y2, m_x1, m_x2, m_b1, m_b2, m_freq, m_reson;
};

struct TwoZero : public Unit
{
	float m_x1, m_x2, m_b1, m_b2, m_freq, m_reson;
};

struct LPZ1 : public Unit
{
	float m_x1;
};

struct HPZ1 : public Unit
{
	float m_x1;
};

struct HPZ2 : public Unit
{
	float m_x1, m_x2;
};

struct BPZ2 : public Unit
{
	float m_x1, m_x2;
};

struct BRZ2 : public Unit
{
	float m_x1, m_x2;
};

struct LPZ2 : public Unit
{
	float m_x1, m_x2;
};

struct Flip : public Unit
{
};

struct Delay1 : public Unit
{
	float m_x1;
};

struct Delay2 : public Unit
{
	float m_x1, m_x2;
};

struct Slope : public Unit
{
	float m_x1;
};

struct Slew : public Unit
{
	float mLevel;
};

struct RLPF : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq, m_reson;
};

struct RHPF : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq, m_reson;
};

struct LPF : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq;
};

struct HPF : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq;
};

struct BPF : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq, m_bw;
};

struct BRF : public Unit
{
	float m_y1, m_y2, m_a0, m_a1, m_b2, m_freq, m_bw;
};

struct LowShelf : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq, m_db;
};

struct HighShelf : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq, m_db;
};

struct MidEQ : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq, m_bw, m_db;
};

struct Resonz : public Unit
{
	float m_y1, m_y2, m_a0, m_b1, m_b2, m_freq, m_rq;
};

struct Ringz : public Unit
{
	float m_y1, m_y2, m_b1, m_b2, m_freq, m_decayTime;
};

struct FOS : public Unit
{
	float m_y1, m_a0, m_a1, m_b1;
};

struct SOS : public Unit
{
	float m_y1, m_y2, m_a0, m_a1, m_a2, m_b1, m_b2;
};

struct Formlet : public Unit
{
	float m_freq, m_decayTime, m_attackTime;
	float m_y01, m_y02, m_b01, m_b02;
	float m_y11, m_y12, m_b11, m_b12;
};

const int kMAXMEDIANSIZE = 32;

struct Median : public Unit
{
	float m_medianValue[kMAXMEDIANSIZE];
	long m_medianAge[kMAXMEDIANSIZE];
	long m_medianSize, m_medianIndex;
};

struct Compander : public Unit
{
	float m_prevmaxval, m_gain, m_clamp, m_clampcoef, m_relax, m_relaxcoef;
};

struct Normalizer : public Unit
{
	float *m_table, *m_xinbuf, *m_xoutbuf, *m_xmidbuf;
	long m_flips, m_pos, m_bufsize;
	float m_slope, m_level, m_curmaxval, m_prevmaxval, m_slopefactor;
};

struct Limiter : public Unit
{
	float *m_table, *m_xinbuf, *m_xoutbuf, *m_xmidbuf;
	long m_flips, m_pos, m_bufsize;
	float m_slope, m_level, m_curmaxval, m_prevmaxval, m_slopefactor;
};


struct Amplitude : public Unit
{
	float m_previn, m_clampcoef, m_relaxcoef;
};

struct DetectSilence : public Unit
{
	float mThresh;
	int32 mCounter, mEndCounter;
};

struct Hilbert : public Unit
{
	float m_coefs[12];
	float m_y1[12];
};

//////////////////////////////////////////////////////////////////////////////////////////////////


extern "C"
{
	void load(InterfaceTable *inTable);

	void Ramp_next(Ramp *unit, int inNumSamples);
	void Ramp_next_1(Ramp *unit, int inNumSamples);
	void Ramp_Ctor(Ramp* unit);

	void Lag_next(Lag *unit, int inNumSamples);
	void Lag_Ctor(Lag* unit);

	void Lag2_next(Lag2 *unit, int inNumSamples);
	void Lag2_Ctor(Lag2* unit);

	void Lag3_next(Lag3 *unit, int inNumSamples);
	void Lag3_Ctor(Lag3* unit);

	void OnePole_next_a(OnePole *unit, int inNumSamples);
	void OnePole_next_k(OnePole *unit, int inNumSamples);
	void OnePole_Ctor(OnePole* unit);

	void OneZero_next(OneZero* unit, int inNumSamples);
	void OneZero_Ctor(OneZero* unit);

	void Integrator_next(Integrator *unit, int inNumSamples);
	void Integrator_Ctor(Integrator* unit);

	void Decay_next(Decay *unit, int inNumSamples);
	void Decay_Ctor(Decay* unit);

	void Decay2_next(Decay2 *unit, int inNumSamples);
	void Decay2_Ctor(Decay2* unit);

	void LeakDC_next(LeakDC *unit, int inNumSamples);
	void LeakDC_Ctor(LeakDC* unit);

	void TwoPole_next(TwoPole *unit, int inNumSamples);
	void TwoPole_Ctor(TwoPole* unit);

	void TwoZero_next(TwoZero *unit, int inNumSamples);
	void TwoZero_Ctor(TwoZero* unit);

	void APF_next(APF *unit, int inNumSamples);
	void APF_Ctor(APF* unit);

	void LPZ1_next(LPZ1 *unit, int inNumSamples);
	void LPZ1_Ctor(LPZ1* unit);

	void HPZ1_next(HPZ1 *unit, int inNumSamples);
	void HPZ1_Ctor(HPZ1* unit);

	void Slope_next(Slope *unit, int inNumSamples);
	void Slope_Ctor(Slope* unit);

	void Delay1_next(Delay1 *unit, int inNumSamples);
	void Delay1_Ctor(Delay1* unit);

	void Flip_Ctor(Flip* unit);
	void Flip_next_even(Flip *unit, int inNumSamples);
	void Flip_next_odd(Flip *unit, int inNumSamples);

	void Delay2_next(Delay2 *unit, int inNumSamples);
	void Delay2_Ctor(Delay2* unit);

	void LPZ2_next(LPZ2 *unit, int inNumSamples);
	void LPZ2_Ctor(LPZ2* unit);

	void HPZ2_next(HPZ2 *unit, int inNumSamples);
	void HPZ2_Ctor(HPZ2* unit);

	void BPZ2_next(BPZ2 *unit, int inNumSamples);
	void BPZ2_Ctor(BPZ2* unit);

	void BRZ2_next(BRZ2 *unit, int inNumSamples);
	void BRZ2_Ctor(BRZ2* unit);

	void Slew_next(Slew *unit, int inNumSamples);
	void Slew_Ctor(Slew* unit);

	void RLPF_next(RLPF *unit, int inNumSamples);
	void RLPF_Ctor(RLPF* unit);

	void RHPF_next(RHPF *unit, int inNumSamples);
	void RHPF_Ctor(RHPF* unit);

	void LPF_next(LPF *unit, int inNumSamples);
	void LPF_Ctor(LPF* unit);

	void HPF_next(HPF *unit, int inNumSamples);
	void HPF_Ctor(HPF* unit);

	void BPF_next(BPF *unit, int inNumSamples);
	void BPF_Ctor(BPF* unit);

	void BRF_next(BRF *unit, int inNumSamples);
	void BRF_Ctor(BRF* unit);

	void Median_next(Median *unit, int inNumSamples);
	void Median_Ctor(Median* unit);

	void MidEQ_next(MidEQ *unit, int inNumSamples);
	void MidEQ_Ctor(MidEQ* unit);

	void LowShelf_next(LowShelf *unit, int inNumSamples);
	void LowShelf_Ctor(LowShelf* unit);

	void Resonz_next(Resonz *unit, int inNumSamples);
	void Resonz_Ctor(Resonz* unit);

	void Ringz_next(Ringz *unit, int inNumSamples);
	void Ringz_Ctor(Ringz* unit);

	void Formlet_next(Formlet *unit, int inNumSamples);
	void Formlet_Ctor(Formlet* unit);

	void FOS_next_k(FOS *unit, int inNumSamples);
	void FOS_next_a(FOS *unit, int inNumSamples);
	void FOS_Ctor(FOS* unit);

	void SOS_next_k(SOS *unit, int inNumSamples);
	void SOS_next_a(SOS *unit, int inNumSamples);
	void SOS_Ctor(SOS* unit);

	void Normalizer_next(Normalizer *unit, int inNumSamples);
	void Normalizer_Ctor(Normalizer* unit);
	void Normalizer_Dtor(Normalizer* unit);

	void Limiter_next(Limiter *unit, int inNumSamples);
	void Limiter_Ctor(Limiter* unit);
	void Limiter_Dtor(Limiter* unit);

	void Compander_next(Compander *unit, int inNumSamples);
	void Compander_Ctor(Compander* unit);

	void Amplitude_next(Amplitude *unit, int inNumSamples);
	void Amplitude_Ctor(Amplitude* unit);

	void DetectSilence_next(DetectSilence *unit, int inNumSamples);
	void DetectSilence_done(DetectSilence *unit, int inNumSamples);
	void DetectSilence_Ctor(DetectSilence* unit);

	void Hilbert_Ctor(Hilbert* unit);
	void Hilbert_next(Hilbert *unit, int inNumSamples);
/*
	void Lag_next(Lag *unit, int inNumSamples);
	void Lag_Ctor(Lag* unit);

	void Lag_next(Lag *unit, int inNumSamples);
	void Lag_Ctor(Lag* unit);
*/
}

//////////////////////////////////////////////////////////////////////////////////////////////////

void Ramp_next(Ramp *unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = IN(0);
	float period = ZIN0(1);

	double slope = unit->m_slope;
	double level = unit->m_level;
	int counter = unit->m_counter;
	int remain = inNumSamples;
	while (remain) {
		int nsmps = sc_min(remain, counter);
		LOOP(nsmps, 
			ZXP(out) = level;
			level += slope;
		);
		in += nsmps;
		counter -= nsmps;
		remain -= nsmps;
		if (counter <= 0) {
			counter = (int)(period * SAMPLERATE);
			counter = sc_max(1, counter);
			slope = (*in - level) / counter;
		}
	}
	unit->m_level = level;
	unit->m_slope = slope;
	unit->m_counter = counter;
		
}

void Ramp_next_1(Ramp *unit, int inNumSamples)
{
	float *out = OUT(0);

	*out = unit->m_level;
	unit->m_level += unit->m_slope;
	if (--unit->m_counter <= 0) {
		float in = ZIN0(0);
		float period = ZIN0(1);
		int counter = (int)(period * SAMPLERATE);
		unit->m_counter = counter = sc_max(1, counter);
		unit->m_slope = (in - unit->m_level) / counter;
	}
		
}

void Ramp_Ctor(Ramp* unit)
{
	if (BUFLENGTH == 1) {
		SETCALC(Ramp_next_1);
	} else {
		SETCALC(Ramp_next);
	}
	
	unit->m_counter = 1;
	unit->m_level = ZIN0(0);
	unit->m_slope = 0.f;
	ZOUT0(0) = unit->m_level;

}

//////////////////////////////////////////////////////////////////////////////////////////////////

void Lag_next(Lag *unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float lag = ZIN0(1);


	float y1 = unit->m_y1;
	float b1 = unit->m_b1;

	if (lag == unit->m_lag) {
		LOOP(inNumSamples, 
			float y0 = ZXP(in); 
			ZXP(out) = y1 = y0 + b1 * (y1 - y0);
		);
	} else {
		unit->m_b1 = lag == 0.f ? 0.f : exp(log001 / (lag * unit->mRate->mSampleRate));
		float b1_slope = CALCSLOPE(unit->m_b1, b1);
		unit->m_lag = lag;
		LOOP(inNumSamples, 
			b1 += b1_slope;
			float y0 = ZXP(in); 
			ZXP(out) = y1 = y0 + b1 * (y1 - y0);
		);
	}
	unit->m_y1 = zapgremlins(y1);
		
}

void Lag_Ctor(Lag* unit)
{
	SETCALC(Lag_next);
		
	unit->m_lag = 0.f;
	unit->m_b1 = 0.f;
	unit->m_y1 = ZIN0(0);
	Lag_next(unit, 1);

}

//////////////////////////////////////////////////////////////////////////////////////////////////

void Lag2_next(Lag2 *unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float lag = ZIN0(1);


	float y1a = unit->m_y1a;
	float y1b = unit->m_y1b;
	float b1 = unit->m_b1;

	if (lag == unit->m_lag) {
		LOOP(inNumSamples, 
			float y0a = ZXP(in); 
			y1a = y0a + b1 * (y1a - y0a);
			y1b = y1a + b1 * (y1b - y1a);
			ZXP(out) = y1b;
		);
	} else {
		unit->m_b1 = lag == 0.f ? 0.f : exp(log001 / (lag * unit->mRate->mSampleRate));
		float b1_slope = CALCSLOPE(unit->m_b1, b1);
		unit->m_lag = lag;
		LOOP(inNumSamples, 
			b1 += b1_slope;
			float y0a = ZXP(in); 
			y1a = y0a + b1 * (y1a - y0a);
			y1b = y1a + b1 * (y1b - y1a);
			ZXP(out) = y1b;
		);
	}
	unit->m_y1a = zapgremlins(y1a);
	unit->m_y1b = zapgremlins(y1b);
		
}

void Lag2_Ctor(Lag2* unit)
{
	SETCALC(Lag2_next);
	
	unit->m_lag = 0.f;
	unit->m_b1 = 0.f;
	unit->m_y1a = unit->m_y1b = ZIN0(0);
	Lag2_next(unit, 1);
}

//////////////////////////////////////////////////////////////////////////////////////////////////

void Lag3_next(Lag3 *unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float lag = ZIN0(1);


	float y1a = unit->m_y1a;
	float y1b = unit->m_y1b;
	float y1c = unit->m_y1c;
	float b1 = unit->m_b1;

	if (lag == unit->m_lag) {
		LOOP(inNumSamples, 
			float y0a = ZXP(in); 
			y1a = y0a + b1 * (y1a - y0a);
			y1b = y1a + b1 * (y1b - y1a);
			y1c = y1b + b1 * (y1c - y1b);
			ZXP(out) = y1c;
		);
	} else {
		unit->m_b1 = lag == 0.f ? 0.f : exp(log001 / (lag * unit->mRate->mSampleRate));
		float b1_slope = CALCSLOPE(unit->m_b1, b1);
		unit->m_lag = lag;
		LOOP(inNumSamples, 
			b1 += b1_slope;
			float y0a = ZXP(in); 
			y1a = y0a + b1 * (y1a - y0a);
			y1b = y1a + b1 * (y1b - y1a);
			y1c = y1b + b1 * (y1c - y1b);
			ZXP(out) = y1c;
		);
	}
	unit->m_y1a = zapgremlins(y1a);
	unit->m_y1b = zapgremlins(y1b);
	unit->m_y1c = zapgremlins(y1c);
		
}

void Lag3_Ctor(Lag3* unit)
{
	SETCALC(Lag3_next);
	
	unit->m_lag = 0.f;
	unit->m_b1 = 0.f;
	unit->m_y1a = unit->m_y1b = unit->m_y1c = ZIN0(0);
	Lag3_next(unit, 1);
}

//////////////////////////////////////////////////////////////////////////////////////////////////

void OnePole_next_a(OnePole *unit, int inNumSamples)
{
	//postbuf("OnePole_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float *b1p = ZIN(1);


	float y1 = unit->m_y1;

	LOOP(inNumSamples, 
		float y0 = ZXP(in); 
		float b1 = ZXP(b1p); 
		ZXP(out) = y1 = y0 + b1 * (y1 - y0);
	);
	unit->m_y1 = zapgremlins(y1);
	
}

void OnePole_next_k(OnePole *unit, int inNumSamples)
{

	//postbuf("OnePole_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float b1 = unit->m_b1;
	unit->m_b1 = ZIN0(1);


	float y1 = unit->m_y1;

	if (b1 == unit->m_b1) {
		if (b1 >= 0.f) {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0 + b1 * (y1 - y0);
			);
		} else {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0 + b1 * (y1 + y0);
			);
		}
	} else {
		float b1_slope = CALCSLOPE(unit->m_b1, b1);
		if (b1 >= 0.f && unit->m_b1 >= 0) {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0 + b1 * (y1 - y0);
				b1 += b1_slope;
			);
		} else if (b1 <= 0.f && unit->m_b1 <= 0) {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0 + b1 * (y1 + y0);
				b1 += b1_slope;
			);
		} else {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = (1.f - fabs(b1)) * y0 + b1 * y1;
				b1 += b1_slope;
			);
		}
	}
	unit->m_y1 = zapgremlins(y1);
	

}

void OnePole_Ctor(OnePole* unit)
{
	if (INRATE(1) == calc_FullRate) {
		SETCALC(OnePole_next_a);
	} else {
		SETCALC(OnePole_next_k);
	}
	unit->m_b1 = 0.f;
	unit->m_y1 = 0.f;
	OnePole_next_a(unit, 1);
}

//////////////////////////////////////////////////////////////////////////////////////////////////

void OneZero_Ctor(OneZero* unit)
{	
	SETCALC(OneZero_next);
	unit->m_b1 = 0.f;
	unit->m_x1 = ZIN0(0);
	OneZero_next(unit, 1);
}


void OneZero_next(OneZero* unit, int inNumSamples)
{
	//postbuf("OneZero::next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float b1 = unit->m_b1;
	unit->m_b1 = ZIN0(1);


	float x1 = unit->m_x1;
	if (b1 == unit->m_b1) {
		if (b1 >= 0.f) {
			LOOP(inNumSamples, 
				float x0 = ZXP(in); 
				ZXP(out) = x0 + b1 * (x1 - x0);
				x1 = x0;
			);
		} else {
			LOOP(inNumSamples, 
				float x0 = ZXP(in); 
				ZXP(out) = x0 + b1 * (x1 + x0);
				x1 = x0;
			);
		}
	} else {
		float b1_slope = CALCSLOPE(unit->m_b1, b1);
		if (b1 >= 0.f && unit->m_b1 >= 0) {
			LOOP(inNumSamples, 
				float x0 = ZXP(in); 
				ZXP(out) = x0 + b1 * (x1 - x0);
				x1 = x0;
				b1 += b1_slope;
			);
		} else if (b1 <= 0.f && unit->m_b1 <= 0) {
			LOOP(inNumSamples, 
				float x0 = ZXP(in); 
				ZXP(out) = x0 + b1 * (x1 + x0);
				x1 = x0;
				b1 += b1_slope;
			);
		} else {
			LOOP(inNumSamples, 
				float x0 = ZXP(in); 
				ZXP(out) = (1.f - fabs(b1)) * x0 + b1 * x1;
				x1 = x0;
				b1 += b1_slope;
			);
		}
	}
	unit->m_x1 = x1;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Integrator_Ctor(Integrator* unit)
{	
	//postbuf("Integrator_Reset\n");
	SETCALC(Integrator_next);
	unit->m_b1 = 0.f;
	unit->m_y1 = 0.f;
	Integrator_next(unit, 1);
}

void Integrator_next(Integrator* unit, int inNumSamples)
{
	//postbuf("Integrator_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float b1 = unit->m_b1;
	unit->m_b1 = ZIN0(1);


	float y1 = unit->m_y1;

	if (b1 == unit->m_b1) {
		if (b1 == 1.f) {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0 + y1;
			);
		} else if (b1 == 0.f) {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0;
			);
		} else {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0 + b1 * y1;
			);
		}
	} else {
		float b1_slope = CALCSLOPE(unit->m_b1, b1);
		LOOP(inNumSamples, 
			float y0 = ZXP(in); 
			ZXP(out) = y1 = y0 + b1 * y1;
			b1 += b1_slope;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Decay_Ctor(Decay* unit)
{	
	SETCALC(Decay_next);
	unit->m_decayTime = 0.f;
	unit->m_b1 = 0.f;
	unit->m_y1 = 0.f;
	Decay_next(unit, 1);
}

void Decay_next(Decay* unit, int inNumSamples)
{
	//postbuf("Decay_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float decayTime = ZIN0(1);

	float y1 = unit->m_y1;
	float b1 = unit->m_b1;
	if (decayTime == unit->m_decayTime) {
		if (b1 == 0.f) {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0;
			);
		} else {
			LOOP(inNumSamples, 
				float y0 = ZXP(in); 
				ZXP(out) = y1 = y0 + b1 * y1;
			);
		}
	} else {
		unit->m_b1 = decayTime == 0.f ? 0.f : exp(log001 / (decayTime * SAMPLERATE));
		unit->m_decayTime = decayTime;
		float b1_slope = CALCSLOPE(unit->m_b1, b1);
		//postbuf("decayTime %g  %g %g\n", unit->m_decayTime, next_b1, b1);
		LOOP(inNumSamples, 
			float y0 = ZXP(in); 
			ZXP(out) = y1 = y0 + b1 * y1;
			b1 += b1_slope;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////


void Decay2_Ctor(Decay2 *unit)
{	
	SETCALC(Decay2_next);

	float attackTime = ZIN0(1);
	float decayTime = ZIN0(2);
	unit->m_b1a = decayTime  == 0.f ? 0.f : exp(log001 / (decayTime * SAMPLERATE));
	unit->m_b1b = attackTime == 0.f ? 0.f : exp(log001 / (attackTime * SAMPLERATE));
	unit->m_decayTime = decayTime;
	unit->m_attackTime = attackTime;
	
	float y0 = ZIN0(0); 
	unit->m_y1a = y0;
	unit->m_y1b = y0;
	ZOUT0(0) = 0.f;
}

void Decay2_next(Decay2* unit, int inNumSamples)
{
	//postbuf("Decay2_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float attackTime = ZIN0(1);
	float decayTime = ZIN0(2);

	float y1a = unit->m_y1a;
	float y1b = unit->m_y1b;
	float b1a = unit->m_b1a;
	float b1b = unit->m_b1b;
	if (decayTime == unit->m_decayTime && attackTime == unit->m_attackTime) {
		LOOP(inNumSamples, 
			float y0 = ZXP(in); 
			y1a = y0 + b1a * y1a;
			y1b = y0 + b1b * y1b;
			ZXP(out) = y1a - y1b;
		);
	} else {
		unit->m_decayTime = decayTime;
		unit->m_attackTime = attackTime;
		float next_b1a = decayTime  == 0.f ? 0.f : exp(log001 / (decayTime * SAMPLERATE));
		float next_b1b = attackTime == 0.f ? 0.f : exp(log001 / (attackTime * SAMPLERATE));
		unit->m_decayTime = decayTime;
		float b1a_slope = CALCSLOPE(next_b1a, b1a);
		float b1b_slope = CALCSLOPE(next_b1b, b1b);
		unit->m_b1a = next_b1a;
		unit->m_b1b = next_b1b;
		LOOP(inNumSamples, 
			float y0 = ZXP(in); 
			y1a = y0 + b1a * y1a;
			y1b = y0 + b1b * y1b;
			ZXP(out) = y1a - y1b;
			b1a += b1a_slope;
			b1b += b1b_slope;
		);
	}
	unit->m_y1a = y1a;
	unit->m_y1b = y1b;
	unit->m_b1a = b1a;
	unit->m_b1b = b1b;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void LeakDC_Ctor(LeakDC *unit)
{	
	//postbuf("LeakDC_Ctor\n");
	SETCALC(LeakDC_next);
	unit->m_b1 = 0.f;
	unit->m_x1 = 0.f;
	unit->m_y1 = 0.f;
	LeakDC_next(unit, 1);
}


void LeakDC_next(LeakDC* unit, int inNumSamples)
{
	//postbuf("LeakDC_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float b1 = unit->m_b1;
	unit->m_b1 = ZIN0(1);

	float y1 = unit->m_y1;
	float x1 = unit->m_x1;

	if (b1 == unit->m_b1) {
		LOOP(inNumSamples, 
			float x0 = ZXP(in); 
			ZXP(out) = y1 = x0 - x1 + b1 * y1; 
			x1 = x0;
		);
	} else {
		float b1_slope = CALCSLOPE(unit->m_b1, b1);
		LOOP(inNumSamples, 
			float x0 = ZXP(in); 
			ZXP(out) = y1 = x0 - x1 + b1 * y1; 
			x1 = x0;
			b1 += b1_slope;
		);
	}
	unit->m_x1 = x1;
	unit->m_y1 = zapgremlins(y1);
	
}


////////////////////////////////////////////////////////////////////////////////////////////////////////

// I'm doing my own loop unrolling and CW unrolls the unrolls, which is too much.
#pragma opt_unroll_loops off

void TwoPole_Ctor(TwoPole *unit)
{	
	//postbuf("TwoPole_Reset\n");
	SETCALC(TwoPole_next);
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_reson = 0.f;
	ZOUT0(0) = 0.f;
}


void TwoPole_next(TwoPole* unit, int inNumSamples)
{
	//postbuf("TwoPole_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float reson = ZIN0(2);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;

	if (freq != unit->m_freq || reson != unit->m_reson) {
		float b1 = unit->m_b1;
		float b2 = unit->m_b2;
		float b1_next = 2.f * reson * cos(freq * unit->mRate->mRadiansPerSample);
		float b2_next = -(reson * reson);
		float b1_slope = (b1_next - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (b2_next - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			ZXP(out) = y0 = ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y2 = ZXP(in) + b1 * y0 + b2 * y1;
			ZXP(out) = y1 = ZXP(in) + b1 * y2 + b2 * y0;
			
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			ZXP(out) = y0 = ZXP(in) + b1 * y1 + b2 * y2;
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_reson = reson;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		float b1 = unit->m_b1;
		float b2 = unit->m_b2;
		LOOP(unit->mRate->mFilterLoops,
			ZXP(out) = y0 = ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y2 = ZXP(in) + b1 * y0 + b2 * y1;
			ZXP(out) = y1 = ZXP(in) + b1 * y2 + b2 * y0;
		);
		LOOP(unit->mRate->mFilterRemain,
			ZXP(out) = y0 = ZXP(in) + b1 * y1 + b2 * y2;
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}


////////////////////////////////////////////////////////////////////////////////////////////////////////


void TwoZero_Ctor(TwoZero* unit)
{	
	//postbuf("TwoZero_Reset\n");
	SETCALC(TwoZero_next);
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_x1 = 0.f;
	unit->m_x2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_reson = 0.f;
	ZOUT0(0) = 0.f;
}

void TwoZero_next(TwoZero* unit, int inNumSamples)
{
	//postbuf("TwoZero_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float reson = ZIN0(2);

	float x0;
	float x1 = unit->m_x1;
	float x2 = unit->m_x2;

	if (freq != unit->m_freq || reson != unit->m_reson) {
		float b1 = unit->m_b1;
		float b2 = unit->m_b2;
		float b1_next = -2.f * reson * cos(freq * unit->mRate->mRadiansPerSample);
		float b2_next = (reson * reson);
		float b1_slope = (b1_next - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (b2_next - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			x0 = ZXP(in);
			ZXP(out) = x0 + b1 * x1 + b2 * x2;
			x2 = ZXP(in);
			ZXP(out) = x2 + b1 * x0 + b2 * x1;
			x1 = ZXP(in);
			ZXP(out) = x1 + b1 * x2 + b2 * x0;
			
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			x0 = ZXP(in);
			ZXP(out) = x0 + b1 * x1 + b2 * x2;
			x2 = x1; 
			x1 = x0;
		);

		unit->m_freq = freq;
		unit->m_reson = reson;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		float b1 = unit->m_b1;
		float b2 = unit->m_b2;
		LOOP(unit->mRate->mFilterLoops,
			x0 = ZXP(in);
			ZXP(out) = x0 + b1 * x1 + b2 * x2;
			x2 = ZXP(in);
			ZXP(out) = x2 + b1 * x0 + b2 * x1;
			x1 = ZXP(in);
			ZXP(out) = x1 + b1 * x2 + b2 * x0;
		);
		LOOP(unit->mRate->mFilterRemain,
			x0 = ZXP(in);
			ZXP(out) = x0 + b1 * x1 + b2 * x2;
			x2 = x1; 
			x1 = x0;
		);
	}
	unit->m_x1 = x1;
	unit->m_x2 = x2;
	
}


////////////////////////////////////////////////////////////////////////////////////////////////////////


void APF_Ctor(APF* unit)
{	
	//postbuf("APF_Reset\n");
	SETCALC(APF_next);
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_x1 = 0.f;
	unit->m_x2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_reson = 0.f;
	ZOUT0(0) = 0.f;
}


void APF_next(APF* unit, int inNumSamples)
{
	//postbuf("APF_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float reson = ZIN0(2);

	float x0, y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float x1 = unit->m_x1;
	float x2 = unit->m_x2;

	if (freq != unit->m_freq || reson != unit->m_reson) {
		float b1 = unit->m_b1;
		float b2 = unit->m_b2;
		float b1_next = 2.f * reson * cos(freq * unit->mRate->mRadiansPerSample);
		float b2_next = -(reson * reson);
		float b1_slope = (b1_next - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (b2_next - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			x0 = ZXP(in);
			ZXP(out) = y0 = x0 + b1 * (y1 - x1) + b2 * (y2 - x2);
			x2 = ZXP(in);
			ZXP(out) = y2 = x2 + b1 * (y0 - x0) + b2 * (y2 - x1);
			x1 = ZXP(in);
			ZXP(out) = y1 = x1 + b1 * (y2 - x2) + b2 * (y2 - x0);
			
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			x0 = ZXP(in);
			ZXP(out) = y0 = x0 + b1 * (y1 - x1) + b2 * (y2 - x2);
			y2 = y1; 
			y1 = y0;
			x2 = x1; 
			x1 = x0;
		);

		unit->m_freq = freq;
		unit->m_reson = reson;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		float b1 = unit->m_b1;
		float b2 = unit->m_b2;
		LOOP(unit->mRate->mFilterLoops,
			x0 = ZXP(in);
			ZXP(out) = y0 = x0 + b1 * (y1 - x1) + b2 * (y2 - x2);
			x2 = ZXP(in);
			ZXP(out) = y2 = x2 + b1 * (y0 - x0) + b2 * (y2 - x1);
			x1 = ZXP(in);
			ZXP(out) = y1 = x1 + b1 * (y2 - x2) + b2 * (y2 - x0);
		);
		LOOP(unit->mRate->mFilterRemain,
			x0 = ZXP(in);
			ZXP(out) = y0 = x0 + b1 * (y1 - x1) + b2 * (y2 - x2);
			y2 = y1; 
			y1 = y0;
			x2 = x1; 
			x1 = x0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	unit->m_x1 = x1;
	unit->m_x2 = x2;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void LPZ1_Ctor(LPZ1* unit)
{	
	//postbuf("LPZ1_Reset\n");
	SETCALC(LPZ1_next);
	unit->m_x1 = ZIN0(0);
	LPZ1_next(unit, 1);
}


void LPZ1_next(LPZ1* unit, int inNumSamples)
{
	//postbuf("LPZ1_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x0;
	float x1 = unit->m_x1;
	
	LOOP(inNumSamples >> 2, 
		x0 = ZXP(in); 
		ZXP(out) = 0.5f * (x0 + x1);
		x1 = ZXP(in); 
		ZXP(out) = 0.5f * (x1 + x0);
		x0 = ZXP(in); 
		ZXP(out) = 0.5f * (x0 + x1);
		x1 = ZXP(in); 
		ZXP(out) = 0.5f * (x1 + x0);
	);
	LOOP(inNumSamples & 3, 
		x0 = ZXP(in); 
		ZXP(out) = 0.5f * (x0 + x1);
		x1 = x0;
	);

	unit->m_x1 = x1;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////


void HPZ1_Ctor(HPZ1* unit)
{	
	//postbuf("HPZ1_Reset\n");
	SETCALC(HPZ1_next);
	unit->m_x1 = ZIN0(0);
	HPZ1_next(unit, 1);
}


void HPZ1_next(HPZ1* unit, int inNumSamples)
{
	//postbuf("HPZ1_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x0;
	float x1 = unit->m_x1;
	
	LOOP(inNumSamples >> 2, 
		x0 = ZXP(in); 
		ZXP(out) = 0.5f * (x0 - x1);
		x1 = ZXP(in); 
		ZXP(out) = 0.5f * (x1 - x0);
		x0 = ZXP(in); 
		ZXP(out) = 0.5f * (x0 - x1);
		x1 = ZXP(in); 
		ZXP(out) = 0.5f * (x1 - x0);
	);
	LOOP(inNumSamples & 3, 
		x0 = ZXP(in); 
		//postbuf("%d %d %g %g\n", this, inNumSamples, x0, x1);
		ZXP(out) = 0.5f * (x0 - x1);
		x1 = x0;
	);

	unit->m_x1 = x1;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Slope_Ctor(Slope* unit)
{	
	//postbuf("Slope_Reset\n");
	SETCALC(Slope_next);
	unit->m_x1 = ZIN0(0);
	//postbuf("Slope_Reset %g\n", unit->m_x1);
	Slope_next(unit, 1);
}


void Slope_next(Slope* unit, int inNumSamples)
{
	//postbuf("Slope_next_a %g\n", unit->m_x1);

	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x1 = unit->m_x1;
	
	float sr = SAMPLERATE;
	LOOP(inNumSamples, 
		float x0 = ZXP(in); 
		ZXP(out) = sr * (x0 - x1);
		x1 = x0;
	);

	unit->m_x1 = x1;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Delay1_Ctor(Delay1* unit)
{	
	//postbuf("Delay1_Reset\n");
	SETCALC(Delay1_next);
	unit->m_x1 = ZIN0(0);
	Delay1_next(unit, 1);
}


void Delay1_next(Delay1* unit, int inNumSamples)
{
	//postbuf("Delay1_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x0;
	float x1 = unit->m_x1;
	
	LOOP(inNumSamples >> 2, 
		x0 = ZXP(in); 
		ZXP(out) = x1;
		x1 = ZXP(in); 
		ZXP(out) = x0;
		x0 = ZXP(in); 
		ZXP(out) = x1;
		x1 = ZXP(in); 
		ZXP(out) = x0;
	);
	LOOP(inNumSamples & 3, 
		x0 = ZXP(in); 
		ZXP(out) = x1;
		x1 = x0;
	);

	unit->m_x1 = x1;
	
}
////////////////////////////////////////////////////////////////////////////////////////////////////////

void Flip_Ctor(Flip* unit)
{
	if (unit->mBufLength & 1) {
		SETCALC(Flip_next_odd);
		Flip_next_odd(unit, 1);
	} else {
		SETCALC(Flip_next_even);
		ZOUT0(0) = ZIN0(0);
	}
}


void Flip_next_even(Flip* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	
	LOOP(inNumSamples >> 1, 
		ZXP(out) = -ZXP(in);
		ZXP(out) =  ZXP(in);
	);
}

void Flip_next_odd(Flip* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	
	if (unit->mWorld->mBufCounter & 1) {
		ZXP(out) = ZXP(in);
		LOOP(inNumSamples >> 1, 
			ZXP(out) = -ZXP(in);
			ZXP(out) =  ZXP(in);
		);
	} else {
		LOOP(inNumSamples >> 1, 
			ZXP(out) = -ZXP(in);
			ZXP(out) =  ZXP(in);
		);
		ZXP(out) = -ZXP(in);
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Delay2_Ctor(Delay2* unit)
{	
	SETCALC(Delay2_next);
	unit->m_x1 = ZIN0(0);
	ZOUT0(0) = 0.f;
}


void Delay2_next(Delay2* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x0;
	float x1 = unit->m_x1;
	float x2 = unit->m_x2;
	
	LOOP(unit->mRate->mFilterLoops, 
		x0 = ZXP(in); 
		ZXP(out) = x2;
		x2 = ZXP(in); 
		ZXP(out) = x1;
		x1 = ZXP(in); 
		ZXP(out) = x0;
	);
	LOOP(unit->mRate->mFilterRemain, 
		x0 = ZXP(in); 
		ZXP(out) = x2;
		x2 = x1;
		x1 = x0;
	);

	unit->m_x1 = x1;
	unit->m_x2 = x2;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////


void LPZ2_Ctor(LPZ2* unit)
{	
	//postbuf("LPZ2_Reset\n");
	SETCALC(LPZ2_next);
	unit->m_x1 = unit->m_x2 = ZIN0(0);
	ZOUT0(0) = 0.f;
}


void LPZ2_next(LPZ2* unit, int inNumSamples)
{
	//postbuf("LPZ2_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x0;
	float x1 = unit->m_x1;
	float x2 = unit->m_x2;
	
	LOOP(unit->mRate->mFilterLoops, 
		x0 = ZXP(in);
		ZXP(out) = (x0 + 2.f * x1 + x2) * 0.25f;
		x2 = ZXP(in);
		ZXP(out) = (x2 + 2.f * x0 + x1) * 0.25f;
		x1 = ZXP(in);
		ZXP(out) = (x1 + 2.f * x2 + x0) * 0.25f;
	);
	LOOP(unit->mRate->mFilterRemain, 
		x0 = ZXP(in); 
		ZXP(out) = (x0 + 2.f * x1 + x2) * 0.25f;
		x2 = x1;
		x1 = x0;
	);

	unit->m_x1 = x1;
	unit->m_x2 = x2;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void HPZ2_Ctor(HPZ2* unit)
{	
	//postbuf("HPZ2_Reset\n");
	SETCALC(HPZ2_next);
	unit->m_x1 = unit->m_x2 = ZIN0(0);
	ZOUT0(0) = 0.f;
}


void HPZ2_next(HPZ2* unit, int inNumSamples)
{
	//postbuf("HPZ2_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x0;
	float x1 = unit->m_x1;
	float x2 = unit->m_x2;
	
	LOOP(unit->mRate->mFilterLoops, 
		x0 = ZXP(in);
		ZXP(out) = (x0 - 2.f * x1 + x2) * 0.25f;
		x2 = ZXP(in);
		ZXP(out) = (x2 - 2.f * x0 + x1) * 0.25f;
		x1 = ZXP(in);
		ZXP(out) = (x1 - 2.f * x2 + x0) * 0.25f;
	);
	LOOP(unit->mRate->mFilterRemain, 
		x0 = ZXP(in); 
		ZXP(out) = (x0 - 2.f * x1 + x2) * 0.25f;
		x2 = x1;
		x1 = x0;
	);

	unit->m_x1 = x1;
	unit->m_x2 = x2;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void BPZ2_Ctor(BPZ2* unit)
{	
	//postbuf("BPZ2_Reset\n");
	SETCALC(BPZ2_next);
	unit->m_x1 = unit->m_x2 = ZIN0(0);
	ZOUT0(0) = 0.f;
}


void BPZ2_next(BPZ2* unit, int inNumSamples)
{
	//postbuf("BPZ2_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x0;
	float x1 = unit->m_x1;
	float x2 = unit->m_x2;
	
	LOOP(unit->mRate->mFilterLoops, 
		x0 = ZXP(in);
		ZXP(out) = (x0 - x2) * 0.5f;
		x2 = ZXP(in);
		ZXP(out) = (x2 - x1) * 0.5f;
		x1 = ZXP(in);
		ZXP(out) = (x1 - x0) * 0.5f;
	);
	LOOP(unit->mRate->mFilterRemain, 
		x0 = ZXP(in); 
		ZXP(out) = (x0 - x2) * 0.5f;
		x2 = x1;
		x1 = x0;
	);

	unit->m_x1 = x1;
	unit->m_x2 = x2;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void BRZ2_Ctor(BRZ2* unit)
{	
	//postbuf("BRZ2_Reset\n");
	SETCALC(BRZ2_next);
	unit->m_x1 = unit->m_x2 = ZIN0(0);
	ZOUT0(0) = 0.f;
}


void BRZ2_next(BRZ2* unit, int inNumSamples)
{
	//postbuf("BRZ2_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);

	float x0;
	float x1 = unit->m_x1;
	float x2 = unit->m_x2;
	
	LOOP(unit->mRate->mFilterLoops, 
		x0 = ZXP(in);
		ZXP(out) = (x0 + x2) * 0.5f;
		x2 = ZXP(in);
		ZXP(out) = (x2 + x1) * 0.5f;
		x1 = ZXP(in);
		ZXP(out) = (x1 + x0) * 0.5f;
	);
	LOOP(unit->mRate->mFilterRemain, 
		x0 = ZXP(in); 
		ZXP(out) = (x0 + x2) * 0.5f;
		x2 = x1;
		x1 = x0;
	);

	unit->m_x1 = x1;
	unit->m_x2 = x2;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Slew_Ctor(Slew* unit)
{	
	//postbuf("Slew_Reset\n");
	SETCALC(Slew_next);
	unit->mLevel = 0.f;
	Slew_next(unit, 1);
}


void Slew_next(Slew* unit, int inNumSamples)
{
	//postbuf("Slew_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float upf = ZIN0(1) *  unit->mRate->mSampleDur;
	float dnf = 0.f - ZIN0(2) *  unit->mRate->mSampleDur;
	float level = unit->mLevel;
	LOOP(inNumSamples, 
		float slope = ZXP(in) - level;
		level += sc_clip(slope,dnf,upf); 
		ZXP(out) = level; 
	);
	unit->mLevel = level;

}

/*void Slew_next_1(Slew* unit, int inNumSamples)
{
	//postbuf("Slew_next_a\n");

	float *out = ZOUT(0);
	float in = ZIN0(0);
	float upf = ZIN0(1);
	float dnf = ZIN0(2);
	float level = unit->mLevel;
	float slope = ZXP(in) - level;
	if (slope > upf) slope = upf;
	else if (slope < dnf) slope = dnf;
	level += slope; 
	ZXP(out) = level; 
	unit->mLevel = level;

}*/

////////////////////////////////////////////////////////////////////////////////////////////////////////

void RLPF_Ctor(RLPF* unit)
{	
	//postbuf("RLPF_Reset\n");
	SETCALC(RLPF_next);
	unit->m_a0 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_reson = 0.f;
	ZOUT0(0) = 0.f;
}


void RLPF_next(RLPF* unit, int inNumSamples)
{
	//postbuf("RLPF_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float reson = ZIN0(2);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq || reson != unit->m_reson) {

		float qres = sc_max(0.001, reson);
		float pfreq = freq * unit->mRate->mRadiansPerSample;
		
		float D = tan(pfreq * qres * 0.5);
		float C = ((1.f-D)/(1.f+D));
		float cosf = cos(pfreq);
		
		float next_b1 = (1.f + C) * cosf;	
		float next_b2 = -C;
		float next_a0 = (1.f + C - next_b1) * .25;
		//post("%g %g %g   %g %g   %g %g %g   %g %g\n", *freq, pfreq, qres, D, C, cosf, next_b1, next_b2, next_a0, y1, y2);

		float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
		float b1_slope = (next_b1 - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			y0 = a0 * ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y0 + 2.f * y1 + y2;
					
			y2 = a0 * ZXP(in) + b1 * y0 + b2 * y1;
			ZXP(out) = y2 + 2.f * y0 + y1;
					
			y1 = a0 * ZXP(in) + b1 * y2 + b2 * y0;
			ZXP(out) = y1 + 2.f * y2 + y0;
					
			a0 += a0_slope; 
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = a0 * ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y0 + 2.f * y1 + y2;
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_reson = reson;
		unit->m_a0 = a0;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			y0 = a0 * ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y0 + 2.f * y1 + y2;
					
			y2 = a0 * ZXP(in) + b1 * y0 + b2 * y1;
			ZXP(out) = y2 + 2.f * y0 + y1;
					
			y1 = a0 * ZXP(in) + b1 * y2 + b2 * y0;
			ZXP(out) = y1 + 2.f * y2 + y0;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = a0 * ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y0 + 2.f * y1 + y2;
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}


////////////////////////////////////////////////////////////////////////////////////////////////////////

void RHPF_Ctor(RHPF* unit)
{	
	//postbuf("RHPF_Reset\n");
	SETCALC(RHPF_next);
	unit->m_a0 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_reson = 0.f;
	ZOUT0(0) = 0.f;
}


void RHPF_next(RHPF* unit, int inNumSamples)
{
	//postbuf("RHPFs_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float reson = ZIN0(2);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq || reson != unit->m_reson) {

		float qres = sc_max(0.001, reson);
		float pfreq = freq * unit->mRate->mRadiansPerSample;
		
		float D = tan(pfreq * qres * 0.5);
		float C = ((1.f-D)/(1.f+D));
		float cosf = cos(pfreq);
		
		float next_b1 = (1.f + C) * cosf;	
		float next_b2 = -C;
		float next_a0 = (1.f + C + next_b1) * .25;

		//post("%g %g %g   %g %g   %g %g %g   %g %g\n", *freq, pfreq, qres, D, C, cosf, next_b1, next_b2, next_a0, y1, y2);

		float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
		float b1_slope = (next_b1 - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			y0 = a0 * ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y0 - 2.f * y1 + y2;
					
			y2 = a0 * ZXP(in) + b1 * y0 + b2 * y1;
			ZXP(out) = y2 - 2.f * y0 + y1;
					
			y1 = a0 * ZXP(in) + b1 * y2 + b2 * y0;
			ZXP(out) = y1 - 2.f * y2 + y0;
					
			a0 += a0_slope; 
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = a0 * ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y0 - 2.f * y1 + y2;
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_reson = reson;
		unit->m_a0 = a0;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			y0 = a0 * ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y0 - 2.f * y1 + y2;
					
			y2 = a0 * ZXP(in) + b1 * y0 + b2 * y1;
			ZXP(out) = y2 - 2.f * y0 + y1;
					
			y1 = a0 * ZXP(in) + b1 * y2 + b2 * y0;
			ZXP(out) = y1 - 2.f * y2 + y0;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = a0 * ZXP(in) + b1 * y1 + b2 * y2;
			ZXP(out) = y0 - 2.f * y1 + y2;
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void LPF_Ctor(LPF* unit)
{	
	////postbuf("LPF_Reset\n");
	SETCALC(LPF_next);
	unit->m_a0 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	ZOUT0(0) = 0.f;
}


void LPF_next(LPF* unit, int inNumSamples)
{
	//postbuf("LPF_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq) {

		float pfreq = freq * unit->mRate->mRadiansPerSample * 0.5;
		
		float C = 1.f / tan(pfreq);
		float C2 = C * C;
		float sqrt2C = C * sqrt2;
		float next_a0 = 1.f / (1.f + sqrt2C + C2);
		float next_b1 = -2.f * (1.f - C2) * next_a0 ;
		float next_b2 = -(1.f - sqrt2C + C2) * next_a0;

		//post("%g %g %g   %g %g   %g %g %g   %g %g\n", *freq, pfreq, qres, D, C, cosf, next_b1, next_b2, next_a0, y1, y2);

		float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
		float b1_slope = (next_b1 - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 + 2.f * y1 + y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 + 2.f * y0 + y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 + 2.f * y2 + y0);
					
			a0 += a0_slope; 
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 + 2.f * y1 + y2);
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_a0 = a0;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 + 2.f * y1 + y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 + 2.f * y0 + y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 + 2.f * y2 + y0);
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 + 2.f * y1 + y2);
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void HPF_Ctor(HPF* unit)
{	
	//postbuf("HPF_Reset\n");
	SETCALC(HPF_next);
	unit->m_a0 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = -1e6f;
	ZOUT0(0) = 0.f;
}


void HPF_next(HPF* unit, int inNumSamples)
{
	//postbuf("HPF_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq) {

		float pfreq = freq * unit->mRate->mRadiansPerSample * 0.5;

		float C = tan(pfreq);
		float C2 = C * C;
		float sqrt2C = C * sqrt2;
		float next_a0 = 1.f / (1.f + sqrt2C + C2);
		float next_b1 = 2.f * (1.f - C2) * next_a0 ;
		float next_b2 = -(1.f - sqrt2C + C2) * next_a0;

		//post("%g %g %g   %g %g   %g %g %g   %g %g\n", *freq, pfreq, qres, D, C, cosf, next_b1, next_b2, next_a0, y1, y2);

		float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
		float b1_slope = (next_b1 - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - 2.f * y1 + y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 - 2.f * y0 + y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 - 2.f * y2 + y0);
					
			a0 += a0_slope; 
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - 2.f * y1 + y2);
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_a0 = a0;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - 2.f * y1 + y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 - 2.f * y0 + y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 - 2.f * y2 + y0);
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - 2.f * y1 + y2);
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void BPF_Ctor(BPF* unit)
{	
	//postbuf("BPF_Reset\n");
	SETCALC(BPF_next);
	unit->m_a0 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_bw = 0.f;
	ZOUT0(0) = 0.f;
}


void BPF_next(BPF* unit, int inNumSamples)
{
	//postbuf("BPF_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float bw = ZIN0(2);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq || bw != unit->m_bw) {

		float pfreq = freq * unit->mRate->mRadiansPerSample;
		float pbw   = bw   * pfreq * 0.5;
		
		float C = 1.f / tan(pbw);
		float D = 2.f * cos(pfreq);
	
		float next_a0 = 1.f / (1.f + C);
		float next_b1 = C * D * next_a0 ;
		float next_b2 = (1.f - C) * next_a0;

		float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
		float b1_slope = (next_b1 - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 - y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 - y0);
					
			a0 += a0_slope; 
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_bw = bw;
		unit->m_a0 = a0;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 - y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 - y0);
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}


////////////////////////////////////////////////////////////////////////////////////////////////////////

void BRF_Ctor(BRF* unit)
{	
	//postbuf("BRF_Reset\n");
	SETCALC(BRF_next);
	unit->m_a0 = 0.f;
	unit->m_a1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_bw = 0.f;
	ZOUT0(0) = 0.f;
}


void BRF_next(BRF* unit, int inNumSamples)
{
	//postbuf("BRF_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float bw = ZIN0(2);

	float ay;
	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float a1 = unit->m_a1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq || bw != unit->m_bw) {
		float pfreq = freq * unit->mRate->mRadiansPerSample;
		float pbw   = bw   * pfreq * 0.5;
		
		float C = tan(pbw);
		float D = 2.f * cos(pfreq);
	
		float next_a0 = 1.f / (1.f + C);
		float next_a1 = -D * next_a0;
		float next_b2 = (1.f - C) * next_a0;

		float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
		float a1_slope = (next_a1 - a1) * unit->mRate->mFilterSlope;
		float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
		
		LOOP(unit->mRate->mFilterLoops,
			ay = a1 * y1;
			y0 = ZXP(in) - ay - b2 * y2;
			ZXP(out) = a0 * (y0 + y2) + ay;
			
			ay = a1 * y0;
			y2 = ZXP(in) - ay - b2 * y1;
			ZXP(out) = a0 * (y2 + y1) + ay;
			
			ay = a1 * y2;
			y1 = ZXP(in) - ay - b2 * y0; 
			ZXP(out) = a0 * (y1 + y0) + ay;
					
			a0 += a0_slope; 
			a1 += a1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			ay = a1 * y1;
			y0 = ZXP(in) - ay - b2 * y2;
			ZXP(out) = a0 * (y0 + y2) + ay;
			y2 = y1; 
			y1 = y0;
		);
		
		unit->m_freq = freq;
		unit->m_bw = bw;
		unit->m_a0 = a0;
		unit->m_a1 = a1;
		unit->m_b2 = b2;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			ay = a1 * y1;
			y0 = ZXP(in) - ay - b2 * y2;
			ZXP(out) = a0 * (y0 + y2) + ay;
			
			ay = a1 * y0;
			y2 = ZXP(in) - ay - b2 * y1;
			ZXP(out) = a0 * (y2 + y1) + ay;
			
			ay = a1 * y2;
			y1 = ZXP(in) - ay - b2 * y0; 
			ZXP(out) = a0 * (y1 + y0) + ay;
		);
		LOOP(unit->mRate->mFilterRemain,
			ay = a1 * y1;
			y0 = ZXP(in) - ay - b2 * y2;
			ZXP(out) = a0 * (y0 + y2) + ay;
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}


////////////////////////////////////////////////////////////////////////////////////////////////////////

void MidEQ_Ctor(MidEQ* unit)
{	
	//postbuf("MidEQ_Reset\n");
	SETCALC(MidEQ_next);
	unit->m_a0 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_bw = 0.f;
	unit->m_db = 0.f;
	ZOUT0(0) = 0.f;
}


void MidEQ_next(MidEQ* unit, int inNumSamples)
{
	//postbuf("MidEQ_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float bw = ZIN0(2);
	float db = ZIN0(3);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;
	if (freq != unit->m_freq || bw != unit->m_bw || db != unit->m_db) {

		float amp = sc_dbamp(db) - 1.0;
		float pfreq = freq * unit->mRate->mRadiansPerSample;
		float pbw   = bw   * pfreq * 0.5;
		
		float C = 1.f / tan(pbw);
		float D = 2.f * cos(pfreq);
	
		float next_a0 = 1.f / (1.f + C);
		float next_b1 = C * D * next_a0 ;
		float next_b2 = (1.f - C) * next_a0;
		next_a0 *= amp;
		float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
		float b1_slope = (next_b1 - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
		float zin;
		LOOP(unit->mRate->mFilterLoops,
			zin = ZXP(in);
			y0 = zin + b1 * y1 + b2 * y2; 
			ZXP(out) = zin + a0 * (y0 - y2);
			
			zin = ZXP(in);
			y2 = zin + b1 * y0 + b2 * y1; 
			ZXP(out) = zin + a0 * (y2 - y1);
			
			zin = ZXP(in);
			y1 = zin + b1 * y2 + b2 * y0; 
			ZXP(out) = zin + a0 * (y1 - y0);
					
			a0 += a0_slope; 
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			zin = ZXP(in);
			y0 = zin + b1 * y1 + b2 * y2; 
			ZXP(out) = zin + a0 * (y0 - y2);
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_bw = bw;
		unit->m_a0 = a0;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		float zin;
		LOOP(unit->mRate->mFilterLoops,
			zin = ZXP(in);
			y0 = zin + b1 * y1 + b2 * y2; 
			ZXP(out) = zin + a0 * (y0 - y2);
			
			zin = ZXP(in);
			y2 = zin + b1 * y0 + b2 * y1; 
			ZXP(out) = zin + a0 * (y2 - y1);
			
			zin = ZXP(in);
			y1 = zin + b1 * y2 + b2 * y0; 
			ZXP(out) = zin + a0 * (y1 - y0);
		);
		LOOP(unit->mRate->mFilterRemain,
			zin = ZXP(in);
			y0 = zin + b1 * y1 + b2 * y2; 
			ZXP(out) = zin + a0 * (y0 - y2);
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
#if 0

void LowShelf_Ctor(LowShelf* unit)
{	
	////postbuf("LowShelf_Reset\n");
	SETCALC(LowShelf_next);
	unit->m_a0 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	ZOUT0(0) = 0.f;
}


void LowShelf_next(LowShelf* unit, int inNumSamples)
{
	//postbuf("LPF_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float db = ZIN0(2);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq || db != unit->m_db) {

		float pfreq = freq * unit->mRate->mRadiansPerSample * 0.5;
		
		float C = 1.f / tan(pfreq);
		float C2 = C * C;
		float sqrt2C = C * sqrt2;
		float next_a0 = 1.f / (1.f + sqrt2C + C2);
		float next_b1 = -2.f * (1.f - C2) * next_a0 ;
		float next_b2 = -(1.f - sqrt2C + C2) * next_a0;

		//post("%g %g %g   %g %g   %g %g %g   %g %g\n", *freq, pfreq, qres, D, C, cosf, next_b1, next_b2, next_a0, y1, y2);

		float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
		float b1_slope = (next_b1 - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 + 2.f * y1 + y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 + 2.f * y0 + y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 + 2.f * y2 + y0);
					
			a0 += a0_slope; 
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 + 2.f * y1 + y2);
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_a0 = a0;
		unit->m_b1 = b1;
		unit->m_b2 = b2;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 + 2.f * y1 + y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 + 2.f * y0 + y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 + 2.f * y2 + y0);
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 + 2.f * y1 + y2);
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}

#endif

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Median_InitMedian(Median* unit, long size, float value);
float Median_InsertMedian(Median* unit, float value);

void Median_Ctor(Median* unit)
{	
	//postbuf("Median_Reset\n");
	SETCALC(Median_next);
	float in = ZIN0(1);
	unit->m_medianSize = sc_clip((int)ZIN0(0), 0, kMAXMEDIANSIZE);
	Median_InitMedian(unit, unit->m_medianSize, in);
	ZOUT0(0) = Median_InsertMedian(unit, in);
}


float Median_InsertMedian(Median* unit, float value)
{
	long i, last, pos=-1;
	
	// keeps a sorted list of the previous n=size values
	// the oldest is removed and the newest is inserted.
	// values between the oldest and the newest are shifted over by one.
	
	// values and ages are both arrays that are 'size' long.
	// the median value is always values[size>>1]
	
	last = unit->m_medianSize - 1;
	// find oldest bin and age the other bins.
	for (i=0; i<unit->m_medianSize; ++i) {
		if (unit->m_medianAge[i] == last) { // is it the oldest bin ?
			pos = i;	
		} else {
			unit->m_medianAge[i]++;	// age the bin
		}
	}
	// move values to fill in place of the oldest and make a space for the newest
	// search lower if value is too small for the open space
	while (pos != 0 && value < unit->m_medianValue[pos-1]) {
		unit->m_medianValue[pos] = unit->m_medianValue[pos-1];
		unit->m_medianAge[pos] = unit->m_medianAge[pos-1];
		pos--;
	}
	// search higher if value is too big for the open space
	while (pos != last && value > unit->m_medianValue[pos+1]) {
		unit->m_medianValue[pos] = unit->m_medianValue[pos+1];
		unit->m_medianAge[pos] = unit->m_medianAge[pos+1];
		pos++;
	}
	unit->m_medianValue[pos] = value;
	unit->m_medianAge[pos] = 0;		// this is the newest bin, age = 0
	return unit->m_medianValue[unit->m_medianSize>>1];
}

void Median_InitMedian(Median* unit, long size, float value)
{
	// initialize the arrays with the first value
	unit->m_medianSize = size;
	for (int i=0; i<size; ++i) {
		unit->m_medianValue[i] = value;
		unit->m_medianAge[i] = i;
	}
}

void Median_next(Median* unit, int inNumSamples)
{
	//postbuf("Median_next_a\n");

	float *out = ZOUT(0);
	float *in = ZIN(1);

	LOOP(inNumSamples, 
		ZXP(out) = Median_InsertMedian(unit, ZXP(in));
	);
	
}


////////////////////////////////////////////////////////////////////////////////////////////////////////

void Resonz_Ctor(Resonz* unit)
{	
	//postbuf("Resonz_Reset\n");
	SETCALC(Resonz_next);
	unit->m_a0 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_rq = 0.f;
	ZOUT0(0) = 0.f;
}


void Resonz_next(Resonz* unit, int inNumSamples)
{
	//postbuf("Resonz_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float rq = ZIN0(2);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq || rq != unit->m_rq) {
		float ffreq = freq * unit->mRate->mRadiansPerSample;
		float B = ffreq * rq;
		float R = 1.f - B * 0.5f;
		float twoR = 2.f * R;
		float R2 = R * R;
		float cost = (twoR * cos(ffreq)) / (1.f + R2);
		float b1_next = twoR * cost;
		float b2_next = -R2;
		float a0_next = (1.f - R2) * 0.5f;
		float a0_slope = (a0_next - a0) * unit->mRate->mFilterSlope;
		float b1_slope = (b1_next - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (b2_next - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 - y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 - y0);
			
			a0 += a0_slope; 
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_rq = rq;
		unit->m_a0 = a0_next;
		unit->m_b1 = b1_next;
		unit->m_b2 = b2_next;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 - y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 - y0);
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}


////////////////////////////////////////////////////////////////////////////////////////////////////////

void Ringz_Ctor(Ringz* unit)
{	
	//postbuf("Ringz_Reset\n");
	SETCALC(Ringz_next);
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	unit->m_y1 = 0.f;
	unit->m_y2 = 0.f;
	unit->m_freq = 0.f;
	unit->m_decayTime = 0.f;
	ZOUT0(0) = 0.f;
}


void Ringz_next(Ringz* unit, int inNumSamples)
{
	//postbuf("Ringz_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float decayTime = ZIN0(2);

	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = 0.5f;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;

	if (freq != unit->m_freq || decayTime != unit->m_decayTime) {
		float ffreq = freq * unit->mRate->mRadiansPerSample;
		float R = decayTime == 0.f ? 0.f : exp(log001/(decayTime * SAMPLERATE));
		float twoR = 2.f * R;
		float R2 = R * R;
		float cost = (twoR * cos(ffreq)) / (1.f + R2);
		float b1_next = twoR * cost;
		float b2_next = -R2;
		float b1_slope = (b1_next - b1) * unit->mRate->mFilterSlope;
		float b2_slope = (b2_next - b2) * unit->mRate->mFilterSlope;
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 - y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 - y0);
			
			b1 += b1_slope; 
			b2 += b2_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			y2 = y1; 
			y1 = y0;
		);

		unit->m_freq = freq;
		unit->m_decayTime = decayTime;
		unit->m_b1 = b1_next;
		unit->m_b2 = b2_next;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			
			y2 = ZXP(in) + b1 * y0 + b2 * y1; 
			ZXP(out) = a0 * (y2 - y1);
			
			y1 = ZXP(in) + b1 * y2 + b2 * y0; 
			ZXP(out) = a0 * (y1 - y0);
		);
		LOOP(unit->mRate->mFilterRemain,
			y0 = ZXP(in) + b1 * y1 + b2 * y2; 
			ZXP(out) = a0 * (y0 - y2);
			y2 = y1; 
			y1 = y0;
		);
	}
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Formlet_Ctor(Formlet* unit)
{	
	//postbuf("Formlet_Reset\n");
	SETCALC(Formlet_next);
	unit->m_b01 = 0.f;
	unit->m_b02 = 0.f;
	unit->m_y01 = 0.f;
	unit->m_y02 = 0.f;
	unit->m_b11 = 0.f;
	unit->m_b12 = 0.f;
	unit->m_y11 = 0.f;
	unit->m_y12 = 0.f;
	unit->m_freq = 0.f;
	unit->m_attackTime = 0.f;
	unit->m_decayTime = 0.f;
	ZOUT0(0) = 0.f;
}

void Formlet_next(Formlet* unit, int inNumSamples)
{
	//postbuf("Formlet_next\n");

	float *out = ZOUT(0);
	float *in = ZIN(0);
	float freq = ZIN0(1);
	float attackTime = ZIN0(2);
	float decayTime = ZIN0(3);

	float y00;
	float y10;
	float y01 = unit->m_y01;
	float y11 = unit->m_y11;
	float y02 = unit->m_y02;
	float y12 = unit->m_y12;

	float b01 = unit->m_b01;
	float b11 = unit->m_b11;
	float b02 = unit->m_b02;
	float b12 = unit->m_b12;
	float ain;
	
	if (freq != unit->m_freq || decayTime != unit->m_decayTime || attackTime != unit->m_attackTime) {
		float ffreq = freq * unit->mRate->mRadiansPerSample;
		
		float R = decayTime == 0.f ? 0.f : exp(log001/(decayTime * SAMPLERATE));
		float twoR = 2.f * R;
		float R2 = R * R;
		float cost = (twoR * cos(ffreq)) / (1.f + R2);
		float b01_next = twoR * cost;
		float b02_next = -R2;
		float b01_slope = (b01_next - b01) * unit->mRate->mFilterSlope;
		float b02_slope = (b02_next - b02) * unit->mRate->mFilterSlope;
		
		R = attackTime == 0.f ? 0.f : exp(log001/(attackTime * SAMPLERATE));
		twoR = 2.f * R;
		R2 = R * R;
		cost = (twoR * cos(ffreq)) / (1.f + R2);
		float b11_next = twoR * cost;
		float b12_next = -R2;
		float b11_slope = (b11_next - b11) * unit->mRate->mFilterSlope;
		float b12_slope = (b12_next - b12) * unit->mRate->mFilterSlope;
		
		LOOP(unit->mRate->mFilterLoops,
			ain = ZXP(in);
			y00 = ain + b01 * y01 + b02 * y02; 
			y10 = ain + b11 * y11 + b12 * y12; 
			ZXP(out) = 0.25f * ((y00 - y02) - (y10 - y12));
			
			ain = ZXP(in);
			y02 = ain + b01 * y00 + b02 * y01; 
			y12 = ain + b11 * y10 + b12 * y11; 
			ZXP(out) = 0.25f * ((y02 - y01) - (y12 - y11));
			
			ain = ZXP(in);
			y01 = ain + b01 * y02 + b02 * y00; 
			y11 = ain + b11 * y12 + b12 * y10; 
			ZXP(out) = 0.25f * ((y01 - y00) - (y11 - y10));
			
			b01 += b01_slope; 
			b02 += b02_slope;
			b11 += b11_slope; 
			b12 += b12_slope;
		);
		LOOP(unit->mRate->mFilterRemain,
			ain = ZXP(in);
			y00 = ain + b01 * y01 + b02 * y02; 
			y10 = ain + b11 * y11 + b12 * y12; 
			ZXP(out) = 0.25f * ((y00 - y02) - (y10 - y12));
			y02 = y01; 
			y01 = y00;
			y12 = y11; 
			y11 = y10;
		);

		unit->m_freq = freq;
		unit->m_attackTime = attackTime;
		unit->m_decayTime = decayTime;
		unit->m_b01 = b01_next;
		unit->m_b02 = b02_next;
		unit->m_b11 = b11_next;
		unit->m_b12 = b12_next;
	} else {
		LOOP(unit->mRate->mFilterLoops,
			ain = ZXP(in);
			y00 = ain + b01 * y01 + b02 * y02; 
			y10 = ain + b11 * y11 + b12 * y12; 
			ZXP(out) = 0.25f * ((y00 - y02) - (y10 - y12));
			
			ain = ZXP(in);
			y02 = ain + b01 * y00 + b02 * y01; 
			y12 = ain + b11 * y10 + b12 * y11; 
			ZXP(out) = 0.25f * ((y02 - y01) - (y12 - y11));
			
			ain = ZXP(in);
			y01 = ain + b01 * y02 + b02 * y00; 
			y11 = ain + b11 * y12 + b12 * y10; 
			ZXP(out) = 0.25f * ((y01 - y00) - (y11 - y10));
		);
		LOOP(unit->mRate->mFilterRemain,
			ain = ZXP(in);
			y00 = ain + b01 * y01 + b02 * y02; 
			y10 = ain + b11 * y11 + b12 * y12; 
			ZXP(out) = 0.25f * ((y00 - y02) - (y10 - y12));
			y02 = y01; 
			y01 = y00;
			y12 = y11; 
			y11 = y10;
		);
	}
	unit->m_y01 = y01;
	unit->m_y02 = y02;
	unit->m_y11 = y11;
	unit->m_y12 = y12;

}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void FOS_Ctor(FOS* unit)
{	
	//postbuf("FOS_Reset\n");
	if (INRATE(1) == calc_FullRate 
			&& INRATE(2) == calc_FullRate 
			&& INRATE(3) == calc_FullRate) {
		SETCALC(FOS_next_a);
	} else {
		SETCALC(FOS_next_k);
	}
	unit->m_y1 = 0.f;
	unit->m_a0 = 0.f;
	unit->m_a1 = 0.f;
	unit->m_b1 = 0.f;
	ZOUT0(0) = 0.f;
}

void FOS_next_a(FOS* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float *a0 = ZIN(1);
	float *a1 = ZIN(2);
	float *b1 = ZIN(3);
	
	float y1 = unit->m_y1;
	LOOP(inNumSamples,
		float y0 = ZXP(in) + ZXP(b1) * y1; 
		ZXP(out) = ZXP(a0) * y0 + ZXP(a1) * y1;
		y1 = y0;
	);
	unit->m_y1 = zapgremlins(y1);

}


void FOS_next_k(FOS* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float next_a0 = ZIN0(1);
	float next_a1 = ZIN0(2);
	float next_b1 = ZIN0(3);
	
	float y1 = unit->m_y1;
	float a0 = unit->m_a0;
	float a1 = unit->m_a1;
	float b1 = unit->m_b1;
	float a0_slope = CALCSLOPE(next_a0, a0);
	float a1_slope = CALCSLOPE(next_a1, a1);
	float b1_slope = CALCSLOPE(next_b1, b1);
	LOOP(inNumSamples,
		float y0 = ZXP(in) + b1 * y1; 
		ZXP(out) = a0 * y0 + a1 * y1;
		y1 = y0;

		a0 += a0_slope;
		a1 += a1_slope;
		b1 += b1_slope;
	);
	unit->m_y1 = zapgremlins(y1);
	unit->m_a0 = next_a0;
	unit->m_a1 = next_a1;
	unit->m_b1 = next_b1;

}
	
////////////////////////////////////////////////////////////////////////////////////////////////////////

void SOS_Ctor(SOS* unit)
{	
	//postbuf("SOS_Reset\n");
	if (INRATE(1) == calc_FullRate 
			&& INRATE(2) == calc_FullRate 
			&& INRATE(3) == calc_FullRate 
			&& INRATE(4) == calc_FullRate 
			&& INRATE(5) == calc_FullRate) {
		SETCALC(SOS_next_a);
	} else {
		SETCALC(SOS_next_k);
	}
	unit->m_y1 = 0.f;
	unit->m_a0 = 0.f;
	unit->m_a1 = 0.f;
	unit->m_a2 = 0.f;
	unit->m_b1 = 0.f;
	unit->m_b2 = 0.f;
	ZOUT0(0) = 0.f;
}

void SOS_next_a(SOS *unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float *a0 = ZIN(1);
	float *a1 = ZIN(2);
	float *a2 = ZIN(3);
	float *b1 = ZIN(4);
	float *b2 = ZIN(5);
	
	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	LOOP(unit->mRate->mFilterLoops,
		y0 = ZXP(in) + ZXP(b1) * y1 + ZXP(b2) * y2; 
		ZXP(out) = ZXP(a0) * y0 + ZXP(a1) * y1 + ZXP(a2) * y2;
		
		y2 = ZXP(in) + ZXP(b1) * y0 + ZXP(b2) * y1; 
		ZXP(out) = ZXP(a0) * y2 + ZXP(a1) * y0 + ZXP(a2) * y1;
		
		y1 = ZXP(in) + ZXP(b1) * y2 + ZXP(b2) * y0; 
		ZXP(out) = ZXP(a0) * y1 + ZXP(a1) * y2 + ZXP(a2) * y0;
	);
	LOOP(unit->mRate->mFilterRemain,
		y0 = ZXP(in) + ZXP(b1) * y1 + ZXP(b2) * y2; 
		ZXP(out) = ZXP(a0) * y0 + ZXP(a1) * y1 + ZXP(a2) * y2;
		y2 = y1; 
		y1 = y0;
	);
	
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
}

void SOS_next_k(SOS *unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float next_a0 = ZIN0(1);
	float next_a1 = ZIN0(2);
	float next_a2 = ZIN0(3);
	float next_b1 = ZIN0(4);
	float next_b2 = ZIN0(5);
	
	float y0;
	float y1 = unit->m_y1;
	float y2 = unit->m_y2;
	float a0 = unit->m_a0;
	float a1 = unit->m_a1;
	float a2 = unit->m_a2;
	float b1 = unit->m_b1;
	float b2 = unit->m_b2;
	float a0_slope = (next_a0 - a0) * unit->mRate->mFilterSlope;
	float a1_slope = (next_a1 - a1) * unit->mRate->mFilterSlope;
	float a2_slope = (next_a2 - a2) * unit->mRate->mFilterSlope;
	float b1_slope = (next_b1 - b1) * unit->mRate->mFilterSlope;
	float b2_slope = (next_b2 - b2) * unit->mRate->mFilterSlope;
	LOOP(unit->mRate->mFilterLoops,
		y0 = ZXP(in) + b1 * y1 + b2 * y2; 
		ZXP(out) = a0 * y0 + a1 * y1 + a2 * y2;
		
		y2 = ZXP(in) + b1 * y0 + b2 * y1; 
		ZXP(out) = a0 * y2 + a1 * y0 + a2 * y1;
		
		y1 = ZXP(in) + b1 * y2 + b2 * y0; 
		ZXP(out) = a0 * y1 + a1 * y2 + a2 * y0;

		a0 += a0_slope;
		a1 += a1_slope;
		a2 += a2_slope;
		b1 += b1_slope;
		b2 += b2_slope;
	);
	LOOP(unit->mRate->mFilterRemain,
		y0 = ZXP(in) + b1 * y1 + b2 * y2; 
		ZXP(out) = a0 * y0 + a1 * y1 + a2 * y2;
		y2 = y1; 
		y1 = y0;
	);
	
	unit->m_a0 = a0;
	unit->m_a1 = a1;
	unit->m_a2 = a2;
	unit->m_b1 = b1;
	unit->m_b2 = b2;
	unit->m_y1 = zapgremlins(y1);
	unit->m_y2 = zapgremlins(y2);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Compander_Ctor(Compander* unit)
{	
	//postbuf("Compander_Reset\n");
	SETCALC(Compander_next);
	unit->m_clamp = 0.f;
	unit->m_clampcoef = 0.f;
	unit->m_relax = 0.f;
	unit->m_relaxcoef = 0.f;
	unit->m_prevmaxval = 0.f;
	unit->m_gain = 0.f;
	ZOUT0(0) = 0.f;
}

void Compander_next(Compander* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float *control = ZIN(1);
	float thresh = ZIN0(2);
	float slope_below = ZIN0(3);
	float slope_above = ZIN0(4);
	float clamp = ZIN0(5);
	float relax = ZIN0(6);
	
	if (clamp != unit->m_clamp) {
		unit->m_clampcoef  = clamp == 0.0 ? 0.0 : exp(log1/(clamp * SAMPLERATE));
		unit->m_clamp = clamp;
	}
	if (relax != unit->m_relax) {
		unit->m_relaxcoef = relax == 0.0 ? 0.0 : exp(log1/(relax *  SAMPLERATE));
		unit->m_relax = relax;
	}
	
	float gain = unit->m_gain;

	float relaxcoef = unit->m_relaxcoef;
	float clampcoef = unit->m_clampcoef;

	float prevmaxval = unit->m_prevmaxval;
	
	float val;
	LOOP(inNumSamples,
		val = fabs(ZXP(control)); 
		if (val < prevmaxval) {
			val = val + (prevmaxval - val) * relaxcoef;
		} else {
			val = val + (prevmaxval - val) * clampcoef;
		}
		prevmaxval = val;
	);
	
	unit->m_prevmaxval = prevmaxval;

	float next_gain;//,absx;
	if (prevmaxval < thresh) {
		if (slope_below == 1.f) {
			next_gain = 1.f;
		} else {
			next_gain = pow(prevmaxval / thresh, slope_below - 1.f);
            //blows up here
            float32 absx = fabs(next_gain);
            //zap gremlins, but returns 1. on failure
            next_gain =  (absx > (float32)1e-15 && absx < (float32)1e15) ? next_gain : (float32)1.;    
		}
	} else {
		if (slope_above == 1.f) {
			next_gain = 1.f;
		} else {
			next_gain = pow(prevmaxval / thresh, slope_above - 1.f);
		}
	}
		
	float gain_slope = CALCSLOPE(next_gain, gain);
	LOOP(inNumSamples, ZXP(out) = ZXP(in) * gain; gain += gain_slope;);
	unit->m_gain = gain;
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////////


void Normalizer_Dtor(Normalizer* unit)
{
	RTFree(unit->mWorld, unit->m_table);
}

void Normalizer_Ctor(Normalizer* unit)
{	
	SETCALC(Normalizer_next);
	//postbuf("Normalizer_Reset\n");
	
	float dur = ZIN0(2);
	unit->m_bufsize = (long)(ceil(dur * SAMPLERATE));
	long allocsize = unit->m_bufsize * 3;
	//allocsize = NEXTPOWEROFTWO(allocsize);
	
	unit->m_table = (float*)RTAlloc(unit->mWorld, allocsize * sizeof(float));
	
	unit->m_pos = 0;
	unit->m_flips = 0;
	unit->m_level = 1.f;
	unit->m_slope = 0.f;
	unit->m_prevmaxval = 0.0;
	unit->m_curmaxval = 0.0;
	unit->m_slopefactor = 1.f / unit->m_bufsize;

	unit->m_xinbuf = unit->m_table - ZOFF;	
	unit->m_xmidbuf = unit->m_xinbuf + unit->m_bufsize;
	unit->m_xoutbuf = unit->m_xmidbuf + unit->m_bufsize;
	ZOUT0(0) = 0.f;
}


void Normalizer_next(Normalizer* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float amp = ZIN0(1);

	long pos = unit->m_pos;
	float slope = unit->m_slope;
	float level = unit->m_level;
	float curmaxval = unit->m_curmaxval;
	float val;
	
	long bufsize = unit->m_bufsize;
	long buf_remain = bufsize - pos;
	
	long remain = inNumSamples;
	while (remain > 0) {
		long nsmps = sc_min(remain, buf_remain);
		float* xinbuf = unit->m_xinbuf + pos;
		float* xoutbuf = unit->m_xoutbuf + pos;
		if (unit->m_flips >= 2) {
			LOOP(nsmps, 
				ZXP(xinbuf) = val = ZXP(in);
				ZXP(out) = level * ZXP(xoutbuf); 
				level += slope;
				val = fabs(val);
				if (val > curmaxval) curmaxval = val;
			);
		} else {
			LOOP(nsmps, 
				ZXP(xinbuf) = val = ZXP(in);
				ZXP(out) = 0.f;
				level += slope;
				val = fabs(val);
				if (val > curmaxval) curmaxval = val;
			);
		}
		pos += nsmps;
		if (pos >= bufsize) {
			pos = 0;
			buf_remain = bufsize;
			
			float maxval2 = sc_max(unit->m_prevmaxval, curmaxval);
			unit->m_prevmaxval = curmaxval;
			unit->m_curmaxval = curmaxval = 0.f;
			
			float next_level;
			if (maxval2 <= 0.00001) next_level = 100000. * amp;
			else next_level = amp / maxval2;

			slope = unit->m_slope = (next_level - level) * unit->m_slopefactor;
			
			float* temp = unit->m_xoutbuf;
			unit->m_xoutbuf = unit->m_xmidbuf;
			unit->m_xmidbuf = unit->m_xinbuf;
			unit->m_xinbuf = temp;
			
			unit->m_flips++;
		}
		remain -= nsmps;
	}

	unit->m_pos = pos;
	unit->m_level = level;
	unit->m_curmaxval = curmaxval;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Limiter_Dtor(Limiter* unit)
{
	RTFree(unit->mWorld, unit->m_table);
}

void Limiter_Ctor(Limiter* unit)
{	
	//postbuf("Limiter_Reset\n");
	SETCALC(Limiter_next);
	
	float dur = ZIN0(2);
	unit->m_bufsize = (long)(ceil(dur * SAMPLERATE));
	long allocsize = unit->m_bufsize * 3;
	allocsize = NEXTPOWEROFTWO(allocsize);
	
	unit->m_table = (float*)RTAlloc(unit->mWorld, allocsize * sizeof(float));
	
	unit->m_flips = 0;
	unit->m_pos = 0;
	unit->m_slope = 0.f;
	unit->m_level = 1.f;
	unit->m_prevmaxval = 0.0;
	unit->m_curmaxval = 0.0;
	unit->m_slopefactor = 1.f / unit->m_bufsize;

	unit->m_xinbuf = unit->m_table - ZOFF;	
	unit->m_xmidbuf = unit->m_xinbuf + unit->m_bufsize;
	unit->m_xoutbuf = unit->m_xmidbuf + unit->m_bufsize;
	ZOUT0(0) = 0.f;
}


void Limiter_next(Limiter* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	float amp = ZIN0(1);

	long pos = unit->m_pos;
	float slope = unit->m_slope;
	float level = unit->m_level;
	float curmaxval = unit->m_curmaxval;
	float val;
	
	long bufsize = unit->m_bufsize;
	long buf_remain = bufsize - pos;
	
	long remain = inNumSamples;
	while (remain > 0) {
		long nsmps = sc_min(remain, buf_remain);
		float* xinbuf = unit->m_xinbuf + pos;
		float* xoutbuf = unit->m_xoutbuf + pos;
		if (unit->m_flips >= 2) {
			LOOP(nsmps, 
				ZXP(xinbuf) = val = ZXP(in);
				ZXP(out) = level * ZXP(xoutbuf); 
				level += slope;
				val = fabs(val);
				if (val > curmaxval) curmaxval = val;
			);
		} else {
			LOOP(nsmps, 
				ZXP(xinbuf) = val = ZXP(in);
				ZXP(out) = 0.f;
				level += slope;
				val = fabs(val);
				if (val > curmaxval) curmaxval = val;
			);
		}
		pos += nsmps;
		if (pos >= bufsize) {
			pos = 0;
			buf_remain = bufsize;
			
			float maxval2 = sc_max(unit->m_prevmaxval, curmaxval);
			unit->m_prevmaxval = curmaxval;
			unit->m_curmaxval = curmaxval = 0.f;
			
			float next_level;
			if (maxval2 > amp) next_level = amp / maxval2;
			else next_level = 1.0;
			
			slope = unit->m_slope = (next_level - level) * unit->m_slopefactor;
			
			float* temp = unit->m_xoutbuf;
			unit->m_xoutbuf = unit->m_xmidbuf;
			unit->m_xmidbuf = unit->m_xinbuf;
			unit->m_xinbuf = temp;
			
			unit->m_flips++;
		}
		remain -= nsmps;
	}

	unit->m_pos = pos;
	unit->m_level = level;
	unit->m_curmaxval = curmaxval;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void Amplitude_Ctor(Amplitude* unit)
{
	//postbuf("Amplitude_Reset\n");
	SETCALC(Amplitude_next);
		
	float clamp = ZIN0(1);
	unit->m_clampcoef  = clamp == 0.0 ? 0.0 : exp(log1/(clamp * SAMPLERATE));

	float relax = ZIN0(2);
	unit->m_relaxcoef = relax == 0.0 ? 0.0 : exp(log1/(relax * SAMPLERATE));

	ZOUT0(0) = unit->m_previn = ZIN0(0);
}

#if 1

void Amplitude_next(Amplitude* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	
	float relaxcoef = unit->m_relaxcoef;
	float clampcoef = unit->m_clampcoef;
	float previn = unit->m_previn;
	
	float val;
	LOOP(inNumSamples,
		val = fabs(ZXP(in)); 
		if (val < previn) {
			val = val + (previn - val) * relaxcoef;
		} else {
			val = val + (previn - val) * clampcoef;
		}
		ZXP(out) = previn = val;
	);
	
	unit->m_previn = previn;
}

#else 

void Amplitude_next(Amplitude* unit, int inNumSamples)
{
	float *out = ZOUT(0);
	float *in = ZIN(0);
	
	float maxval = 0.f;
	LOOP(inNumSamples,
		float val = fabs(ZXP(in)); 
		if (val > maxval) maxval = val;
	);
	
	float previn = unit->m_previn;
	float coef = maxval < previn ? unit->m_relaxcoef : unit->m_clampcoef ;

	LOOP(inNumSamples, ZXP(out) = previn = maxval + (previn - maxval) * coef; );
	
	unit->m_previn = previn;
}

#endif


////////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma opt_unroll_loops reset

////////////////////////////////////////////////////////////////////////////////////////////////////////

void DetectSilence_Ctor(DetectSilence* unit)
{	
	//postbuf("DetectSilence_Reset\n");
	SETCALC(DetectSilence_next);
	
	unit->mThresh = ZIN0(1);
	unit->mEndCounter = (int32)(SAMPLERATE * ZIN0(2));
	unit->mCounter = -1;		 
}


void DetectSilence_next(DetectSilence* unit, int inNumSamples)
{
	float thresh = unit->mThresh;
	int counter = unit->mCounter;

	// I thought of a better way to do this...
	/*
	for (int i=0; i<inNumSamples; ++i) {
		float val = fabs(ZXP(in)); 
		if (val >= thresh) counter = 0;
		else if (counter >= 0) {
			if (++counter >= unit->mEndCounter && doneAction) {
				int doneAction = (int)ZIN0(3);
				DoneAction(doneAction, unit);
				SETCALC(DetectSilence_done);
			}
		}
		ZXP(out) = 0.f;
	}
	*/
	float *in = IN(0);
	for (int i=0; i<inNumSamples; ++i) {
		float val = fabs(*in++); 
		if (val > thresh) {
			counter = 0;
			break;
		} else if (counter >= 0) {
			if (++counter >= unit->mEndCounter) {
				int doneAction = (int)ZIN0(3);
				DoneAction(doneAction, unit);
				SETCALC(DetectSilence_done);
			}
		}
	}
	unit->mCounter = counter;
}

void DetectSilence_done(DetectSilence* unit, int inNumSamples)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
// Based on HilbertIIR from SC2
// a 12 pole (6 per side) Hilbert IIR filter
// based on Sean Costello and Bernie Hutchins
// created by jl anderson - 7 jan 2001
// UGen created by Josh Parmenter

void Hilbert_Ctor(Hilbert *unit)
{
	// calculate coefs based on SampleRate, store in the struct
    SETCALC(Hilbert_next);
    
    float gamconst = (15.0 * pi) / SAMPLERATE;
    float gamma01 = gamconst * 0.3609;
    float gamma02 = gamconst * 2.7412;
    float gamma03 = gamconst * 11.1573;
    float gamma04 = gamconst * 44.7581;
    float gamma05 = gamconst * 179.6242;
    float gamma06 = gamconst * 798.4578;
    float gamma07 = gamconst * 1.2524;
    float gamma08 = gamconst * 5.5671;
    float gamma09 = gamconst * 22.3423;
    float gamma10 = gamconst * 89.6271;
    float gamma11 = gamconst * 364.7914;
    float gamma12 = gamconst * 2770.1114;
    unit->m_coefs[0] = (gamma01 - 1) / (gamma01 + 1);
    unit->m_coefs[1] = (gamma02 - 1) / (gamma02 + 1);
    unit->m_coefs[2] = (gamma03 - 1) / (gamma03 + 1);
    unit->m_coefs[3] = (gamma04 - 1) / (gamma04 + 1);
    unit->m_coefs[4] = (gamma05 - 1) / (gamma05 + 1);
    unit->m_coefs[5] = (gamma06 - 1) / (gamma06 + 1);
    unit->m_coefs[6] = (gamma07 - 1) / (gamma07 + 1);
    unit->m_coefs[7] = (gamma08 - 1) / (gamma08 + 1);
    unit->m_coefs[8] = (gamma09 - 1) / (gamma09 + 1);
    unit->m_coefs[9] = (gamma10 - 1) / (gamma10 + 1);
    unit->m_coefs[10] = (gamma11 - 1) / (gamma11 + 1);
    unit->m_coefs[11] = (gamma12 - 1) / (gamma12 + 1);
    for(int i = 0; i < 12; ++ i) {
		unit->m_y1[i] = 0.0;
    }
    Hilbert_next(unit, 1);
}

void Hilbert_next(Hilbert *unit, int inNumSamples)
{
    float *in = IN(0);
    float *outcos = OUT(0);
    float *outsin = OUT(1);
    float y1[12];
    float coefs[12];
        
    // each filter's last sample
    for(int i = 0; i < 12; ++i){
	y1[i] = unit->m_y1[i];
	coefs[i] = unit->m_coefs[i];
	}
    
    float ay1, ay2, ay3, ay4, ay5, ay6;
    float ay7, ay8, ay9, ay10, ay11, ay12;
    
    float y0_1, y0_2, y0_3, y0_4, y0_5, y0_6;
    float y0_7, y0_8, y0_9, y0_10, y0_11, y0_12;
	    
    for (int i = 0; i < inNumSamples; ++i) {

		float thisin = in[i];
	
		y0_1 = thisin - (coefs[0]) * y1[0];
		ay1 = coefs[0] * y0_1 + 1 * y1[0]; 
		y1[0] = y0_1;
		y0_2 = ay1 - (coefs[1]) * y1[1];
		ay2 = coefs[1] * y0_2 + 1 * y1[1]; 
		y1[1] = y0_2;
		y0_3 = ay2 - (coefs[2]) * y1[2];
		ay3 = coefs[2] * y0_3 + 1 * y1[2]; 
		y1[2] = y0_3;	
		y0_4 = ay3 - (coefs[3]) * y1[3];
		ay4 =coefs[3] *  y0_4 + 1 * y1[3]; 
		y1[3] = y0_4;
		y0_5 = ay4 - (coefs[4]) * y1[4];
		ay5 = coefs[4] * y0_5 + 1 * y1[4]; 
		y1[4] = y0_5;	
		y0_6 = ay5 - (coefs[5]) * y1[5];
		outcos[i] = ay6 = coefs[5] * y0_6 + 1 * y1[5]; 
		y1[5] = y0_6;
		
		y0_7 = thisin - (coefs[6]) * y1[6];
		ay7 = coefs[6] * y0_7 + 1 * y1[6]; 
		y1[6] = y0_7;
		y0_8 = ay7 - (coefs[7]) * y1[7];
		ay8 = coefs[7] * y0_8 + 1 * y1[7]; 
		y1[7] = y0_8;
		y0_9 = ay8 - (coefs[8]) * y1[8];
		ay9 = coefs[8] * y0_9 + 1 * y1[8]; 
		y1[8] = y0_9;
		y0_10 = ay9 - (coefs[9]) * y1[9];
		ay10 = coefs[9] * y0_10 + 1 * y1[9]; 
		y1[9] = y0_10;
		y0_11 = ay10 - (coefs[10]) * y1[10];
		ay11 = coefs[10] * y0_11  + 1 * y1[10]; 
		y1[10] = y0_11;
		y0_12 = ay11 - (coefs[11]) * y1[11];
		outsin[i] = ay12 = coefs[11] * y0_12 + 1 * y1[11]; 
		y1[11] = y0_12;
	}
	
    for(int i = 0; i < 12; ++i) {
		unit->m_y1[i] = zapgremlins(y1[i]);
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void load(InterfaceTable *inTable)
{
	ft = inTable;

	DefineSimpleUnit(Ramp);
	DefineSimpleUnit(Lag);
	DefineSimpleUnit(Lag2);
	DefineSimpleUnit(Lag3);
	DefineSimpleUnit(OnePole);
	DefineSimpleUnit(OneZero);
	DefineSimpleUnit(TwoPole);
	DefineSimpleUnit(TwoZero);
	DefineSimpleUnit(Decay);
	DefineSimpleUnit(Decay2);
	DefineSimpleUnit(Flip);
	DefineSimpleUnit(Delay1);
	DefineSimpleUnit(Delay2);
	DefineSimpleUnit(Integrator);
	DefineSimpleUnit(LeakDC);
	DefineSimpleUnit(LPZ1);
	DefineSimpleUnit(HPZ1);
	DefineSimpleUnit(LPZ2);
	DefineSimpleUnit(HPZ2);
	DefineSimpleUnit(BPZ2);
	DefineSimpleUnit(BRZ2);
	DefineSimpleUnit(APF);
	DefineSimpleUnit(LPF);
	DefineSimpleUnit(HPF);
	DefineSimpleUnit(BPF);
	DefineSimpleUnit(BRF);
	DefineSimpleUnit(RLPF);
	DefineSimpleUnit(RHPF);
	
	DefineSimpleUnit(Slew);
	DefineSimpleUnit(Slope);

	DefineSimpleUnit(MidEQ);
	//DefineSimpleUnit(LowShelf);
	//DefineSimpleUnit(HighShelf);
	DefineSimpleUnit(Median);

	DefineSimpleUnit(Resonz);
	DefineSimpleUnit(Ringz);
	DefineSimpleUnit(Formlet);

	DefineSimpleUnit(FOS);
	DefineSimpleUnit(SOS);

	DefineSimpleUnit(Compander);
	DefineDtorUnit(Limiter);
	DefineDtorUnit(Normalizer);

	DefineSimpleUnit(Amplitude);
	DefineSimpleUnit(DetectSilence);

	DefineSimpleUnit(Hilbert);
}

//////////////////////////////////////////////////////////////////////////////////////////////////

