/***************************************************************************
 *   Copyright (C) 2003 by Hans Karlsson                                   *
 *   karlsson.h@home.se                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <Python.h>
#include "imagelabel.h"
#include <qpixmap.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <kpixmapeffect.h>
#include <kdebug.h>
#include <kimageeffect.h>

// Effect
Effect::Effect(ImageLabel* img, int msec) :
	myImage(img)
{
	if (msec > 0)
	{
		// remove the effect after the given time
		//QTimer::singleShot (millisec, myImage, SLOT(slotEffectExpired()));
		//timer -> changeInterval(millisec);
    millisec = msec;
  }
  else
  {
    millisec = msec;
  }
}

Effect::~Effect()
{}

void Effect::startTimer()
{
  if (millisec > 0)
  {
    QTimer::singleShot (millisec, myImage, SLOT(slotEffectExpired()));
    millisec = 0;
  }
}

// Intensity
Intensity::Intensity(ImageLabel* img, float r, int millisec) : Effect(img, millisec)
{
	ratio = r;
	ratio = (ratio > 1)  ? 1 : ratio;
	ratio = (ratio < -1) ? -1 : ratio;
}

KPixmap Intensity::apply(KPixmap pixmap)
{
	return KPixmapEffect::intensity(pixmap, ratio);
}

// ChannelIntensity
ChannelIntensity::ChannelIntensity(ImageLabel* img, float r, QString c, int millisec) : Effect(img, millisec)
{
	ratio = r;
	ratio = (ratio > 1)  ? 1 : ratio;
	ratio = (ratio < -1) ? -1 : ratio;

	channel = 0;
	if (c.find("red", 0 , false))
	{
		channel = 0;
	}
	else if (c.find("green", 0, false))
	{
		channel = 1;
	}
	else if (c.find("blue", 0, false))
	{
		channel = 2;
	}
}

KPixmap ChannelIntensity::apply(KPixmap pixmap)
{
	return KPixmapEffect::channelIntensity(pixmap, ratio, (KPixmapEffect::RGBComponent)channel);
}

// ToGray
ToGray::ToGray(ImageLabel* img, int millisec) : Effect(img, millisec)
{
}

KPixmap ToGray::apply(KPixmap pixmap)
{
	return KPixmapEffect::toGray(pixmap);
}

/***********************************************************************/


ImageLabel::ImageLabel(int ix,int iy,int iw,int ih)
 : Meter(ix,iy,iw,ih), zoomed(false), rollover(false)
 {
   background = 0;
  hidden  = 0;
  cblend = 0;
  //scaleMat.reset();
  //rotMat.reset();
  scale_w = 1;
  scale_h = 1;
  rot_angle = 0;
  
  doScale = false;
  doRotate = false;

  imageEffect = 0;
}

ImageLabel::ImageLabel() : Meter(), zoomed(false), rollover(false){
  hidden  = 0;
  cblend = 0;
  background = 0;
}
ImageLabel::~ImageLabel(){
  if (imageEffect != 0)
    {
      delete imageEffect;
      imageEffect = 0;
    }  
}

void ImageLabel::setValue(int v){
 setValue( QString::number( v ) );
}

void ImageLabel::showImage(){
  hidden = 0;
  setEnabled(true);
}

void ImageLabel::hideImage(){
  hidden = 1;
  setEnabled(false);
}

void ImageLabel::rotate(int deg){
  doRotate = !(deg == 0);

  rot_angle = deg;

  applyTransformations();
}

void ImageLabel::scale(int w, int h){
  doScale = !(w == realpixmap.width() && h == realpixmap.height());

  scale_w = w;
  scale_h = h;

  applyTransformations();
}

void ImageLabel::smoothScale(int w, int h){
    doScale = !(w == realpixmap.width() && h == realpixmap.height());

    scale_w = w;
    scale_h = h;

    applyTransformations(true);

//  double widthFactor = ((double)w) / ((double)realpixmap.width());
//  double heightFactor = ((double)h) / ((double)realpixmap.height());

//  pixmap.convertFromImage(realpixmap.convertToImage().smoothScale(w, h));

//  setWidth(pixmap.width());
//  setHeight(pixmap.height());

}

void ImageLabel::removeImageTransformations()
{
    doScale = false;
    doRotate = false;

    scale_w = 1;
    scale_h = 1;
    rot_angle = 0;
    pixmap = realpixmap;
}

void ImageLabel::applyTransformations(bool useSmoothScale)
{
    pixmap = realpixmap;
    if (doRotate)
    {
        // KDE and QT seem to miss a high quality image rotation
        QWMatrix rotMat;
        rotMat.rotate(rot_angle);
        pixmap = pixmap.xForm(rotMat);
    }
    if (doScale)
    {
        if (myKaramba -> useSmoothTransforms() || useSmoothScale)
        {
            pixmap.convertFromImage(pixmap.convertToImage().smoothScale(scale_w, scale_h));
        }
        else
        {
            double widthFactor = ((double)scale_w) / ((double)realpixmap.width());
            double heightFactor = ((double)scale_h) / ((double)realpixmap.height());
            QWMatrix scaleMat;
            scaleMat.scale(widthFactor, heightFactor);
            pixmap = pixmap.xForm(scaleMat);
        }
    }
    if (imageEffect != 0)
    {
        pixmap = imageEffect -> apply(pixmap);
    }
    setWidth(pixmap.width());
    setHeight(pixmap.height());
}

void ImageLabel::setValue(QString fn)
{
  // use the first line
  QStringList sList = QStringList::split( "\n", fn );
  QString fileName = *sList.begin();
  QFileInfo fileInfo( fileName );
  QString path;

  QRegExp rx("^http://",false);
  bool fileOnNet = (rx.search(fileName)!=-1)?true:false;


  if( fileInfo.isRelative() && !fileOnNet )
  {
     path = themePath + "/" + fileName;
  }
  else
  {
     path = fileName;
  }

  if ( fileOnNet )
  {
      QString tmpFile;
      if(KIO::NetAccess::download(KURL(path), tmpFile))
      {
          realpixmap = KPixmap(tmpFile);
	      pixmap = realpixmap;
	      setWidth(pixmap.width());
	      setHeight(pixmap.height());

          KIO::NetAccess::removeTempFile(tmpFile);
           qDebug( "Downloaded: " + path + " to " + tmpFile );
      }
      else
      {
          qDebug( "Error Downloading: " + path );
      }
  }
  else
  {
    realpixmap = KPixmap( path );
    pixmap = realpixmap;
    setWidth(pixmap.width());
    setHeight(pixmap.height());

  }

  pixmapWidth = pixmap.width();
  pixmapHeight = pixmap.height();

  rect_off = QRect(getX(),getY(),pixmapWidth,pixmapHeight);
}


//Matthew Kay: a new version of setValue to be used by createTaskIcon()
/**
 * set the internal pixmap of this image to the given QPixmap pix
 */
void ImageLabel::setValue(QPixmap& pix) {
  realpixmap = KPixmap(pix);
  pixmap = realpixmap;
  setWidth(pixmap.width());
  setHeight(pixmap.height());

  pixmapWidth = pixmap.width();
  pixmapHeight = pixmap.height();
  rect_off = QRect(getX(),getY(),pixmapWidth,pixmapHeight);
}


void ImageLabel::mUpdate(QPainter* p, int backgroundUpdate)
{
  if (backgroundUpdate == 1){
  //only draw image if not hidden
  if (hidden != 90) {
    if (cblend == 0)
      //draw the pixmap
      p->drawPixmap(getX(),getY(),pixmap);
    else {
      //Blend this image with a color

      QImage image = pixmap.convertToImage();

      QImage result = KImageEffect::blend(QColor(255,0,0), image, 0.5f);
      p->drawImage(getX(),getY(),result);

      //p->drawRect(boundingBox);
    }


  }
  // start Timer
  if (imageEffect != 0)
  {
    imageEffect -> startTimer();
  }
}
}

void ImageLabel::mUpdate(QPainter* p)
{
  //only draw image if not hidden
  if (hidden == 0 && background == 0) {
    if (cblend == 0)
      //draw the pixmap
      p->drawPixmap(getX(),getY(),pixmap);
    else {
      //Blend this image with a color

      QImage image = pixmap.convertToImage();

      QImage result = KImageEffect::blend(QColor(255,0,0), image, 0.5f);
      p->drawImage(getX(),getY(),result);

      //p->drawRect(boundingBox);
    }
  }
  // start Timer
  if (imageEffect != 0)
  {
    imageEffect -> startTimer();
  }
}

bool ImageLabel::click(QMouseEvent* e)
{
    if (getBoundingBox().contains(e -> x(), e -> y()) && isEnabled())
    {
        QString program;
        if (e -> button() == Qt::LeftButton)
        {
            program = leftButtonAction;
        }
        else if (e -> button() == Qt::MidButton)
        {
            program = middleButtonAction;
        }
        else if (e -> button() == Qt::RightButton)
        {
            program = rightButtonAction;
        }

        if( program != "" )
        {
            KRun::runCommand(program);
        } else {
	  return true;
	}
    }

    return false;
}

void ImageLabel::parseImages(QString fn, QString fn_roll, int _xoff, int _yoff, int _xon, int _yon)
{
	//fn = filename;
	//fn_roll = filename_roll;

	xoff = _xoff;
	yoff = _yoff;
	xon = _xon;
	yon = _yon;

	// use the first line
	QStringList sList = QStringList::split( "\n", fn );
	QString fileName = *sList.begin();
	QFileInfo fileInfo( fileName );
	QString path;

	QRegExp rx("^http://",false);
	bool fileOnNet = (rx.search(fileName)!=-1)?true:false;


	if( fileInfo.isRelative() && !fileOnNet )
	{
	 path = themePath + "/" + fileName;
	}
	else
	{
	 path = fileName;
	}

	if ( fileOnNet )
	{
	  QString tmpFile;
	  if(KIO::NetAccess::download(KURL(path), tmpFile))
	  {
		  pixmap_off = KPixmap(tmpFile);
		  KIO::NetAccess::removeTempFile(tmpFile);
		   qDebug( "Downloaded: " + path + " to " + tmpFile );
	  }
	  else
	  {
		  qDebug( "Error Downloading: " + path );
	  }
	}
	else
	{
		pixmap_off = KPixmap( path );
	}

	pixmapOffWidth = pixmap.width();
	pixmapOffHeight = pixmap.height();

	rect_off = QRect(xoff,yoff,pixmapWidth,pixmapHeight);
/////////////////////////////
	if (fn_roll.isEmpty())
		return;

	rollover=true;
	sList = QStringList::split( "\n", fn_roll );
	fileName = *sList.begin();
	fileInfo = QFileInfo( fileName );

	fileOnNet = (rx.search(fileName)!=-1)?true:false;


	if( fileInfo.isRelative() && !fileOnNet )
	{
	 path = themePath + "/" + fileName;
	}
	else
	{
	 path = fileName;
	}

	if ( fileOnNet )
	{
	  QString tmpFile;
	  if(KIO::NetAccess::download(KURL(path), tmpFile))
	  {
		  pixmap_on = KPixmap(tmpFile);
		  KIO::NetAccess::removeTempFile(tmpFile);
		   qDebug( "Downloaded: " + path + " to " + tmpFile );
	  }
	  else
	  {
		  qDebug( "Error Downloading: " + path );
	  }
	}
	else
	{
		pixmap_on = KPixmap( path );
	}

	pixmapOnWidth = pixmap_on.width();
	pixmapOnHeight = pixmap_on.height();

	rect_on = QRect(xon,yon,pixmapOnWidth,pixmapOnHeight);
}

void ImageLabel::setKaramba(karamba* k)
{
	myKaramba = k;
}

void ImageLabel::setBackground(int b)
{
  background = b;
}

void ImageLabel::rolloverImage(QMouseEvent *e)
{
	if (!rollover)
		return;

	if (zoomed)
	{
		if (!rect_off.contains(e->pos()))
		{
			// rollover the image to the zoomed image
			//setValue(fn_roll);
			setX(xoff);
			setY(yoff);
			pixmap = pixmap_off;
			pixmapWidth = pixmapOffWidth;
			pixmapHeight = pixmapOffHeight;
			zoomed = false;
			myKaramba->step();
		}
	}
	else
	{
		if (rect_off.contains(e->pos()))
		{
			// rollover the image to the zoomed image
			//setValue(fn_roll);
			setX(xon);
			setY(yon);
			pixmap = pixmap_on;
			pixmapWidth = pixmapOnWidth;
			pixmapHeight = pixmapOnHeight;
			zoomed = true;
			myKaramba->step();
		}
	}
}

void ImageLabel::setTooltip(QString txt)
{
	QRect rect(getX(),getY(),pixmapWidth,pixmapHeight);
	QToolTip::add(myKaramba, rect, txt);
}


void ImageLabel::removeEffects()
{
	if (imageEffect != 0)
	{
		delete imageEffect;
		imageEffect = 0;
	}
  	applyTransformations();
};

void ImageLabel::intensity(float ratio, int millisec)
{
	if (imageEffect != 0)
	{
		delete imageEffect;
		imageEffect = 0;
	}
	//KPixmapEffect::intensity(pixmap, ratio);
	imageEffect = new Intensity(this, ratio, millisec);
  	applyTransformations();
};

void ImageLabel::channelIntensity(float ratio, QString channel, int millisec)
{
	if (imageEffect != 0)
	{
		delete imageEffect;
		imageEffect = 0;
	}
	//KPixmapEffect::channelIntensity(pixmap, ratio, rgbChannel);
	imageEffect = new ChannelIntensity(this, ratio, channel, millisec);
  	applyTransformations();
}

void ImageLabel::toGray(int millisec)
{
	if (imageEffect != 0)
	{
		delete imageEffect;
		imageEffect = 0;
	}
	//KPixmapEffect::toGray(pixmap);
	imageEffect = new ToGray(this, millisec);
	applyTransformations();
}

void ImageLabel::slotEffectExpired()
{
	removeEffects();
	myKaramba -> externalStep();
}

void ImageLabel::attachClickArea(QString leftMouseButton, QString middleMouseButton, QString rightMouseButton)
{
    leftButtonAction = leftMouseButton;
    middleButtonAction = middleMouseButton;
    rightButtonAction = rightMouseButton;
}

