/*
 * Copyright (C) 2003 Hans Karlsson <karlsson.h@home.se>
 */


#include <Python.h>

#include <kdebug.h>
#include <qdir.h>
#include <qwidgetlist.h>
#include <krun.h>
#include <kparts/componentfactory.h>
#include <kparts/part.h>

#include "richtextlabel.h"

#include "xpm.h"
#include "karamba.h"


//Matthew Kay: For setting window dock type
#include <kwin.h>


// Menu IDs
#define EDITSCRIPT 1
#define THEMECONF  2

/* now a method we need to expose to Python */
long moveSystray(long widget, long x, long y, long w, long h) {
  karamba* currTheme = (karamba*)widget;

  if (currTheme->systray != 0) {
    currTheme->systray->move((int)x,(int)y);
    currTheme->systray->setMinimumSize((int)w,(int)h);
    currTheme->systray->show();
  }
  
  return 1;
}

static PyObject* py_move_systray(PyObject *self, PyObject *args)
{
  long widget, x, y, w, h;
  if (!PyArg_ParseTuple(args, "lllll:moveSystray", &widget, &x, &y, &w, &h))
    return NULL;
  return Py_BuildValue("l", moveSystray(widget, x, y, w, h));
}

/* now a method we need to expose to Python */
long showSystray(long widget) {
  karamba* currTheme = (karamba*)widget;

  if (currTheme->systray != 0) {
    currTheme->systray->show();
  }
  
  return 1;
}

static PyObject* py_show_systray(PyObject *self, PyObject *args)
{
  long widget;
  if (!PyArg_ParseTuple(args, "l:showSystray", &widget))
    return NULL;
  return Py_BuildValue("l", showSystray(widget));
}

/* now a method we need to expose to Python */
long hideSystray(long widget) {
  karamba* currTheme = (karamba*)widget;

  if (currTheme->systray != 0) {
    currTheme->systray->hide();
  }
  
  return 1;
}

static PyObject* py_hide_systray(PyObject *self, PyObject *args)
{
  long widget;
  if (!PyArg_ParseTuple(args, "l:hideSystray", &widget))
    return NULL;
  return Py_BuildValue("l", hideSystray(widget));
}

/* now a method we need to expose to Python */
long createSystray(long widget, long x, long y, long w, long h) {
  karamba* currTheme = (karamba*)widget;

  //Don't create more than one systray
  if (currTheme->systray == 0) {
    currTheme->systray = new Systemtray(currTheme);    
    currTheme->systray->move((int)x,(int)y);
    currTheme->systray->setMinimumSize((int)w,(int)h);
    currTheme->systray->initSystray();
    currTheme->systray->show();
  }
  
  return 1;
}

static PyObject* py_create_systray(PyObject *self, PyObject *args)
{
  long widget, x, y, w, h;
  if (!PyArg_ParseTuple(args, "lllll:createSystray", &widget, &x, &y, &w, &h))
    return NULL;
  return Py_BuildValue("l", createSystray(widget, x, y, w, h));
}


/* now a method we need to expose to Python */
long acceptDrops(long widget) {
  karamba* currTheme = (karamba*)widget;

  currTheme->setAcceptDrops(TRUE);

  return 1;
}

static PyObject* py_accept_drops(PyObject *self, PyObject *args)
{
  long widget;

  if (!PyArg_ParseTuple(args, "l", &widget))
    return NULL;
  return Py_BuildValue("l", acceptDrops(widget));
}


// Converts a Python String to a QString with Unicode support
QString PyString2QString(PyObject* text)
{
    QString qtext;
    if (PyString_CheckExact(text))
    {
        //qWarning("PyString2QString: Object is String");
        char* t = PyString_AsString(text);
        //qtext.setLatin1(t); // maybe better
        qtext.setAscii(t);
    }
    else if (PyUnicode_CheckExact(text))
    {
        //qWarning("PyString2QString: Object is Unicode");
        Py_UNICODE* t = PyUnicode_AsUnicode(text);
        int length = PyUnicode_GetSize(text);
        qtext.setUnicodeCodes((ushort*)t, length);
    }
    else
    {
        ///qWarning("PyString2QString: Object Type Error");
        // Error raise execption ...
    }
    return qtext;
}

/* now a method we need to expose to Python */
long deleteRichText(long widget, long text) {
  karamba* currTheme = (karamba*)widget;
  int res1;

  RichTextLabel *tmp = (RichTextLabel*)text;

  currTheme -> clickList -> removeRef(tmp);
  res1 = currTheme -> meterList -> removeRef(tmp);

  qWarning("deleteRichText");

  //meterList is autoDelete(true) so we don't have to delete the image here

  return 1;
}

static PyObject* py_delete_rich_text(PyObject *self, PyObject *args)
{
  long widget, text;

  if (!PyArg_ParseTuple(args, "ll:deleteRichText", &widget, &text))
    return NULL;
  return Py_BuildValue("l", deleteRichText(widget, text));
}

// Shows a Rich Text Label
static PyObject* py_show_rich_text(PyObject* self, PyObject* args)
{
    long widget, text;
    if (!PyArg_ParseTuple(args, "ll:showRichText", &widget, &text))
        return NULL;

    RichTextLabel* richText = (RichTextLabel*) text;

    richText -> showText();
    return Py_BuildValue("l", 1);
}

// Hides a Rich Text Label
static PyObject* py_hide_rich_text(PyObject* self, PyObject* args)
{
    long widget, text;
    if (!PyArg_ParseTuple(args, "ll:hideRichText", &widget, &text))
        return NULL;

    RichTextLabel* richText = (RichTextLabel*) text;

    richText -> hideText();
    return Py_BuildValue("l", 1);
}

// Set the width of a Rich Text Label
static PyObject* py_set_rich_text_width(PyObject* self, PyObject* args)
{
    long widget, text, size;
    if (!PyArg_ParseTuple(args, "lll:setRichTextWidth", &widget, &text, &size))
        return NULL;

    RichTextLabel* richText = (RichTextLabel*) text;

    richText -> setWidth(size);
    return Py_BuildValue("l", 1);
}

// Returns the size of a Rich Text Label
static PyObject* py_get_rich_text_size(PyObject* self, PyObject* args)
{
    long widget, text;
    if (!PyArg_ParseTuple(args, "ll:getRichTextSize", &widget, &text))
        return NULL;

    RichTextLabel* richText = (RichTextLabel*) text;
    int width = richText -> getWidth();
    int height = richText -> getHeight();

    return Py_BuildValue("ll", width, height);
}

// Moves a Rich Text Label
long moveRichText(long widget, long text, long x, long y) {
    karamba* currTheme = (karamba*)widget;
    RichTextLabel* currSensor = (RichTextLabel*)text;

    currSensor->setX((int)x);
    currSensor->setY((int)y);

    return 1;
}

static PyObject* py_move_rich_text(PyObject* self, PyObject* args)
{
    long widget, text, x, y;
    if (!PyArg_ParseTuple(args, "llll:moveRichText", &widget, &text, &x, &y))
        return NULL;
    return Py_BuildValue("l", moveRichText(widget, text, x, y));
}

// Change a Rich Text Label
long changeRichText(long widget, RichTextLabel* richText, QString text)
{
    karamba* currTheme = (karamba*) widget;

    richText -> setText(text);
    return (int)richText;
}

static PyObject* py_change_rich_text(PyObject* self, PyObject* args)
{
    long widget, rt;
    PyObject* t;
    if (!PyArg_ParseTuple(args, "llO:changeRichText", &widget, &rt, &t))
        return NULL;

    RichTextLabel* richText = (RichTextLabel*) rt;

    QString text = PyString2QString(t);
    return Py_BuildValue("l", changeRichText(widget, richText, text));
}

// Creates a Rich Text Label
long createRichText(long widget, QString text, bool underlineLinks)
{
    karamba* currTheme = (karamba*) widget;

    RichTextLabel* richText = new RichTextLabel(currTheme);
    richText -> setText(text, underlineLinks);

    currTheme -> meterList -> append(richText);
    currTheme -> clickList -> append(richText);

    return (int)richText;
}

static PyObject* py_create_rich_text(PyObject* self, PyObject* args)
{
    long widget;
    PyObject* t;
    bool underlineLinks = false;
    if (!PyArg_ParseTuple(args, "lO|l:createRichText", &widget, &t, &underlineLinks))
        return NULL;

    QString text = PyString2QString(t);
    return Py_BuildValue("l", createRichText(widget, text, underlineLinks));
}

// Runs a command, returns 0 if it could not start command
static PyObject* py_execute_command(PyObject* self, PyObject* args)
{
  char* command;
  if (!PyArg_ParseTuple(args, "s:execute", &command))
      return NULL;
  return Py_BuildValue("l", KRun::runCommand(command));
}



// Runs a command, returns 0 if it could not start command
static PyObject* py_execute_command_interactive(PyObject* self, PyObject* args)
{
  long widget;
  //if (!PyArg_ParseTuple(args, "ls", &widget, &command))
  //  return NULL;

  int numLines;       /* how many lines we passed for parsing */
  char * line;        /* pointer to the line as a string */
  
  PyObject * listObj; /* the list of strings */
  PyObject * strObj;  /* one string in the list */
  
  /* the O! parses for a Python object (listObj) checked
     to be of type PyList_Type */
  if (! PyArg_ParseTuple( args, "lO!", &widget, &PyList_Type, &listObj)) 
    return NULL;
  

  karamba* currTheme = (karamba*)widget;
  
  currTheme->currProcess = new KProcess;
  


  /* get the number of lines passed to us */
  numLines = PyList_Size(listObj);
  
  /* should raise an error here. */
  if (numLines < 0) return NULL; /* Not a list */

  /* iterate over items of the list, grabbing strings, and parsing
     for numbers */
  for (int i=0; i<numLines; i++){
    
    /* grab the string object from the next element of the list */
    strObj = PyList_GetItem(listObj, i); /* Can't fail */
    
    /* make it a string */
    line = PyString_AsString( strObj );
    
    /* now do the parsing */
    *(currTheme->currProcess) << line;
    
  }




  QApplication::connect(currTheme->currProcess, SIGNAL(processExited(KProcess *)),
			currTheme, SLOT(processExited(KProcess *)));
  QApplication::connect(currTheme->currProcess, SIGNAL(receivedStdout(KProcess *, char *, int)),
			currTheme, SLOT(receivedStdout(KProcess *, char *, int)));
  currTheme->currProcess->start(KProcess::NotifyOnExit, KProcess::Stdout);

  return Py_BuildValue("l", (int)(currTheme->currProcess->pid()));
}


long attachClickArea(long widget, long meter, QString LeftButton, QString MiddleButton, QString RightButton)
{
  karamba* currTheme = (karamba*) widget;
  Meter* currMeter = (Meter*) meter;

  // Look if currMeter has an ClickArea attached.
  bool meterAlreadyClickable = currTheme -> clickList -> containsRef(currMeter);

  // if currMeter is of type ImageLabel*
  if (ImageLabel* image = dynamic_cast<ImageLabel*>(currMeter))
  {
      image -> attachClickArea(LeftButton, MiddleButton, RightButton);
      if (!meterAlreadyClickable)
      {
          //qWarning("attachClickArea : meter is image");
          currTheme -> clickList -> append(image);
      }
  }
  // else if currMeter is of type TextLabel*
  else if (TextLabel* text = dynamic_cast<TextLabel*>(currMeter))
  {
      text -> attachClickArea(LeftButton, MiddleButton, RightButton);
      if (!meterAlreadyClickable)
      {
          //qWarning("attachClickArea : meter is text");
          currTheme -> clickList -> append(text);
      }
  }
  else
  {
      //The given meter does not support attached clickAreas.
      qWarning("The given meter is not of type image or text");
      return 0;
  }
  return 1;
}

static PyObject* py_attach_clickArea(PyObject* self, PyObject* args, PyObject* dict)
{
  long widget;
  long meter;
  char* LeftButton = NULL;
  char* MiddleButton = NULL;
  char* RightButton = NULL;
  static char* mouseButtons[] = {"Widget", "Meter", "LeftButton", "MiddleButton", "RightButton", NULL};
  if (!PyArg_ParseTupleAndKeywords(args, dict, "ll|sss:attachClickArea", mouseButtons,
                 &widget, &meter, &LeftButton, &MiddleButton, &RightButton))
    return NULL;
  QString lB, mB, rB;
  if (LeftButton != NULL)
  {
      lB.setAscii(LeftButton);
  }
  else
  {
      lB.setAscii("");
  }
  if (MiddleButton != NULL)
  {
      mB.setAscii(MiddleButton);
  }
  else
  {
      mB.setAscii("");
  }
  if (RightButton != NULL)
  {
       rB.setAscii(RightButton);
  }
  else
  {
       rB.setAscii("");
  }
  return Py_BuildValue("l", attachClickArea(widget, meter, lB, mB, rB));
}

#include "pyapi.cpp"

long createMenu(long widget){
  karamba* currTheme = (karamba*)widget;

  KPopupMenu* tmp = new KPopupMenu(currTheme);
  currTheme->menuList->append (tmp );

  currTheme->connect(tmp, SIGNAL(activated(int)), currTheme, SLOT(passMenuItemClicked(int)));

  return (int)tmp;
}

static PyObject* py_create_menu(PyObject *self, PyObject *args)
{
  long widget;
  if (!PyArg_ParseTuple(args, "l:createMenu", &widget))
    return NULL;
  return Py_BuildValue("l", createMenu(widget));
}

bool menuExists(karamba* currTheme, KPopupMenu* menu){
  bool foundMenu = false;
  KPopupMenu* tmp;

  for(int i = 0; i < currTheme->menuList->count(); i++){
    if(i==0){
      tmp = (KPopupMenu*) currTheme->menuList->first();
    } else {
      tmp = (KPopupMenu*) currTheme->menuList->next();
    }
    if(tmp != 0){
      if(tmp == menu){
        foundMenu = true;
        break;
      }
    }
  }
  return foundMenu;
}


long deleteMenu(long widget, long menu){
  karamba* currTheme = (karamba*)widget;
  KPopupMenu* tmp = (KPopupMenu*)menu;

  currTheme->menuList->removeRef(tmp);

  return 1;
}

static PyObject* py_delete_menu(PyObject *self, PyObject *args)
{
  long widget, menu;
  if (!PyArg_ParseTuple(args, "ll:deleteMenu", &widget, &menu))
    return NULL;
  return Py_BuildValue("l", deleteMenu(widget, menu));
}

long addMenuItem(long widget, long menu, QString text, QString icon){
  karamba* currTheme = (karamba*)widget;
  KPopupMenu* tmp = (KPopupMenu*)menu;

  long id = 0;
  if(menuExists(currTheme, tmp))
  {
    id = tmp->insertItem(SmallIconSet(icon), text);
  }

  return id;
}

static PyObject* py_add_menu_item(PyObject *self, PyObject *args)
{
  long widget, menu;
  char* i;
  PyObject* t;
  if (!PyArg_ParseTuple(args, "llOs:addMenuItem", &widget, &menu, &t, &i))
    return NULL;
  QString icon;
  QString text;
  icon.setAscii(i);
  text = PyString2QString(t);
  return Py_BuildValue("l", addMenuItem(widget, menu, text, icon));
}

long addMenuSeparator(long widget, long menu){
  karamba* currTheme = (karamba*)widget;
  KPopupMenu* tmp = (KPopupMenu*)menu;

  long id = 0;
  if(menuExists(currTheme, tmp))
  {
    id = tmp->insertSeparator();
  }

  return id;
}

static PyObject* py_add_menu_separator(PyObject *self, PyObject *args)
{
  long widget, menu;
  
  if (!PyArg_ParseTuple(args, "ll:addMenuSeparator", &widget, &menu))
    return NULL;
  
  return Py_BuildValue("l", addMenuSeparator(widget, menu));
}

long removeMenuItem(long widget, long menu, long id){
  karamba* currTheme = (karamba*)widget;
  KPopupMenu* tmp = (KPopupMenu*)menu;

  if(menuExists(currTheme,tmp)){
    tmp->removeItem(id);
    return 1;
  } else {
    return 0;
  }
}

static PyObject* py_remove_menu_item(PyObject *self, PyObject *args)
{
  long widget, menu, id;
  if (!PyArg_ParseTuple(args, "lll:removeMenuItem", &widget, &menu, &id))
    return NULL;
  return Py_BuildValue("l", removeMenuItem(widget, menu, id));
}

long popupMenu(long widget, long menu, long x, long y){
  karamba* currTheme = (karamba*)widget;
  KPopupMenu* tmp = (KPopupMenu*)menu;

  if(menuExists(currTheme,tmp)){
    tmp->popup(currTheme->mapToGlobal( QPoint(x,y) ));
    return 1;
  } else {
    return 0;
  }
}

static PyObject* py_popup_menu(PyObject *self, PyObject *args)
{
  long widget, menu, x, y;
  if (!PyArg_ParseTuple(args, "llll:popupMenu", &widget, &menu, &x, &y))
    return NULL;
  return Py_BuildValue("l", popupMenu(widget, menu, x, y));
}



/* now a method we need to expose to Python */
long getThemeText(long widget, char* name) {
 karamba* currTheme = (karamba*)widget;
 QObjectListIt it( *currTheme->meterList ); // iterate over meters

 while ( it != 0 )
 {
  if (strcmp(((Meter*) *it)->name(), name) == 0)
   return (long)*it;
  ++it;
 }
 return 0;
}

static PyObject* py_get_theme_text(PyObject *self, PyObject *args)
{
 long widget;
 char* name;
 if (!PyArg_ParseTuple(args, "ls:getThemeText", &widget, &name))
  return NULL;
 return Py_BuildValue("l", getThemeText(widget, name));
}


/* now a method we need to expose to Python */
long getThemeImage(long widget, char* name) {
 karamba* currTheme = (karamba*)widget;
 QObjectListIt it( *currTheme->meterList ); // iterate over meters

 while ( it != 0 )
 {
  if (strcmp(((Meter*) *it)->name(), name) == 0)
   return (long)*it;
  ++it;
 }
 return 0;
}


static PyObject* py_get_theme_image(PyObject *self, PyObject *args)
{
 long widget;
 char* name;
 if (!PyArg_ParseTuple(args, "ls:getThemeImage", &widget, &name))
  return NULL;
 return Py_BuildValue("l", getThemeImage(widget, name));
}


// API-Function addMenuConfigOption
long addMenuConfigOption(long widget, QString key, QString name)
{
  karamba* currTheme = (karamba*)widget;

  currTheme -> addMenuConfigOption(key, name);

  return 1;
}

static PyObject* py_add_menu_config_option(PyObject *self, PyObject *args)
{
  long widget;
  char* key;
  PyObject* name;

  if (!PyArg_ParseTuple(args, "lsO:addMenuConfigOption", &widget, &key, &name))
    return NULL;

  QString k, n;
  k.setAscii(key);
  n = PyString2QString(name);

  return Py_BuildValue("l", addMenuConfigOption(widget, k, n));
}

long setMenuConfigOption(long widget, QString key, bool value)
{
	karamba* currTheme = (karamba*)widget;

	return currTheme -> setMenuConfigOption(key, value);
}

static PyObject* py_set_menu_config_option(PyObject *self, PyObject *args)
{
  long widget;
  char* key;
  bool value;

  if (!PyArg_ParseTuple(args, "lss:setMenuConfigOption", &widget, &key, &value))
    return NULL;

  QString k;
  k.setAscii(key);

  return Py_BuildValue("l", setMenuConfigOption(widget, k, value));
}

long readMenuConfigOption(long widget, QString key)
{
	karamba* currTheme = (karamba*)widget;

	return currTheme -> readMenuConfigOption(key);
}

static PyObject* py_read_menu_config_option(PyObject *self, PyObject *args)
{
	long widget;
	char* key;

	if (!PyArg_ParseTuple(args, "ls:readMenuConfigOption", &widget, &key))
		return NULL;

	QString k;
	k.setAscii(key);

	return Py_BuildValue("l", readMenuConfigOption(widget, k));
}

// API-Function writeConfigEntry
long writeConfigEntry(long widget, QString key, QString value)
{
  karamba* currTheme = (karamba*)widget;

  currTheme -> config -> setGroup("theme");
  currTheme -> config -> writeEntry(key, value);

  return 1;
}

static PyObject* py_write_config_entry(PyObject *self, PyObject *args)
{
  long widget;
  char* key;
  char* value;
  PyObject* py_value;
  //QVariant value;
  if (!PyArg_ParseTuple(args, "lss:writeConfigEntry", &widget, &key, &value))
    return NULL;
  QString k, v;
  k.setAscii(key);
  v.setAscii(value);

  // Check for Types
  /*
  if (PyNumber_Check(py_value))
  {
      int i;
      if (!PyArg_ParseTuple(py_value, "l", &i))
      {
          qWarning("py_write_config_entry: ASSERT (Must not happen!)");
          return NULL;
      }
      value = i;
  }
  else if (PyString_Check(py_value))
  {
      char* s;
      if (!PyArg_ParseTuple(py_value, "s", &s))
      {
          qWarning("py_write_config_entry: ASSERT (Must not happen!)");
          return NULL;
      }
      value = s;
  }
  else
  {
      qWarning("py_write_config_entry: Need more types!");
      return NULL;
  }
  */
  return Py_BuildValue("l", writeConfigEntry(widget, k, value));
}

// API-Function readConfigEntry
QVariant readConfigEntry(long widget, QString key)
{
  karamba* currTheme = (karamba*)widget;

  currTheme -> config -> setGroup("theme");
  return currTheme -> config -> readEntry(key);
}

static PyObject* py_read_config_entry(PyObject *self, PyObject *args)
{
  long widget;
  char* key;
  if (!PyArg_ParseTuple(args, "ls:readConfigEntry", &widget, &key))
    return NULL;
  QString k;
  k.setAscii(key);

  QVariant entry = readConfigEntry(widget, k);
  QString type;
  type.setAscii(entry.typeName());
  //qWarning(type);

  if (type == "Bool")
  {
    //qWarning("Type is Bool");
    return Py_BuildValue("l", (int)entry.toBool());
  }

  // somehow it treats integer entries as strings by default
  /*
  if (type == "Int")
  {
    qWarning("Type is Int");
    return Py_BuildValue("l", entry.toInt());
  }
  */
  bool isint = false;
  int i = entry.toInt(&isint);
  if (isint)
  {
    //qWarning("Type is Int");
    return Py_BuildValue("l", i);
  }

  if (type == "QString")
  {
    //qWarning("Type is QString");
    return Py_BuildValue("s", entry.toString().ascii());
  }
  // Add more types if needed
  return NULL;
}

/* now a method we need to expose to Python */
long toggleShowDesktop(long widget) {
  karamba* currTheme = (karamba*)widget;

  ShowDesktop *s = ShowDesktop::the();

  s->toggle();

  return 1;
}

static PyObject* py_toggle_show_desktop(PyObject *self, PyObject *args)
{
  long widget, image, deg;
  if (!PyArg_ParseTuple(args, "l:toggleShowDesktop", &widget))
    return NULL;
  return Py_BuildValue("l", toggleShowDesktop(widget));
}


// This does something with a task, such as minimize or close it
int performTaskAction(long widget, long ctask, long action) {
  karamba* currTheme = (karamba*)widget;
  Task* currTask = 0;
  Task* task;

  TaskList taskList = currTheme -> taskManager.tasks();

  for (task = taskList.first(); task; task = taskList.next())
  {
    if ((int)task == (int)ctask)
    {
      currTask = task;
    }

  }

  if (currTask != 0) {

    switch (action) {
    case 1:
      currTask->maximize();
      break;

    case 2:
      currTask->restore();
      break;

    case 3:
      currTask->iconify();
      break;

    case 4:
      currTask->close();
      break;

    case 5:
      currTask->activate();
      break;

    case 6:
      currTask->raise();
      break;

    case 7:
      currTask->lower();
      break;

    case 8:
      currTask->activateRaiseOrIconify();
      break;

    case 9:
      currTask->toggleAlwaysOnTop();
      break;

    case 10:
      currTask->toggleShaded();
      break;

    default:
      printf("You are trying to perform an invalid action in performTaskAction\n");
    }


    return 0;

  } else {
    return 1;
  }

}

static PyObject* py_perform_task_action(PyObject *self, PyObject *args)
{
  long widget, task, action;
  if (!PyArg_ParseTuple(args, "lll:performTaskAction", &widget, &task, &action))
    return NULL;
  return Py_BuildValue("l", performTaskAction(widget, task, action));
}


// This returns all the info about a certain task
// Return type is a Python List
PyObject* getTaskInfo(long widget, long ctask) {
  karamba* currTheme = (karamba*)widget;
  Task* currTask = 0;
  Task* task;

  TaskList taskList = currTheme -> taskManager.tasks();

  for (task = taskList.first(); task; task = taskList.next())
  {
    if ((int)task == (int)ctask)
    {
      currTask = task;
    }

  }

  if (currTask != 0) {


    PyObject* pList = PyList_New(0);

    //Task Name
    if (currTask->name() != NULL) {
      PyList_Append(pList, PyString_FromString(currTask->name().latin1()));
    } else {
      PyList_Append(pList, PyString_FromString(""));
    }

    //Icon Name
    if (currTask->iconName() != NULL) {
      PyList_Append(pList, PyString_FromString(currTask->iconName().latin1()));
    } else {
      PyList_Append(pList, PyString_FromString(""));
    }

    //Class Name
    if (currTask->className() != NULL) {
      PyList_Append(pList, PyString_FromString(currTask->className().latin1()));
    } else {
      PyList_Append(pList, PyString_FromString(""));
    }

    // Desktop this task is on
    PyList_Append(pList, PyInt_FromLong(currTask->desktop()));

    // is it maximized?
    PyList_Append(pList, PyInt_FromLong(currTask->isMaximized()));

    // is it iconified?
    PyList_Append(pList, PyInt_FromLong(currTask->isIconified()));

    // is it shaded?
    PyList_Append(pList, PyInt_FromLong(currTask->isShaded()));

    // is it focused?
    PyList_Append(pList, PyInt_FromLong(currTask->isActive()));

    // a reference back to itself
    PyList_Append(pList, PyInt_FromLong((int)currTask));

    return pList;

  } else {
    qWarning("Task not found.");
    return NULL;
  }

}


static PyObject* py_get_task_info(PyObject *self, PyObject *args)
{
  long widget, task;
  if (!PyArg_ParseTuple(args, "ll:getTaskInfo", &widget, &task))
    return NULL;
  return getTaskInfo(widget, task);
}

// This returns all the info about a certain startup
// Return type is a Python List
PyObject* getStartupInfo(long widget, long cstartup) {
    karamba* currTheme = (karamba*)widget;
    Startup* currentStartup = (Startup*) cstartup;
    Startup* startup;

    StartupList startupList = currTheme -> taskManager.startups();

    for (startup = startupList.first(); startup; startup = startupList.next())
    {
        if ((int)startup == (int)cstartup)
        {
            break;
        }
    }

    startup = currentStartup;

    if (startup != 0)
    {
        PyObject* pList = PyList_New(0);

        //Startup Name
        if (startup -> text() != NULL)
        {
            PyList_Append(pList, PyString_FromString(startup -> text().latin1()));
        }
        else
        {
            PyList_Append(pList, PyString_FromString(""));
        }

        //Icon Name
        if (startup -> icon() != NULL)
        {
            PyList_Append(pList, PyString_FromString(startup -> icon().latin1()));
        }
        else
        {
            PyList_Append(pList, PyString_FromString(""));
        }

        //Executable Name
        if (startup -> bin() != NULL)
        {
            PyList_Append(pList, PyString_FromString(startup -> bin().latin1()));
        }
        else
        {
            PyList_Append(pList, PyString_FromString(""));
        }

        // a reference back to itself
        PyList_Append(pList, PyInt_FromLong((int) startup));

        return pList;

    }
    else
    {
        return NULL;
    }
}

static PyObject* py_get_startup_info(PyObject* self, PyObject* args)
{
    long widget, startup;
    if (!PyArg_ParseTuple(args, "ll:getStartupInfo", &widget, &startup))
        return NULL;
    return getStartupInfo(widget, startup);
}

// This gets a system task list
// It returns a String List of task names
PyObject* getTaskNames(long widget)
{
    karamba* currTheme = (karamba*)widget;
    PyObject* pList = PyList_New(0);
    PyObject* pString;

    TaskList taskList = currTheme -> taskManager.tasks();

    Task* task;
    for (task = taskList.first(); task; task = taskList.next())
    {
        pString = PyString_FromString(task->name().latin1());
        PyList_Append(pList, pString);
    }

    return pList;
}

static PyObject* py_get_task_names(PyObject *self, PyObject *args)
{
  long widget;
  if(!PyArg_ParseTuple(args, "l:getTaskNames", &widget))
    return NULL;
  return getTaskNames(widget);
}

// This gets a system task list
PyObject* getTaskList(long widget)
{
    karamba* currTheme = (karamba*)widget;
    PyObject* pList = PyList_New(0);
    PyObject* pString;

    TaskList taskList = currTheme -> taskManager.tasks();

    Task* task;
    for (task = taskList.first(); task; task = taskList.next())
    {
        pString = PyInt_FromLong((int)task);
        PyList_Append(pList, pString);
    }


    return pList;
}

static PyObject* py_get_task_list(PyObject *self, PyObject *args)
{
  long widget;
  if(!PyArg_ParseTuple(args, "l:getTaskList", &widget))
    return NULL;
  return getTaskList(widget);
}

// This gets a system startup list
PyObject* getStartupList(long widget)
{
    karamba* currTheme = (karamba*)widget;
    PyObject* pList = PyList_New(0);
    PyObject* pString;

    StartupList startupList = currTheme -> taskManager.startups();

    Startup* startup;

    for (startup = startupList.first(); startup; startup = startupList.next())
    {
        pString = PyInt_FromLong((int) startup);
        PyList_Append(pList, pString);
    }

    return pList;
}

static PyObject* py_get_startup_list(PyObject *self, PyObject *args)
{
  long widget;
  if(!PyArg_ParseTuple(args, "l:getStartupList", &widget))
    return NULL;
  return getStartupList(widget);
}

/* now a method we need to expose to Python */
int getWidgetXCoordinate(long widget) {
  karamba* currTheme = (karamba*)widget;

  return currTheme->x();
}

/* now a method we need to expose to Python */
int getWidgetYCoordinate(long widget) {
  karamba* currTheme = (karamba*)widget;

  return currTheme->y();
}

static PyObject* py_get_widget_position(PyObject *self, PyObject *args)
{
  long widget;
  if(!PyArg_ParseTuple(args, "l:getWidgetPosition", &widget))
    return NULL;
  return Py_BuildValue("(i,i)", getWidgetXCoordinate(widget), getWidgetYCoordinate(widget));
}


/* now a method we need to expose to Python */
const char* getThemePath(long widget) {
  karamba* currTheme = (karamba*)widget;

  return currTheme->themePath.ascii();
}

static PyObject* py_get_theme_path(PyObject *self, PyObject *args)
{
  long widget;
  if (!PyArg_ParseTuple(args, "l:getThemePath", &widget))
    return NULL;
  return Py_BuildValue("s", getThemePath(widget));
}


/* now a method we need to expose to Python */
long createWidgetMask(long widget, char* path) {
  karamba* currTheme = (karamba*)widget;

  currTheme->clearMask();

  if (currTheme->widgetMask != 0) {
    delete currTheme->widgetMask;
  }

  QString maskpath;
  maskpath.setAscii(path);

  currTheme->widgetMask = new QBitmap(path);

  currTheme->kroot->stop();
  delete currTheme->kroot;

  //  currTheme->setMask(*(currTheme->widgetMask));

  QBitmap blah = QBitmap("/home/ageitgey/mask.png");
  currTheme->setMask(blah);


  currTheme->kroot = new KarambaRootPixmap((QWidget*)currTheme);
  currTheme->kroot->start();

  return (int)currTheme->widgetMask;
}

static PyObject* py_create_widget_mask(PyObject *self, PyObject *args)
{
  long widget;
  char *text;
  if (!PyArg_ParseTuple(args, "ls:createWidgetMask", &widget, &text))
    return NULL;
  return Py_BuildValue("l", createWidgetMask(widget, text));
}


/* now a method we need to expose to Python */
long createClickArea(long widget, long x, long y, long w, long h, char* text) {

  karamba* currTheme = (karamba*)widget;
  ClickArea *tmp = new ClickArea( x, y, w, h );
  QString onclick;

  onclick.setAscii(text);

  tmp->setOnClick(onclick );

  currTheme -> clickList -> append(tmp);
  return (int)tmp;
}

static PyObject* py_create_click_area(PyObject *self, PyObject *args)
{
  long widget, x, y, w, h;
  char *text;
  if (!PyArg_ParseTuple(args, "llllls:createClickArea", &widget, &x, &y, &w, &h, &text))
    return NULL;
  return Py_BuildValue("l", createClickArea(widget, x, y, w, h, text));
}


/* now a method we need to expose to Python */
long removeImageEffects(long widget, long image) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->removeEffects();

  return 1;
}

static PyObject* py_remove_image_effects(PyObject *self, PyObject *args)
{
  long widget, image;
  if (!PyArg_ParseTuple(args, "ll:removeImageEffects", &widget, &image))
    return NULL;
  return Py_BuildValue("l", removeImageEffects(widget, image));
}

/* now a method we need to expose to Python */
long changeImageIntensity(long widget, long image, float ratio, int millisec) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->intensity(ratio, millisec);

  return 1;
}

static PyObject* py_change_image_intensity(PyObject *self, PyObject *args)
{
  long widget, image, millisec;
	float ratio;
  millisec = 0;
  if (!PyArg_ParseTuple(args, "llf|l:changeImageIntensity", &widget, &image, &ratio, &millisec))
    return NULL;
  return Py_BuildValue("l", changeImageIntensity(widget, image, ratio, millisec));
}

/* now a method we need to expose to Python */
long changeImageChannelIntensity(long widget, long image, float ratio, char* channel, int millisec) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

	QString qChannel;
	qChannel.setAscii(channel);

  currSensor->channelIntensity(ratio, qChannel, millisec);

  return 1;
}

static PyObject* py_change_image_channel_intensity(PyObject *self, PyObject *args)
{
  long widget, image, millisec;
	float ratio;
	char* channel;
  millisec = 0;
  if (!PyArg_ParseTuple(args, "llfs|l:changeImageChannelIntensity", &widget, &image, &ratio, &channel, &millisec))
    return NULL;
  return Py_BuildValue("l", changeImageChannelIntensity(widget, image, ratio, channel, millisec));
}

/* now a method we need to expose to Python */
long changeImageToGray(long widget, long image, int millisec) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->toGray(millisec);

  return 1;
}

static PyObject* py_change_image_toGray(PyObject *self, PyObject *args)
{
  long widget, image, millisec;
  millisec = 0;
  if (!PyArg_ParseTuple(args, "ll|l:changeImageToGray", &widget, &image, &millisec))
    return NULL;
  return Py_BuildValue("l", changeImageToGray(widget, image, millisec));
}




long removeImageTransformations(long widget, long image) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->removeImageTransformations();

  return 1;
}

static PyObject* py_remove_image_transformations(PyObject *self, PyObject *args)
{
  long widget, image;
  if (!PyArg_ParseTuple(args, "ll:removeImageTransformation", &widget, &image))
    return NULL;
  return Py_BuildValue("l", removeImageTransformations(widget, image));
}


/* now a method we need to expose to Python */
long rotateImage(long widget, long image, long deg) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->rotate((int)deg);

  return 1;
}

static PyObject* py_rotate_image(PyObject *self, PyObject *args)
{
  long widget, image, deg;
  if (!PyArg_ParseTuple(args, "lll:rotateImage", &widget, &image, &deg))
    return NULL;
  return Py_BuildValue("l", rotateImage(widget, image, deg));
}

/* now a method we need to expose to Python */
long getImageHeight(long widget, long image) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  return currSensor->getHeight();
}

static PyObject* py_get_image_height(PyObject *self, PyObject *args)
{
  long widget, image;
  if (!PyArg_ParseTuple(args, "ll:getImageHeight", &widget, &image))
    return NULL;
  return Py_BuildValue("l", getImageHeight(widget, image));
}

/* now a method we need to expose to Python */
long getImageWidth(long widget, long image) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  return currSensor->getWidth();
}

static PyObject* py_get_image_width(PyObject *self, PyObject *args)
{
  long widget, image;
  if (!PyArg_ParseTuple(args, "ll:getImageWidth", &widget, &image))
    return NULL;
  return Py_BuildValue("l", getImageWidth(widget, image));
}

/* now a method we need to expose to Python */
long resizeImageSmooth(long widget, long image, long w, long h) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->smoothScale((int)w, (int)h);

  return 1;
}

static PyObject* py_resize_image_smooth(PyObject *self, PyObject *args)
{
  long widget, image, w, h;
  if (!PyArg_ParseTuple(args, "llll:resizeImageSmooth", &widget, &image, &w, &h))
    return NULL;
  return Py_BuildValue("l", resizeImageSmooth(widget, image, w, h));
}


/* now a method we need to expose to Python */
long resizeImage(long widget, long image, long w, long h) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->scale((int)w, (int)h);

  return 1;
}

static PyObject* py_resize_image(PyObject *self, PyObject *args)
{
  long widget, image, w, h;
  if (!PyArg_ParseTuple(args, "llll:resizeImage", &widget, &image, &w, &h))
    return NULL;
  return Py_BuildValue("l", resizeImage(widget, image, w, h));
}


/* now a method we need to expose to Python */
long showImage(long widget, long image) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->showImage();

  return 1;
}

static PyObject* py_show_image(PyObject *self, PyObject *args)
{
  long widget, image;
  if (!PyArg_ParseTuple(args, "ll:showImage", &widget, &image))
    return NULL;
  return Py_BuildValue("l", showImage(widget, image));
}

/* now a method we need to expose to Python */
long hideImage(long widget, long image) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->hideImage();

  return 1;
}

static PyObject* py_hide_image(PyObject *self, PyObject *args)
{
  long widget, image;
  if (!PyArg_ParseTuple(args, "ll:hideImage", &widget, &image))
    return NULL;
  return Py_BuildValue("l", hideImage(widget, image));
}

/* now a method we need to expose to Python */
long moveImage(long widget, long image, long x, long y) {
  karamba* currTheme = (karamba*)widget;
  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor -> setX((int)x);
  currSensor -> setY((int)y);

  return 1;
}

static PyObject* py_move_image(PyObject *self, PyObject *args)
{
  long widget, image, x, y;
  if (!PyArg_ParseTuple(args, "llll:moveImage", &widget, &image, &x, &y))
    return NULL;
  return Py_BuildValue("l", moveImage(widget, image, x, y));
}

/* now a method we need to expose to Python */
long addImageTooltip(long widget, long image, QString chartiptext) {
  karamba* currTheme = (karamba*)widget;

  QString tiptext;
  tiptext.setAscii(chartiptext);

  ImageLabel* currSensor = (ImageLabel*)image;

  currSensor->setTooltip(tiptext);

  return (int)1;
}

static PyObject* py_add_image_tooltip(PyObject *self, PyObject *args)
{
  long widget, image;
  PyObject* t;
  if (!PyArg_ParseTuple(args, "llO:addImageTooltip", &widget, &image, &t))
    return NULL;
  QString text;
  text = PyString2QString(t);
  return Py_BuildValue("l", addImageTooltip(widget, image, text));
}


/* now a method we need to expose to Python */
long deleteImage(long widget, long image) {
  karamba* currTheme = (karamba*)widget;
  int res1, res2;

  ImageLabel *tmp = (ImageLabel*)image;

  currTheme -> clickList -> removeRef(tmp);
  res2 = currTheme->imageList->removeRef(tmp);
  res1 = currTheme->meterList->removeRef(tmp);

  //meterList is autoDelete(true) so we don't have to delete the image here

  return 1;
}

static PyObject* py_delete_image(PyObject *self, PyObject *args)
{
  long widget, image;

  if (!PyArg_ParseTuple(args, "ll:deleteImage", &widget, &image))
    return NULL;
  return Py_BuildValue("l", deleteImage(widget, image));
}


/* now a method we need to expose to Python */
long createBackgroundImage(long widget, long x, long y, char* path) {
  karamba* currTheme = (karamba*)widget;

  QString file;
  file.setAscii(path);

  QString fakefile;

  ImageLabel *tmp = new ImageLabel(x,y,0,0);
  tmp->setThemePath( currTheme->themePath );
  tmp->setValue( file );
  tmp->parseImages(file, fakefile, x,y, 0, 0);
  tmp->setKaramba(currTheme);
  tmp->setBackground(1);

  currTheme->setSensor(file, (Meter*) tmp );
  currTheme->meterList->append (tmp );
  currTheme->imageList->append (tmp );

  return (int)tmp;
}

static PyObject* py_create_background_image(PyObject *self, PyObject *args)
{
  long widget, x, y;
  char *text;
  if (!PyArg_ParseTuple(args, "llls:createBackgroundImage", &widget, &x, &y, &text))
    return NULL;
  return Py_BuildValue("l", createBackgroundImage(widget, x, y, text));
}


/* now a method we need to expose to Python */
long createImage(long widget, long x, long y, char* path) {
  karamba* currTheme = (karamba*)widget;

  QString file;
  file.setAscii(path);

  QString fakefile;

  ImageLabel *tmp = new ImageLabel(x,y,0,0);
  tmp->setThemePath( currTheme->themePath );
  tmp->setValue( file );
  tmp->parseImages(file, fakefile, x,y, 0, 0);
  tmp->setKaramba(currTheme);

  currTheme->setSensor(file, (Meter*) tmp );
  currTheme->meterList->append (tmp );
  currTheme->imageList->append (tmp );

  return (int)tmp;
}

static PyObject* py_create_image(PyObject *self, PyObject *args)
{
  long widget, x, y;
  char *text;
  if (!PyArg_ParseTuple(args, "llls:createImage", &widget, &x, &y, &text))
    return NULL;
  return Py_BuildValue("l", createImage(widget, x, y, text));
}

//Matthew Kay: new function for creating icons for tasks
/**
 * creates the icon for the specified task as a karamba image
 * @param ctask		the window id of the task to create the icon for
 */
long createTaskIcon(long widget, long x, long y, long ctask) {
  karamba* currTheme = (karamba*)widget;

  //get the specified task and insure it exists
  TaskList taskList = currTheme -> taskManager.tasks();
  Task* task = 0;
  Task* currTask = 0;
  for (task = taskList.first(); task; task = taskList.next()) {
    if ((int)task == (int)ctask) {
      //task found
      currTask = task;
      break;
    }
  }
  if (currTask == 0) {
    //no task was found
    qWarning("Task not found.");
    return (long)NULL ;
  }

  //retrieve the QPixmap that represents this image
  QPixmap iconPixmap = KWin::icon(currTask->window());

  //create the ImageLabel for this image
  ImageLabel *tmp = new ImageLabel(x, y, 0, 0);
  tmp->setThemePath( currTheme->themePath );
  tmp->setValue(iconPixmap);
  tmp->setKaramba(currTheme);

  currTheme->meterList->append (tmp );
  currTheme->imageList->append (tmp );

  //return the image's "id" for python
  return (int)tmp;
}
/**
 * proxy function to accept python calls to createTaskIcon()
 */
static PyObject* py_create_task_icon(PyObject *self, PyObject *args)
{
  long widget, x, y, ctask;
  if (!PyArg_ParseTuple(args, "llll:createTaskIcon", &widget, &x, &y, &ctask))
    return NULL;
  return Py_BuildValue("l", createTaskIcon(widget, x, y, ctask));
}

/* now a method we need to expose to Python */
long openTheme(char* path) {

  QString filename;
  karamba* currTheme = 0; 

  filename.setAscii(path);
  
  QFileInfo file( filename );
  
  if( file.exists() )
    {
      currTheme = new karamba( filename, false );
      currTheme->show();
    }
  
  return (int)currTheme;
}

static PyObject* py_open_theme(PyObject *self, PyObject *args)
{
  char *themePath;
  if (!PyArg_ParseTuple(args, "s:openTheme", &themePath))
    return NULL;
  return Py_BuildValue("l", openTheme(themePath));
}



/* now a method we need to expose to Python */
long showText(long widget, long text) {
  karamba* currTheme = (karamba*)widget;
  TextLabel* currSensor = (TextLabel*)text;

  currSensor->showText();

  return 1;
}

static PyObject* py_show_text(PyObject *self, PyObject *args)
{
  long widget, text;
  if (!PyArg_ParseTuple(args, "ll:showText", &widget, &text))
    return NULL;
  return Py_BuildValue("l", showText(widget, text));
}

/* now a method we need to expose to Python */
long hideText(long widget, long text) {
  karamba* currTheme = (karamba*)widget;
  TextLabel* currSensor = (TextLabel*)text;

  currSensor->hideText();

  return 1;
}

static PyObject* py_hide_text(PyObject *self, PyObject *args)
{
  long widget, text;
  if (!PyArg_ParseTuple(args, "ll:hideText", &widget, &text))
    return NULL;
  return Py_BuildValue("l", hideText(widget, text));
}


/* now a method we need to expose to Python */
long moveText(long widget, long text, long x, long y) {
  karamba* currTheme = (karamba*)widget;
  TextLabel* currSensor = (TextLabel*)text;

  currSensor->setX((int)x);
  currSensor->setY((int)y);

  return 1;
}

static PyObject* py_move_text(PyObject *self, PyObject *args)
{
  long widget, text, x, y;
  if (!PyArg_ParseTuple(args, "llll:moveText", &widget, &text, &x, &y))
    return NULL;
  return Py_BuildValue("l", moveText(widget, text, x, y));
}


/* now a method we need to expose to Python */
long changeTextShadow(long widget, long textSensor, long size) {
  TextLabel* currSensor = (TextLabel*)textSensor;
  karamba* currTheme = (karamba*)widget;

  currSensor->setShadow( size );


  return 1;
}

static PyObject* py_change_text_shadow(PyObject *self, PyObject *args)
{
  long widget, textSensor;
  long shadow;
  if (!PyArg_ParseTuple(args, "lll:changeTextShadow", &widget, &textSensor, &shadow))
    return NULL;
  return Py_BuildValue("l", changeTextShadow(widget, textSensor, shadow));
}


/* now a method we need to expose to Python */
long changeTextSize(long widget, long textSensor, long size) {
  TextLabel* currSensor = (TextLabel*)textSensor;
  karamba* currTheme = (karamba*)widget;

  currSensor->setFontSize( size );


  return 1;
}

static PyObject* py_change_text_size(PyObject *self, PyObject *args)
{
  long widget, textSensor;
  long size;
  if (!PyArg_ParseTuple(args, "lll:changeTextSize", &widget, &textSensor, &size))
    return NULL;
  return Py_BuildValue("l", changeTextSize(widget, textSensor, size));
}


/* now a method we need to expose to Python */
long changeTextColor(long widget, long textSensor, long r, long g, long b) {
  TextLabel* currSensor = (TextLabel*)textSensor;
  karamba* currTheme = (karamba*)widget;

  currSensor->setColor( r, g, b );

  return 1;
}

static PyObject* py_change_text_color(PyObject *self, PyObject *args)
{
  long widget, textSensor;
  long r, g, b;
  if (!PyArg_ParseTuple(args, "lllll:changeTextColor", &widget, &textSensor, &r, &g, &b))
    return NULL;
  return Py_BuildValue("l", changeTextColor(widget, textSensor, r, g, b));
}


/* now a method we need to expose to Python */
long changeTextFont(long widget, long textSensor, char* newFont) {
  TextLabel* currSensor = (TextLabel*)textSensor;
  karamba* currTheme = (karamba*)widget;

  QString value;
  value.setAscii(newFont);
  currSensor->setFont( value );

  return 1;
}

static PyObject* py_change_text_font(PyObject *self, PyObject *args)
{
  long widget, textSensor;
  char* text;
  if (!PyArg_ParseTuple(args, "lls:changeTextFont", &widget, &textSensor, &text))
    return NULL;
  return Py_BuildValue("l", changeTextFont(widget, textSensor, text));
}


/* now a method we need to expose to Python */
long changeText(long widget, long textSensor, QString newText) {
  TextLabel* currSensor = (TextLabel*)textSensor;
  karamba* currTheme = (karamba*)widget;

  currSensor->setValue(newText);

  return 1;
}

static PyObject* py_change_text(PyObject *self, PyObject *args)
{
  long widget, textSensor;
  PyObject* t;
  if (!PyArg_ParseTuple(args, "llO:changeText", &widget, &textSensor, &t))
    return NULL;
  QString text;
  text = PyString2QString(t);
  return Py_BuildValue("l", changeText(widget, textSensor, text));
}

/* now a method we need to expose to Python */
long deleteText(long widget, long text) {
  karamba* currTheme = (karamba*)widget;
  int res1;

  TextLabel *tmp = (TextLabel*)text;

  currTheme -> clickList -> removeRef(tmp);
  res1 = currTheme->meterList->removeRef(tmp);

  //meterList is autoDelete(true) so we don't have to delete the image here

  return 1;
}

static PyObject* py_delete_text(PyObject *self, PyObject *args)
{
  long widget, text;

  if (!PyArg_ParseTuple(args, "ll:deleteText", &widget, &text))
    return NULL;
  return Py_BuildValue("l", deleteText(widget, text));
}


/* now a method we need to expose to Python */
long createText(long widget, long x, long y, long w, long h, QString text) {
  karamba* currTheme = (karamba*)widget;


  TextLabel *tmp = new TextLabel((int)x, (int)y, (int)w, (int)h );
//   bool paramFound = false;

//   get3Int( "COLOR", line, r, g, b, paramFound );
//   if ( paramFound || !defaultTextLabel )
//     tmp->setColor ( r, g, b );
//   else
//     if ( defaultTextLabel )
//       {
// 	defaultTextLabel->getColor().rgb ( &r, &g, &b );
  tmp->setColor( 128, 128, 128 );
//       }

//   get3Int( "BGCOLOR", line, r, g, b, paramFound );
//   if ( paramFound  || !defaultTextLabel )
//     tmp->setBGColor( r, g, b );
//   else
//     if ( defaultTextLabel )
//       {
// 	defaultTextLabel->getBGColor().rgb ( &r, &g, &b );
// 	tmp->setBGColor( r, g, b );
//       }

//   QString dFont = getString("FONT", line, paramFound);
//   if( paramFound || !defaultTextLabel )
//     tmp->setFont( dFont );
//   else
//     if( defaultTextLabel )
//       tmp->setFont( defaultTextLabel->getFont() );


//   int dFontSize = getInt( "FONTSIZE", line, paramFound );
//   if( paramFound || !defaultTextLabel )
//     tmp->setFontSize( dFontSize );
//   else
//     if( defaultTextLabel )
//       tmp->setFontSize( defaultTextLabel->getFontSize() );

//   QString dAlign = getString( "ALIGN", line, paramFound );
//   if( paramFound || !defaultTextLabel )
                    tmp->setAlignment( "LEFT" );
//   else
//     if( defaultTextLabel )
//       tmp->setAlignment( defaultTextLabel->getAlignment() );

//   bool dFp = getBoolean( "FIXEDPITCH", line, paramFound );
//   if( paramFound || !defaultTextLabel )
//     tmp->setFixedPitch( dFp );
//   else
//     if( defaultTextLabel )
//       tmp->setFixedPitch( defaultTextLabel->getFixedPitch() );

//   int dShadow = getInt( "SHADOW", line, paramFound );
//   if( paramFound || !defaultTextLabel )
     tmp->setShadow( 0 );
//   else
//     if( defaultTextLabel )
//       tmp->setShadow( defaultTextLabel->getShadow() );

//  printf("%s\n", text);

  //QString line;
  //QString value;
  //line.setAscii(text);
  //value.setAscii(text);
  tmp->setValue(text);
  //  currTheme->setSensor(line, (Meter*)tmp );
  currTheme->meterList->append ( tmp );

  return (int)tmp;
}

static PyObject* py_create_text(PyObject *self, PyObject *args)
{
  long widget, x, y, w, h;
  PyObject* t = NULL;
  long textlength;
  if (!PyArg_ParseTuple(args, "lllllO:createText", &widget, &x, &y, &w, &h, &t))
    return NULL;

  QString text = PyString2QString(t);
  return Py_BuildValue("l", createText(widget, x, y, w, h, text));
}

/* now a method we need to expose to Python */
long redrawWidgetBackground(long widget) {
  karamba* currTheme = (karamba*)widget;
  currTheme->kroot->repaint(true);
  return 1;
}

static PyObject* py_redraw_widget_background(PyObject *self, PyObject *args)
{
  long widget;
  if (!PyArg_ParseTuple(args, "l:redrawWidgetBackground", &widget))
    return NULL;
  return Py_BuildValue("l", redrawWidgetBackground(widget));
}


/* now a method we need to expose to Python */
long redrawWidget(long widget) {
  karamba* currTheme = (karamba*)widget;
  currTheme->externalStep();
  return 1;
}

static PyObject* py_redraw_widget(PyObject *self, PyObject *args)
{
  long widget;
  if (!PyArg_ParseTuple(args, "l:redrawWidget", &widget))
    return NULL;
  return Py_BuildValue("l", redrawWidget(widget));
}


/* now a method we need to expose to Python */
long resizeWidget(long widget, long x, long y) {
  karamba* currTheme = (karamba*)widget;
  currTheme->setFixedSize((int)x,(int)y);
  return 1;
}

static PyObject* py_resize_widget(PyObject *self, PyObject *args)
{
  long widget, x, y;
  if (!PyArg_ParseTuple(args, "lll:resizeWidget", &widget, &x, &y))
    return NULL;
  return Py_BuildValue("l", resizeWidget(widget, x, y));
}

/* now a method we need to expose to Python */
long moveWidget(long widget, long x, long y) {
  karamba* currTheme = (karamba*)widget;
  currTheme->move((int)x, (int)y);
  return 1;
}

static PyObject* py_move_widget(PyObject *self, PyObject *args)
{
  long widget, x, y;
  if (!PyArg_ParseTuple(args, "lll:moveWidget", &widget, &x, &y))
    return NULL;
  return Py_BuildValue("l", moveWidget(widget, x, y));
}

static PyMethodDef karamba_methods[] = {
    /**
      * Define basic Bar methods (createBar, deleteBar, getThemeBar, etc)
      * See: pyapi_templates.h
      */
    PY_API_DECLARE_BASIC(Bar)

    {"setBarImage", py_setBarImage, METH_VARARGS, "Set bar image"},
    {"getBarImage", py_getBarImage, METH_VARARGS, "Get bar image"},
    {"setBarVertical", py_setBarVertical, METH_VARARGS, "Set bar orientation"},
    {"getBarVertical", py_getBarVertical, METH_VARARGS, "Get bar orientation"},
    {"setBarValue", py_setBarValue, METH_VARARGS, "Set bar value"},
    {"getBarValue", py_getBarValue, METH_VARARGS, "Get bar value"},
    {"setBarMinMax", py_setBarMinMax, METH_VARARGS, "Set bar min & max"},
    {"getBarMinMax", py_getBarMinMax, METH_VARARGS, "Get bar min & max"},

    {"acceptDrops", py_accept_drops, METH_VARARGS, "Allows widget to receive Drop (I.E. Drag and Drop) events"},

    {"hideRichText", py_hide_rich_text, METH_VARARGS, "hides a Rich Text Sensor"},
    {"showRichText", py_show_rich_text, METH_VARARGS, "shows a Rich Text Sensor"},
    {"changeRichText", py_change_rich_text, METH_VARARGS, "Change the content of a Rich Text Sensor"},
    {"setRichTextWidth", py_set_rich_text_width, METH_VARARGS, "Sets the width of a Rich Text Sensor"},
    {"getRichTextSize", py_get_rich_text_size, METH_VARARGS, "Get the (width, height) of a Rich Text Sensor"},
    {"moveRichText", py_move_rich_text, METH_VARARGS, "Moves a Rich Text Sensor"},
    {"createRichText", py_create_rich_text, METH_VARARGS, "Create a Rich Text Sensor"},
    {"deleteRichText", py_delete_rich_text, METH_VARARGS, "Deletes a Rich Text Sensor"},
    {"execute", py_execute_command, METH_VARARGS, "Execute a command"},
    {"executeInteractive", py_execute_command_interactive, METH_VARARGS, "Execute a command and get it's output (stdout)"},
    {"attachClickArea", (PyCFunction)py_attach_clickArea, METH_VARARGS|METH_KEYWORDS, "Add a clickArea to the given text or image"},
    {"addMenuConfigOption", py_add_menu_config_option, METH_VARARGS, "Add a configuration entry to the menu"},
    {"setMenuConfigOption", py_set_menu_config_option, METH_VARARGS, "Set a configuration entry in the menu"},
    {"readMenuConfigOption", py_read_menu_config_option, METH_VARARGS, "Read a configuration entry in the menu"},
    {"readConfigEntry", py_read_config_entry, METH_VARARGS, "Read a configuration entry"},
    {"writeConfigEntry", py_write_config_entry, METH_VARARGS, "Writes a configuration entry"},
    {"moveWidget", py_move_widget, METH_VARARGS, "Move Widget to x,y"},
    {"resizeWidget", py_resize_widget, METH_VARARGS, "Resize Widget to width,height"},
    {"createWidgetMask", py_create_widget_mask, METH_VARARGS, "Create a clipping mask for this widget"},
    {"redrawWidget", py_redraw_widget, METH_VARARGS, "Update Widget to reflect your changes"},
    {"redrawWidgetBackground", py_redraw_widget_background, METH_VARARGS, "Update Widget to reflect background image changes"},
    {"getThemePath", py_get_theme_path, METH_VARARGS,  "Get the file path of the theme"},
    {"toggleShowDesktop", py_toggle_show_desktop, METH_VARARGS, "Show/Hide the desktop"},
    {"createClickArea", py_create_click_area, METH_VARARGS, "Create a Click Area Sensor"},
    {"getWidgetPosition", py_get_widget_position, METH_VARARGS, "Get Widget Position"},
    {"createMenu", py_create_menu, METH_VARARGS, "Create a popup menu"},
    {"deleteMenu", py_delete_menu, METH_VARARGS, "Delete a popup menu"},
    {"addMenuItem", py_add_menu_item, METH_VARARGS, "Add a popup menu entry"},
    {"addMenuSeparator", py_add_menu_separator, METH_VARARGS, "Add a popup menu seperator item"},
    {"removeMenuItem", py_remove_menu_item, METH_VARARGS, "Remove a popup menu entry"},
    {"popupMenu", py_popup_menu, METH_VARARGS, "Popup a menu at a specified location"},
    {"removeImageTransformations", py_remove_image_transformations, METH_VARARGS, "Restore original size and orientation of an Image"},
    {"removeImageEffects", py_remove_image_effects, METH_VARARGS, "Remove Effects of an Image"},
    {"changeImageIntensity", py_change_image_intensity, METH_VARARGS, "Change Intensity of an Image"},
    {"changeImageChannelIntensity", py_change_image_channel_intensity, METH_VARARGS, "Change Intensity of an Image Channel"},
    {"changeImageToGray", py_change_image_toGray, METH_VARARGS, "Converts an Image to Grayscale"},

    {"openTheme", py_open_theme, METH_VARARGS, "Open a new theme"},

    {"createText", py_create_text, METH_VARARGS, "Create a Text Sensor"},
    {"deleteText", py_delete_text, METH_VARARGS, "Delete a Text object"},
    {"getThemeText", py_get_theme_text, METH_VARARGS, "Get text from .theme using it's name"},

    {"moveText", py_move_text, METH_VARARGS, "Move a Text object"},
    {"changeText", py_change_text, METH_VARARGS, "Change a Text Sensor's Text"},
    {"hideText", py_hide_text, METH_VARARGS, "Hide a Text object"},
    {"showText", py_show_text, METH_VARARGS, "Show a Text object"},
    {"changeTextShadow", py_change_text_shadow, METH_VARARGS, "Change a Text Shadow size"},
    {"changeTextFont", py_change_text_font, METH_VARARGS, "Change a Text Sensor's Font"},
    {"changeTextColor", py_change_text_color, METH_VARARGS, "Change a Text Sensor's Color"},
    {"changeTextSize", py_change_text_size, METH_VARARGS, "Change a Text Sensor's Font Size"},
    {"createImage", py_create_image, METH_VARARGS, "Create an Image"},
    {"createTaskIcon", py_create_task_icon, METH_VARARGS, "Create an Image of the Icon for a Task"},
    {"createBackgroundImage", py_create_background_image, METH_VARARGS, "Create an Image (only redraw it when background changes)"},
    {"deleteImage", py_delete_image, METH_VARARGS, "Delete an Image"},
    {"addImageTooltip", py_add_image_tooltip, METH_VARARGS, "Create a Tooltip for an Image"},
    {"moveImage", py_move_image, METH_VARARGS, "Move an Image"},
    {"hideImage", py_hide_image, METH_VARARGS, "Hide an Image"},
    {"showImage", py_show_image, METH_VARARGS, "Show an Image"},
    {"resizeImage", py_resize_image, METH_VARARGS, "Scale an Image"},
    {"resizeImageSmooth", py_resize_image_smooth, METH_VARARGS, "Scale an Image (slower, better looking)"},
    {"rotateImage", py_rotate_image, METH_VARARGS, "Rotate an Image"},
    {"getImageWidth", py_get_image_width, METH_VARARGS, "Get the width of an Image"},
    {"getImageHeight", py_get_image_height, METH_VARARGS, "Get the height of an Image"},
    {"getStartupList", py_get_startup_list, METH_VARARGS, "Get the system startup list"},
    {"getStartupInfo", py_get_startup_info, METH_VARARGS, "Get all the info for a startup"},
    {"getTaskList", py_get_task_list, METH_VARARGS, "Get the system task list"},
    {"getTaskNames", py_get_task_names, METH_VARARGS, "Get the system task list in name form"},
    {"getTaskInfo", py_get_task_info, METH_VARARGS, "Get all the info for a task"},
    {"performTaskAction", py_perform_task_action, METH_VARARGS, "Do something with a task, such as minimize it"},
    {"getThemeText", py_get_theme_text, METH_VARARGS, "Get text meter from .theme using it's name"},
    {"getThemeImage", py_get_theme_image, METH_VARARGS, "Get image meter from .theme using it's name"},
    {"createSystray", py_create_systray, METH_VARARGS, "Create a Systray"},
    {"hideSystray", py_hide_systray, METH_VARARGS, "Hide the Systray"},
    {"showSystray", py_show_systray, METH_VARARGS, "Show the Systray"},
    {"moveSystray", py_move_systray, METH_VARARGS, "Move the Systray"},
    {NULL, NULL, 0 ,NULL}
};



karamba::karamba(QString fn, bool reloading)
        :  QWidget( 0,"karamba", Qt::WGroupLeader | WStyle_Customize |  WRepaintNoErase| WStyle_NoBorder | WDestructiveClose  )
{

    char pypath[1024];
    themePath = QFileInfo( fn ).dirPath(true) + "/";
    themeName = QFileInfo( fn ).baseName();
    themeFile = fn;



//     //Is this a plain .theme or a compressed .ctheme?
//     QString ext = QFileInfo(fn).extension( FALSE );

//     if (ext == "ctheme") {      printf("Compressed theme!");
//     } else if (ext == "theme") {
//       printf("Regular theme!");
//     } else
//       printf("Unknown theme!");

    QFile themeConfigFile(QDir::home().absPath() + "/.superkaramba/" + themeName + ".rc");
    // Tests if config file Exists
    if (!QFileInfo(themeConfigFile).exists())
    {
       // Create config file
       themeConfigFile.open(IO_ReadWrite);
       themeConfigFile.close();
    }

    // Creates KConfig Object
    config = new KConfig(QFileInfo(themeConfigFile).filePath(), false, false);
    config -> setGroup("internal");

    connect(qApp,SIGNAL(lastWindowClosed()),qApp,SLOT(quit()));

    pythonThemeExtensionLoaded = false;

    //Add theme path to python path so that we can find the python file
    snprintf(pypath, 1023, "sys.path.insert(0, '%s')", themePath.ascii());

    // get the global python thread lock
    PyEval_AcquireLock();

    // get a reference to the PyInterpreterState
    extern PyThreadState * mainThreadState;
    PyInterpreterState * mainInterpreterState = mainThreadState->interp;

    // create a thread state object for this thread
    PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
    PyThreadState_Swap(myThreadState);

    // load the .py file for this .theme
    PyRun_SimpleString("import sys");
    PyRun_SimpleString(pypath);
    PyRun_SimpleString("sys.path.insert(0, '')");


    PyImport_AddModule("karamba");
    Py_InitModule("karamba", karamba_methods);


    pName = PyString_FromString(themeName.ascii());

    pModule = PyImport_Import(pName);

    //Make sure the module is up to date.
    if (reloading)
      PyImport_ReloadModule(pModule);

    if (pModule != NULL) {
	pDict = PyModule_GetDict(pModule);
	if (pDict != NULL) {
	  pythonThemeExtensionLoaded = true;
	}	
	//HERE	
    }
     else {
       PyErr_Print();
       fprintf(stderr, "------------------------------------------------------\n");       
       fprintf(stderr, "What does ImportError mean?\n");       
       fprintf(stderr, "\n");       
       fprintf(stderr, "It means that I couldn't load a python add-on for %s.theme\n", themeName.ascii());       
       fprintf(stderr, "If this is a regular theme and doesn't use python\n");       
       fprintf(stderr, "extensions, then nothing is wrong.\n");       
       fprintf(stderr, "------------------------------------------------------\n");       
       
     }

    //Clean up this thread's python interpreter reference
    PyThreadState_Swap(NULL);
    PyThreadState_Clear(myThreadState);
    PyThreadState_Delete(myThreadState);
    PyEval_ReleaseLock();
    //This thread's python refence is now gone.  A new reference must be created for each thread.


    //Matthew Kay: set window type to "dock" (plays better with taskbar themes this way)
    KWin::setType(winId(), NET::Dock);

    widgetMask = 0;
    info = new NETWinInfo( qt_xdisplay(), winId(), qt_xrootwin(), NET::WMState );

    // could be replaced with TaskManager
    kWinModule = new KWinModule();
    desktop = 0;

        connect( kWinModule,SIGNAL(currentDesktopChanged(int)), this,SLOT(currentDesktopChanged(int)) );

    // Setup of the Task Manager Callbacks
    connect(&taskManager, SIGNAL(activeTaskChanged(Task*)), this, SLOT(activeTaskChanged(Task*)) );
    connect(&taskManager, SIGNAL(taskAdded(Task*)), this, SLOT(taskAdded(Task*)) );
    connect(&taskManager, SIGNAL(taskRemoved(Task*)), this, SLOT(taskRemoved(Task*)) );
    connect(&taskManager, SIGNAL(startupAdded(Startup*)), this, SLOT(startupAdded(Startup*)) );
    connect(&taskManager, SIGNAL(startupRemoved(Startup*)), this, SLOT(startupRemoved(Startup*)) );

    themeConfMenu = new KPopupMenu( this);
    themeConfMenu -> setCheckable(true);

    /* XXX - need to be able to delete all these DesktopChangeSlot objects */
    DesktopChangeSlot *dslot;
    
    int mid;

    toDesktopMenu = new KPopupMenu (this);
    toDesktopMenu -> setCheckable(true);
    mid = toDesktopMenu -> insertItem ("&All Desktops", 
                                 dslot = new DesktopChangeSlot(this,0), 
                                 SLOT(receive()));
    dslot->setMenuId(mid);

    toDesktopMenu -> insertSeparator();
    for (int ndesktop=1; ndesktop <= kWinModule->numberOfDesktops(); ndesktop++)
    {
      QString name = "Desktop &";
      name += ('0' + ndesktop);
      
      mid = toDesktopMenu -> insertItem (name, dslot = new DesktopChangeSlot(this, ndesktop), SLOT(receive()));
      dslot->setMenuId(mid);
    }
    

    kpop = new KPopupMenu( this );
    kpop -> setCheckable(true);

    keditpop = new KPopupMenu ( this );

    keditpop -> insertItem(SmallIconSet("edit"),tr("Edit &theme"),this,SLOT(editConfig()), CTRL+Key_E );
    keditpop -> insertItem(SmallIconSet("edit"),tr("Edit &script"),this,SLOT(editScript()), ALT+Key_E, EDITSCRIPT);

    // Test if Theme Script exists
    QFileInfo scriptFile(themePath + "/" + themeName + ".py");
    keditpop -> setItemEnabled(EDITSCRIPT, scriptFile.exists());

    accColl = new KActionCollection( this );
    menuAccColl = new KActionCollection( this );
    
    kpop->insertItem( SmallIconSet("reload"),tr("Update"), this, SLOT(updateSensors()), Key_F5 );
    toggleLocked = new KToggleAction (  tr("Toggle &locked position"), SmallIconSet("locked"),
                   CTRL+Key_L, this, SLOT( slotToggleLocked() ), accColl  , "Locked position" );
    accColl->insert(toggleLocked);
    toggleLocked -> setChecked(true);

    toggleLocked->plug(kpop);

    toggleFastTransforms = new KToggleAction(tr("Use &fast image scaling"),
       CTRL+Key_F, this, SLOT( slotToggleFastTransforms() ), accColl, "Fast transformations");

    accColl->insert(toggleFastTransforms);
    toggleFastTransforms -> setChecked(true);

    toggleFastTransforms -> plug(kpop);

    kpop->insertSeparator();

    kpop->insertItem( SmallIconSet("fileopen"),tr("&Open new theme"), this, SLOT(startNewKaramba()), CTRL+Key_O );
    //    kpop->insertItem( SmallIconSet("edit"),"&Edit configuration",this,SLOT(editConfig()), CTRL+Key_E );
    kpop->insertItem(tr("&Edit"), keditpop);
    kpop->insertItem(tr("Configure &theme"), themeConfMenu, THEMECONF);
    kpop->setItemEnabled(THEMECONF, false);
    kpop->insertItem(tr("To Des&ktop"), toDesktopMenu);

    kpop->insertItem( SmallIconSet("reload3"),tr("&Reload theme"),this,SLOT(reloadConfig()), CTRL+Key_R );
    kpop->insertItem( SmallIconSet("fileclose"),tr("&Close this theme"), this, SLOT(killWidget()), CTRL+Key_C );

    kpop->insertSeparator();
    kpop->insertItem( SmallIconSet("find"),tr("&Download more themes"), this, SLOT(downloadThemes()), CTRL+Key_D );

    kpop->insertSeparator();

    kpop->insertItem( SmallIconSet("exit"),tr("&Quit"), this, SLOT(quitKaramba()), CTRL+Key_Q );

    kpop->polish();

    numberOfConfMenuItems = 0;

    systray = 0;
    foundKaramba = false;
    onTop = false;
    managed = false;
    fixedPosition = false;



    meterList = new QObjectList();
    meterList->setAutoDelete( true );
    sensorList = new QObjectList();
    sensorList->setAutoDelete( true );
    clickList = new QObjectList();
    timeList = new QObjectList();
    imageList = new QObjectList();
    menuList = new QObjectList();
    menuList->setAutoDelete( true );

    client = kapp->dcopClient();
    if (!client->isAttached())
        client->attach();
    appId = client->registerAs(qApp->name());


    this->setBackgroundMode( NoBackground);
    if( !(onTop || managed))
        this->lower();


    if( !parseConfig() )
    {
        setFixedSize(0,0);
        QTimer::singleShot( 100, this, SLOT(killWidget()) );

        qDebug("Could not read config file.");
    }
    else
    {
        kroot = new KarambaRootPixmap((QWidget*)this);
        kroot->start();


    }
    
    // Karamba specific Config Entries
    if (!config -> readBoolEntry("lockedPosition", true))
    {
        toggleLocked -> setChecked(false);
        slotToggleLocked();
    }

    if (!config -> readBoolEntry("fastTransforms", true))
    {
        toggleFastTransforms -> setChecked(false);
        slotToggleFastTransforms();
    }

    desktop = config -> readNumEntry("desktop", desktop);
    if (desktop > kWinModule->numberOfDesktops())
    {
        desktop = 0;
    }

    if (desktop)
    {
        info->setDesktop( desktop );
    }
    else
        info->setDesktop( NETWinInfo::OnAllDesktops);

    // Read Themespecific Config Entries
    config -> setGroup("theme");
    if (config -> hasKey("widgetPosX") && config -> hasKey("widgetPosY"))
    {
       move(config -> readNumEntry("widgetPosX"), config -> readNumEntry("widgetPosY"));
    }
    //    if (config -> hasKey("widgetWidth") && config -> hasKey("widgetHeight"))
    //    {
    //       setFixedSize(config -> readNumEntry("widgetWidth"), config -> readNumEntry("widgetHeight"));
    //    }
    
    haveUpdated = 0;
    this->setMouseTracking(true);


//     KParts::ReadOnlyPart* m_part;

//     KTrader::OfferList offers = KTrader::self()->query( "KTextEditor/Document" );

//     KService::Ptr service = *offers.begin();

//     KLibFactory *factory = KLibLoader::self()->factory( service->library() );

//     //    m_part = static_cast<KTextEditor::Document*>( factory->create( this, 0, "KTextEditor::Document" ) );
//     m_part = (KParts::ReadOnlyPart *)factory->create( this, "konsolepart", "KParts::ReadOnlyPart" );

//     QWidget * view = m_part->widget();
//     view->setMinimumSize(200,200);
//     view->show();


    setFocusPolicy(QWidget::StrongFocus);
}

karamba::~karamba()
{
  widgetClosed();
    writeConfigData();
    delete config;

    if(meterList != 0)
    {
        meterList->clear();
        delete meterList;
    }

    if( sensorList != 0 )
    {
        sensorList->clear();
        delete sensorList;
    }

    if( imageList != 0 )
    {
        imageList->clear();
        delete imageList;
    }
	
	if( clickList != 0 )
    {
        clickList->clear();
        delete clickList;
    }
	
	if( timeList != 0 )
    {
        timeList->clear();
        delete timeList;
    }

    if( toggleLocked != 0)
        delete toggleLocked;
    if( accColl != 0)
        delete accColl;
    if ( menuAccColl != 0)
	delete menuAccColl;
    
    if( keditpop != 0)
      delete keditpop;

    if( themeConfMenu != 0)
      delete themeConfMenu;

    if( kpop != 0)
        delete kpop;

    if( widgetMask != 0)
      delete widgetMask;

    delete kWinModule;

    //Clean up Python references
    if (pythonThemeExtensionLoaded) {
      // get the global python thread lock
      PyEval_AcquireLock();
      
      // get a reference to the PyInterpreterState
      extern PyThreadState * mainThreadState;
      PyInterpreterState * mainInterpreterState = mainThreadState->interp;
      
      // create a thread state object for this thread
      PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
      PyThreadState_Swap(myThreadState);
      
      //Displose of current python module so we can reload in constructor.
      Py_DECREF(pModule);
      Py_DECREF(pName);
      
      //Clean up this thread's python interpreter reference
      PyThreadState_Swap(NULL);
      PyThreadState_Clear(myThreadState);
      PyThreadState_Delete(myThreadState);
      PyEval_ReleaseLock();
      //This thread's python refence is now gone.  A new reference must be created for each thread.
    }
}

bool karamba::parseConfig()
{
    QTimer *m_sysTimer = new QTimer(this);

    QFile file(themeFile);
    QString line;
    QString meter;
    int interval = 0;
    //    TextLabel *defaultTextLabel = 0;
    TextField *defaultTextField = 0;

    if ( file.open(IO_ReadOnly | IO_Translate) )
    {
        QTextStream t( &file );        // use a text stream
        QValueStack<QPoint> offsetStack;
        offsetStack.push(QPoint(0,0));

        int x=0;
        int y=0;
        int w=0;
        int h=0;
        while( ( line = t.readLine() ) !=0 )
        {
            QRegExp rx("^\\s*(\\S+)");
            rx.search(line);
            meter = rx.cap(1).upper();

            x = getInt("X",line)+offsetStack.top().x();
            y = getInt("Y",line)+offsetStack.top().y();
            w = getInt("W",line);
            h = getInt("H",line);


            if( meter == "KARAMBA" && !foundKaramba )
            {
	      //                qDebug("karamba");
                toggleLocked->setChecked( getBoolean( "LOCKED", line ) );
                slotToggleLocked();


                getInt( "X",line );
                getInt( "Y",line );

                x = ( x < 0 ) ? 0:x;
                y = ( y < 0 ) ? 0:y;
                move(x,y);


                if( w == 0 ||  h == 0)
                {
                    w = 300;
                    h = 300;
                }
                setFixedSize(w,h);

                if( getBoolean( "RIGHT", line ) )
                {
                    QDesktopWidget *d = QApplication::desktop();
                    x = d->width() - w;
                }
                else
                    if( getBoolean( "LEFT", line ) )
                    {
                        x = 0;
                    }

                if( getBoolean( "BOTTOM", line ) )
                {
                    QDesktopWidget *d = QApplication::desktop();
                    y = d->height() - h;
                }
                else
                    if( getBoolean( "TOP", line ) )
                    {
                        y = 0;
                    }

                pm = QPixmap(size());

                if( getBoolean("ONTOP", line ) )
                {
                    onTop = true;
                    KWin::setState( winId(), NET::StaysOnTop );
                }

                if( getBoolean("MANAGED", line ) ) {
		  managed = true;
		  reparent(0, Qt::WType_Dialog | WStyle_Customize | WStyle_NormalBorder|  WRepaintNoErase | WDestructiveClose, pos());
		  //		    reparent(0,WDestructiveClose, pos());
		    
		    
		} else {
		  //info->setDesktop( NETWinInfo::OnAllDesktops );
		  info->setState( NETWinInfo::SkipTaskbar | NETWinInfo::SkipPager,NETWinInfo::SkipTaskbar | NETWinInfo::SkipPager );
		  if (onTop) {
                    KWin::setState( winId(), NET::StaysOnTop );

		  }
		}

                if (getBoolean("ONALLDESKTOPS", line))
                {
                    desktop = 200; // ugly
                }


                bool dfound=false;
                int desktop = getInt("DESKTOP", line, dfound);
                if (dfound)
                {
                    info->setDesktop( dfound );
                }
                if( getBoolean("TOPBAR", line ) )
                {
                    move(x,0);
                    KWin::setStrut( winId(), 0, 0, h, 0 );
                    toggleLocked->setChecked( true );
                    slotToggleLocked();
                    toggleLocked->setEnabled(false);

                }

                if( getBoolean("BOTTOMBAR", line ) )
                {
                    int dh = QApplication::desktop()->height();
                    move( x, dh - h );
                    KWin::setStrut( winId(), 0, 0, 0, h );
                    toggleLocked->setChecked( true );
                    slotToggleLocked();
                    toggleLocked->setEnabled(false);


                }


		bool found = false;
                QString path = getString( "MASK", line, found );

                if( 1 )
                {		  

		  QFileInfo info(path);
		  if( info.isRelative())
                    path = themePath +"/" + path;

		  widgetMask = new QBitmap(path);
  
		  setMask(*widgetMask);
		  
                }

                interval = getInt("INTERVAL",line);
                interval = ( interval == 0 ) ? 5000 : interval;
                foundKaramba = true;
            }

            if( meter == "THEME" )
            {
                QString path = getString( "PATH", line );
                QFileInfo info(path);
                if( info.isRelative())
                    path = themePath +"/" + path;
		//                qDebug("path: "+path);
                (new karamba( path, false ))->show();
            }

            if( meter == "<GROUP>" )
            {
                int offsetX = offsetStack.top().x();
                int offsetY = offsetStack.top().y();
                offsetStack.push( QPoint( offsetX + getInt("X",line),
                                          offsetY + getInt("Y",line) ) );
            }

            if( meter == "</GROUP>" )
            {
                offsetStack.pop();
            }

            if( meter == "CLICKAREA" )
            {
                if( !hasMouseTracking() )
                    setMouseTracking(true);
                ClickArea *tmp = new ClickArea( x, y, w, h );
                tmp->setOnClick( getString( "ONCLICK", line ) );

                setSensor( line, (Meter*) tmp );
                clickList->append( tmp );
                if( getBoolean( "PREVIEW", line ) )
                    meterList->append( tmp );
            }

            // program sensor without a meter
            if(meter == "SENSOR=PROGRAM")
            {
                setSensor(line, 0 );
            }

            if(meter == "IMAGE")
            {
                QString file = getString("PATH",line);
                QString file_roll = getString("PATHROLL",line);
                int xon = getInt("XROLL",line);
                int yon = getInt("YROLL",line);
                QString tiptext = getString("TOOLTIP",line);
                QString name = getString("NAME",line);
                xon = ( xon <= 0 ) ? x:xon;
                yon = ( yon <= 0 ) ? y:yon;

                ImageLabel *tmp = new ImageLabel(x,y,0,0);
                tmp->setThemePath( themePath );
                tmp->setValue( file );
				tmp->parseImages(file, file_roll, x,y, xon, yon);
				tmp->setKaramba(this);
				if (name != "")
				  tmp->setName(name.ascii());
				if (!tiptext.isEmpty())
					tmp->setTooltip(tiptext);

                //bool sensorFound = false;
                //getString( "SENSOR", line, sensorFound );
                //if( !sensorFound )

                setSensor(line, (Meter*) tmp );
                meterList->append (tmp );
                imageList->append (tmp );
            }

            if( meter == "DEFAULTFONT" )
            {
                int r,g,b;

                delete defaultTextField;
                defaultTextField = new TextField( );

                get3Int( "COLOR", line, r, g, b );
                defaultTextField->setColor( r, g, b );
                get3Int( "BGCOLOR", line, r, g, b );
                defaultTextField->setBGColor( r, g, b );

                defaultTextField->setFont( getString( "FONT", line ) );
                defaultTextField->setFontSize( getInt( "FONTSIZE", line ) );
                defaultTextField->setAlignment( getString( "ALIGN", line ) );
                defaultTextField->setFixedPitch( getBoolean( "FIXEDPITCH", line ) );
                defaultTextField->setShadow( getInt( "SHADOW", line ) );

            }


            if(meter == "TEXT" || meter == "CLICKMAP")
            {
                int r,g,b;

                TextField* tmpText = new TextField();
                bool paramFound = false;

                get3Int( "COLOR", line, r, g, b, paramFound );
                if ( paramFound || !defaultTextField ) {
                    tmpText->setColor ( r, g, b );
		}
                else
                    if ( defaultTextField )
                    {
                        defaultTextField->getColor().rgb ( &r, &g, &b );
                        tmpText->setColor( r, g, b );
                    }

                get3Int( "BGCOLOR", line, r, g, b, paramFound );
                if ( paramFound  || !defaultTextField )
                    tmpText->setBGColor( r, g, b );
                else
                    if ( defaultTextField )
                    {
                        defaultTextField->getBGColor().rgb ( &r, &g, &b );
                        tmpText->setBGColor( r, g, b );
                    }

                QString dFont = getString("FONT", line, paramFound);
                if( paramFound || !defaultTextField )
                    tmpText->setFont( dFont );
                else
                    if( defaultTextField )
                        tmpText->setFont( defaultTextField->getFont() );


                int dFontSize = getInt( "FONTSIZE", line, paramFound );
                if( paramFound || !defaultTextField )
                    tmpText->setFontSize( dFontSize );
                else
                    if( defaultTextField )
                        tmpText->setFontSize( defaultTextField->getFontSize() );

                QString dAlign = getString( "ALIGN", line, paramFound );
                if( paramFound || !defaultTextField )
                    tmpText->setAlignment( dAlign );
                else
                    if( defaultTextField )
                        tmpText->setAlignment( defaultTextField->getAlignment() );

                bool dFp = getBoolean( "FIXEDPITCH", line, paramFound );
                if( paramFound || !defaultTextField )
                    tmpText->setFixedPitch( dFp );
                else
                    if( defaultTextField )
                        tmpText->setFixedPitch( defaultTextField->getFixedPitch() );

                int dShadow = getInt( "SHADOW", line, paramFound );
                if( paramFound || !defaultTextField )
                    tmpText->setShadow( dShadow );
                else
                    if( defaultTextField )
                        tmpText->setShadow( defaultTextField->getShadow() );

                // ////////////////////////////////////////////////////
                // Now handle the specifics
                if( meter == "TEXT" )
                {
                    
                    TextLabel *tmp = new TextLabel( x, y, w, h );
                    tmp->setTextProps( tmpText );
                    tmp->setValue( getString( "VALUE", line ) );

                    QString name = getString( "NAME", line);
                    if (name != "")
                        tmp->setName(name.ascii());

                    setSensor( line, (Meter*)tmp );
                    meterList->append ( tmp );
                }

                if(meter == "CLICKMAP")
                {
                    if( !hasMouseTracking() )
                        setMouseTracking(true);
                    ClickMap *tmp = new ClickMap( x, y, w, h );
                    tmp->setTextProps( tmpText );

                    setSensor( line, (Meter*) tmp );
                    // set all params
                    clickList -> append(tmp);
                    meterList->append( tmp );

                }

            }

            if(meter == "BAR")
            {
                Bar *tmp =
                    new Bar( x, y, w, h );
                tmp->setThemePath( themePath );
                tmp->setImage( getString( "PATH", line ) );
                tmp->setVertical( getBoolean( "VERTICAL", line ) );
                bool maxFound = false;
                int max = getInt( "MAX", line, maxFound );
                if( maxFound )
                    tmp->setMax( max );
                bool minFound = false;
                int min = getInt( "MIN", line, minFound );
                if( minFound )
                    tmp->setMin( min );
                tmp->setValue( getInt( "VALUE", line ) );
                QString name = getString("NAME",line);
                if (name != "")
                    tmp->setName(name.ascii());
                setSensor( line, (Meter*)tmp );
                meterList->append ( tmp );
            }

            if(meter == "GRAPH")
            {
                int r, g, b;
                int points = getInt("POINTS",line);
                get3Int("COLOR", line, r, g, b);

                Graph *tmp = new Graph( x, y, w, h, points );
                bool maxFound = false;
                int max = getInt( "MAX", line, maxFound );
                if( maxFound )
                    tmp->setMax( max );
                bool minFound = false;
                int min = getInt( "MIN", line, minFound );
                if( minFound )
                    tmp->setMin( min );

                tmp->setColor( r , g, b );

                setSensor( line, (Graph*)tmp );
                meterList->append ( tmp );
            }
        }
    }
    if( !foundKaramba )
    {
        //  interval = initKaramba( "", 0, 0, 0, 0 );
        //   this->close(true);
        //delete this;
        return false;
    }
    else
    {
        connect(m_sysTimer, SIGNAL(timeout()), SLOT(step()));

        interval = interval == 0 ? 1000 : interval;
        m_sysTimer->start(interval);
        QTimer::singleShot( 0, this, SLOT(step()) );
        //QTimer::singleShot( 5000, this, SLOT(step()) );
        //QTimer::singleShot( 10000, this, SLOT(step()) );

        if( !(onTop || managed) )
            lowerTimer.start();
        //update();
        //QTimer::singleShot( 0, this, SLOT(update()) );
        return true;
    }
}

void karamba::startNewKaramba()
{
    QStringList fileNames;
    fileNames = KFileDialog::getOpenFileNames(QString::null, "*.theme *.ctheme|Themes", 0, "Open Themes");
    for ( QStringList::Iterator it = fileNames.begin(); it != fileNames.end(); ++it )
    {
        QFileInfo file( *it );
        if( file.exists() )
        {
            (new karamba( *it, false ))->show();
        }
    }
}

void karamba::popupNotify(int item)
{
  
}

void karamba::reloadConfig()
{
    writeConfigData();
    QFileInfo file( themeFile );
    //qDebug(themeFile);
    if( file.exists() )
    {
        (new karamba( themeFile, true ))->show();
    }
    killWidget();
}

void karamba::killWidget() {


  close();
}

void karamba::quitKaramba()
{
	//qWarning("Quit Karamba.");
	qApp -> closeAllWindows();

	qApp -> quit();
}


void karamba::editConfig()
{
    QFileInfo fileInfo( themeFile );
    QString path;

    if( fileInfo.isRelative() )
    {
        path = themePath + "/" + themeFile;
    }
    else
    {
        path = themeFile;
    }
    //    qDebug("krun:"+path);
    KRun::runURL( KURL( path ), "text/plain" );

}

void karamba::editScript()
{
    QFileInfo fileInfo( themeFile );
    QString path;

    if( fileInfo.isRelative() )
    {
        path = themePath + "/" + themeName + ".py";
    }
    else
    {
        path = QFileInfo(themeFile).dirPath() + "/" + themeName + ".py";
    }
    //    qDebug("krun:"+path);
    KRun::runURL( KURL( path ), "text/plain" );

}


int karamba::getInt(QString w, QString &line )
{
    bool b;
    return getInt( w, line, b );
}

int karamba::getInt(QString w, QString &line, bool &found )
{
    QRegExp rx( "\\W+" + w +"=([-]?\\d+)", false );
    found = (rx.search(line)==-1)?false:true;
    return rx.cap(1).toInt();
}

void karamba::get3Int(QString w, QString &line, int &a, int &b, int &c )
{
    bool bo;
    get3Int ( w, line, a, b, c, bo );
}

void karamba::get3Int(QString w, QString &line, int &a, int &b, int &c, bool &found )
{
    QRegExp rx( "\\W+" + w + "=([-]?\\d+),([-]?\\d+),([-]?\\d+)", false );
    found = (rx.search(line)==-1)?false:true;
    a = rx.cap(1).toInt();
    b = rx.cap(2).toInt();
    c = rx.cap(3).toInt();
}

QString karamba::getString(QString w, QString &line)
{
    bool b;
    return getString ( w, line, b );
}

QString karamba::getString(QString w, QString &line, bool &found)
{
    QRegExp rx( "\\W+" + w + "=\"([^\"]*)\"", false );
    found = (rx.search(line)==-1)?false:true;
    if (rx.cap(1).isEmpty())
    {
        rx = QRegExp(w + "=(\\S+)", false);
        rx.search(line);
        return rx.cap(1);
    }
    else
    {
        return rx.cap(1);
    }

}

bool karamba::getBoolean( QString w, QString &line )
{
    bool b;
    return getBoolean ( w, line, b );
}

bool karamba::getBoolean( QString w, QString &line, bool &found )
{
    return  ( getString( w, line, found ).lower() == "true")?true:false;
}

QString karamba::findSensorFromMap(Sensor* sensor)
{
  QMap<QString,Sensor*>::Iterator it;

  for ( it = sensorMap.begin(); it != sensorMap.end(); ++it )
    {
        if (it.data() == sensor)
          return it.key();
    }
  return "";
}

Sensor* karamba::findSensorFromList(Meter* meter)
{
  QObjectListIt it( *sensorList ); // iterate over meters

  while ( it != 0 )
    {
        if (((Sensor*) *it)->hasMeter(meter))
            return ((Sensor*)*it);
        ++it;
    }
  return NULL;
}

void karamba::deleteMeterFromSensors(Meter* meter)
{
  Sensor* sensor = findSensorFromList(meter);

  if (sensor)
    {
        sensor->deleteMeter(meter);
        if (sensor->isEmpty())
          {
            QString s = findSensorFromMap(sensor);
            sensorMap.erase(s);
            sensorList->removeRef(sensor);
          }
    }
}

void karamba::setSensor(QString &line, Meter* meter)
{
    Sensor* sensor = 0;

    QString sens = getString("SENSOR",line).upper();

    if( sens == "CPU" )
    {
        QString cpuNbr = getString( "CPU", line );
        sensor = sensorMap["CPU"+cpuNbr];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?1000:interval;
            sensor = ( sensorMap["CPU"+cpuNbr] = new CPUSensor( cpuNbr, interval ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam("FORMAT",getString("FORMAT",line));
        sp->addParam("DECIMALS",getString("DECIMALS",line));

        sensor->addMeter(sp);
        sensor->setMaxValue(sp);

    }

    if( sens == "MEMORY" )
    {
        sensor = sensorMap["MEMORY"];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?3000:interval;
            sensor = ( sensorMap["MEMORY"] = new MemSensor( interval ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam("FORMAT",getString("FORMAT",line));

        sensor->addMeter(sp);
        sensor->setMaxValue(sp);
    }


    if( sens == "DISK" )
    {
        sensor = sensorMap["DISK"];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?5000:interval;
            sensor = ( sensorMap["DISK"] = new DiskSensor( interval ) );
            sensorList->append( sensor );
        }
        // meter->setMax( ((DiskSensor*)sensor)->getTotalSpace(mntPt)/1024 );
        SensorParams *sp = new SensorParams(meter);
        QString mntPt = getString("MOUNTPOINT",line);
        if( mntPt == ""  )
            mntPt = "/";

        sp->addParam("MOUNTPOINT",mntPt);
        sp->addParam("FORMAT",getString("FORMAT",line));
        sensor->addMeter(sp);
        sensor->setMaxValue(sp);
    }

    if( sens == "NETWORK")
    {
        int interval = getInt("INTERVAL",line );
        interval = (interval == 0)?2000:interval;
        QString device = getString("DEVICE", line );
        sensor = sensorMap["NETWORK"+device];
        if (sensor == 0)
        {
            sensor = ( sensorMap["NETWORK"+device] = new NetworkSensor( device, interval ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam( "FORMAT", getString( "FORMAT", line ) );
        sp->addParam( "DECIMALS", getString( "DECIMALS", line ) );
        sensor->addMeter(sp);
    }

    if( sens == "UPTIME" )
    {
        sensor = sensorMap["UPTIME"];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?60000:interval;
            sensor = ( sensorMap["UPTIME"] = new UptimeSensor( interval ));
            sensorList->append( sensor );

        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam( "FORMAT", getString( "FORMAT", line ) );
        sensor->addMeter(sp);
    }

    if( sens == "SENSOR" )
    {
        sensor = sensorMap["SENSOR"];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?30000:interval;
            sensor = ( sensorMap["SENSOR"] = new SensorSensor( interval ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam("FORMAT",getString("FORMAT",line));
        sp->addParam("TYPE",getString("TYPE",line));
        sensor->addMeter(sp);
    }


    if( sens == "TEXTFILE" )
    {
        QString path = getString( "PATH", line );
        bool rdf = getBoolean( "RDF", line );
        sensor = sensorMap["TEXTFILE"+path];
        if (sensor == 0)
        {
            int interval = getInt( "INTERVAL", line );
            interval = ( interval == 0 )?60000:interval;
            QString encoding = getString( "ENCODING", line );

            sensor = ( sensorMap["TEXTFILE"+path] =
                           new TextFileSensor( path, rdf, interval, encoding ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam("LINE",QString::number(getInt("LINE",line)));
        sensor->addMeter(sp);
    }


    if( sens == "TIME")
    {
        sensor = sensorMap["DATE"];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?60000:interval;
            sensor = ( sensorMap["DATE"] = new DateSensor( interval ) );
            sensorList->append( sensor );
			timeList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam("FORMAT",getString("FORMAT",line));
        sp->addParam("CALWIDTH",getString("CALWIDTH",line));
        sp->addParam("CALHEIGHT",getString("CALHEIGHT",line));
        sensor->addMeter(sp);
    }

#ifdef HAVE_XMMS

    if( sens == "XMMS" )
    {
        sensor = sensorMap["XMMS"];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?1000:interval;
            QString encoding = getString( "ENCODING", line );

            sensor = ( sensorMap["XMMS"] = new XMMSSensor( interval, encoding ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam("FORMAT",getString("FORMAT",line));
        sensor->addMeter(sp);
        sensor->setMaxValue(sp);
    }
#endif // HAVE_XMMS


    if( sens == "NOATUN" )
    {
        sensor = sensorMap["NOATUN"];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?1000:interval;
            sensor = ( sensorMap["NOATUN"] = new NoatunSensor( interval, client ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam("FORMAT",getString("FORMAT",line));
        sensor->addMeter(sp);
        sensor->setMaxValue(sp);
    }

    if( sens == "PROGRAM")
    {
        QString progName = getString("PROGRAM",line);
        sensor = sensorMap["PROGRAM"+progName];
        if (sensor == 0)
        {
            int interval = getInt("INTERVAL",line);
            interval = (interval == 0)?3600000:interval;
            QString encoding = getString( "ENCODING", line );

            sensor = (sensorMap["PROGRAM"+progName] =
                          new ProgramSensor( progName, interval, encoding ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam( "LINE", QString::number( getInt( "LINE", line ) ) );
        sp->addParam( "THEMAPATH", themePath );
        sensor->addMeter(sp);
    }

    if( sens == "RSS" )
    {
        QString source = getString( "SOURCE", line );
        QString format = getString( "FORMAT", line );

        sensor = sensorMap["RSS"+source];
        if (sensor == 0)
        {
            int interval = getInt( "INTERVAL", line );
            interval = ( interval == 0 )?60000:interval;
            QString encoding = getString( "ENCODING", line );

            sensor = ( sensorMap["RSS"+source] =
                           new RssSensor( source, interval, format, encoding ) );
            sensorList->append( sensor );
        }
        SensorParams *sp = new SensorParams(meter);
        sp->addParam("SOURCE",getString("SOURCE",line));
        sensor->addMeter(sp);
    }

    if (sensor != 0)
    {
        QTimer::singleShot( 0, sensor, SLOT(update()) );
        sensor->start();
    }
}

void karamba::passMenuOptionChanged(QString key, bool value)
{
//Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded) {

      // get the global lock
      PyEval_AcquireLock();
      // get a reference to the PyInterpreterState

      //Python thread references
      extern PyThreadState * mainThreadState;

      PyInterpreterState * mainInterpreterState = mainThreadState->interp;
      // create a thread state object for this thread
      PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
      PyThreadState_Swap(myThreadState);

      pFunc = PyDict_GetItemString(pDict, "menuOptionChanged");

      if (pFunc && PyCallable_Check(pFunc)) {
	        pArgs = PyTuple_New(3);

          pValue = PyInt_FromLong((int)this);
          PyTuple_SetItem(pArgs, 0, pValue);

          pValue = PyString_FromString(key.ascii());
          PyTuple_SetItem(pArgs, 1, pValue);

          pValue = PyInt_FromLong((int)value);
          PyTuple_SetItem(pArgs, 2, pValue);

          pValue = PyObject_CallObject(pFunc, pArgs);

          if (pValue != NULL) {
            Py_DECREF(pValue);
          }
          else {
            qWarning("Call to menuOptionChanged failed");
	    PyErr_Print();
          }
          Py_DECREF(pArgs);
      }
      else {
	//callback wasn't found
      }


      // swap my thread state out of the interpreter
      PyThreadState_Swap(NULL);
      // clear out any cruft from thread state object
      PyThreadState_Clear(myThreadState);
      // delete my thread state object
      PyThreadState_Delete(myThreadState);
      // release the lock
      PyEval_ReleaseLock();
    }
}

void karamba::meterClicked(QMouseEvent* e, Meter* meter)
{
    //qWarning("karamba::meterClicked");
    if (pythonThemeExtensionLoaded && haveUpdated)
    {
        // get the global lock
        PyEval_AcquireLock();
        // get a reference to the PyInterpreterState

        //Python thread references
        extern PyThreadState * mainThreadState;

        PyInterpreterState * mainInterpreterState = mainThreadState -> interp;
        // create a thread state object for this thread
        PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
        PyThreadState_Swap(myThreadState);

        pFunc = PyDict_GetItemString(pDict, "meterClicked");

        if (pFunc && PyCallable_Check(pFunc))
        {
	  //qWarning ("meterClicked is going to be called");

            pArgs = PyTuple_New(3);

            pValue = PyInt_FromLong((int)this);
            PyTuple_SetItem(pArgs, 0, pValue);

            //pValue = PyInt_FromLong(e->x());
            //PyTuple_SetItem(pArgs, 1, pValue);

            //pValue = PyInt_FromLong(e->y());
            //PyTuple_SetItem(pArgs, 2, pValue);

            // if meter is of type RichTextLabel
            if (RichTextLabel* richText = dynamic_cast<RichTextLabel*>(meter))
            {
                pValue = PyString_FromString(richText -> anchorAt(e -> x(), e -> y()).ascii());
            }
            else
            {
                pValue = PyInt_FromLong((int)meter);
            }
            PyTuple_SetItem(pArgs, 1, pValue);

            int button = 0;

            if( e->button() == Qt::LeftButton )
                button = 1;
            else if( e->button() == Qt::MidButton )
                button = 2;
            else if( e->button() == Qt::RightButton )
                button = 3;
            pValue = PyInt_FromLong(button);
            PyTuple_SetItem(pArgs, 2, pValue);


            pValue = PyObject_CallObject(pFunc, pArgs);
            if (pValue != NULL)
            {
                Py_DECREF(pValue);
            }
            else
            {
                qWarning("Call to meterClicked failed");
		PyErr_Print();
            }
            Py_DECREF(pArgs);
        }
        else
        {
	  //callback not found
        }

        // swap my thread state out of the interpreter
        PyThreadState_Swap(NULL);
        // clear out any cruft from thread state object
        PyThreadState_Clear(myThreadState);
        // delete my thread state object
        PyThreadState_Delete(myThreadState);
        // release the lock
        PyEval_ReleaseLock();
    }
}

void karamba::passClick( QMouseEvent *e )
{
	QObjectListIt it2( *timeList ); // iterate over meters
    while ( it2 != 0 )
    {
        (( DateSensor* ) *it2)->toggleCalendar( e );
        ++it2;
    }

    QObjectListIt it(*clickList);
    while (it != 0)
    {
        if (((Meter*)(*it)) -> click(e))
        {
            // callback
            meterClicked(e, (Meter*)(*it));
        }
        ++it;
    }

    //Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded && haveUpdated) {

      // get the global lock
      PyEval_AcquireLock();
      // get a reference to the PyInterpreterState

      //Python thread references
      extern PyThreadState * mainThreadState;

      PyInterpreterState * mainInterpreterState = mainThreadState->interp;
      // create a thread state object for this thread
      PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
      PyThreadState_Swap(myThreadState);

      pFunc = PyDict_GetItemString(pDict, "widgetClicked");

      if (pFunc && PyCallable_Check(pFunc)) {
	pArgs = PyTuple_New(4);

	pValue = PyInt_FromLong((int)this);
	PyTuple_SetItem(pArgs, 0, pValue);

	pValue = PyInt_FromLong(e->x());
	PyTuple_SetItem(pArgs, 1, pValue);

	pValue = PyInt_FromLong(e->y());
	PyTuple_SetItem(pArgs, 2, pValue);

	int button = 0;

	if( e->button() == Qt::LeftButton )
	  button = 1;
	else if( e->button() == Qt::MidButton )
	  button = 2;
	else if( e->button() == Qt::RightButton )
	  button = 3;

	pValue = PyInt_FromLong(button);
	PyTuple_SetItem(pArgs, 3, pValue);


	pValue = PyObject_CallObject(pFunc, pArgs);
	if (pValue != NULL) {
	  Py_DECREF(pValue);
	}
	else {
	  qWarning("Call to widgetClicked failed");
	  PyErr_Print();
	}
	Py_DECREF(pArgs);
      }
      else {
	//callback not found
      }


      // swap my thread state out of the interpreter
      PyThreadState_Swap(NULL);
      // clear out any cruft from thread state object
      PyThreadState_Clear(myThreadState);
      // delete my thread state object
      PyThreadState_Delete(myThreadState);
      // release the lock
      PyEval_ReleaseLock();
    }
}


void karamba::passWheelClick( QWheelEvent *e )
{
    //Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded && haveUpdated) {

      // get the global lock
      PyEval_AcquireLock();
      // get a reference to the PyInterpreterState

      //Python thread references
      extern PyThreadState * mainThreadState;

      PyInterpreterState * mainInterpreterState = mainThreadState->interp;
      // create a thread state object for this thread
      PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
      PyThreadState_Swap(myThreadState);


      //    PyRun_SimpleString("iconbar.yo()");

      pFunc = PyDict_GetItemString(pDict, "widgetClicked");

      if (pFunc && PyCallable_Check(pFunc)) {
	pArgs = PyTuple_New(4);

	pValue = PyInt_FromLong((int)this);
	PyTuple_SetItem(pArgs, 0, pValue);

	pValue = PyInt_FromLong(e->x());
	PyTuple_SetItem(pArgs, 1, pValue);

	pValue = PyInt_FromLong(e->y());
	PyTuple_SetItem(pArgs, 2, pValue);

	int button = 0;

	if( e->delta() > 0 )

	  button = 4;
	else
	  button = 5;



	pValue = PyInt_FromLong(button);
	PyTuple_SetItem(pArgs, 3, pValue);


	pValue = PyObject_CallObject(pFunc, pArgs);
	if (pValue != NULL) {
	  Py_DECREF(pValue);
	}
	else {
	  qWarning("Call to widgetClicked failed");
	  PyErr_Print();
	}
	Py_DECREF(pArgs);
      }
      else {
	//callback not found
      }


      // swap my thread state out of the interpreter
      PyThreadState_Swap(NULL);
      // clear out any cruft from thread state object
      PyThreadState_Clear(myThreadState);
      // delete my thread state object
      PyThreadState_Delete(myThreadState);
      // release the lock
      PyEval_ReleaseLock();
    }
}

void karamba::mousePressEvent( QMouseEvent *e )
{

    if( e->button() == RightButton )
    {

        kpop->exec(QCursor::pos());
    }
    else
    {
        clickPos = e->pos();
        if( toggleLocked -> isChecked() )
            passClick( e );
        if( !(onTop || managed))
            lower();
    }
}

void karamba::wheelEvent( QWheelEvent *e )
{
  passWheelClick( e );
}

void karamba::mouseReleaseEvent( QMouseEvent *e )
{
    /*if( ks != 0)
    {
        ks->loadFromShared("DESKTOP1",
                           QRect(QPoint(x(),y()), QPoint(x()+width()-1,y()+height()-1)));
    }
    */
    clickPos = e->pos();
    //    qDebug("Click   X="+QString::number( clickPos.x() )+" Y="+QString::number( clickPos .y() ) );
    //    qDebug("Window  X="+QString::number( ( e->globalPos() - clickPos ).x())+" Y="+QString::number( ( e->globalPos() - clickPos ).y()));

    //update();
    step();
}

void karamba::mouseDoubleClickEvent( QMouseEvent *e )
{
    if( !toggleLocked -> isChecked() )
    {
        passClick( e );
    }
}


void karamba::keyPressEvent(QKeyEvent *e) 
{
  //Everything below is to call the python callback function

  if (pythonThemeExtensionLoaded) {

    // get the global lock
    PyEval_AcquireLock();
    // get a reference to the PyInterpreterState

    //Python thread references
    extern PyThreadState * mainThreadState;

    PyInterpreterState * mainInterpreterState = mainThreadState->interp;
    // create a thread state object for this thread
    PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
    PyThreadState_Swap(myThreadState);

    pFunc = PyDict_GetItemString(pDict, "keyPressed");

    if (pFunc && PyCallable_Check(pFunc)) {
      pArgs = PyTuple_New(2);

      pValue = PyInt_FromLong((int)this);
      PyTuple_SetItem(pArgs, 0, pValue);

      pValue = PyString_FromString(e->text().ascii());
      PyTuple_SetItem(pArgs, 1, pValue);

      pValue = PyObject_CallObject(pFunc, pArgs);
      if (pValue != NULL) {
	Py_DECREF(pValue);
      }
      else {
	qWarning("Call to keyPressed failed");
	PyErr_Print();
      }
      Py_DECREF(pArgs);
      }
    else {
      //callback not found
    }


    // swap my thread state out of the interpreter
    PyThreadState_Swap(NULL);
    // clear out any cruft from thread state object
    PyThreadState_Clear(myThreadState);
    // delete my thread state object
    PyThreadState_Delete(myThreadState);
    // release the lock
    PyEval_ReleaseLock();
  }

  

}


void karamba::mouseMoveEvent( QMouseEvent *e )
{
    if( e->state() !=  0 && e->state() < 16 && !toggleLocked -> isChecked() )
    {
        move( e->globalPos() - clickPos );
    }
    else
    {
        // Change cursor over ClickArea
        QObjectListIt it(*clickList);
        bool insideArea = false;

        while (it != 0)
        {
            insideArea = ((Meter*)(*it)) -> insideActiveArea(e -> x(), e -> y());
            if (insideArea)
            {
                break;
            }
            ++it;
        }

        if(insideArea)
        {
            if( cursor().shape() != PointingHandCursor )
                setCursor( PointingHandCursor );
        }
        else
        {
            if( cursor().shape() != ArrowCursor )
                setCursor( ArrowCursor );
        }

        QObjectListIt image_it( *imageList);	// iterate over image sensors
        while ( image_it != 0 )
        {
            ((ImageLabel*) *image_it)->rolloverImage(e);
            ++image_it;
        }
    }


    if (pythonThemeExtensionLoaded)
    {
        // get the global lock
        PyEval_AcquireLock();
        // get a reference to the PyInterpreterState

        //Python thread references
        extern PyThreadState * mainThreadState;

        PyInterpreterState * mainInterpreterState = mainThreadState->interp;
        // create a thread state object for this thread
        PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
        PyThreadState_Swap(myThreadState);

        pFunc = PyDict_GetItemString(pDict, "widgetMouseMoved");

        if (pFunc && PyCallable_Check(pFunc))
        {
            pArgs = PyTuple_New(4);

            pValue = PyInt_FromLong((int)this);
            PyTuple_SetItem(pArgs, 0, pValue);

            pValue = PyInt_FromLong(e->x());
            PyTuple_SetItem(pArgs, 1, pValue);

            pValue = PyInt_FromLong(e->y());
            PyTuple_SetItem(pArgs, 2, pValue);

            int button = 0;

            if( e->button() == Qt::LeftButton )
                button = 1;
            else if( e->button() == Qt::MidButton )
                button = 2;
            else if( e->button() == Qt::RightButton )
                button = 3;

            pValue = PyInt_FromLong(button);
            PyTuple_SetItem(pArgs, 3, pValue);

            pValue = PyObject_CallObject(pFunc, pArgs);
            if (pValue != NULL)
            {
                Py_DECREF(pValue);
            }
            else
            {
                //callback failed
            }
            Py_DECREF(pArgs);
        }
        else
        {
            //callback not found
        }

        // swap my thread state out of the interpreter
        PyThreadState_Swap(NULL);
        // clear out any cruft from thread state object
        PyThreadState_Clear(myThreadState);
        // delete my thread state object
        PyThreadState_Delete(myThreadState);
        // release the lock
        PyEval_ReleaseLock();
    }
}

void karamba::closeEvent ( QCloseEvent *  qc)
{
  qc->accept();

//  this->close(true);
//  delete this;
}

void karamba::paintEvent ( QPaintEvent *e)
{
    if( !(onTop || managed))
        if( lowerTimer.elapsed() > 100 )
        {
            this->lower();
            lowerTimer.restart();
        }
    QRect rect = e->rect();
    bitBlt(this,rect.topLeft(),&pm,rect,Qt::CopyROP);
}

void karamba::updateSensors()
{
    QObjectListIt it( *sensorList ); // iterate over meters
    while ( it != 0 )
    {
        ((Sensor*) *it)->update();
        ++it;
    }
    QTimer::singleShot( 500, this, SLOT(step()) );
}

void karamba::updateBackground(KSharedPixmap* kpm)
{
  background = QPixmap(*kpm);

  QPixmap buffer = QPixmap(size());

  pm = QPixmap(size());
  buffer.fill(Qt::black);

    QObjectListIt it( *imageList ); // iterate over meters
    p.begin(&buffer);
    bitBlt(&buffer,0,0,&background,0,Qt::CopyROP);

    while ( it != 0 )
    {
      if (((ImageLabel*) *it)->background == 1)
        ((ImageLabel*) *it)->mUpdate(&p, 1);
      ++it;
    }
    p.end();


    bitBlt(&pm,0,0,&buffer,0,Qt::CopyROP);
    background = pm;

    QPixmap buffer2 = QPixmap(size());

    pm = QPixmap(size());
    buffer2.fill(Qt::black);

    QObjectListIt it2( *meterList ); // iterate over meters
    p.begin(&buffer2);
    bitBlt(&buffer2,0,0,&background,0,Qt::CopyROP);

    while ( it2 != 0 )
    {
        ((Meter*) *it2)->mUpdate(&p);
        ++it2;
    }
    p.end();

    bitBlt(&pm,0,0,&buffer2,0,Qt::CopyROP);
    if (systray != 0){
      systray->updateBackgroundPixmap(buffer2);
    }
    repaint();

}

void karamba::currentDesktopChanged( int )
{
    kroot->repaint( true );
}


void karamba::externalStep()
{
  QPixmap buffer = QPixmap(size());

  pm = QPixmap(size());
  buffer.fill(Qt::black);

    QObjectListIt it( *meterList ); // iterate over meters
    p.begin(&buffer);
    bitBlt(&buffer,0,0,&background,0,Qt::CopyROP);

    while ( it != 0 )
    {
        ((Meter*) *it)->mUpdate(&p);
        ++it;
    }
    p.end();

    bitBlt(&pm,0,0,&buffer,0,Qt::CopyROP);
    repaint();
}


void karamba::step()
{
    pm = QPixmap(size());
    QPixmap buffer = QPixmap(size());
    buffer.fill(Qt::black);

    QObjectListIt it( *meterList ); // iterate over meters
    p.begin(&buffer);

    bitBlt(&buffer,0,0,&background,0,Qt::CopyROP);


    while ( it != 0 )
    {
        ((Meter*) *it)->mUpdate(&p);
        ++it;
    }
    p.end();

    bitBlt(&pm,0,0,&buffer,0,Qt::CopyROP);
    update();


    if (pythonThemeExtensionLoaded) {

      // get the global lock
      PyEval_AcquireLock();
      // get a reference to the PyInterpreterState

      //Python thread references
      extern PyThreadState * mainThreadState;

      PyInterpreterState * mainInterpreterState = mainThreadState->interp;
      // create a thread state object for this thread
      PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
      PyThreadState_Swap(myThreadState);

      if (haveUpdated == 0) {
	pFunc = PyDict_GetItemString(pDict, "initWidget");
	haveUpdated = 1;
      }
      else
	pFunc = PyDict_GetItemString(pDict, "widgetUpdated");


      if (pFunc && PyCallable_Check(pFunc)) {
	pArgs = PyTuple_New(1);

	  pValue = PyInt_FromLong((int)this);
	  PyTuple_SetItem(pArgs, 0, pValue);
	
	pValue = PyObject_CallObject(pFunc, pArgs);
	if (pValue != NULL) {
	  Py_DECREF(pValue);
	}
	else {
	  qWarning("Call to widgetUpdated failed");
	  PyErr_Print();
	}
	Py_DECREF(pArgs);

      }
	else {
	  //callback not found
	}

      // swap my thread state out of the interpreter
      PyThreadState_Swap(NULL);
      // clear out any cruft from thread state object
      PyThreadState_Clear(myThreadState);
      // delete my thread state object
      PyThreadState_Delete(myThreadState);
      // release the lock
      PyEval_ReleaseLock();

    
    }
}

void karamba::widgetClosed() {
    if (pythonThemeExtensionLoaded) {

      // get the global lock
      PyEval_AcquireLock();
      // get a reference to the PyInterpreterState

      //Python thread references
      extern PyThreadState * mainThreadState;

      PyInterpreterState * mainInterpreterState = mainThreadState->interp;
      // create a thread state object for this thread
      PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
      PyThreadState_Swap(myThreadState);

      pFunc = PyDict_GetItemString(pDict, "widgetClosed");


      if (pFunc && PyCallable_Check(pFunc)) {
	pArgs = PyTuple_New(1);

	  pValue = PyInt_FromLong((int)this);
	  PyTuple_SetItem(pArgs, 0, pValue);
	
	pValue = PyObject_CallObject(pFunc, pArgs);
	if (pValue != NULL) {
	  Py_DECREF(pValue);
	}
	else {
	  qWarning("Call to widgetClosed failed");
	  //	  PyErr_Print();
	}
	Py_DECREF(pArgs);

      }
	else {
	  //callback not found
	}

      // swap my thread state out of the interpreter
      PyThreadState_Swap(NULL);
      // clear out any cruft from thread state object
      PyThreadState_Clear(myThreadState);
      // delete my thread state object
      PyThreadState_Delete(myThreadState);
      // release the lock
      PyEval_ReleaseLock();

    
    }
}
void karamba::slotToggleLocked()
{
  //    qDebug("toggleLocked");
    bool locked = toggleLocked->isChecked();
    if( toggleLocked -> isChecked() )
    {
        toggleLocked->setIconSet(SmallIconSet("lock"));
    }
    else
    {
        toggleLocked->setIconSet(SmallIconSet("move"));
    }

}

void karamba::slotToggleFastTransforms()
{
//	bool fastTransforms = toggleFastTransforms -> isChecked();
// 	if (toggleFastTransforms -> isChecked())
// 	{
//     toggleFastTransforms -> setIconSet(SmallIconSet("ok"));
// 	}
// 	else
// 	{
//     QPixmap ok_disabled;
// 		toggleFastTransforms -> setIconSet(ok_disabled);
// 	}
  //config.setGroup("internal");
  //config.writeEntry("fastTransforms", toggleFastTransforms -> isChecked());
}


bool karamba::useSmoothTransforms()
{
  return !toggleFastTransforms -> isChecked();
}

void karamba::writeConfigData()
{
    config -> setGroup("internal");
    config -> writeEntry("lockedPosition", toggleLocked -> isChecked() );
    config -> writeEntry("fastTransforms", toggleFastTransforms -> isChecked() );
    config -> writeEntry("desktop", desktop );
    config -> setGroup("theme");
    // Widget Position
    config -> writeEntry("widgetPosX", x());
    config -> writeEntry("widgetPosY", y());

    // Widget Size
    config -> writeEntry("widgetWidth", width());
    config -> writeEntry("widgetHeight", height());

    // write changes to DiskSensor
    config -> sync();
    qWarning("Config File ~/.superkaramba/%s.rc written.", themeName.ascii());
}

void karamba::slotToggleConfigOption(QString key, bool value)
{
    config -> setGroup("config menu");
    config -> writeEntry(key, value);
    passMenuOptionChanged(key, value);
}

void karamba::addMenuConfigOption(QString key, QString name)
{
    kpop -> setItemEnabled(THEMECONF, true);

    SignalBridge* action = new SignalBridge(this, key, menuAccColl);
    KToggleAction* confItem = new KToggleAction (name, KShortcut::null(), action, SLOT(receive()),
                                   menuAccColl, key);
    confItem -> setName(key);

    menuAccColl -> insert(confItem);

    connect(action, SIGNAL( enabled(QString, bool) ), this, SLOT( slotToggleConfigOption(QString, bool) ));

    config -> setGroup("config menu");
    confItem -> setChecked(config -> readBoolEntry(key));

    confItem -> plug(themeConfMenu);

    numberOfConfMenuItems++;
}

bool karamba::setMenuConfigOption(QString key, bool value)
{
    KToggleAction* menuAction = ((KToggleAction*)menuAccColl -> action(key));
    if (menuAction == NULL)
    {
	    qWarning("Menu action %s not found.", key.ascii());
	    return false;
    }
    else
    {
	    menuAction -> setChecked(value);
	    return true;
    }
}

bool karamba::readMenuConfigOption(QString key)
{
	KToggleAction* menuAction = ((KToggleAction*)menuAccColl -> action(key));
	if (menuAction == NULL)
	{
		qWarning("Menu action %s not found.", key.ascii());
		return false;
	}
	else
	{
		return menuAction -> isChecked();
	}
}

void karamba::passMenuItemClicked(int id)
{
    //Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded)
    {
        // get the global lock
        PyEval_AcquireLock();
        // get a reference to the PyInterpreterState

        //Python thread references
        extern PyThreadState * mainThreadState;

        PyInterpreterState * mainInterpreterState = mainThreadState->interp;
        // create a thread state object for this thread
        PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
        PyThreadState_Swap(myThreadState);

        pFunc = PyDict_GetItemString(pDict, "menuItemClicked");

        if (pFunc && PyCallable_Check(pFunc))
        {
            pArgs = PyTuple_New(3);

            pValue = PyInt_FromLong((int)this);
            PyTuple_SetItem(pArgs, 0, pValue);

            int menu = 0;
            for(int i = 0; i < menuList->count(); i++)
            {
                KPopupMenu* tmp;
                if(i==0)
                {
                    tmp = (KPopupMenu*) menuList->first();
                }
                else
                {
                    tmp = (KPopupMenu*) menuList->next();
                }
                if(tmp != 0)
                {
                    if(tmp->isItemVisible(id))
                    {
                        menu = (int) tmp;
                        break;
                    }
                }
            }

            pValue = PyInt_FromLong(menu);
            PyTuple_SetItem(pArgs, 1, pValue);

            pValue = PyInt_FromLong(id);
            PyTuple_SetItem(pArgs, 2, pValue);


            pValue = PyObject_CallObject(pFunc, pArgs);
            if (pValue != NULL)
            {
                Py_DECREF(pValue);
            }
            else
            {
                qWarning("Call to menuItemClicked failed");
		PyErr_Print();
            }
            Py_DECREF(pArgs);
        }
        else
        {
	  //can't find callback
        }

        // swap my thread state out of the interpreter
        PyThreadState_Swap(NULL);
        // clear out any cruft from thread state object
        PyThreadState_Clear(myThreadState);
        // delete my thread state object
        PyThreadState_Delete(myThreadState);
        // release the lock
        PyEval_ReleaseLock();
    }
}

void karamba::activeTaskChanged(Task* t)
{
    //Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded)
    {

        // get the global lock
        PyEval_AcquireLock();
        // get a reference to the PyInterpreterState

        //Python thread references
        extern PyThreadState* mainThreadState;

        PyInterpreterState* mainInterpreterState = mainThreadState -> interp;
        // create a thread state object for this thread
        PyThreadState* myThreadState = PyThreadState_New(mainInterpreterState);
        PyThreadState_Swap(myThreadState);

        pFunc = PyDict_GetItemString(pDict, "activeTaskChanged");

        if (pFunc && PyCallable_Check(pFunc))
        {
            pArgs = PyTuple_New(2);

            pValue = PyInt_FromLong((int) this);
            PyTuple_SetItem(pArgs, 0, pValue);

            pValue = PyInt_FromLong((int) t);
            PyTuple_SetItem(pArgs, 1, pValue);

            pValue = PyObject_CallObject(pFunc, pArgs);
            if (pValue != NULL)
            {
                Py_DECREF(pValue);
            }
            else
            {
                qWarning("Call to activeTaskChanged failed");
		PyErr_Print();
            }
            Py_DECREF(pArgs);
        }
        else
        {
	  //callback not found
        }

        // swap my thread state out of the interpreter
        PyThreadState_Swap(NULL);
        // clear out any cruft from thread state object
        PyThreadState_Clear(myThreadState);
        // delete my thread state object
        PyThreadState_Delete(myThreadState);
        // release the lock
        PyEval_ReleaseLock();
    }
}

void karamba::taskAdded(Task* t)
{
    //Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded)
    {

        // get the global lock
        PyEval_AcquireLock();
        // get a reference to the PyInterpreterState

        //Python thread references
        extern PyThreadState* mainThreadState;

        PyInterpreterState* mainInterpreterState = mainThreadState -> interp;
        // create a thread state object for this thread
        PyThreadState* myThreadState = PyThreadState_New(mainInterpreterState);
        PyThreadState_Swap(myThreadState);

        pFunc = PyDict_GetItemString(pDict, "taskAdded");

        if (pFunc && PyCallable_Check(pFunc))
        {
            pArgs = PyTuple_New(2);

            pValue = PyInt_FromLong((int) this);
            PyTuple_SetItem(pArgs, 0, pValue);

            pValue = PyInt_FromLong((int) t);
            PyTuple_SetItem(pArgs, 1, pValue);

            pValue = PyObject_CallObject(pFunc, pArgs);
            if (pValue != NULL)
            {
                Py_DECREF(pValue);
            }
            else
            {
                qWarning("Call to taskAdded failed");
		PyErr_Print();
            }
            Py_DECREF(pArgs);
        }
        else
        {
	  //callback not found
        }

        // swap my thread state out of the interpreter
        PyThreadState_Swap(NULL);
        // clear out any cruft from thread state object
        PyThreadState_Clear(myThreadState);
        // delete my thread state object
        PyThreadState_Delete(myThreadState);
        // release the lock
        PyEval_ReleaseLock();
    }
}

void karamba::taskRemoved(Task* t)
{
    //Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded)
    {

        // get the global lock
        PyEval_AcquireLock();
        // get a reference to the PyInterpreterState

        //Python thread references
        extern PyThreadState* mainThreadState;

        PyInterpreterState* mainInterpreterState = mainThreadState -> interp;
        // create a thread state object for this thread
        PyThreadState* myThreadState = PyThreadState_New(mainInterpreterState);
        PyThreadState_Swap(myThreadState);

        pFunc = PyDict_GetItemString(pDict, "taskRemoved");

        if (pFunc && PyCallable_Check(pFunc))
        {
            pArgs = PyTuple_New(2);

            pValue = PyInt_FromLong((int) this);
            PyTuple_SetItem(pArgs, 0, pValue);

            pValue = PyInt_FromLong((int) t);
            PyTuple_SetItem(pArgs, 1, pValue);

            pValue = PyObject_CallObject(pFunc, pArgs);
            if (pValue != NULL)
            {
                Py_DECREF(pValue);
            }
            else
            {
                qWarning("Call to taskRemoved failed");
		PyErr_Print();
            }
            Py_DECREF(pArgs);
        }
        else
        {
	  //callback not found
        }

        // swap my thread state out of the interpreter
        PyThreadState_Swap(NULL);
        // clear out any cruft from thread state object
        PyThreadState_Clear(myThreadState);
        // delete my thread state object
        PyThreadState_Delete(myThreadState);
        // release the lock
        PyEval_ReleaseLock();
    }
}

void karamba::startupAdded(Startup* t)
{
    //Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded)
    {

        // get the global lock
        PyEval_AcquireLock();
        // get a reference to the PyInterpreterState

        //Python thread references
        extern PyThreadState* mainThreadState;

        PyInterpreterState* mainInterpreterState = mainThreadState -> interp;
        // create a thread state object for this thread
        PyThreadState* myThreadState = PyThreadState_New(mainInterpreterState);
        PyThreadState_Swap(myThreadState);

        pFunc = PyDict_GetItemString(pDict, "startupAdded");

        if (pFunc && PyCallable_Check(pFunc))
        {
            pArgs = PyTuple_New(2);

            pValue = PyInt_FromLong((int) this);
            PyTuple_SetItem(pArgs, 0, pValue);

            pValue = PyInt_FromLong((int) t);
            PyTuple_SetItem(pArgs, 1, pValue);

            pValue = PyObject_CallObject(pFunc, pArgs);
            if (pValue != NULL)
            {
                Py_DECREF(pValue);
            }
            else
            {
                qWarning("Call to startupAdded failed");
		PyErr_Print();
            }
            Py_DECREF(pArgs);
        }
        else
        {
	  //callback not found
        }

        // swap my thread state out of the interpreter
        PyThreadState_Swap(NULL);
        // clear out any cruft from thread state object
        PyThreadState_Clear(myThreadState);
        // delete my thread state object
        PyThreadState_Delete(myThreadState);
        // release the lock
        PyEval_ReleaseLock();
    }
}

void karamba::startupRemoved(Startup* t)
{
    //Everything below is to call the python callback function

    if (pythonThemeExtensionLoaded)
    {

        // get the global lock
        PyEval_AcquireLock();
        // get a reference to the PyInterpreterState

        //Python thread references
        extern PyThreadState* mainThreadState;

        PyInterpreterState* mainInterpreterState = mainThreadState -> interp;
        // create a thread state object for this thread
        PyThreadState* myThreadState = PyThreadState_New(mainInterpreterState);
        PyThreadState_Swap(myThreadState);

        pFunc = PyDict_GetItemString(pDict, "startupRemoved");

        if (pFunc && PyCallable_Check(pFunc))
        {
            pArgs = PyTuple_New(2);

            pValue = PyInt_FromLong((int) this);
            PyTuple_SetItem(pArgs, 0, pValue);

            pValue = PyInt_FromLong((int) t);
            PyTuple_SetItem(pArgs, 1, pValue);

            pValue = PyObject_CallObject(pFunc, pArgs);
            if (pValue != NULL)
            {
                Py_DECREF(pValue);
            }
            else
            {
                qWarning("Call to startupRemoved failed");
		PyErr_Print();
            }
            Py_DECREF(pArgs);
        }
        else
        {
	  //cannot find callback
        }

        // swap my thread state out of the interpreter
        PyThreadState_Swap(NULL);
        // clear out any cruft from thread state object
        PyThreadState_Clear(myThreadState);
        // delete my thread state object
        PyThreadState_Delete(myThreadState);
        // release the lock
        PyEval_ReleaseLock();
    }
}

void  karamba::processExited (KProcess *proc) {
  //Nothing needed here at the moment.
  //Could add a "commandFinished" callback here if it becomes needed at some point.
}

void  karamba::receivedStdout (KProcess *proc, char *buffer, int buflen) {

  //Everything below is to call the python callback function

  if (pythonThemeExtensionLoaded) {

    // get the global lock
    PyEval_AcquireLock();
    // get a reference to the PyInterpreterState

    //Python thread references
    extern PyThreadState * mainThreadState;

    PyInterpreterState * mainInterpreterState = mainThreadState->interp;
    // create a thread state object for this thread
    PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
    PyThreadState_Swap(myThreadState);

    pFunc = PyDict_GetItemString(pDict, "commandOutput");

    if (pFunc && PyCallable_Check(pFunc)) {
      pArgs = PyTuple_New(3);

      pValue = PyInt_FromLong((int)this);
      PyTuple_SetItem(pArgs, 0, pValue);

      pValue = PyInt_FromLong((int)proc->pid());
      PyTuple_SetItem(pArgs, 1, pValue);

      pValue = PyString_FromString(buffer);
      PyTuple_SetItem(pArgs, 2, pValue);

      pValue = PyObject_CallObject(pFunc, pArgs);
      if (pValue != NULL) {
	Py_DECREF(pValue);
      }
      else {
	qWarning("Call to commandOutput failed");
	PyErr_Print();
      }
      Py_DECREF(pArgs);
      }
    else {
      //callback not found
    }


    // swap my thread state out of the interpreter
    PyThreadState_Swap(NULL);
    // clear out any cruft from thread state object
    PyThreadState_Clear(myThreadState);
    // delete my thread state object
    PyThreadState_Delete(myThreadState);
    // release the lock
    PyEval_ReleaseLock();
  }

}


//For KDE session management
void karamba::saveProperties(KConfig* config) {
  config->writeEntry("theme", themeFile);
  writeConfigData();
}

//For KDE session management
void karamba::readProperties(KConfig* config) {
  QString atheme = config->readEntry("theme","");


}

//Register types of events that can be dragged on our widget
void karamba::dragEnterEvent(QDragEnterEvent* event)
{
    event->accept(
        QTextDrag::canDecode(event)
    );
}

//Handle the drop part of a drag and drop event.
void karamba::dropEvent(QDropEvent* event)
{
    QString text;

    if ( QTextDrag::decode(event, text) ) {

      //Everything below is to call the python callback function

      if (pythonThemeExtensionLoaded) {

	// get the global lock
	PyEval_AcquireLock();
	// get a reference to the PyInterpreterState

	//Python thread references
	extern PyThreadState * mainThreadState;

	PyInterpreterState * mainInterpreterState = mainThreadState->interp;
	// create a thread state object for this thread
	PyThreadState * myThreadState = PyThreadState_New(mainInterpreterState);
	PyThreadState_Swap(myThreadState);

	pFunc = PyDict_GetItemString(pDict, "itemDropped");

	if (pFunc && PyCallable_Check(pFunc)) {
	  pArgs = PyTuple_New(2);

	  pValue = PyInt_FromLong((int)this);
	  PyTuple_SetItem(pArgs, 0, pValue);

	  pValue = PyString_FromString(text.ascii());
	  PyTuple_SetItem(pArgs, 1, pValue);

	  pValue = PyObject_CallObject(pFunc, pArgs);
	  if (pValue != NULL) {
	    Py_DECREF(pValue);
	  }
	  else {
	    qWarning("Call to itemDropped failed");
	    PyErr_Print();
	  }
	  Py_DECREF(pArgs);
	}
	else {
	  //callback failed
	}


	// swap my thread state out of the interpreter
	PyThreadState_Swap(NULL);
	// clear out any cruft from thread state object
	PyThreadState_Clear(myThreadState);
	// delete my thread state object
	PyThreadState_Delete(myThreadState);
	// release the lock
	PyEval_ReleaseLock();
      }

    }
}

void karamba::toDesktop(int id, int menuid)
{
    int i;

    desktop = id;
    for (i=0; ; i++)
    {
        int mid = toDesktopMenu->idAt(i);
        if (mid == -1)
            break;

        toDesktopMenu->setItemChecked(mid, false);
    }
    toDesktopMenu->setItemChecked(menuid, true);

    if (desktop)
        info->setDesktop( desktop);
    else
        info->setDesktop( NETWinInfo::OnAllDesktops );
}

//Eventually maybe add a theme browser here
void karamba::downloadThemes() {
  //Go to SK Themes webpage for now
  KRun::runURL( KURL( "http://netdragon.sourceforge.net/index.php?page=Themes" ), "text/html" );
}


SignalBridge::SignalBridge(QObject* parent, QString name, KActionCollection* ac) : QObject(parent, name),
   collection(ac)
{
  setName(name);
}

void SignalBridge::receive()
{
  emit enabled(name(), ((KToggleAction*)collection -> action(name())) -> isChecked());
}

DesktopChangeSlot::DesktopChangeSlot(QObject *parent, int id)
    : QObject(parent, "")
{
    desktopid = id;
}

void DesktopChangeSlot::receive()
{
    karamba *k = (karamba *)parent();

    /* XXX - check type cast */

    k->toDesktop(desktopid, menuid);
}

void DesktopChangeSlot::setMenuId(int id)
{
    menuid = id;
}

int DesktopChangeSlot::menuId()
{
    return menuid;
}

#include "karamba.moc"
